<?php
/**
 * Admin notices handler.
 *
 * @package ProductFeesToolkitForWooCommerce
 */

declare(strict_types=1);

namespace PFTFW\Infrastructure;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin notices manager for printing activation errors.
 *
 * Handles both single-site and network admin notices.
 *
 * @since 1.0.0
 */
final class Admin_Notices {
	/**
	 * Wire admin notice hooks.
	 *
	 * @return void
	 */
	public static function register(): void {
		add_action( 'admin_notices', array( self::class, 'maybe_print_activation_error' ) );
		add_action( 'network_admin_notices', array( self::class, 'maybe_print_network_activation_error' ) );
	}

	/**
	 * Print and clear single-site activation errors.
	 *
	 * @return void
	 */
	public static function maybe_print_activation_error(): void {
		$message = get_transient( 'pftfw_activation_error' );
		if ( $message ) {
			delete_transient( 'pftfw_activation_error' );
			echo '<div class="notice notice-error"><p>' . esc_html( (string) $message ) . '</p></div>';
		}
	}

	/**
	 * Print and clear network activation errors (multisite).
	 *
	 * @return void
	 */
	public static function maybe_print_network_activation_error(): void {
		if ( ! is_multisite() ) {
			return;
		}

		$message = get_site_transient( 'pftfw_network_activation_error' );
		if ( $message ) {
			delete_site_transient( 'pftfw_network_activation_error' );
			echo '<div class="notice notice-error"><p>' . esc_html( (string) $message ) . '</p></div>';
		}
	}
}
