<?php
/**
 * Activation checks and bootstrap.
 *
 * @package ProductFeesToolkitForWooCommerce
 */

declare(strict_types=1);

namespace PFTFW\Infrastructure;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Activation bootstrap and environment checks.
 *
 * Validates WordPress and WooCommerce requirements on plugin activation
 * and records admin notice messages when checks fail.
 *
 * @since 1.0.0
 */
final class Activator {
	/**
	 * Perform activation checks and set transient messages when failing.
	 *
	 * @param bool $network_wide Whether network activated.
	 * @return void
	 */
	public static function activate( $network_wide = false ): void { // phpcs:ignore WordPress.NamingConventions.ValidFunctionName.MethodNameInvalid
		$ok     = true;
		$errors = array();

		// Requirements class should be available via autoloader.
		if ( ! class_exists( '\\PFTFW\\Infrastructure\\Requirements' ) ) {
			$ok       = false;
			$errors[] = __( 'Product Fees Toolkit for WooCommerce files are missing or could not be loaded. Please reinstall the plugin.', 'product-fees-toolkit-for-woocommerce' );
		}

		if ( $ok && ! Requirements::wp_version_ok() ) {
			$errors[] = sprintf(
				/* translators: %s: Required WordPress version number. */
				__( 'Product Fees Toolkit for WooCommerce requires WordPress %s or newer.', 'product-fees-toolkit-for-woocommerce' ),
				Requirements::MIN_WP
			);
			$ok = false;
		}

		if ( $ok && ! Requirements::wc_active() ) {
			$errors[] = __( 'WooCommerce must be installed and active.', 'product-fees-toolkit-for-woocommerce' );
			$ok       = false;
		} elseif ( $ok && ! Requirements::wc_version_ok() ) {
			$errors[] = sprintf(
				/* translators: %s: Required WooCommerce version number. */
				__( 'Product Fees Toolkit for WooCommerce requires WooCommerce %s or newer.', 'product-fees-toolkit-for-woocommerce' ),
				Requirements::MIN_WC
			);
			$ok = false;
		}

		if ( ! $ok ) {
			$message = implode( ' ', $errors );
			if ( is_multisite() && $network_wide ) {
				set_site_transient( 'pftfw_network_activation_error', $message, 5 * MINUTE_IN_SECONDS );
			} else {
				set_transient( 'pftfw_activation_error', $message, 5 * MINUTE_IN_SECONDS );
			}

			deactivate_plugins( plugin_basename( PFTFW_PLUGIN_FILE ), false, (bool) $network_wide );
		}
	}
}
