<?php
/**
 * Main plugin class file.
 *
 * Handles bootstrapping and runtime hooks for Product Fees Toolkit for WooCommerce.
 *
 * @package ProductFeesToolkitForWooCommerce
 */

declare(strict_types=1);

namespace PFTFW;

use WC_Cart;
use WC_Coupon;
use WC_Product;

/**
 * Main plugin class responsible for bootstrapping and runtime hooks.
 */
final class Plugin {

	public const VERSION = '1.0';

	/**
	 * Register all hooks.
	 *
	 * @return void
	 */
	public function init(): void {

		// Fees.
		add_action( 'woocommerce_cart_calculate_fees', array( $this, 'add_fees' ), 15 );

		// Admin.
		if ( is_admin() ) {
			( new Admin\Product_Settings() )->init();
			( new Admin\Global_Settings() )->init();
		}

		// Frontend: optionally add Product Fees tab on single product page.
		if ( ! is_admin() ) {
			add_filter( 'woocommerce_product_tabs', array( $this, 'add_product_fees_tab' ) );
		}
	}

	/**
	 * Add the fees to the cart.
	 *
	 * @param WC_Cart $cart WooCommerce cart object.
	 * @return void
	 */
	public function add_fees( WC_Cart $cart ): void {
		$fees = $this->get_fees( $cart );
		if ( empty( $fees ) ) {
			return;
		}

		foreach ( $fees as $fee ) {
			$amount = (float) wc_format_decimal( (float) $fee['amount'], wc_get_price_decimals() );
			if ( 0.0 === $amount ) {
				continue;
			}
			$cart->add_fee( $fee['name'], $amount, (bool) $fee['taxable'], (string) $fee['tax_class'] );
		}
	}

	/**
	 * Register a "Product Fees" tab when the global setting is enabled and the product has fees.
	 *
	 * @param array $tabs Existing product tabs.
	 * @return array Modified tabs, potentially including the Product Fees tab.
	 */
	public function add_product_fees_tab( array $tabs ): array {
		if ( 'yes' !== (string) get_option( 'pftfw_display_fees_single', 'no' ) ) {
			return $tabs;
		}

		$pftfw_product = wc_get_product( get_the_ID() );
		if ( ! $pftfw_product instanceof WC_Product ) {
			return $tabs;
		}

		$has_fees = $this->product_contains_multiple_fee_data( (int) $pftfw_product->get_id() ) || $this->product_contains_fee_data( (int) $pftfw_product->get_id() );
		if ( ! $has_fees ) {
			return $tabs;
		}

		$tabs['pftfw_fees'] = array(
			'title'    => esc_html__( 'Product Fees', 'product-fees-toolkit-for-woocommerce' ),
			'priority' => 25,
			'callback' => array( $this, 'render_product_fees_tab' ),
		);

		return $tabs;
	}

	/**
	 * Output the Product Fees tab content as a table similar to the Additional information tab.
	 *
	 * @return void
	 */
	public function render_product_fees_tab(): void {
		$pftfw_product = wc_get_product( get_the_ID() );
		if ( ! $pftfw_product instanceof WC_Product ) {
			return;
		}

		$price    = (float) $pftfw_product->get_price();
		$fee_rows = array();

		$structured = get_post_meta( $pftfw_product->get_id(), 'pftfw_product_fees', true );
		if ( is_array( $structured ) && ! empty( $structured ) ) {
			$fee_rows = $structured;
		} else {
			$legacy_name    = (string) get_post_meta( $pftfw_product->get_id(), 'product-fee-name', true );
			$legacy_amount  = (string) get_post_meta( $pftfw_product->get_id(), 'product-fee-amount', true );
			$legacy_type    = (string) get_post_meta( $pftfw_product->get_id(), 'product-fee-type', true );
			$legacy_percent = (string) get_post_meta( $pftfw_product->get_id(), 'product-fee-percent', true );
			if ( '' !== $legacy_name && ( '' !== $legacy_amount || '' !== $legacy_percent ) ) {
				$fee_rows[] = array(
					'name'       => $legacy_name,
					'type'       => $legacy_type,
					'amount'     => $legacy_amount,
					'percent'    => $legacy_percent,
					'multiplier' => get_post_meta( $pftfw_product->get_id(), 'product-fee-multiplier', true ),
				);
			}
		}

		$items = array();
		foreach ( $fee_rows as $fee ) {
			$name = (string) ( $fee['name'] ?? '' );
			if ( '' === $name ) {
				continue;
			}
			$type        = (string) ( $fee['type'] ?? 'fixed' );
			$fixed_raw   = (string) ( $fee['amount'] ?? '' );
			$percent_raw = (string) ( $fee['percent'] ?? '' );

			$display = '';
			if ( 'fixed_plus_percent' === $type ) {
				$fixed_amt   = $this->calculate_fee_amount( $fixed_raw, $price, 'fixed' );
				$percent_amt = $this->calculate_fee_amount( $percent_raw, $price, 'percent' );
				if ( 0.0 < $price ) {
					$display = wc_price( (float) $fixed_amt + (float) $percent_amt );
				} else {
					$display = wc_price( (float) $fixed_amt );
					if ( '' !== $percent_raw ) {
						$display .= ' + ' . esc_html( rtrim( $percent_raw, '%' ) . '%' );
					}
				}
			} elseif ( 'percent' === $type ) {
				if ( 0.0 < $price ) {
					$amt     = $this->calculate_fee_amount( '' !== $percent_raw ? $percent_raw : $fixed_raw, $price, 'percent' );
					$display = wc_price( (float) $amt );
				} else {
					$raw     = '' !== $percent_raw ? $percent_raw : $fixed_raw;
					$display = esc_html( rtrim( (string) $raw, '%' ) . '%' );
				}
			} else {
				$amt     = $this->calculate_fee_amount( $fixed_raw, $price, 'fixed' );
				$display = wc_price( (float) $amt );
			}

			$items[] = array(
				'name'    => $name,
				'display' => $display,
			);
		}

		$items = apply_filters( 'pftfw_single_product_fee_items', $items, $pftfw_product );

		if ( empty( $items ) ) {
			return;
		}

		echo '<table class="woocommerce-product-attributes shop_attributes" aria-label="' . esc_attr__( 'Product Fees', 'product-fees-toolkit-for-woocommerce' ) . '">';
		echo '<colgroup><col style="width:30%"><col></colgroup>';
		echo '<tbody>';
		foreach ( $items as $it ) {
			echo '<tr class="woocommerce-product-attributes-item pftfw-fee-row">';
			echo '<th class="woocommerce-product-attributes-item__label" scope="row">' . esc_html( $it['name'] ) . '</th>';
			echo '<td class="woocommerce-product-attributes-item__value"><p>' . wp_kses_post( $it['display'] ) . '</p></td>';
			echo '</tr>';
		}
		echo '</tbody>';
		echo '</table>';
	}

	/**
	 * Render fees on the single product page when enabled in settings.
	 *
	 * @return void
	 */
	public function render_single_product_fees(): void {
		if ( 'yes' !== (string) get_option( 'pftfw_display_fees_single', 'no' ) ) {
			return;
		}

		if ( ! function_exists( 'is_product' ) || ! is_product() ) {
			return;
		}

		$pftfw_product = wc_get_product( get_the_ID() );
		if ( ! $pftfw_product instanceof WC_Product ) {
			return;
		}

		$price    = (float) $pftfw_product->get_price();
		$fee_rows = array();

		$structured = get_post_meta( $pftfw_product->get_id(), 'pftfw_product_fees', true );
		if ( is_array( $structured ) && ! empty( $structured ) ) {
			$fee_rows = $structured;
		} else {
			$legacy_name    = (string) get_post_meta( $pftfw_product->get_id(), 'product-fee-name', true );
			$legacy_amount  = (string) get_post_meta( $pftfw_product->get_id(), 'product-fee-amount', true );
			$legacy_type    = (string) get_post_meta( $pftfw_product->get_id(), 'product-fee-type', true );
			$legacy_percent = (string) get_post_meta( $pftfw_product->get_id(), 'product-fee-percent', true );
			if ( '' !== $legacy_name && ( '' !== $legacy_amount || '' !== $legacy_percent ) ) {
				$fee_rows[] = array(
					'name'       => $legacy_name,
					'type'       => $legacy_type,
					'amount'     => $legacy_amount,
					'percent'    => $legacy_percent,
					'multiplier' => get_post_meta( $pftfw_product->get_id(), 'product-fee-multiplier', true ),
				);
			}
		}

		$items = array();
		foreach ( $fee_rows as $fee ) {
			$name = (string) ( $fee['name'] ?? '' );
			if ( '' === $name ) {
				continue;
			}
			$type        = (string) ( $fee['type'] ?? 'fixed' );
			$fixed_raw   = (string) ( $fee['amount'] ?? '' );
			$percent_raw = (string) ( $fee['percent'] ?? '' );

			$display = '';
			if ( 'fixed_plus_percent' === $type ) {
				$fixed_amt   = $this->calculate_fee_amount( $fixed_raw, $price, 'fixed' );
				$percent_amt = $this->calculate_fee_amount( $percent_raw, $price, 'percent' );
				if ( 0.0 < $price ) {
					$display = wc_price( (float) $fixed_amt + (float) $percent_amt );
				} else {
					$display = wc_price( (float) $fixed_amt );
					if ( '' !== $percent_raw ) {
						$display .= ' + ' . esc_html( rtrim( $percent_raw, '%' ) . '%' );
					}
				}
			} elseif ( 'percent' === $type ) {
				if ( 0.0 < $price ) {
					$amt     = $this->calculate_fee_amount( '' !== $percent_raw ? $percent_raw : $fixed_raw, $price, 'percent' );
					$display = wc_price( (float) $amt );
				} else {
					$raw     = '' !== $percent_raw ? $percent_raw : $fixed_raw;
					$display = esc_html( rtrim( (string) $raw, '%' ) . '%' );
				}
			} else {
				$amt     = $this->calculate_fee_amount( $fixed_raw, $price, 'fixed' );
				$display = wc_price( (float) $amt );
			}

			$items[] = array(
				'name'    => $name,
				'display' => $display,
			);
		}

		$items = apply_filters( 'pftfw_single_product_fee_items', $items, $pftfw_product );

		if ( empty( $items ) ) {
			return;
		}

		echo '<div class="pftfw-product-fees">';
		echo '<h3 class="pftfw-fees__title">' . esc_html__( 'Additional fees', 'product-fees-toolkit-for-woocommerce' ) . '</h3>';
		echo '<ul class="pftfw-fees__list">';
		foreach ( $items as $it ) {
			echo '<li class="pftfw-fees__item"><span class="pftfw-fees__name">' . esc_html( $it['name'] ) . '</span>: <span class="pftfw-fees__amount">' . wp_kses_post( $it['display'] ) . '</span></li>';
		}
		echo '</ul>';
		echo '</div>';
	}

	/**
	 * Get all the fees for the cart items.
	 *
	 * @param WC_Cart $cart WooCommerce cart object.
	 * @return array<int, array{name:string,amount:float,taxable:bool,tax_class:string}> List of fee arrays for WC_Cart::add_fee.
	 */
	public function get_fees( WC_Cart $cart ): array {
		$fees          = array();
		$combine_names = ( 'combine' === get_option( 'pftfw_name_conflicts', 'combine' ) );
		$accumulator   = $combine_names ? array() : null;

		if ( $this->maybe_remove_fees_for_coupon( $cart ) ) {
			return $fees;
		}

		foreach ( $cart->get_cart() as $cart_item ) {
			$product   = $cart_item['data'];
			$item_data = array(
				'id'           => (int) $product->get_id(),
				'variation_id' => (int) ( $cart_item['variation_id'] ?? 0 ),
				'parent_id'    => (int) $product->get_parent_id(),
				'qty'          => (int) ( $cart_item['quantity'] ?? 1 ),
				'price'        => (float) $product->get_price(),
			);

			$fees_for_item = $this->get_fee_data_list( $item_data );
			// Back-compat: if no structured fees exist, fallback to legacy single fee meta.
			if ( empty( $fees_for_item ) ) {
				$single = $this->get_fee_data( $item_data );
				if ( $single ) {
					$fees_for_item = array(
						array(
							'name'       => (string) $single['name'],
							'amount'     => (float) $single['amount'],
							'multiplier' => (string) $single['multiplier'],
						),
					);
				}
			}

			if ( ! empty( $fees_for_item ) ) {
				$fee_tax_class = $this->get_fee_tax_class( $product );
				foreach ( $fees_for_item as $fee ) {
					$prepared = apply_filters(
						'pftfw_filter_fee_data',
						array(
							'name'      => (string) $fee['name'],
							'amount'    => (float) $fee['amount'],
							'taxable'   => ( '_no_tax' === $this->get_fee_tax_class( $product ) ) ? false : true,
							'tax_class' => (string) $fee_tax_class,
						),
						$item_data
					);

					if ( $combine_names ) {
						$fee_id = sanitize_key( $prepared['name'] );
						if ( ! isset( $accumulator[ $fee_id ] ) ) {
							$accumulator[ $fee_id ] = $prepared;
						} else {
							$accumulator[ $fee_id ]['amount'] += (float) $prepared['amount'];
						}
					} else {
						$fees[] = $prepared; // Keep separate entries even if names match.
					}
				}
			}
		}

		if ( $combine_names ) {
			$fees = array_values( $accumulator );
		}

		return $fees;
	}

	/**
	 * Get the legacy single-fee data from a product/variation.
	 *
	 * @param array{id:int,variation_id:int,parent_id:int,qty:int,price:float} $item Cart item data.
	 * @return array{name:string,amount:float,multiplier:string}|false Legacy fee data or false when none.
	 */
	public function get_fee_data( array $item ) {
		$fee_data = false;

		// If variation has no fee data, fallback to parent.
		if ( 0 !== $item['variation_id'] && ! $this->product_contains_fee_data( $item['id'] ) ) {
			$item['id'] = $item['parent_id'];
		}

		if ( $this->product_contains_fee_data( $item['id'] ) ) {
			$name        = (string) get_post_meta( $item['id'], 'product-fee-name', true );
			$fixed_raw   = (string) get_post_meta( $item['id'], 'product-fee-amount', true );
			$type        = (string) get_post_meta( $item['id'], 'product-fee-type', true );
			$percent_raw = (string) get_post_meta( $item['id'], 'product-fee-percent', true );
			$multiplier  = (string) get_post_meta( $item['id'], 'product-fee-multiplier', true );

			$amount         = 0.0;
			$effective_type = in_array( $type, array( 'fixed', 'percent', 'fixed_plus_percent' ), true )
				? $type
				: '';

			if ( 'fixed_plus_percent' === $effective_type ) {
				$fixed_amount   = $this->calculate_fee_amount( $fixed_raw, $item['price'], 'fixed' );
				$percent_amount = $this->calculate_fee_amount( $percent_raw, $item['price'], 'percent' );
				$amount         = (float) $fixed_amount + (float) $percent_amount;
			} elseif ( 'percent' === $effective_type ) {
				$amount = $this->calculate_fee_amount( $fixed_raw, $item['price'], 'percent' );
			} else {
				// Default/back-compat: treat as fixed unless a % is present in saved amount.
				$amount = $this->calculate_fee_amount( $fixed_raw, $item['price'], $type );
			}

			$amount = $this->maybe_multiply_by_quantity( (float) $amount, $multiplier, $item['qty'] );
			$amount = max( 0.0, (float) $amount ); // Enforce non-negative fees.

			$fee_data = array(
				'name'       => $name,
				'amount'     => (float) $amount,
				'multiplier' => $multiplier,
			);
		}

		return $fee_data;
	}

	/**
	 * Check if legacy fee meta exists and is non-empty for a product.
	 *
	 * @param int $product_id Product ID.
	 * @return bool True if fee data exists, false otherwise.
	 */
	public function product_contains_fee_data( int $product_id ): bool {
		$fee_name    = (string) get_post_meta( $product_id, 'product-fee-name', true );
		$fee_amount  = (string) get_post_meta( $product_id, 'product-fee-amount', true );
		$fee_type    = (string) get_post_meta( $product_id, 'product-fee-type', true );
		$fee_percent = (string) get_post_meta( $product_id, 'product-fee-percent', true );

		$has_fixed   = ( '' !== $fee_amount && (float) $this->normalize_amount( str_replace( '%', '', $fee_amount ) ) > 0.0 );
		$has_percent = ( '' !== $fee_percent && (float) $this->normalize_amount( str_replace( '%', '', $fee_percent ) ) > 0.0 );

		if ( 'fixed_plus_percent' === $fee_type ) {
			return ( '' !== $fee_name && ( $has_fixed || $has_percent ) );
		}

		return ( '' !== $fee_name && (
			( 'percent' === $fee_type && ( $has_percent || $has_fixed ) ) ||
			( 'fixed' === $fee_type && $has_fixed ) ||
			( '' === $fee_type && $has_fixed ) // Back-compat: legacy without explicit type.
		) );
	}

	/**
	 * Check if structured multiple-fee rows exist for a product.
	 *
	 * @param int $product_id Product ID.
	 * @return bool True if at least one structured fee row is present.
	 */
	private function product_contains_multiple_fee_data( int $product_id ): bool {
		$fees = get_post_meta( $product_id, 'pftfw_product_fees', true );
		if ( ! is_array( $fees ) || empty( $fees ) ) {
			return false;
		}
		foreach ( $fees as $fee ) {
			$name    = (string) ( $fee['name'] ?? '' );
			$amount  = (string) ( $fee['amount'] ?? '' );
			$percent = (string) ( $fee['percent'] ?? '' );
			if ( '' !== $name && ( '' !== $amount || '' !== $percent ) ) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Convert a fee amount from percentage (e.g. "10%") to a number using item price when needed.
	 * Accepts numeric strings using the store decimal separator.
	 *
	 * @param string $fee_amount Raw fee amount (may include a trailing %).
	 * @param float  $item_price Product item price used for percentage calculations.
	 * @param string $type       Fee type: 'fixed', 'percent', or empty to auto-detect.
	 * @return float Normalized numeric amount.
	 */
	public function calculate_fee_amount( string $fee_amount, float $item_price, string $type = '' ): float {
		$normalized = $this->normalize_amount( $fee_amount );

		// Back-compat: if no explicit type saved, infer percentage when a % is present in the saved value.
		$effective_type = in_array( $type, array( 'fixed', 'percent' ), true )
			? $type
			: ( false !== strpos( $fee_amount, '%' ) ? 'percent' : 'fixed' );

		if ( 'percent' === $effective_type ) {
			$percent = (float) str_replace( '%', '', $normalized );
			return ( $percent / 100 ) * $item_price;
		}

		// Fixed amount (strip any stray % just in case).
		return (float) str_replace( '%', '', $normalized );
	}

	/**
	 * Get multiple fee rows for a product/variation.
	 *
	 * @param array{id:int,variation_id:int,parent_id:int,qty:int,price:float} $item Cart item data.
	 * @return array<int, array{name:string,amount:float,multiplier:string}> Prepared fee rows.
	 */
	public function get_fee_data_list( array $item ): array {
		$rows = array();

		$target_id = $item['id'];
		if ( 0 !== $item['variation_id'] && ! $this->product_contains_multiple_fee_data( $item['id'] ) ) {
			$target_id = $item['parent_id'];
		}

		$fees = get_post_meta( $target_id, 'pftfw_product_fees', true );
		if ( ! is_array( $fees ) || empty( $fees ) ) {
			return array();
		}

		foreach ( $fees as $fee ) {
			$name        = (string) ( $fee['name'] ?? '' );
			$type        = (string) ( $fee['type'] ?? 'fixed' );
			$fixed_raw   = (string) ( $fee['amount'] ?? '' );
			$percent_raw = (string) ( $fee['percent'] ?? '' );
			$multiplier  = (string) ( ( ( $fee['multiplier'] ?? 'no' ) === 'yes' ) ? 'yes' : 'no' );
			if ( '' === $name ) {
				continue;
			}

			$amount         = 0.0;
			$effective_type = in_array( $type, array( 'fixed', 'percent', 'fixed_plus_percent' ), true ) ? $type : '';

			if ( 'fixed_plus_percent' === $effective_type ) {
				$fixed_amount   = $this->calculate_fee_amount( $fixed_raw, $item['price'], 'fixed' );
				$percent_amount = $this->calculate_fee_amount( $percent_raw, $item['price'], 'percent' );
				$amount         = (float) $fixed_amount + (float) $percent_amount;
			} elseif ( 'percent' === $effective_type ) {
				$amount = $this->calculate_fee_amount( '' !== $percent_raw ? $percent_raw : $fixed_raw, $item['price'], 'percent' );
			} else {
				$amount = $this->calculate_fee_amount( $fixed_raw, $item['price'], 'fixed' );
			}

			$amount = $this->maybe_multiply_by_quantity( (float) $amount, $multiplier, $item['qty'] );
			$amount = max( 0.0, (float) $amount );

			if ( 0 < $amount ) {
				$rows[] = array(
					'name'       => $name,
					'amount'     => (float) $amount,
					'multiplier' => $multiplier,
				);
			}
		}

		return $rows;
	}

	/**
	 * Multiply the fee by the cart item quantity if requested.
	 *
	 * @param float  $amount     Base amount.
	 * @param string $multiplier Either 'yes' or 'no' from settings.
	 * @param int    $qty        Quantity of the cart line item.
	 * @return float Adjusted amount.
	 */
	public function maybe_multiply_by_quantity( float $amount, string $multiplier, int $qty ): float {
		if ( 'yes' === $multiplier ) {
			return $qty * $amount;
		}
		return $amount;
	}

	/**
	 * Determine if a coupon in the cart should remove fees.
	 *
	 * @param WC_Cart $cart Cart object.
	 * @return bool True when a coupon indicates fees should be removed.
	 */
	public function maybe_remove_fees_for_coupon( WC_Cart $cart ): bool {
		$cart_coupons = $cart->get_coupons();
		if ( ! empty( $cart_coupons ) ) {
			foreach ( $cart_coupons as $coupon ) {
				if ( $coupon instanceof WC_Coupon && 'yes' === $coupon->get_meta( 'pftfw_coupon_remove_fees' ) ) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * Get the fee's tax class.
	 *
	 * @param WC_Product $product Product instance used to determine tax class when inheriting.
	 * @return string WC-compatible tax class or '_no_tax'.
	 */
	public function get_fee_tax_class( WC_Product $product ): string {
		$fee_tax_class = (string) get_option( 'pftfw_fee_tax_class', '_no_tax' );

		if ( ! wc_tax_enabled() ) {
			return '_no_tax';
		}

		if ( 'inherit_product_tax' === $fee_tax_class ) {
			if ( 'taxable' === $product->get_tax_status() ) {
				$fee_tax_class = (string) $product->get_tax_class();
			} else {
				$fee_tax_class = '_no_tax';
			}
		}

		return $fee_tax_class;
	}

	/**
	 * Normalize raw amounts to a dot-decimal numeric string for safe math operations.
	 * Converts store decimal separator to '.', removes thousand separators and spaces.
	 *
	 * @param string $raw Raw amount from settings/meta.
	 * @return string Normalized numeric string.
	 */
	private function normalize_amount( string $raw ): string {
		$raw = trim( (string) $raw );
		if ( '' === $raw ) {
			return $raw;
		}

		$store = wc_get_price_decimal_separator();
		$alt   = ( '.' === $store ) ? ',' : '.';

		// Normalize alternate decimal to the store decimal first (handles legacy-saved values and imports).
		$raw = str_replace( $alt, $store, $raw );
		// Remove common thousands-separator spaces.
		$raw = str_replace( array( "\xC2\xA0", ' ' ), '', $raw );

		// Finally, convert the store decimal to a dot for math operations.
		return str_replace( $store, '.', $raw );
	}
}
