<?php
/**
 * Admin: per-product Product Fees settings (simple products and variations).
 *
 * Adds a Product Fees tab to the product data panel, renders settings UI, and
 * saves both structured multi-row fee data and legacy single-field meta for
 * backward compatibility.
 *
 * @package ProductFeesToolkitForWooCommerce\Admin
 * @since 1.3.0
 */

declare(strict_types=1);

namespace PFTFW\Admin;

/**
 * Handles WooCommerce product-level fee settings (simple and variations).
 */
class Product_Settings {
	/**
	 * Register admin hooks for product and variation settings UIs.
	 *
	 * @return void
	 */
	public function init(): void {
		// Product panel.
		add_action( 'woocommerce_product_write_panel_tabs', array( $this, 'create_product_panel_tab' ) );
		add_action( 'woocommerce_product_data_panels', array( $this, 'product_settings_fields' ) );
		add_action( 'woocommerce_process_product_meta', array( $this, 'save_product_settings_fields' ) );

		// Variations.
		add_action( 'woocommerce_product_after_variable_attributes', array( $this, 'variation_settings_fields' ), 10, 3 );
		add_action( 'woocommerce_save_product_variation', array( $this, 'save_variation_settings_fields' ), 10, 2 );

		// Admin assets.
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
	}

	/**
	 * Add the Product Fees tab to the product data panel.
	 *
	 * @return void
	 */
	public function create_product_panel_tab(): void {
		// Always render the tab; visibility is managed dynamically via JS based on product type.
		echo '<li class="fees_product_tab product_fee_options"><a href="#fees_product_data"><span>' . esc_html__( 'Product Fees', 'product-fees-toolkit-for-woocommerce' ) . '</span></a></li>';
	}

	/**
	 * Output the settings panel for managing product fees.
	 *
	 * @return void
	 */
	public function product_settings_fields(): void {
		echo '<div id="fees_product_data" class="fee_panel panel woocommerce_options_panel wc-metaboxes-wrapper">';
		wp_nonce_field( 'pftfw_product_fee_meta', 'pftfw_product_fee_nonce' );

		$post_id = get_the_ID();
		$saved   = $this->get_saved_product_fees( (int) $post_id );

		// No back-compat: do not prefill from legacy single meta.

		echo '<div class="pftfw-fees-repeatable">';
			echo '<div class="pftfw-fee-blocks">';
		if ( ! empty( $saved ) ) {
			foreach ( $saved as $i => $fee ) {
				$this->render_fee_block( $i, $fee );
			}
		} else {
			$this->render_fee_block(
				0,
				array(
					'name'       => '',
					'type'       => 'fixed',
					'amount'     => '',
					'percent'    => '',
					'multiplier' => 'no',
				)
			);
		}
			echo '</div>';
			echo '<p><button type="button" class="button button-secondary pftfw-add-fee">' . esc_html__( 'Add fee', 'product-fees-toolkit-for-woocommerce' ) . '</button></p>';
			echo '<script type="text/template" id="pftfw-fee-block-template">' . wp_kses( $this->get_fee_block_template(), $this->get_allowed_template_html() ) . '</script>';
			do_action( 'pftfw_add_product_settings_group_one' );
			echo '</div>'; // .pftfw-fees-repeatable.

		echo '</div>';
	}

	/**
	 * Save the product-level settings posted from the Product Fees panel.
	 *
	 * @param int $post_id Product ID.
	 * @return void
	 */
	public function save_product_settings_fields( int $post_id ): void {
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}
		// Only access specific POST fields needed for this function.
		$nonce = isset( $_POST['pftfw_product_fee_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['pftfw_product_fee_nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'pftfw_product_fee_meta' ) ) {
			return;
		}

		$updated = false;

		// Structured multiple fees.
		$raw_fees = array();
		if ( isset( $_POST['pftfw_product_fees'] ) && is_array( $_POST['pftfw_product_fees'] ) ) {
			$raw_fees = map_deep( wp_unslash( $_POST['pftfw_product_fees'] ), 'sanitize_text_field' );
		}
		if ( is_array( $raw_fees ) ) {
			$fees = array();
			foreach ( $raw_fees as $i => $row ) {
				if ( ! is_array( $row ) ) {
					continue;
				}
				$name        = isset( $row['name'] ) ? sanitize_text_field( wp_unslash( $row['name'] ) ) : '';
				$type        = isset( $row['type'] ) ? sanitize_text_field( wp_unslash( $row['type'] ) ) : '';
				$type        = in_array( $type, array( 'fixed', 'percent', 'fixed_plus_percent' ), true ) ? $type : 'fixed';
				$percent_raw = isset( $row['percent'] ) ? sanitize_text_field( wp_unslash( $row['percent'] ) ) : '';
				$amount_raw  = isset( $row['amount'] ) ? sanitize_text_field( wp_unslash( $row['amount'] ) ) : '';
				$percent     = $this->sanitize_amount_or_percent( $percent_raw );
				$amount      = $this->sanitize_amount_or_percent( $amount_raw );
				if ( 'fixed' === $type && false === strpos( $amount, '%' ) ) {
					$amount = $this->format_fixed_amount_for_storage( $amount );
				}
				$mult = ( ! empty( $row['multiplier'] ) && 'yes' === $row['multiplier'] ) ? 'yes' : 'no';

				// Skip completely empty rows.
				if ( '' === $name && '' === $amount && '' === $percent ) {
					continue;
				}
				// Default a missing name to a unique label so multiple rows don't collide when combining.
				if ( '' === $name && ( '' !== $amount || '' !== $percent ) ) {
					/* translators: %d: fee sequence number */
					$name = sprintf( esc_html__( 'Product Fee #%d', 'product-fees-toolkit-for-woocommerce' ), (int) ( $i + 1 ) );
				}

				$fees[] = array(
					'name'       => $name,
					'type'       => $type,
					'amount'     => $amount,
					'percent'    => $percent,
					'multiplier' => $mult,
				);
			}
			update_post_meta( $post_id, 'pftfw_product_fees', $fees );
			$updated = true;
		}

		// No back-compat sync to legacy single meta.
	}

	/**
	 * Render the per-variation fee settings UI.
	 *
	 * @param int   $loop           Variation loop index.
	 * @param array $variation_data Variation data array.
	 * @param mixed $variation      Variation object (typically WP_Post/WC object with ->ID).
	 * @return void
	 */
	public function variation_settings_fields( $loop, $variation_data, $variation ): void {
		wp_nonce_field( 'pftfw_product_fee_meta', 'pftfw_product_fee_nonce' );

		$vid   = (int) $variation->ID;
		$saved = $this->get_saved_product_fees( $vid );

		echo '<div class="pftfw-fees-repeatable" data-variation-id="' . esc_attr( (string) $vid ) . '">';
			echo '<div class="pftfw-fee-blocks">';
		if ( ! empty( $saved ) ) {
			foreach ( $saved as $i => $fee ) {
				$this->render_variation_fee_block( $vid, (int) $loop, (int) $i, (array) $fee );
			}
		} else {
			$this->render_variation_fee_block(
				$vid,
				(int) $loop,
				0,
				array(
					'name'       => '',
					'type'       => 'fixed',
					'amount'     => '',
					'percent'    => '',
					'multiplier' => 'no',
				)
			);
		}
			echo '</div>';
			echo '<p><button type="button" class="button button-secondary pftfw-add-fee">' . esc_html__( 'Add fee', 'product-fees-toolkit-for-woocommerce' ) . '</button></p>';
			// Per-variation template.
			echo '<script type="text/template" id="pftfw-variation-fee-template-' . esc_attr( (string) $vid ) . '" class="pftfw-fee-template">' . wp_kses( $this->get_variation_fee_block_template( $vid, (int) $loop ), $this->get_allowed_template_html() ) . '</script>';
			do_action( 'pftfw_add_variation_settings' );
		echo '</div>';
	}

	/**
	 * Save variation fee settings for a given variation.
	 *
	 * @param int   $post_id Variation ID.
	 * @param mixed $i       Variation loop index from the form.
	 * @return void
	 */
	public function save_variation_settings_fields( int $post_id, $i ): void {
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}
		// Only access specific POST fields needed for this function.
		$nonce = isset( $_POST['pftfw_product_fee_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['pftfw_product_fee_nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'pftfw_product_fee_meta' ) ) {
			return;
		}

		$updated = false;

		// New: multiple fees per variation.
		$posted_rows = null;
		// Resolve the loop index that corresponds to this variation ID to avoid index drift.
		$loop_index        = $i;
		$variable_post_ids = array();
		if ( isset( $_POST['variable_post_id'] ) && is_array( $_POST['variable_post_id'] ) ) {
			$variable_post_ids = array_map( 'absint', wp_unslash( $_POST['variable_post_id'] ) );
		}
		if ( ! empty( $variable_post_ids ) ) {
			$found = array_search( $post_id, $variable_post_ids, true );
			if ( false !== $found ) {
				$loop_index = (int) $found;
			}
		}
		// Preferred: WooCommerce variations form posts by loop index under variable_* namespaces.
		$variable_fees         = array();
		$variation_fees_legacy = array();
		if ( isset( $_POST['variable_pftfw_fees'] ) && is_array( $_POST['variable_pftfw_fees'] ) ) {
			$variable_fees = map_deep( wp_unslash( $_POST['variable_pftfw_fees'] ), 'sanitize_text_field' );
		}
		if ( isset( $_POST['pftfw_variation_fees'] ) && is_array( $_POST['pftfw_variation_fees'] ) ) {
			$variation_fees_legacy = map_deep( wp_unslash( $_POST['pftfw_variation_fees'] ), 'sanitize_text_field' );
		}

		if ( is_array( $variable_fees ) && isset( $variable_fees[ $loop_index ] ) && is_array( $variable_fees[ $loop_index ] ) ) {
			$posted_rows = $variable_fees[ $loop_index ];
		} elseif ( is_array( $variation_fees_legacy ) && isset( $variation_fees_legacy[ $post_id ] ) && is_array( $variation_fees_legacy[ $post_id ] ) ) {
			// Back-compat: capture by variation ID.
			$posted_rows = $variation_fees_legacy[ $post_id ];
		} elseif ( is_array( $variation_fees_legacy ) && isset( $variation_fees_legacy[ $i ] ) && is_array( $variation_fees_legacy[ $i ] ) ) {
			// Back-compat: capture by loop index using the older name.
			$posted_rows = $variation_fees_legacy[ $i ];
		}

		if ( is_array( $posted_rows ) ) {
			$fees = array();
			foreach ( $posted_rows as $idx => $row ) {
				if ( ! is_array( $row ) ) {
					continue;
				}
				$name        = isset( $row['name'] ) ? sanitize_text_field( wp_unslash( $row['name'] ) ) : '';
				$type        = isset( $row['type'] ) ? sanitize_text_field( wp_unslash( $row['type'] ) ) : '';
				$type        = in_array( $type, array( 'fixed', 'percent', 'fixed_plus_percent' ), true ) ? $type : 'fixed';
				$percent_raw = isset( $row['percent'] ) ? sanitize_text_field( wp_unslash( $row['percent'] ) ) : '';
				$amount_raw  = isset( $row['amount'] ) ? sanitize_text_field( wp_unslash( $row['amount'] ) ) : '';
				$percent     = $this->sanitize_amount_or_percent( $percent_raw );
				$amount      = $this->sanitize_amount_or_percent( $amount_raw );
				if ( 'fixed' === $type && false === strpos( $amount, '%' ) ) {
					$amount = $this->format_fixed_amount_for_storage( $amount );
				}
				$mult = ( ! empty( $row['multiplier'] ) && in_array( $row['multiplier'], array( 'yes', 'on', '1' ), true ) ) ? 'yes' : 'no';

				if ( '' === $name && '' === $amount && '' === $percent ) {
					continue;
				}
				// Default a missing name to a unique label so multiple rows don't collide when combining.
				if ( '' === $name && ( '' !== $amount || '' !== $percent ) ) {
					/* translators: %d: fee sequence number */
					$name = sprintf( esc_html__( 'Product Fee #%d', 'product-fees-toolkit-for-woocommerce' ), (int) ( $idx + 1 ) );
				}

				$fees[] = array(
					'name'       => $name,
					'type'       => $type,
					'amount'     => $amount,
					'percent'    => $percent,
					'multiplier' => $mult,
				);
			}
			update_post_meta( $post_id, 'pftfw_product_fees', array_values( $fees ) );
			$updated = true;
		}

		// Legacy single fields (kept for back-compat editing existing data).
		$product_fee_name = isset( $_POST['product-fee-name'] ) && is_array( $_POST['product-fee-name'] ) && isset( $_POST['product-fee-name'][ $post_id ] ) ? sanitize_text_field( wp_unslash( $_POST['product-fee-name'][ $post_id ] ) ) : null;
		if ( null !== $product_fee_name ) {
			update_post_meta( $post_id, 'product-fee-name', $product_fee_name );
			$updated = true;
		}

		$product_fee_type_raw = isset( $_POST['product-fee-type'] ) && is_array( $_POST['product-fee-type'] ) && isset( $_POST['product-fee-type'][ $post_id ] ) ? sanitize_text_field( wp_unslash( $_POST['product-fee-type'][ $post_id ] ) ) : '';
		if ( '' !== $product_fee_type_raw ) {
			$product_fee_type = in_array( $product_fee_type_raw, array( 'fixed', 'percent', 'fixed_plus_percent' ), true ) ? $product_fee_type_raw : '';
			update_post_meta( $post_id, 'product-fee-type', $product_fee_type );
			$updated = true;
		}

		$posted_type             = in_array( $product_fee_type_raw, array( 'fixed', 'percent', 'fixed_plus_percent' ), true ) ? $product_fee_type_raw : '';
		$product_fee_percent_raw = isset( $_POST['product-fee-percent'] ) && is_array( $_POST['product-fee-percent'] ) && isset( $_POST['product-fee-percent'][ $post_id ] ) ? sanitize_text_field( wp_unslash( $_POST['product-fee-percent'][ $post_id ] ) ) : null;
		if ( null !== $product_fee_percent_raw ) {
			$percent = $this->sanitize_amount_or_percent( $product_fee_percent_raw );
			update_post_meta( $post_id, 'product-fee-percent', $percent );
			$updated = true;
		}

		$product_fee_amount_raw = isset( $_POST['product-fee-amount'] ) && is_array( $_POST['product-fee-amount'] ) && isset( $_POST['product-fee-amount'][ $post_id ] ) ? sanitize_text_field( wp_unslash( $_POST['product-fee-amount'][ $post_id ] ) ) : null;
		if ( null !== $product_fee_amount_raw ) {
			$amount         = $this->sanitize_amount_or_percent( $product_fee_amount_raw );
			$saved_type     = (string) get_post_meta( $post_id, 'product-fee-type', true );
			$effective_type = in_array( $posted_type, array( 'fixed', 'percent', 'fixed_plus_percent' ), true )
				? $posted_type
				: ( in_array( $saved_type, array( 'fixed', 'percent', 'fixed_plus_percent' ), true ) ? $saved_type : ( false !== strpos( $amount, '%' ) ? 'percent' : 'fixed' ) );

			if ( 'fixed' === $effective_type && false === strpos( $amount, '%' ) ) {
				$amount = $this->format_fixed_amount_for_storage( $amount );
			}

			if ( 'percent' === $effective_type && isset( $percent ) ) {
				$amount = $percent;
			}

			update_post_meta( $post_id, 'product-fee-amount', $amount );
			$updated = true;
		}

		if ( $updated ) {
			$multiplier = 'no';
			if ( isset( $_POST['product-fee-multiplier'] ) && is_array( $_POST['product-fee-multiplier'] ) ) {
				$multiplier_array = array_map( 'sanitize_text_field', wp_unslash( $_POST['product-fee-multiplier'] ) );
				$multiplier       = isset( $multiplier_array[ $post_id ] ) ? 'yes' : 'no';
			}
			update_post_meta( $post_id, 'product-fee-multiplier', $multiplier );
		}
	}

	/**
	 * Enqueue admin assets for the product editing screen.
	 *
	 * @return void
	 */
	public function enqueue_admin_assets(): void {
		$screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
		if ( ! $screen || 'product' !== $screen->post_type ) {
			return;
		}

		wp_enqueue_style(
			'pftfw-admin-product-settings',
			PFTFW_PLUGIN_URL . 'assets/css/admin-product-settings.css',
			array(),
			PFTFW_VERSION
		);

		wp_enqueue_script(
			'pftfw-admin-product-settings',
			PFTFW_PLUGIN_URL . 'assets/js/admin-product-settings.js',
			array( 'jquery' ),
			PFTFW_VERSION,
			true
		);

		$dec      = wc_get_price_decimal_separator();
		$decs     = (int) wc_get_price_decimals();
		$alt      = ( '.' === $dec ) ? ',' : '.';
		$step     = 1 / pow( 10, max( 0, $decs ) );
		$step_str = number_format( $step, $decs, '.', '' );

		wp_localize_script(
			'pftfw-admin-product-settings',
			'pftfwFeeVars',
			array(
				'dec'          => $dec,
				'alt'          => $alt,
				'decs'         => $decs,
				'step'         => $step_str,
				'allowedTypes' => $this->get_allowed_product_types(),
			)
		);
	}

	/**
	 * Get product types for which the Product Fees tab should be available.
	 *
	 * @return array<int,string>
	 */
	private function get_allowed_product_types(): array {
		$types = array( 'simple' );
		/**
		 * Filter the product types that should show the Product Fees tab.
		 *
		 * @param array<int,string> $types Default ['simple'].
		 */
		return (array) apply_filters( 'pftfw_product_types_with_tab', $types );
	}

	/**
	 * Allowed HTML tags and attributes for admin fee templates.
	 *
	 * @return array<string,array<string,bool>>
	 */
	private function get_allowed_template_html(): array {
		return array(
			'div'    => array(
				'class'      => true,
				'data-index' => true,
			),
			'h3'     => array(
				'class' => true,
			),
			'span'   => array(
				'class'    => true,
				'data-tip' => true,
			),
			'button' => array(
				'type'       => true,
				'class'      => true,
				'aria-label' => true,
			),
			'p'      => array(
				'class' => true,
			),
			'label'  => array(
				'for'   => true,
				'class' => true,
			),
			'input'  => array(
				'type'        => true,
				'class'       => true,
				'id'          => true,
				'name'        => true,
				'value'       => true,
				'placeholder' => true,
				'inputmode'   => true,
				'pattern'     => true,
				'checked'     => true,
			),
			'select' => array(
				'id'    => true,
				'name'  => true,
				'class' => true,
			),
			'option' => array(
				'value' => true,
			),
		);
	}

	/**
	 * Sanitize a numeric or percentage string for storage.
	 *
	 * @param string $input Raw user input.
	 * @return string Sanitized value preserving decimal separator and optional %.
	 */
	private function sanitize_amount_or_percent( string $input ): string {
		$input = trim( wp_strip_all_tags( $input ) );
		// Allow numbers, both decimal separators ('.' and ','), and optional percent sign. Negatives are not allowed.
		$sanitized = preg_replace( '/[^0-9\.,%]/', '', $input );

		$store_decimal = wc_get_price_decimal_separator();
		$alt_decimal   = ( '.' === $store_decimal ) ? ',' : '.';

		// Normalize all alt separators to the store decimal.
		$sanitized = str_replace( $alt_decimal, $store_decimal, $sanitized );

		// Collapse multiple decimal separators (keep first occurrence only).
		if ( substr_count( $sanitized, $store_decimal ) > 1 ) {
			$parts     = explode( $store_decimal, $sanitized );
			$sanitized = array_shift( $parts ) . $store_decimal . implode( '', $parts );
		}

		return $sanitized;
	}

	/**
	 * Normalize fixed amount for storage using the store decimal separator and configured precision.
	 *
	 * @param string $amount Raw amount.
	 * @return string Normalized fixed amount.
	 */
	private function format_fixed_amount_for_storage( string $amount ): string {
		$sep      = wc_get_price_decimal_separator();
		$decimals = (int) wc_get_price_decimals();

		$amount = trim( $amount );
		if ( '' === $amount ) {
			return $amount;
		}

		// Do not alter if explicitly a percentage string.
		if ( false !== strpos( $amount, '%' ) ) {
			return $amount;
		}

		// Disallow negative values: strip any leading '-' and treat as positive.
		if ( 0 === strpos( $amount, '-' ) ) {
			$amount = ltrim( $amount, '-' );
		}

		if ( '' === $amount ) {
			return '';
		}

		if ( $decimals <= 0 ) {
			// No decimals configured: strip any existing separator and fraction.
			if ( false !== strpos( $amount, $sep ) ) {
				list($int) = explode( $sep, $amount, 2 );
				$amount    = $int;
			}
			return $amount;
		}

		if ( false === strpos( $amount, $sep ) ) {
			$amount = $amount . $sep . str_repeat( '0', $decimals );
		} else {
			list($int, $frac) = array_pad( explode( $sep, $amount, 2 ), 2, '' );
			if ( strlen( $frac ) < $decimals ) {
				$frac = str_pad( $frac, $decimals, '0' );
			} elseif ( strlen( $frac ) > $decimals ) {
				$frac = substr( $frac, 0, $decimals );
			}
			$amount = $int . $sep . $frac;
		}

		return $amount;
	}

	/**
	 * Get saved structured product fees for a product/variation.
	 *
	 * @param int $post_id Product or variation ID.
	 * @return array<int,array<string,string>> Saved rows.
	 */
	private function get_saved_product_fees( int $post_id ): array {
		$fees = get_post_meta( $post_id, 'pftfw_product_fees', true );
		return is_array( $fees ) ? $fees : array();
	}

	/**
	 * Render a single product fee block row.
	 *
	 * @param int   $index Row index.
	 * @param array $fee   Fee row data.
	 * @return void
	 */
	private function render_fee_block( int $index, array $fee ): void {
		$name       = esc_attr( $fee['name'] ?? '' );
		$type       = esc_attr( $fee['type'] ?? 'fixed' );
		$amount     = esc_attr( $fee['amount'] ?? '' );
		$percent    = esc_attr( $fee['percent'] ?? '' );
		$multiplier = ( 'yes' === ( $fee['multiplier'] ?? 'no' ) ) ? 'yes' : 'no';

		echo '<div class="pftfw-fee-block wc-metabox" data-index="' . esc_attr( (string) $index ) . '">';
			// Header with remove button.
			echo '<h3 class="pftfw-fee-block__title"><span class="pftfw-fee-label">' .
		/* translators: %d: fee sequence number */
		sprintf( esc_html__( 'Product fee #%d', 'product-fees-toolkit-for-woocommerce' ), (int) ( $index + 1 ) ) .
		'</span> <button type="button" class="button-link delete pftfw-remove-fee" aria-label="' . esc_attr__( 'Remove fee', 'product-fees-toolkit-for-woocommerce' ) . '">' . esc_html__( 'Remove', 'product-fees-toolkit-for-woocommerce' ) . '</button></h3>';

				echo '<div class="options_group">';

				// Fee Name.
				woocommerce_wp_text_input(
					array(
						'id'          => 'pftfw_product_fees_' . esc_attr( (string) $index ) . '_name',
						'name'        => 'pftfw_product_fees[' . esc_attr( (string) $index ) . '][name]',
						'label'       => esc_html__( 'Fee Name', 'product-fees-toolkit-for-woocommerce' ),
						'placeholder' => esc_html__( 'Product Fee', 'product-fees-toolkit-for-woocommerce' ),
						'value'       => $name,
						'class'       => 'short',
					)
				);

				// Fee Type.
				woocommerce_wp_select(
					array(
						'id'            => 'pftfw_product_fees_' . esc_attr( (string) $index ) . '_type',
						'name'          => 'pftfw_product_fees[' . esc_attr( (string) $index ) . '][type]',
						'label'         => esc_html__( 'Fee Type', 'product-fees-toolkit-for-woocommerce' ),
						'options'       => array(
							'fixed'              => esc_html__( 'Fixed value', 'product-fees-toolkit-for-woocommerce' ),
							'percent'            => esc_html__( 'Percentage', 'product-fees-toolkit-for-woocommerce' ),
							'fixed_plus_percent' => esc_html__( 'Fixed + Percentage', 'product-fees-toolkit-for-woocommerce' ),
						),
						'value'         => $type,
						'wrapper_class' => 'pftfw-fee-type',
						'class'         => 'pftfw-fee-type-select',
						'desc_tip'      => 'true',
						/* translators: %s: store decimal separator */
						'description'   => sprintf( esc_html__( 'Choose how the fee should be calculated. Fixed uses your store decimal separator (%s).', 'product-fees-toolkit-for-woocommerce' ), wc_get_price_decimal_separator() ),
					)
				);

				// Fixed Amount.
				woocommerce_wp_text_input(
					array(
						'id'                => 'pftfw_product_fees_' . esc_attr( (string) $index ) . '_amount',
						'name'              => 'pftfw_product_fees[' . esc_attr( (string) $index ) . '][amount]',
						/* translators: %s: currency symbol */
						'label'             => sprintf( esc_html__( 'Fixed Amount (%s)', 'product-fees-toolkit-for-woocommerce' ), get_woocommerce_currency_symbol() ),
						'placeholder'       => esc_html__( 'e.g. 3.50', 'product-fees-toolkit-for-woocommerce' ),
						'value'             => $amount,
						'class'             => 'short pftfw-fee-amount',
						'wrapper_class'     => 'pftfw-field-fixed',
						'custom_attributes' => array(
							'inputmode' => 'decimal',
							'pattern'   => '[0-9.,%]*',
						),
						'desc_tip'          => 'true',
						/* translators: 1: store decimal separator */
						'description'       => sprintf( esc_html__( 'Enter a decimal using your store decimal separator (%1$s). Used when Fee Type is Fixed or Fixed + Percentage.', 'product-fees-toolkit-for-woocommerce' ), wc_get_price_decimal_separator() ),
					)
				);

				// Percent Amount.
				woocommerce_wp_text_input(
					array(
						'id'                => 'pftfw_product_fees_' . esc_attr( (string) $index ) . '_percent',
						'name'              => 'pftfw_product_fees[' . esc_attr( (string) $index ) . '][percent]',
						'label'             => esc_html__( 'Percentage (%)', 'product-fees-toolkit-for-woocommerce' ),
						'placeholder'       => esc_html__( 'e.g. 10', 'product-fees-toolkit-for-woocommerce' ),
						'value'             => $percent,
						'class'             => 'short pftfw-fee-amount',
						'wrapper_class'     => 'pftfw-field-percent',
						'custom_attributes' => array(
							'inputmode' => 'decimal',
							'pattern'   => '[0-9.,%]*',
						),
						'desc_tip'          => 'true',
						'description'       => esc_html__( 'Enter a number like 10 for 10%. Used when Fee Type is Percentage or Fixed + Percentage.', 'product-fees-toolkit-for-woocommerce' ),
					)
				);

				// Multiply by quantity.
				woocommerce_wp_checkbox(
					array(
						'id'            => 'pftfw_product_fees_' . esc_attr( (string) $index ) . '_multiplier',
						'name'          => 'pftfw_product_fees[' . esc_attr( (string) $index ) . '][multiplier]',
						'label'         => esc_html__( 'Multiply Fee by Quantity', 'product-fees-toolkit-for-woocommerce' ),
						'value'         => $multiplier,
						'wrapper_class' => 'product-fee-multiplier',
						'desc_tip'      => 'true',
						'description'   => esc_html__( 'Multiply the fee by the quantity of this product that is added to the cart.', 'product-fees-toolkit-for-woocommerce' ),
					)
				);

				echo '</div>';

			echo '</div>';
	}

	/**
	 * Get the HTML template for a new fee row on the product panel.
	 *
	 * @return string Template HTML.
	 */
	private function get_fee_block_template(): string {
		// Use __INDEX__ placeholder to be replaced in JS.
		ob_start();
		?>
		<div class="pftfw-fee-block wc-metabox" data-index="__INDEX__">
			<h3 class="pftfw-fee-block__title"><span class="pftfw-fee-label"><?php echo esc_html__( 'Product fee #1', 'product-fees-toolkit-for-woocommerce' ); ?></span> <button type="button" class="button-link delete pftfw-remove-fee" aria-label="<?php echo esc_attr__( 'Remove fee', 'product-fees-toolkit-for-woocommerce' ); ?>"><?php echo esc_html__( 'Remove', 'product-fees-toolkit-for-woocommerce' ); ?></button></h3>
			<div class="options_group">
				<?php
				woocommerce_wp_text_input(
					array(
						'id'          => 'pftfw_product_fees___INDEX___name',
						'name'        => 'pftfw_product_fees[__INDEX__][name]',
						'label'       => esc_html__( 'Fee Name', 'product-fees-toolkit-for-woocommerce' ),
						'placeholder' => esc_html__( 'Product Fee', 'product-fees-toolkit-for-woocommerce' ),
						'value'       => '',
						'class'       => 'short',
					)
				);

				woocommerce_wp_select(
					array(
						'id'            => 'pftfw_product_fees___INDEX___type',
						'name'          => 'pftfw_product_fees[__INDEX__][type]',
						'label'         => esc_html__( 'Fee Type', 'product-fees-toolkit-for-woocommerce' ),
						'options'       => array(
							'fixed'              => esc_html__( 'Fixed value', 'product-fees-toolkit-for-woocommerce' ),
							'percent'            => esc_html__( 'Percentage', 'product-fees-toolkit-for-woocommerce' ),
							'fixed_plus_percent' => esc_html__( 'Fixed + Percentage', 'product-fees-toolkit-for-woocommerce' ),
						),
						'value'         => 'fixed',
						'wrapper_class' => 'pftfw-fee-type',
						'class'         => 'pftfw-fee-type-select',
						'desc_tip'      => 'true',
						/* translators: %s: store decimal separator */
						'description'   => sprintf( esc_html__( 'Choose how the fee should be calculated. Fixed uses your store decimal separator (%s).', 'product-fees-toolkit-for-woocommerce' ), wc_get_price_decimal_separator() ),
					)
				);

				woocommerce_wp_text_input(
					array(
						'id'                => 'pftfw_product_fees___INDEX___amount',
						'name'              => 'pftfw_product_fees[__INDEX__][amount]',
						/* translators: %s: currency symbol */
						'label'             => sprintf( esc_html__( 'Fixed Amount (%s)', 'product-fees-toolkit-for-woocommerce' ), get_woocommerce_currency_symbol() ),
						'placeholder'       => esc_html__( 'e.g. 3.50', 'product-fees-toolkit-for-woocommerce' ),
						'value'             => '',
						'class'             => 'short pftfw-fee-amount',
						'wrapper_class'     => 'pftfw-field-fixed',
						'custom_attributes' => array(
							'inputmode' => 'decimal',
							'pattern'   => '[0-9.,%]*',
						),
						'desc_tip'          => 'true',
						/* translators: 1: store decimal separator */
						'description'       => sprintf( esc_html__( 'Enter a decimal using your store decimal separator (%1$s). Used when Fee Type is Fixed or Fixed + Percentage.', 'product-fees-toolkit-for-woocommerce' ), wc_get_price_decimal_separator() ),
					)
				);

				woocommerce_wp_text_input(
					array(
						'id'                => 'pftfw_product_fees___INDEX___percent',
						'name'              => 'pftfw_product_fees[__INDEX__][percent]',
						'label'             => esc_html__( 'Percentage (%)', 'product-fees-toolkit-for-woocommerce' ),
						'placeholder'       => esc_html__( 'e.g. 10', 'product-fees-toolkit-for-woocommerce' ),
						'value'             => '',
						'class'             => 'short pftfw-fee-amount',
						'wrapper_class'     => 'pftfw-field-percent',
						'custom_attributes' => array(
							'inputmode' => 'decimal',
							'pattern'   => '[0-9.,%]*',
						),
						'desc_tip'          => 'true',
						'description'       => esc_html__( 'Enter a number like 10 for 10%. Used when Fee Type is Percentage or Fixed + Percentage.', 'product-fees-toolkit-for-woocommerce' ),
					)
				);

				woocommerce_wp_checkbox(
					array(
						'id'            => 'pftfw_product_fees___INDEX___multiplier',
						'name'          => 'pftfw_product_fees[__INDEX__][multiplier]',
						'label'         => esc_html__( 'Multiply Fee by Quantity', 'product-fees-toolkit-for-woocommerce' ),
						'value'         => 'no',
						'wrapper_class' => 'product-fee-multiplier',
						'desc_tip'      => 'true',
						'description'   => esc_html__( 'Multiply the fee by the quantity of this product that is added to the cart.', 'product-fees-toolkit-for-woocommerce' ),
					)
				);
				?>
			</div>
		</div>
		<?php
		return (string) ob_get_clean();
	}

	/**
	 * Render a single variation fee row.
	 *
	 * @param int   $variation_id Variation ID.
	 * @param int   $loop_index   Variation loop index.
	 * @param int   $index        Row index.
	 * @param array $fee          Fee row data.
	 * @return void
	 */
	private function render_variation_fee_block( int $variation_id, int $loop_index, int $index, array $fee ): void {
		$name       = esc_attr( $fee['name'] ?? '' );
		$type       = esc_attr( $fee['type'] ?? 'fixed' );
		$amount     = esc_attr( $fee['amount'] ?? '' );
		$percent    = esc_attr( $fee['percent'] ?? '' );
		$multiplier = ( 'yes' === ( $fee['multiplier'] ?? 'no' ) ) ? 'yes' : 'no';

		echo '<div class="pftfw-fee-block wc-metabox" data-index="' . esc_attr( (string) $index ) . '">';
			// Header with remove button.
			echo '<h3 class="pftfw-fee-block__title"><span class="pftfw-fee-label">' .
		/* translators: %d: fee sequence number */
		sprintf( esc_html__( 'Product fee #%d', 'product-fees-toolkit-for-woocommerce' ), (int) ( $index + 1 ) ) .
		'</span> <button type="button" class="button-link delete pftfw-remove-fee" aria-label="' . esc_attr__( 'Remove fee', 'product-fees-toolkit-for-woocommerce' ) . '">' . esc_html__( 'Remove', 'product-fees-toolkit-for-woocommerce' ) . '</button></h3>';

				echo '<div class="options_group">';

				// Fee Name (1/2).
				woocommerce_wp_text_input(
					array(
						'id'            => 'pftfw_variation_fees_' . esc_attr( (string) $variation_id ) . '_' . esc_attr( (string) $index ) . '_name',
						'name'          => 'variable_pftfw_fees[' . esc_attr( (string) $loop_index ) . '][' . esc_attr( (string) $index ) . '][name]',
						'label'         => esc_html__( 'Fee Name', 'product-fees-toolkit-for-woocommerce' ),
						'placeholder'   => esc_html__( 'Product Fee', 'product-fees-toolkit-for-woocommerce' ),
						'value'         => $name,
						'class'         => 'short',
						'wrapper_class' => 'form-row form-row-first',
					)
				);

				// Fee Type (1/2).
				woocommerce_wp_select(
					array(
						'id'            => 'pftfw_variation_fees_' . esc_attr( (string) $variation_id ) . '_' . esc_attr( (string) $index ) . '_type',
						'name'          => 'variable_pftfw_fees[' . esc_attr( (string) $loop_index ) . '][' . esc_attr( (string) $index ) . '][type]',
						'label'         => esc_html__( 'Fee Type', 'product-fees-toolkit-for-woocommerce' ),
						'options'       => array(
							'fixed'              => esc_html__( 'Fixed value', 'product-fees-toolkit-for-woocommerce' ),
							'percent'            => esc_html__( 'Percentage', 'product-fees-toolkit-for-woocommerce' ),
							'fixed_plus_percent' => esc_html__( 'Fixed + Percentage', 'product-fees-toolkit-for-woocommerce' ),
						),
						'value'         => $type,
						'wrapper_class' => 'form-row form-row-last pftfw-fee-type',
						'class'         => 'pftfw-fee-type-select',
						'desc_tip'      => 'true',
						/* translators: %s: store decimal separator */
						'description'   => sprintf( esc_html__( 'Choose how the fee should be calculated. Fixed uses your store decimal separator (%s).', 'product-fees-toolkit-for-woocommerce' ), wc_get_price_decimal_separator() ),
					)
				);

				// Fixed Amount (1/2).
				woocommerce_wp_text_input(
					array(
						'id'                => 'pftfw_variation_fees_' . esc_attr( (string) $variation_id ) . '_' . esc_attr( (string) $index ) . '_amount',
						'name'              => 'variable_pftfw_fees[' . esc_attr( (string) $loop_index ) . '][' . esc_attr( (string) $index ) . '][amount]',
						/* translators: %s: currency symbol */
						'label'             => sprintf( esc_html__( 'Fixed Amount (%s)', 'product-fees-toolkit-for-woocommerce' ), get_woocommerce_currency_symbol() ),
						'placeholder'       => esc_html__( 'e.g. 3.50', 'product-fees-toolkit-for-woocommerce' ),
						'value'             => $amount,
						'class'             => 'short pftfw-fee-amount',
						'wrapper_class'     => 'form-row form-row-first pftfw-field-fixed',
						'custom_attributes' => array(
							'inputmode' => 'decimal',
							'pattern'   => '[0-9.,%]*',
						),
						'desc_tip'          => 'true',
						/* translators: 1: store decimal separator */
						'description'       => sprintf( esc_html__( 'Enter a decimal using your store decimal separator (%1$s). Used when Fee Type is Fixed or Fixed + Percentage.', 'product-fees-toolkit-for-woocommerce' ), wc_get_price_decimal_separator() ),
					)
				);

				// Percent Amount (1/2).
				woocommerce_wp_text_input(
					array(
						'id'                => 'pftfw_variation_fees_' . esc_attr( (string) $variation_id ) . '_' . esc_attr( (string) $index ) . '_percent',
						'name'              => 'variable_pftfw_fees[' . esc_attr( (string) $loop_index ) . '][' . esc_attr( (string) $index ) . '][percent]',
						'label'             => esc_html__( 'Percentage (%)', 'product-fees-toolkit-for-woocommerce' ),
						'placeholder'       => esc_html__( 'e.g. 10', 'product-fees-toolkit-for-woocommerce' ),
						'value'             => $percent,
						'class'             => 'short pftfw-fee-amount',
						'wrapper_class'     => 'form-row form-row-last pftfw-field-percent',
						'custom_attributes' => array(
							'inputmode' => 'decimal',
							'pattern'   => '[0-9.,%]*',
						),
						'desc_tip'          => 'true',
						'description'       => esc_html__( 'Enter a number like 10 for 10%. Used when Fee Type is Percentage or Fixed + Percentage.', 'product-fees-toolkit-for-woocommerce' ),
					)
				);

				// Multiply by quantity.
				woocommerce_wp_checkbox(
					array(
						'id'            => 'pftfw_variation_fees_' . esc_attr( (string) $variation_id ) . '_' . esc_attr( (string) $index ) . '_multiplier',
						'name'          => 'variable_pftfw_fees[' . esc_attr( (string) $loop_index ) . '][' . esc_attr( (string) $index ) . '][multiplier]',
						'label'         => esc_html__( 'Multiply Fee by Quantity', 'product-fees-toolkit-for-woocommerce' ),
						'value'         => $multiplier,
						'cbvalue'       => 'yes',
						'wrapper_class' => 'product-fee-multiplier',
						'desc_tip'      => 'true',
						'description'   => esc_html__( 'Multiply the fee by the quantity of this product that is added to the cart.', 'product-fees-toolkit-for-woocommerce' ),
					)
				);

				echo '</div>';

			echo '</div>';
	}

	/**
	 * Get the HTML template for a new variation fee row.
	 *
	 * @param int $variation_id Variation ID.
	 * @param int $loop_index   Variation loop index.
	 * @return string Template HTML.
	 */
	private function get_variation_fee_block_template( int $variation_id, int $loop_index ): string {
		// IMPORTANT: Output raw HTML here (no woocommerce_wp_* helpers) so the template contains valid fields.
		$currency   = get_woocommerce_currency_symbol();
		$dec_sep    = wc_get_price_decimal_separator();
		$title      = esc_html__( 'Product fee #1', 'product-fees-toolkit-for-woocommerce' );
		$label_fee  = esc_html__( 'Fee Name', 'product-fees-toolkit-for-woocommerce' );
		$ph_fee     = esc_html__( 'Product Fee', 'product-fees-toolkit-for-woocommerce' );
		$label_type = esc_html__( 'Fee Type', 'product-fees-toolkit-for-woocommerce' );
		/* translators: %s: currency symbol */
		$label_fixed = sprintf( esc_html__( 'Fixed Amount (%s)', 'product-fees-toolkit-for-woocommerce' ), $currency );
		$label_pct   = esc_html__( 'Percentage (%)', 'product-fees-toolkit-for-woocommerce' );
		$ph_fixed    = esc_html__( 'e.g. 3.50', 'product-fees-toolkit-for-woocommerce' );
		$ph_pct      = esc_html__( 'e.g. 10', 'product-fees-toolkit-for-woocommerce' );
		/* translators: %s: store decimal separator */
		$desc_type = sprintf( esc_html__( 'Choose how the fee should be calculated. Fixed uses your store decimal separator (%s).', 'product-fees-toolkit-for-woocommerce' ), $dec_sep );
		/* translators: 1: store decimal separator */
		$desc_fixed = sprintf( esc_html__( 'Enter a decimal using your store decimal separator (%1$s). Used when Fee Type is Fixed or Fixed + Percentage.', 'product-fees-toolkit-for-woocommerce' ), $dec_sep );
		$desc_pct   = esc_html__( 'Enter a number like 10 for 10%. Used when Fee Type is Percentage or Fixed + Percentage.', 'product-fees-toolkit-for-woocommerce' );
		$label_mult = esc_html__( 'Multiply Fee by Quantity', 'product-fees-toolkit-for-woocommerce' );
		$desc_mult  = esc_html__( 'Multiply the fee by the quantity of this product that is added to the cart.', 'product-fees-toolkit-for-woocommerce' );

		$vid_attr  = esc_attr( (string) $variation_id );
		$loop_attr = esc_attr( (string) $loop_index );

		return '
		<div class="pftfw-fee-block wc-metabox" data-index="__INDEX__">
			<h3 class="pftfw-fee-block__title"><span class="pftfw-fee-label">' . $title . '</span> <button type="button" class="button-link delete pftfw-remove-fee" aria-label="' . esc_attr__( 'Remove fee', 'product-fees-toolkit-for-woocommerce' ) . '">' . esc_html__( 'Remove', 'product-fees-toolkit-for-woocommerce' ) . '</button></h3>
			<div class="options_group">
				<p class="form-field form-row form-row-first">
					<label for="pftfw_variation_fees_' . $vid_attr . '___INDEX___name">' . $label_fee . '</label>
					<input type="text" class="short" id="pftfw_variation_fees_' . $vid_attr . '___INDEX___name" name="variable_pftfw_fees[' . $loop_attr . '][__INDEX__][name]" value="" placeholder="' . $ph_fee . '" />
				</p>
				<p class="form-field form-row form-row-last pftfw-fee-type">
					<label for="pftfw_variation_fees_' . $vid_attr . '___INDEX___type">' . $label_type . ' <span class="woocommerce-help-tip" data-tip="' . esc_attr( $desc_type ) . '"></span></label>
					<select class="pftfw-fee-type-select" id="pftfw_variation_fees_' . $vid_attr . '___INDEX___type" name="variable_pftfw_fees[' . $loop_attr . '][__INDEX__][type]">
						<option value="fixed">' . esc_html__( 'Fixed value', 'product-fees-toolkit-for-woocommerce' ) . '</option>
						<option value="percent">' . esc_html__( 'Percentage', 'product-fees-toolkit-for-woocommerce' ) . '</option>
						<option value="fixed_plus_percent">' . esc_html__( 'Fixed + Percentage', 'product-fees-toolkit-for-woocommerce' ) . '</option>
					</select>
				</p>
				<p class="form-field form-row form-row-first pftfw-field-fixed">
					<label for="pftfw_variation_fees_' . $vid_attr . '___INDEX___amount">' . $label_fixed . ' <span class="woocommerce-help-tip" data-tip="' . esc_attr( $desc_fixed ) . '"></span></label>
					<input type="text" class="short pftfw-fee-amount" id="pftfw_variation_fees_' . $vid_attr . '___INDEX___amount" name="variable_pftfw_fees[' . $loop_attr . '][__INDEX__][amount]" value="" placeholder="' . $ph_fixed . '" inputmode="decimal" pattern="[0-9.,%]*" />
				</p>
				<p class="form-field form-row form-row-last pftfw-field-percent">
					<label for="pftfw_variation_fees_' . $vid_attr . '___INDEX___percent">' . $label_pct . ' <span class="woocommerce-help-tip" data-tip="' . esc_attr( $desc_pct ) . '"></span></label>
					<input type="text" class="short pftfw-fee-amount" id="pftfw_variation_fees_' . $vid_attr . '___INDEX___percent" name="variable_pftfw_fees[' . $loop_attr . '][__INDEX__][percent]" value="" placeholder="' . $ph_pct . '" inputmode="decimal" pattern="[0-9.,%]*" />
				</p>
				<p class="form-field product-fee-multiplier">
					<label>
						<input type="checkbox" id="pftfw_variation_fees_' . $vid_attr . '___INDEX___multiplier" name="variable_pftfw_fees[' . $loop_attr . '][__INDEX__][multiplier]" value="yes" /> ' . $label_mult . ' <span class="woocommerce-help-tip" data-tip="' . esc_attr( $desc_mult ) . '"></span>
					</label>
				</p>
			</div>
		</div>';
	}
}
