<?php
/**
 * Admin global settings: adds Fees settings section, coupon option and
 * CSV import/export integrations for Product Fees Toolkit for WooCommerce.
 *
 * @package ProductFeesToolkitForWooCommerce\Admin
 * @since 1.0
 */

declare(strict_types=1);

namespace PFTFW\Admin;

use WC_Tax;

/**
 * Global settings and integrations (coupon option, CSV import/export).
 *
 * @since 1.0
 */
class Global_Settings {
	/**
	 * Register admin settings section and integration hooks.
	 *
	 * @since 1.0
	 * @return void
	 */
	public function init(): void {
		// WooCommerce > Settings > Products > Fees section and fields.
		add_filter( 'woocommerce_get_sections_products', array( $this, 'add_product_section' ) );
		add_filter( 'woocommerce_get_settings_products', array( $this, 'product_settings_output' ), 10, 2 );

		// Coupon option.
		add_action( 'woocommerce_coupon_options', array( $this, 'add_coupon_setting' ), 10, 2 );
		add_action( 'woocommerce_coupon_options_save', array( $this, 'save_coupon_setting' ), 10, 2 );

		// CSV Import/Export Support.
		add_filter( 'woocommerce_csv_product_import_mapping_options', array( $this, 'add_columns_to_importer_exporter' ) );
		add_filter( 'woocommerce_csv_product_import_mapping_default_columns', array( $this, 'add_column_to_mapping_screen' ) );
		add_filter( 'woocommerce_product_import_pre_insert_product_object', array( $this, 'process_import' ), 10, 2 );

		add_filter( 'woocommerce_product_export_column_names', array( $this, 'add_columns_to_importer_exporter' ) );
		add_filter( 'woocommerce_product_export_product_default_columns', array( $this, 'add_columns_to_importer_exporter' ) );
		add_filter( 'woocommerce_product_export_product_column_pftfw_fee_name', array( $this, 'add_export_data_fee_name' ), 10, 2 );
		add_filter( 'woocommerce_product_export_product_column_pftfw_fee_amount', array( $this, 'add_export_data_fee_amount' ), 10, 2 );
		add_filter( 'woocommerce_product_export_product_column_pftfw_fee_multiplier', array( $this, 'add_export_data_fee_multiplier' ), 10, 2 );
		add_filter( 'woocommerce_product_export_product_column_pftfw_fee_type', array( $this, 'add_export_data_fee_type' ), 10, 2 );
		add_filter( 'woocommerce_product_export_product_column_pftfw_fee_percent', array( $this, 'add_export_data_fee_percent' ), 10, 2 );
	}

	// Product Fees Settings.
	// -------------------------------------------------------.

	/**
	 * Add the "Product Fees" section to WooCommerce > Settings > Products.
	 *
	 * @since 1.0
	 * @param array $sections Existing product sections.
	 * @return array Modified sections including the Product Fees section.
	 */
	public function add_product_section( array $sections ): array {
		$sections['fees'] = esc_html__( 'Product Fees', 'product-fees-toolkit-for-woocommerce' );
		return $sections;
	}

	/**
	 * Output the settings for our section when selected.
	 *
	 * @since 1.0
	 * @param array  $settings        Existing settings array.
	 * @param string $current_section Current section slug.
	 * @return array Settings for the current section.
	 */
	public function product_settings_output( array $settings, string $current_section ): array {
		if ( 'fees' === $current_section ) {
			$settings = $this->settings_fields();
		}
		return $settings;
	}

	/**
	 * Build the settings fields for the Product Fees section.
	 *
	 * @since 1.0
	 * @return array Settings definition array.
	 */
	public function settings_fields(): array {
		$settings = apply_filters(
			'pftfw_global_product_settings',
			array(
				array(
					'title' => esc_html__( 'Product Fees', 'product-fees-toolkit-for-woocommerce' ),
					'type'  => 'title',
					'desc'  => '',
					'id'    => 'product_fees_options',
				),
				array(
					'title'    => esc_html__( 'Fee Tax Class:', 'product-fees-toolkit-for-woocommerce' ),
					'desc'     => esc_html__( 'Optionally control which tax class gets applied to fees, or leave it so no taxes are applied.', 'product-fees-toolkit-for-woocommerce' ),
					'id'       => 'pftfw_fee_tax_class',
					'css'      => 'min-width:150px;',
					'default'  => 'title',
					'type'     => 'select',
					'class'    => 'wc-enhanced-select',
					'options'  => $this->tax_classes(),
					'desc_tip' => true,
				),
				array(
					'title'    => esc_html__( 'Fee Name Conflicts', 'product-fees-toolkit-for-woocommerce' ),
					'desc'     => esc_html__( 'If option #2 is chosen, whichever product comes first in the cart will take precedence.', 'product-fees-toolkit-for-woocommerce' ),
					'id'       => 'pftfw_name_conflicts',
					'default'  => 'combine',
					'type'     => 'radio',
					'options'  => array(
						'combine'      => esc_html__( 'Combine fees with the same name [Recommended].', 'product-fees-toolkit-for-woocommerce' ),
						'dont_combine' => esc_html__( 'Only add one fee if the names are conflicting.', 'product-fees-toolkit-for-woocommerce' ),
					),
					'desc_tip' => true,
				),
				array(
					'title'   => esc_html__( 'Show on Product Page', 'product-fees-toolkit-for-woocommerce' ),
					'desc'    => esc_html__( 'Show all product fees in a new product tab', 'product-fees-toolkit-for-woocommerce' ),
					'id'      => 'pftfw_display_fees_single',
					'default' => 'no',
					'type'    => 'checkbox',
				),
				array(
					'type' => 'sectionend',
					'id'   => 'product_fees_options',
				),
			)
		);

		return $settings;
	}

	/**
	 * Get available tax classes for use as options in the settings UI.
	 *
	 * @since 1.0
	 * @return array Array of option key => label pairs.
	 */
	public function tax_classes(): array {
		$tax_classes     = WC_Tax::get_tax_classes();
		$classes_options = array();

		$classes_options['_no_tax']             = esc_html__( 'No taxes for fees', 'product-fees-toolkit-for-woocommerce' );
		$classes_options['inherit_product_tax'] = esc_html__( 'Fee tax class based on the fee\'s product', 'product-fees-toolkit-for-woocommerce' );
		$classes_options['']                    = esc_html__( 'Standard', 'product-fees-toolkit-for-woocommerce' );

		if ( ! empty( $tax_classes ) ) {
			foreach ( $tax_classes as $class ) {
				$classes_options[ sanitize_title( $class ) ] = (string) $class;
			}
		}

		return $classes_options;
	}

	// Coupon Settings.
	// -------------------------------------------------------.

	/**
	 * Output the coupon setting to remove product fees.
	 *
	 * @since 1.0
	 * @param int        $coupon_id Coupon ID.
	 * @param \WC_Coupon $coupon    Coupon object.
	 * @return void
	 */
	public function add_coupon_setting( $coupon_id, $coupon ): void { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found -- Provided by action signature.
		woocommerce_wp_checkbox(
			array(
				'id'          => 'pftfw_coupon_remove_fees',
				'label'       => esc_html__( 'Remove product fees', 'product-fees-toolkit-for-woocommerce' ),
				'description' => esc_html__( 'Check this box if the coupon should remove product fees.', 'product-fees-toolkit-for-woocommerce' ),
				'value'       => $coupon->get_meta( 'pftfw_coupon_remove_fees' ),
			)
		);

		// Nonce field to verify on save.
		\wp_nonce_field( 'pftfw_coupon_options', 'pftfw_coupon_options_nonce' );
	}

	/**
	 * Save the coupon setting that controls fee removal.
	 *
	 * @since 1.0
	 * @param int        $post_id Coupon post ID.
	 * @param \WC_Coupon $coupon  Coupon object.
	 * @return void
	 */
	public function save_coupon_setting( $post_id, $coupon ): void {
		// Verify nonce before processing form data.
		if (
			! isset( $_POST['pftfw_coupon_options_nonce'] ) ||
			! \wp_verify_nonce( \sanitize_text_field( \wp_unslash( $_POST['pftfw_coupon_options_nonce'] ) ), 'pftfw_coupon_options' )
		) {
			return;
		}

		// Capability check.
		if ( ! \current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		$value = isset( $_POST['pftfw_coupon_remove_fees'] ) ? 'yes' : '';
		$coupon->add_meta_data( 'pftfw_coupon_remove_fees', $value, true );
		$coupon->save_meta_data();
	}

	// CSV Import/Export Support.
	// -------------------------------------------------------.

	/**
	 * Register custom CSV columns (used by both importer and exporter).
	 *
	 * @since 1.0
	 * @param array $options Existing column options.
	 * @return array Modified column options.
	 */
	public function add_columns_to_importer_exporter( array $options ): array {
		$options['pftfw_fee_name']       = 'Fee Name';
		$options['pftfw_fee_amount']     = 'Fee Amount';
		$options['pftfw_fee_multiplier'] = 'Fee Multiplier';
		$options['pftfw_fee_type']       = 'Fee Type';
		$options['pftfw_fee_percent']    = 'Fee Percent';
		return $options;
	}

	/**
	 * Make the custom columns selectable in the CSV mapping UI (import).
	 *
	 * @since 1.0
	 * @param array $columns Existing mapping columns.
	 * @return array Modified mapping columns.
	 */
	public function add_column_to_mapping_screen( array $columns ): array {
		$columns['Fee Name']       = 'pftfw_fee_name';
		$columns['fee name']       = 'pftfw_fee_name';
		$columns['Fee Amount']     = 'pftfw_fee_amount';
		$columns['fee amount']     = 'pftfw_fee_amount';
		$columns['Fee Multiplier'] = 'pftfw_fee_multiplier';
		$columns['fee multiplier'] = 'pftfw_fee_multiplier';
		$columns['Fee Type']       = 'pftfw_fee_type';
		$columns['fee type']       = 'pftfw_fee_type';
		$columns['Fee Percent']    = 'pftfw_fee_percent';
		$columns['fee percent']    = 'pftfw_fee_percent';
		return $columns;
	}

	/**
	 * Import handler to save our custom CSV columns onto the product object.
	 *
	 * @since 1.0
	 * @param \WC_Product $product Product object being prepared for insert.
	 * @param array       $data    Raw imported data for the current row.
	 * @return \WC_Product Modified product object.
	 */
	public function process_import( $product, array $data ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found -- Signature required by filter.
		if ( ! empty( $data['pftfw_fee_name'] ) ) {
			$product->update_meta_data( 'product-fee-name', $data['pftfw_fee_name'] );
		}
		if ( ! empty( $data['pftfw_fee_amount'] ) ) {
			$product->update_meta_data( 'product-fee-amount', $data['pftfw_fee_amount'] );
		}
		if ( ! empty( $data['pftfw_fee_multiplier'] ) ) {
			$product->update_meta_data( 'product-fee-multiplier', $data['pftfw_fee_multiplier'] );
		}
		if ( ! empty( $data['pftfw_fee_type'] ) ) {
			$type = sanitize_text_field( (string) $data['pftfw_fee_type'] );
			$type = in_array( $type, array( 'fixed', 'percent', 'fixed_plus_percent' ), true ) ? $type : '';
			$product->update_meta_data( 'product-fee-type', $type );
		}
		if ( ! empty( $data['pftfw_fee_percent'] ) ) {
			$product->update_meta_data( 'product-fee-percent', $data['pftfw_fee_percent'] );
		}
		return $product;
	}

	/**
	 * Export column: Fee Name.
	 *
	 * @since 1.0
	 * @param mixed       $value   Existing value (unused).
	 * @param \WC_Product $product Product object.
	 * @return string Exported meta value.
	 */
	public function add_export_data_fee_name( $value, $product ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found -- Filter signature.
		return $product->get_meta( 'product-fee-name', true, 'edit' );
	}

	/**
	 * Export column: Fee Amount.
	 *
	 * @since 1.0
	 * @param mixed       $value   Existing value (unused).
	 * @param \WC_Product $product Product object.
	 * @return string Exported meta value.
	 */
	public function add_export_data_fee_amount( $value, $product ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found -- Filter signature.
		return $product->get_meta( 'product-fee-amount', true, 'edit' );
	}

	/**
	 * Export column: Fee Multiplier.
	 *
	 * @since 1.0
	 * @param mixed       $value   Existing value (unused).
	 * @param \WC_Product $product Product object.
	 * @return string Exported meta value.
	 */
	public function add_export_data_fee_multiplier( $value, $product ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found -- Filter signature.
		return $product->get_meta( 'product-fee-multiplier', true, 'edit' );
	}

	/**
	 * Export column: Fee Type.
	 *
	 * @since 1.0
	 * @param mixed       $value   Existing value (unused).
	 * @param \WC_Product $product Product object.
	 * @return string Exported meta value.
	 */
	public function add_export_data_fee_type( $value, $product ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found -- Filter signature.
		return $product->get_meta( 'product-fee-type', true, 'edit' );
	}

	/**
	 * Export column: Fee Percent.
	 *
	 * @since 1.0
	 * @param mixed       $value   Existing value (unused).
	 * @param \WC_Product $product Product object.
	 * @return string Exported meta value.
	 */
	public function add_export_data_fee_percent( $value, $product ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found -- Filter signature.
		return $product->get_meta( 'product-fee-percent', true, 'edit' );
	}
}
