(function (window, document) {
	'use strict';
	var V = window.pftfwFeeVars || { dec: '.', alt: ',', decs: 2, step: '0.01' };
	var DEC = V.dec || '.';
	var ALT = V.alt || ',';
	var DECS = parseInt(V.decs, 10) || 0;
	var STEP = parseFloat(String(V.step || '0')) || 0;


	function toNumber(val) {
		if (!val) return 0;
		val = val.replace(/%/g, '');
		val = val.split(ALT).join(DEC);
		var n = parseFloat(val.split(DEC).join('.'));
		return isNaN(n) ? 0 : n;
	}

	function format(n) {
		var s = n.toFixed(DECS);
		if (DEC !== '.') {
			s = s.replace('.', DEC);
		}
		return s;
	}

	function clean(val) {
		if (!val) return '';
		val = val.replace(/%/g, '%');
		val = val.split(ALT).join(DEC);
		var re = new RegExp('[^0-9' + (DEC === '.' ? '\\.' : ',') + '%]', 'g');
		val = val.replace(re, '');
		// collapse multiple decimals
		var first = val.indexOf(DEC);
		if (first > -1) {
			var before = val.slice(0, first + 1);
			var after = val.slice(first + 1).replace(new RegExp('\\' + DEC, 'g'), '');
			val = before + after;
		}
		return val;
	}

	function adjust(input, dir) {
		var v = input.value || '';
		var pct = v.indexOf('%') > -1;
		v = v.replace(/%/g, '');
		v = v.split(ALT).join(DEC);
		var num = toNumber(v);
		num = num + (dir * STEP);
		if (!isFinite(num)) num = 0;
		var out = format(num);
		if (pct) {
			out += '%';
		}
		input.value = out;
		input.dispatchEvent(new Event('change', { bubbles: true }));
	}

	function bindAmountInput(el) {
		if (!el || el.dataset.pftfwBound === '1') return;
		el.addEventListener('input', function (e) {
			var v = clean(e.target.value);
			e.target.value = v;
		});
		el.addEventListener('keydown', function (e) {
			if (e.key === 'ArrowUp') {
				e.preventDefault();
				adjust(e.currentTarget, +1);
			} else if (e.key === 'ArrowDown') {
				e.preventDefault();
				adjust(e.currentTarget, -1);
			}
		});
		el.addEventListener('wheel', function (e) {
			if (document.activeElement !== e.currentTarget) return;
			e.preventDefault();
			adjust(e.currentTarget, e.deltaY < 0 ? +1 : -1);
		}, { passive: false });
		el.dataset.pftfwBound = '1';
	}

	function bindAmountInputsWithin(root) {
		var nodes = (root || document).querySelectorAll('input.pftfw-fee-amount');
		for (var i = 0; i < nodes.length; i++) {
			bindAmountInput(nodes[i]);
		}
	}

	function updateVisibilityFor(container, type) {
		var showFixed = type === 'fixed' || type === 'fixed_plus_percent' || type === '';
		var showPercent = type === 'percent' || type === 'fixed_plus_percent';
		var fixedNodes = container.querySelectorAll('.pftfw-field-fixed');
		var percentNodes = container.querySelectorAll('.pftfw-field-percent');
		for (var i = 0; i < fixedNodes.length; i++) {
			fixedNodes[i].style.display = showFixed ? '' : 'none';
		}
		for (var j = 0; j < percentNodes.length; j++) {
			percentNodes[j].style.display = showPercent ? '' : 'none';
		}
	}

	function containerOf(select) {
		return (
			select.closest('.pftfw-fee-block') ||
			select.closest('.woocommerce_variation') ||
			document.getElementById('fees_product_data') ||
			document
		);
	}

	// Product type -> toggle Product Fees tab visibility
	function currentProductType() {
		if (window.jQuery && jQuery('#product-type').length) {
			var v = jQuery('#product-type').val();
			if (v) return v;
		}
		var el = document.getElementById('product-type');
		return el ? el.value : '';
	}

	function isFeesTabAllowed(type) {
		var allowed = (window.pftfwFeeVars && Array.isArray(window.pftfwFeeVars.allowedTypes)) ? window.pftfwFeeVars.allowedTypes : ['simple'];
		return allowed.indexOf(type) > -1;
	}

	function setFeesTabVisible(visible) {
		var tab = document.querySelector('ul.product_data_tabs li.product_fee_options');
		var panel = document.getElementById('fees_product_data');
		if (tab) {
			tab.style.display = visible ? '' : 'none';
			// If hiding and currently active, try switch to another visible tab
			if (!visible && tab.classList.contains('active')) {
				var first = document.querySelector('ul.product_data_tabs li:not([style*="display: none"]) a');
				if (first) { first.click(); }
			}
		}
		if (panel) {
			panel.style.display = visible ? '' : 'none';
		}
	}

	function refreshFeesTabVisibility(type) {
		var t = type || currentProductType();
		setFeesTabVisible(isFeesTabAllowed(t));
	}

	function bindTypeSelectorsWithin(root) {
		var selects = (root || document).querySelectorAll('.pftfw-fee-type-select, select[id^="product-fee-type"]');
		for (var i = 0; i < selects.length; i++) {
			(function (sel) {
				function apply() {
					var cont = containerOf(sel);
					updateVisibilityFor(cont, sel.value);
				}
				sel.addEventListener('change', apply);
				apply();
			})(selects[i]);
		}
	}

	function nextFeeIndex(blocksContainer) {
		var blocks = blocksContainer.querySelectorAll('.pftfw-fee-block');
		var maxIdx = -1;
		for (var i = 0; i < blocks.length; i++) {
			var idx = parseInt(blocks[i].getAttribute('data-index') || String(i), 10);
			if (!isNaN(idx) && idx > maxIdx) maxIdx = idx;
		}
		return maxIdx + 1;
	}

	function addFeeRowFor(wrapper) {
		if (!wrapper) return;
		var blocksContainer = wrapper.querySelector('.pftfw-fee-blocks');
		if (!blocksContainer) return;
		// Find template: product-level uses #pftfw-fee-block-template, variation uses a sibling script.pftfw-fee-template
		var tpl = wrapper.querySelector('script.pftfw-fee-template');
		if (!tpl) {
			tpl = document.getElementById('pftfw-fee-block-template');
		}
		if (!tpl) return;
		var idx = nextFeeIndex(blocksContainer);
		var html = tpl.innerHTML.replace(/__INDEX__/g, String(idx));
		var temp = document.createElement('div');
		temp.innerHTML = html;
		var block = temp.firstElementChild;
		blocksContainer.appendChild(block);
		// Bind behaviors for new block
		bindAmountInputsWithin(block);
		bindTypeSelectorsWithin(block);
		renumberFeeHeaders(blocksContainer);
		return block;
	}

	function handleRemoveClick(e) {
		var target = e.target && e.target.closest('.pftfw-remove-fee');
		if (!target) return;
		e.preventDefault();
		var block = target.closest('.pftfw-fee-block');
		if (block && block.parentNode) {
			var container = block.parentNode; // .pftfw-fee-blocks
			block.parentNode.removeChild(block);
			renumberFeeHeaders(container);
		}
	}

	function bindRepeatableUI() {
		// Product-level add button
		var productContainer = document.getElementById('fees_product_data');
		if (productContainer) {
			var addBtn = productContainer.querySelector('.pftfw-add-fee');
			if (addBtn) {
				addBtn.addEventListener('click', function (e) {
					e.preventDefault();
					addFeeRowFor(productContainer.querySelector('.pftfw-fees-repeatable'));
				});
			}
			productContainer.addEventListener('click', handleRemoveClick);
		}

		// Delegated add for variation wrappers only (avoid duplicating product-level handler)
		document.addEventListener('click', function (e) {
			var btn = e.target && e.target.closest('.pftfw-add-fee');
			if (!btn) return;
			var wrapper = btn.closest('.pftfw-fees-repeatable');
			if (!wrapper) return;
			// Guard: only handle variation containers (they have data-variation-id)
			if (!wrapper.hasAttribute('data-variation-id')) {
				return;
			}
			e.preventDefault();
			addFeeRowFor(wrapper);
		}, true);

		// Delegated remove for dynamically added blocks across the page
		document.addEventListener('click', handleRemoveClick, true);
	}

	function renumberFeeHeaders(container) {
		var root = container || document;
		var blockLists = [];
		if (root.classList && root.classList.contains('pftfw-fee-blocks')) {
			blockLists = [root];
		} else {
			blockLists = root.querySelectorAll ? root.querySelectorAll('.pftfw-fee-blocks') : [];
		}
		for (var k = 0; k < blockLists.length; k++) {
			var blocks = blockLists[k].querySelectorAll('.pftfw-fee-block');
			for (var i = 0; i < blocks.length; i++) {
				var label = blocks[i].querySelector('.pftfw-fee-label');
				if (label) {
					label.textContent = 'Product fee #' + (i + 1);
				}
			}
		}
	}



	function init() {
		bindAmountInputsWithin(document);
		bindTypeSelectorsWithin(document);
		bindRepeatableUI();
		renumberFeeHeaders(document);
		refreshFeesTabVisibility();
		// Track product type changes
		if (window.jQuery) {
			jQuery(document.body).on('woocommerce_product_type_changed', function (e, type) {
				refreshFeesTabVisibility(type);
			});
			jQuery('#product-type').on('change', function () {
				refreshFeesTabVisibility(this.value);
			});
		}
		// Re-bind when WC variations load via AJAX
		if (window.jQuery && jQuery(document).on) {
			jQuery(document).on('woocommerce_variations_loaded woocommerce_variations_added', function () {
				bindAmountInputsWithin(document);
				bindTypeSelectorsWithin(document);
				renumberFeeHeaders(document);
			});
		}
	}

	if (document.readyState === 'loading') {
		document.addEventListener('DOMContentLoaded', init);
	} else {
		init();
	}
})(window, document);
