<?php
/**
 * Render class.
 *
 * @package Product_Attribute_Icons_WooCommerce
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Render class for generating HTML output.
 */
class PAIW_Render {

	/**
	 * Current settings for use in render methods.
	 *
	 * @var array
	 */
	private static $current_settings = array();

	/**
	 * Get icons HTML for a product.
	 *
	 * @param int $product_id Product ID.
	 * @return string HTML output or empty string.
	 */
	public static function get_icons( $product_id ) {
		$product = wc_get_product( $product_id );

		if ( ! $product ) {
			return '';
		}

		// Get settings.
		$settings = get_option( 'paiw_settings', array() );
		
		// Store settings for use in render_icon_item method.
		self::$current_settings = $settings;

		if ( empty( $settings ) || ! isset( $settings['attributes'] ) ) {
			return '';
		}

		// Get product attributes.
		$product_attributes = $product->get_attributes();

		if ( empty( $product_attributes ) ) {
			return '';
		}

		$icons_html = array();

		// Loop through product attributes.
		foreach ( $product_attributes as $attribute_name => $attribute ) {
			// Check if this attribute is enabled in settings.
			if ( ! isset( $settings['attributes'][ $attribute_name ] ) ) {
				continue;
			}

			$attribute_config = $settings['attributes'][ $attribute_name ];

			if ( empty( $attribute_config['enabled'] ) || empty( $attribute_config['icon'] ) ) {
				continue;
			}

			// Get attribute terms.
			if ( $attribute->is_taxonomy() ) {
				$terms = wc_get_product_terms( $product_id, $attribute_name, array( 'fields' => 'all' ) );
			} else {
				// For non-taxonomy attributes, get the values.
				$attribute_values = $attribute->get_options();
				$terms = array();
				foreach ( $attribute_values as $value ) {
					$term = new stdClass();
					$term->name = $value;
					$terms[] = $term;
				}
			}

			if ( empty( $terms ) ) {
				continue;
			}

			// Generate HTML for each term.
			foreach ( $terms as $term ) {
				$term_name = is_object( $term ) ? $term->name : $term;
				$icon_html = self::render_icon_item( $attribute_name, $term_name, $attribute_config );

				if ( ! empty( $icon_html ) ) {
					$icons_html[] = $icon_html;
				}
			}
		}

		if ( empty( $icons_html ) ) {
			return '';
		}

		// Build wrapper classes.
		$wrapper_classes = array( 'paiw-wrapper' );
		$wrapper_classes = apply_filters( 'paiw_render_wrapper_classes', $wrapper_classes, $product_id );

		$wrapper_class_string = esc_attr( implode( ' ', array_unique( $wrapper_classes ) ) );

		// Build inline styles for default colors.
		$inline_styles = array();
		$bg_color = isset( $settings['bg_color'] ) && ! empty( $settings['bg_color'] ) ? $settings['bg_color'] : '';
		$text_color = isset( $settings['text_color'] ) && ! empty( $settings['text_color'] ) ? $settings['text_color'] : '';
		$border_color = isset( $settings['border_color'] ) && ! empty( $settings['border_color'] ) ? $settings['border_color'] : '';

		if ( ! empty( $bg_color ) || ! empty( $text_color ) || ! empty( $border_color ) ) {
			if ( ! empty( $bg_color ) ) {
				$inline_styles[] = '--paiw-bg-color: ' . esc_attr( $bg_color );
			}
			if ( ! empty( $text_color ) ) {
				$inline_styles[] = '--paiw-text-color: ' . esc_attr( $text_color );
			}
			if ( ! empty( $border_color ) ) {
				$inline_styles[] = '--paiw-border-color: ' . esc_attr( $border_color );
			}
		}
		
		$style_attr = ! empty( $inline_styles ) ? ' style="' . esc_attr( implode( '; ', $inline_styles ) ) . '"' : '';

		// Build final HTML.
		$output = '<ul class="' . esc_attr( $wrapper_class_string ) . '"' . $style_attr . '>';
		$output .= implode( '', $icons_html );
		$output .= '</ul>';

		return $output;
	}

	/**
	 * Render a single icon item.
	 *
	 * @param string $attribute_name Attribute slug.
	 * @param string $term_name Term name.
	 * @param array  $attribute_config Attribute configuration.
	 * @return string HTML for icon item.
	 */
	private static function render_icon_item( $attribute_name, $term_name, $attribute_config ) {
		$icon_classes = trim( $attribute_config['icon'] );
		
		// Ensure dashicons have the base class.
		if ( ! empty( $icon_classes ) ) {
			$icon_parts = explode( ' ', $icon_classes );
			// If it starts with dashicons- but doesn't have dashicons base class, add it.
			if ( preg_match( '/^dashicons-/', $icon_parts[0] ) && ! in_array( 'dashicons', $icon_parts, true ) ) {
				array_unshift( $icon_parts, 'dashicons' );
				$icon_classes = implode( ' ', $icon_parts );
			}
		}
		
		$icon_classes = esc_attr( $icon_classes );
		$term_display = esc_html( $term_name );

		// Always use 'both' mode (icon + text).
		$display_mode = 'both';

		// Get border radius from settings.
		$border_radius = isset( self::$current_settings['border_radius'] ) ? self::$current_settings['border_radius'] : array(
			'top_left'     => 3,
			'top_right'    => 3,
			'bottom_right' => 3,
			'bottom_left'  => 3,
		);

		// Build item classes.
		$item_classes = array( 'paiw-item', 'paiw-' . sanitize_html_class( str_replace( 'pa_', '', $attribute_name ) ) );
		
		$item_classes = apply_filters( 'paiw_render_item_classes', $item_classes, $attribute_name, $term_name );
		$item_class_string = esc_attr( implode( ' ', array_unique( $item_classes ) ) );

		// Build inline styles for per-attribute colors and border radius.
		$item_styles = array();
		
		// Check if custom colors are enabled for this attribute
		$use_custom_colors = isset( $attribute_config['use_custom_colors'] ) && $attribute_config['use_custom_colors'];
		
		// Get default colors from settings
		$default_bg_color = isset( self::$current_settings['bg_color'] ) && ! empty( self::$current_settings['bg_color'] ) ? self::$current_settings['bg_color'] : '';
		$default_text_color = isset( self::$current_settings['text_color'] ) && ! empty( self::$current_settings['text_color'] ) ? self::$current_settings['text_color'] : '';
		$default_border_color = isset( self::$current_settings['border_color'] ) && ! empty( self::$current_settings['border_color'] ) ? self::$current_settings['border_color'] : '';
		
		// Use custom colors only if enabled and set, otherwise use defaults
		$attr_bg_color = '';
		$attr_text_color = '';
		$attr_border_color = '';
		
		if ( $use_custom_colors ) {
			$attr_bg_color = isset( $attribute_config['bg_color'] ) && ! empty( $attribute_config['bg_color'] ) ? $attribute_config['bg_color'] : $default_bg_color;
			$attr_text_color = isset( $attribute_config['text_color'] ) && ! empty( $attribute_config['text_color'] ) ? $attribute_config['text_color'] : $default_text_color;
			$attr_border_color = isset( $attribute_config['border_color'] ) && ! empty( $attribute_config['border_color'] ) ? $attribute_config['border_color'] : $default_border_color;
		} else {
			$attr_bg_color = $default_bg_color;
			$attr_text_color = $default_text_color;
			$attr_border_color = $default_border_color;
		}

		if ( ! empty( $attr_bg_color ) ) {
			$item_styles[] = 'background-color: ' . esc_attr( $attr_bg_color );
		}
		if ( ! empty( $attr_text_color ) ) {
			$item_styles[] = 'color: ' . esc_attr( $attr_text_color );
		}
		if ( ! empty( $attr_border_color ) ) {
			$item_styles[] = 'border-color: ' . esc_attr( $attr_border_color );
		}

		// Add border radius.
		$border_radius_css = sprintf(
			'%dpx %dpx %dpx %dpx',
			absint( $border_radius['top_left'] ),
			absint( $border_radius['top_right'] ),
			absint( $border_radius['bottom_right'] ),
			absint( $border_radius['bottom_left'] )
		);
		$item_styles[] = 'border-radius: ' . esc_attr( $border_radius_css );

		$item_style_attr = ! empty( $item_styles ) ? ' style="' . esc_attr( implode( '; ', $item_styles ) ) . '"' : '';

		// Build icon HTML.
		$icon_html = '<span class="' . esc_attr( $icon_classes ) . '" aria-hidden="true"></span>';

		// Allow filtering of icon HTML.
		$icon_html = apply_filters( 'paiw_render_icon_html', $icon_html, $attribute_name, $term_name, $attribute_config );

		// Build item HTML (always show icon + text).
		$item_html = '<li class="' . esc_attr( $item_class_string ) . '"' . $item_style_attr . '>';
		$item_html .= $icon_html;
		$item_html .= '<span class="paiw-label">' . esc_html( $term_display ) . '</span>';
		$item_html .= '</li>';

		return $item_html;
	}
}

