<?php
/**
 * Injector class.
 *
 * @package Product_Attribute_Icons_WooCommerce
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Auto-injection class.
 */
class PAIW_Injector {

	/**
	 * The single instance of the class.
	 *
	 * @var PAIW_Injector
	 */
	private static $instance = null;

	/**
	 * Flag to track if hook has been registered.
	 *
	 * @var bool
	 */
	private static $hook_registered = false;

	/**
	 * Get the singleton instance.
	 *
	 * @return PAIW_Injector
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		$this->init_hooks();
	}

	/**
	 * Initialize hooks.
	 */
	private function init_hooks() {
		// Register hook at wp to ensure settings are current and hook is registered early enough.
		add_action( 'wp', array( $this, 'register_injection_hook' ) );
	}

	/**
	 * Register the injection hook at the correct priority based on settings.
	 */
	public function register_injection_hook() {
		// Prevent duplicate registrations.
		if ( self::$hook_registered ) {
			return;
		}

		// Only on single product pages.
		if ( ! is_product() ) {
			return;
		}

		// Get settings to determine injection position.
		$settings = get_option( 'paiw_settings', array() );
		$position = isset( $settings['injection_position'] ) ? $settings['injection_position'] : 'below_price';

		// Don't hook if position is 'none'.
		if ( 'none' === $position ) {
			return;
		}

		// Remove any existing hooks first to prevent duplicates.
		// Remove from all possible priorities to ensure clean registration.
		for ( $p = 3; $p <= 30; $p++ ) {
			remove_action( 'woocommerce_single_product_summary', array( $this, 'inject_icons' ), $p );
		}

		// Map position to priority.
		// WooCommerce default priorities:
		// - Title: 5
		// - Rating: 10
		// - Price: 10
		// - Excerpt: 20
		// - Add to cart: 30
		$priority_map = array(
			'above_title'      => 3,   // Before title (priority 5) - shows even if title is hidden.
			'above_price'      => 6,   // Before price (priority 10) but after title (priority 5).
			'below_price'      => 15,  // After price (priority 10).
			'below_short_desc' => 25,  // After short description/excerpt (priority 20).
		);

		$priority = isset( $priority_map[ $position ] ) ? $priority_map[ $position ] : 15;

		// Hook into single product summary at the correct priority.
		add_action( 'woocommerce_single_product_summary', array( $this, 'inject_icons' ), $priority );
		
		// Mark as registered.
		self::$hook_registered = true;
	}

	/**
	 * Inject icons into product page.
	 */
	public function inject_icons() {
		// Only on single product pages.
		if ( ! is_product() ) {
			return;
		}

		global $post;

		if ( ! isset( $post->ID ) ) {
			return;
		}

		// Get icons HTML.
		$icons_html = PAIW_Render::get_icons( $post->ID );

		if ( ! empty( $icons_html ) ) {
			// Output is already escaped in Render class.
			echo $icons_html; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}
	}
}
