<?php
/**
 * Core plugin class.
 *
 * @package Product_Attribute_Icons_WooCommerce
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Main plugin class.
 */
class PAIW_Core {

	/**
	 * The single instance of the class.
	 *
	 * @var PAIW_Core
	 */
	private static $instance = null;

	/**
	 * Get the singleton instance.
	 *
	 * @return PAIW_Core
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		$this->load_dependencies();
		$this->init();
	}

	/**
	 * Load required files.
	 */
	private function load_dependencies() {
		require_once PAIW_PATH . 'includes/class-paiw-admin.php';
		require_once PAIW_PATH . 'includes/class-paiw-shortcode.php';
		require_once PAIW_PATH . 'includes/class-paiw-render.php';
		require_once PAIW_PATH . 'includes/class-paiw-injector.php';
	}

	/**
	 * Initialize the plugin.
	 */
	private function init() {
		// Check dependencies first.
		if ( ! $this->check_dependencies() ) {
			return;
		}

		// Load text domain.
		add_action( 'init', array( $this, 'load_textdomain' ) );

		// Initialize admin.
		if ( is_admin() ) {
			PAIW_Admin::get_instance();
		}

		// Initialize shortcode.
		PAIW_Shortcode::get_instance();

		// Initialize injector.
		PAIW_Injector::get_instance();

		// Enqueue frontend assets.
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_assets' ) );
	}

	/**
	 * Check plugin dependencies.
	 *
	 * @return bool True if all dependencies are met.
	 */
	private function check_dependencies() {
		// Check PHP version.
		if ( version_compare( PHP_VERSION, '7.4', '<' ) ) {
			add_action( 'admin_notices', array( $this, 'php_version_notice' ) );
			return false;
		}

		// Check if WooCommerce is active.
		if ( ! class_exists( 'WooCommerce' ) ) {
			add_action( 'admin_notices', array( $this, 'woocommerce_missing_notice' ) );
			return false;
		}

		// Check WooCommerce version.
		if ( defined( 'WC_VERSION' ) && version_compare( WC_VERSION, '5.0', '<' ) ) {
			add_action( 'admin_notices', array( $this, 'woocommerce_version_notice' ) );
			return false;
		}

		return true;
	}

	/**
	 * Display PHP version notice.
	 */
	public function php_version_notice() {
		?>
		<div class="notice notice-error">
			<p>
				<?php
				printf(
					/* translators: 1: Plugin name, 2: Required PHP version, 3: Current PHP version */
					esc_html__( '%1$s requires PHP version %2$s or higher. You are running PHP %3$s.', 'product-attribute-icons-display' ),
					'<strong>' . esc_html__( 'Product Attribute Icons Display', 'product-attribute-icons-display' ) . '</strong>',
					esc_html( '7.4' ),
					esc_html( PHP_VERSION )
				);
				?>
			</p>
		</div>
		<?php
	}

	/**
	 * Display WooCommerce missing notice.
	 */
	public function woocommerce_missing_notice() {
		?>
		<div class="notice notice-error">
			<p>
				<?php
				printf(
					/* translators: 1: Plugin name, 2: WooCommerce plugin name */
					esc_html__( '%1$s requires %2$s to be installed and active.', 'product-attribute-icons-display' ),
					'<strong>' . esc_html__( 'Product Attribute Icons Display', 'product-attribute-icons-display' ) . '</strong>',
					'<strong>WooCommerce</strong>'
				);
				?>
			</p>
		</div>
		<?php
	}

	/**
	 * Display WooCommerce version notice.
	 */
	public function woocommerce_version_notice() {
		?>
		<div class="notice notice-error">
			<p>
				<?php
				printf(
					/* translators: 1: Plugin name, 2: Required WooCommerce version, 3: Current WooCommerce version */
					esc_html__( '%1$s requires WooCommerce version %2$s or higher. You are running WooCommerce %3$s.', 'product-attribute-icons-display' ),
					'<strong>' . esc_html__( 'Product Attribute Icons Display', 'product-attribute-icons-display' ) . '</strong>',
					'5.0',
					defined( 'WC_VERSION' ) ? esc_html( WC_VERSION ) : esc_html__( 'Unknown', 'product-attribute-icons-display' )
				);
				?>
			</p>
		</div>
		<?php
	}

	/**
	 * Load plugin text domain.
	 * Note: For plugins hosted on WordPress.org, translations are automatically loaded.
	 * This function is kept for backward compatibility and self-hosted installations.
	 * 
	 * @return void
	 */
	public function load_textdomain() {
		// WordPress.org automatically loads translations via translate.wordpress.org.
		// Only load manually for self-hosted installations or if explicitly requested.
		// Check if we're on WordPress.org by checking if translate.wordpress.org is available.
		// For now, we'll skip loading to comply with WordPress.org guidelines.
		// Self-hosted users can uncomment the line below or use a filter to enable it.
		
		// Uncomment for self-hosted installations:
		// load_plugin_textdomain(
		// 	'product-attribute-icons-display',
		// 	false,
		// 	dirname( plugin_basename( PAIW_PATH . 'product-attribute-icons-display.php' ) ) . '/languages'
		// );
	}

	/**
	 * Enqueue frontend assets.
	 */
	public function enqueue_assets() {
		// Only enqueue on single product pages.
		if ( ! is_product() ) {
			return;
		}


		wp_enqueue_style(
			'paiw-style',
			PAIW_URL . 'assets/css/style.css',
			array(),
			PAIW_VERSION
		);
	}

}

