<?php
/**
 * Admin class.
 *
 * @package Product_Attribute_Icons_WooCommerce
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Admin settings class.
 */
class PAIW_Admin {

	/**
	 * The single instance of the class.
	 *
	 * @var PAIW_Admin
	 */
	private static $instance = null;

	/**
	 * Settings option name.
	 *
	 * @var string
	 */
	private $option_name = 'paiw_settings';

	/**
	 * Get the singleton instance.
	 *
	 * @return PAIW_Admin
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
	}

	/**
	 * Add admin menu.
	 */
	public function add_admin_menu() {
		add_submenu_page(
			'woocommerce',
			__( 'Product Icons', 'product-attribute-icons-display' ),
			__( 'Product Icons', 'product-attribute-icons-display' ),
			'manage_woocommerce',
			'paiw-settings',
			array( $this, 'render_settings_page' )
		);
	}

	/**
	 * Register settings.
	 */
	public function register_settings() {
		register_setting(
			'paiw_settings_group',
			$this->option_name,
			array(
				'sanitize_callback' => array( $this, 'sanitize_settings' ),
				'default'           => array(),
			)
		);
	}

	/**
	 * Sanitize settings.
	 *
	 * @param array $input Raw input data.
	 * @return array Sanitized settings.
	 */
	public function sanitize_settings( $input ) {
		// Verify nonce for security.
		if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'paiw_settings_group-options' ) ) {
			return get_option( $this->option_name, array() );
		}

		// Check user capabilities.
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			return get_option( $this->option_name, array() );
		}

		$sanitized = array();

		// Sanitize injection position.
		$allowed_positions = array( 'none', 'above_title', 'above_price', 'below_price', 'below_short_desc' );
		if ( isset( $input['injection_position'] ) && in_array( $input['injection_position'], $allowed_positions, true ) ) {
			$sanitized['injection_position'] = $input['injection_position'];
		} else {
			$sanitized['injection_position'] = 'below_price';
		}

		// Sanitize background color.
		if ( isset( $input['bg_color'] ) && ! empty( trim( $input['bg_color'] ) ) ) {
			$sanitized['bg_color'] = sanitize_hex_color( $input['bg_color'] );
			if ( false === $sanitized['bg_color'] ) {
				$sanitized['bg_color'] = '#f5f5f5'; // Default if invalid.
			}
		}

		// Sanitize text/icon color.
		if ( isset( $input['text_color'] ) && ! empty( trim( $input['text_color'] ) ) ) {
			$sanitized['text_color'] = sanitize_hex_color( $input['text_color'] );
			if ( false === $sanitized['text_color'] ) {
				$sanitized['text_color'] = '#333333'; // Default if invalid.
			}
		}

		// Sanitize border color.
		if ( isset( $input['border_color'] ) && ! empty( trim( $input['border_color'] ) ) ) {
			$sanitized['border_color'] = sanitize_hex_color( $input['border_color'] );
			if ( false === $sanitized['border_color'] ) {
				$sanitized['border_color'] = '#dddddd'; // Default if invalid.
			}
		}


		// Sanitize border radius values.
		if ( isset( $input['border_radius'] ) && is_array( $input['border_radius'] ) ) {
			$sanitized['border_radius'] = array(
				'top_left'     => isset( $input['border_radius']['top_left'] ) ? absint( $input['border_radius']['top_left'] ) : 3,
				'top_right'    => isset( $input['border_radius']['top_right'] ) ? absint( $input['border_radius']['top_right'] ) : 3,
				'bottom_right' => isset( $input['border_radius']['bottom_right'] ) ? absint( $input['border_radius']['bottom_right'] ) : 3,
				'bottom_left'  => isset( $input['border_radius']['bottom_left'] ) ? absint( $input['border_radius']['bottom_left'] ) : 3,
			);
		} else {
			$sanitized['border_radius'] = array(
				'top_left'     => 3,
				'top_right'    => 3,
				'bottom_right' => 3,
				'bottom_left'  => 3,
			);
		}

		// Sanitize attributes.
		if ( isset( $input['attributes'] ) && is_array( $input['attributes'] ) ) {
			$sanitized['attributes'] = array();

			foreach ( $input['attributes'] as $attribute_slug => $attribute_data ) {
				if ( ! is_array( $attribute_data ) ) {
					continue;
				}

				$sanitized['attributes'][ $attribute_slug ] = array(
					'enabled'         => isset( $attribute_data['enabled'] ) ? true : false,
					'icon'            => isset( $attribute_data['icon'] ) ? $this->sanitize_icon_class( $attribute_data['icon'] ) : '',
					'use_custom_colors' => isset( $attribute_data['use_custom_colors'] ) ? true : false,
					'bg_color'        => isset( $attribute_data['bg_color'] ) && ! empty( trim( $attribute_data['bg_color'] ) ) ? sanitize_hex_color( $attribute_data['bg_color'] ) : '',
					'text_color'      => isset( $attribute_data['text_color'] ) && ! empty( trim( $attribute_data['text_color'] ) ) ? sanitize_hex_color( $attribute_data['text_color'] ) : '',
					'border_color'    => isset( $attribute_data['border_color'] ) && ! empty( trim( $attribute_data['border_color'] ) ) ? sanitize_hex_color( $attribute_data['border_color'] ) : '',
				);
			}
		}

		return $sanitized;
	}

	/**
	 * Sanitize icon class.
	 *
	 * @param string $icon_class Icon class string.
	 * @return string Sanitized icon class.
	 */
	private function sanitize_icon_class( $icon_class ) {
		// Allow alphanumeric, dashes, underscores, and spaces (for multiple classes).
		$sanitized = preg_replace( '/[^a-z0-9_\s-]/i', '', $icon_class );
		// Remove extra spaces.
		$sanitized = preg_replace( '/\s+/', ' ', trim( $sanitized ) );
		return $sanitized;
	}

	/**
	 * Enqueue admin assets.
	 *
	 * @param string $hook Current admin page hook.
	 */
	public function enqueue_admin_assets( $hook ) {
		if ( 'woocommerce_page_paiw-settings' !== $hook ) {
			return;
		}


		wp_enqueue_style(
			'paiw-admin-style',
			PAIW_URL . 'assets/css/admin.css',
			array(),
			PAIW_VERSION
		);

		// Enqueue WordPress color picker.
		wp_enqueue_style( 'wp-color-picker' );
		wp_enqueue_script( 'wp-color-picker' );
		wp_enqueue_script(
			'paiw-admin-script',
			PAIW_URL . 'assets/js/admin.js',
			array( 'jquery', 'wp-color-picker' ),
			PAIW_VERSION,
			true
		);
	}

	/**
	 * Render settings page.
	 */
	public function render_settings_page() {
		// Check user capabilities.
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'product-attribute-icons-display' ) );
		}

		$settings = get_option( $this->option_name, array() );
		$attributes = wc_get_attribute_taxonomies();

		?>
		<div class="wrap paiw-settings-wrap">
			<div class="paiw-header">
				<h1 class="paiw-title"><?php esc_html_e( 'Product Attribute Icons Display', 'product-attribute-icons-display' ); ?></h1>
				<div class="paiw-meta">
					<?php
					/* translators: %s: Plugin version number */
					printf(
						'<span class="paiw-version">%s</span>',
						esc_html(
							sprintf(
								/* translators: %s: Plugin version number */
								__( 'Version %s', 'product-attribute-icons-display' ),
								PAIW_VERSION
							)
						)
					);
					?>
					<span class="paiw-separator">|</span>
					<span class="paiw-author"><?php esc_html_e( 'Author:', 'product-attribute-icons-display' ); ?> <a href="https://neatplugins.com" target="_blank" rel="noopener noreferrer">NeatPlugins.com</a></span>
				</div>
			</div>

			<form method="post" action="options.php">
				<?php
				settings_fields( 'paiw_settings_group' );
				wp_nonce_field( 'paiw_settings_group-options', '_wpnonce', true, true );
				do_settings_sections( 'paiw_settings_group' );
				?>

				<div class="paiw-settings-card">
					<h2 class="paiw-card-title"><?php esc_html_e( 'General Settings', 'product-attribute-icons-display' ); ?></h2>
					
					<table class="form-table" role="presentation">
						<tbody>
							<tr>
								<th scope="row">
									<label for="paiw_injection_position">
										<?php esc_html_e( 'Injection Position', 'product-attribute-icons-display' ); ?>
									</label>
								</th>
								<td>
									<select name="<?php echo esc_attr( $this->option_name ); ?>[injection_position]" id="paiw_injection_position" class="paiw-select">
										<option value="none" <?php selected( isset( $settings['injection_position'] ) ? $settings['injection_position'] : 'below_price', 'none' ); ?>>
											<?php esc_html_e( 'None (Manual via Shortcode)', 'product-attribute-icons-display' ); ?>
										</option>
										<option value="above_title" <?php selected( isset( $settings['injection_position'] ) ? $settings['injection_position'] : 'below_price', 'above_title' ); ?>>
											<?php esc_html_e( 'Above Product Title', 'product-attribute-icons-display' ); ?>
										</option>
										<option value="above_price" <?php selected( isset( $settings['injection_position'] ) ? $settings['injection_position'] : 'below_price', 'above_price' ); ?>>
											<?php esc_html_e( 'Above Product Price', 'product-attribute-icons-display' ); ?>
										</option>
										<option value="below_price" <?php selected( isset( $settings['injection_position'] ) ? $settings['injection_position'] : 'below_price', 'below_price' ); ?>>
											<?php esc_html_e( 'Below Product Price', 'product-attribute-icons-display' ); ?>
										</option>
										<option value="below_short_desc" <?php selected( isset( $settings['injection_position'] ) ? $settings['injection_position'] : 'below_price', 'below_short_desc' ); ?>>
											<?php esc_html_e( 'Below Short Description', 'product-attribute-icons-display' ); ?>
										</option>
									</select>
									<p class="description">
										<?php esc_html_e( 'Choose where to automatically display attribute icons on single product pages.', 'product-attribute-icons-display' ); ?>
									</p>
								</td>
							</tr>
							<tr>
								<th scope="row">
									<label for="paiw_bg_color">
										<?php esc_html_e( 'Default Background Color', 'product-attribute-icons-display' ); ?>
									</label>
								</th>
								<td>
									<input
										type="text"
										name="<?php echo esc_attr( $this->option_name ); ?>[bg_color]"
										id="paiw_bg_color"
										value="<?php echo esc_attr( isset( $settings['bg_color'] ) ? $settings['bg_color'] : '#f5f5f5' ); ?>"
										class="paiw-color-picker wp-color-picker"
										data-default-color="#f5f5f5"
									/>
									<p class="description">
										<?php esc_html_e( 'Default background color for attribute icon badges (can be overridden per attribute).', 'product-attribute-icons-display' ); ?>
									</p>
								</td>
							</tr>
							<tr>
								<th scope="row">
									<label for="paiw_text_color">
										<?php esc_html_e( 'Default Text & Icon Color', 'product-attribute-icons-display' ); ?>
									</label>
								</th>
								<td>
									<input
										type="text"
										name="<?php echo esc_attr( $this->option_name ); ?>[text_color]"
										id="paiw_text_color"
										value="<?php echo esc_attr( isset( $settings['text_color'] ) ? $settings['text_color'] : '#333333' ); ?>"
										class="paiw-color-picker wp-color-picker"
										data-default-color="#333333"
									/>
									<p class="description">
										<?php esc_html_e( 'Default color for text label and icon (can be overridden per attribute).', 'product-attribute-icons-display' ); ?>
									</p>
								</td>
							</tr>
							<tr>
								<th scope="row">
									<label for="paiw_border_color">
										<?php esc_html_e( 'Default Border Color', 'product-attribute-icons-display' ); ?>
									</label>
								</th>
								<td>
									<input
										type="text"
										name="<?php echo esc_attr( $this->option_name ); ?>[border_color]"
										id="paiw_border_color"
										value="<?php echo esc_attr( isset( $settings['border_color'] ) ? $settings['border_color'] : '#dddddd' ); ?>"
										class="paiw-color-picker wp-color-picker"
										data-default-color="#dddddd"
									/>
									<p class="description">
										<?php esc_html_e( 'Default border color for attribute icon badges (can be overridden per attribute).', 'product-attribute-icons-display' ); ?>
									</p>
								</td>
							</tr>
							<tr>
								<th scope="row">
									<label><?php esc_html_e( 'Border Radius', 'product-attribute-icons-display' ); ?></label>
								</th>
								<td>
									<div class="paiw-border-radius-controls">
										<div class="paiw-radius-field">
											<label for="paiw_radius_top_left"><?php esc_html_e( 'Top Left', 'product-attribute-icons-display' ); ?></label>
											<input
												type="number"
												name="<?php echo esc_attr( $this->option_name ); ?>[border_radius][top_left]"
												id="paiw_radius_top_left"
												value="<?php echo esc_attr( isset( $settings['border_radius']['top_left'] ) ? $settings['border_radius']['top_left'] : 3 ); ?>"
												class="paiw-radius-input"
												min="0"
												step="1"
											/>
											<span class="paiw-radius-unit">px</span>
										</div>
										<div class="paiw-radius-field">
											<label for="paiw_radius_top_right"><?php esc_html_e( 'Top Right', 'product-attribute-icons-display' ); ?></label>
											<input
												type="number"
												name="<?php echo esc_attr( $this->option_name ); ?>[border_radius][top_right]"
												id="paiw_radius_top_right"
												value="<?php echo esc_attr( isset( $settings['border_radius']['top_right'] ) ? $settings['border_radius']['top_right'] : 3 ); ?>"
												class="paiw-radius-input"
												min="0"
												step="1"
											/>
											<span class="paiw-radius-unit">px</span>
										</div>
										<div class="paiw-radius-field">
											<label for="paiw_radius_bottom_right"><?php esc_html_e( 'Bottom Right', 'product-attribute-icons-display' ); ?></label>
											<input
												type="number"
												name="<?php echo esc_attr( $this->option_name ); ?>[border_radius][bottom_right]"
												id="paiw_radius_bottom_right"
												value="<?php echo esc_attr( isset( $settings['border_radius']['bottom_right'] ) ? $settings['border_radius']['bottom_right'] : 3 ); ?>"
												class="paiw-radius-input"
												min="0"
												step="1"
											/>
											<span class="paiw-radius-unit">px</span>
										</div>
										<div class="paiw-radius-field">
											<label for="paiw_radius_bottom_left"><?php esc_html_e( 'Bottom Left', 'product-attribute-icons-display' ); ?></label>
											<input
												type="number"
												name="<?php echo esc_attr( $this->option_name ); ?>[border_radius][bottom_left]"
												id="paiw_radius_bottom_left"
												value="<?php echo esc_attr( isset( $settings['border_radius']['bottom_left'] ) ? $settings['border_radius']['bottom_left'] : 3 ); ?>"
												class="paiw-radius-input"
												min="0"
												step="1"
											/>
											<span class="paiw-radius-unit">px</span>
										</div>
									</div>
									<p class="description">
										<?php esc_html_e( 'Set border radius for each corner of the attribute badges (in pixels).', 'product-attribute-icons-display' ); ?>
									</p>
								</td>
							</tr>
						</tbody>
					</table>
					
					<div class="paiw-icon-help-note">
						<strong><?php esc_html_e( 'Icon Help:', 'product-attribute-icons-display' ); ?></strong>
						<?php esc_html_e( 'Use Dashicons format: dashicons dashicons-{name} (e.g., dashicons dashicons-menu).', 'product-attribute-icons-display' ); ?>
						<a href="https://developer.wordpress.org/resource/dashicons/" target="_blank" rel="noopener noreferrer"><?php esc_html_e( 'Browse Dashicons', 'product-attribute-icons-display' ); ?></a>
					</div>
				</div>

				<div class="paiw-settings-card">
					<div class="paiw-card-header">
						<h2 class="paiw-card-title"><?php esc_html_e( 'Attributes', 'product-attribute-icons-display' ); ?></h2>
						<button type="button" class="paiw-toggle-all" data-state="expand"><?php esc_html_e( 'Expand All', 'product-attribute-icons-display' ); ?></button>
					</div>

					<?php if ( empty( $attributes ) ) : ?>
						<p class="paiw-no-attributes">
							<?php esc_html_e( 'No product attributes found. Please create attributes in WooCommerce first.', 'product-attribute-icons-display' ); ?>
						</p>
					<?php else : ?>
						<table class="paiw-attributes-table">
						<thead>
							<tr>
								<th class="paiw-col-toggle"></th>
								<th class="paiw-col-name"><?php esc_html_e( 'Attribute', 'product-attribute-icons-display' ); ?></th>
								<th class="paiw-col-preview"><?php esc_html_e( 'Icon Preview', 'product-attribute-icons-display' ); ?></th>
								<th class="paiw-col-icon"><?php esc_html_e( 'Icon', 'product-attribute-icons-display' ); ?></th>
							</tr>
						</thead>
							<tbody>
							<?php foreach ( $attributes as $attribute ) : ?>
								<?php
								$attribute_slug = 'pa_' . $attribute->attribute_name;
								$attribute_data = isset( $settings['attributes'][ $attribute_slug ] ) ? $settings['attributes'][ $attribute_slug ] : array();
								$enabled = isset( $attribute_data['enabled'] ) ? $attribute_data['enabled'] : false;
								$icon = isset( $attribute_data['icon'] ) ? $attribute_data['icon'] : '';
								$use_custom_colors = isset( $attribute_data['use_custom_colors'] ) ? $attribute_data['use_custom_colors'] : false;
								$bg_color = isset( $attribute_data['bg_color'] ) ? $attribute_data['bg_color'] : '';
								$text_color = isset( $attribute_data['text_color'] ) ? $attribute_data['text_color'] : '';
								$border_color = isset( $attribute_data['border_color'] ) ? $attribute_data['border_color'] : '';
								
								// Prepare icon for preview
								$icon_preview = trim( $icon );
								if ( ! empty( $icon_preview ) ) {
									$icon_parts = explode( ' ', $icon_preview );
									if ( preg_match( '/^dashicons-/', $icon_parts[0] ) && ! in_array( 'dashicons', $icon_parts, true ) ) {
										array_unshift( $icon_parts, 'dashicons' );
										$icon_preview = implode( ' ', $icon_parts );
									}
								}
								
								// Determine icon library for link (only Dashicons)
								$icon_library = '';
								$icon_link = '';
								if ( ! empty( $icon ) ) {
									if ( strpos( $icon, 'dashicons' ) !== false || preg_match( '/^dashicons-/', $icon ) ) {
										$icon_library = 'Dashicons';
										$icon_link = 'https://developer.wordpress.org/resource/dashicons/';
									}
								}
								?>
								<tr class="paiw-attribute-row" data-attribute="<?php echo esc_attr( $attribute_slug ); ?>">
									<td class="paiw-col-toggle">
										<button type="button" class="paiw-expand-btn" aria-label="<?php esc_attr_e( 'Expand', 'product-attribute-icons-display' ); ?>">
											<span class="dashicons dashicons-arrow-down-alt2"></span>
										</button>
									</td>
									<td class="paiw-col-name">
										<div class="paiw-attribute-name-row">
											<label class="paiw-toggle-switch">
												<input
													type="checkbox"
													name="<?php echo esc_attr( $this->option_name ); ?>[attributes][<?php echo esc_attr( $attribute_slug ); ?>][enabled]"
													value="1"
													<?php checked( $enabled, true ); ?>
													class="paiw-enable-toggle"
												/>
												<span class="paiw-toggle-slider"></span>
											</label>
											<span class="paiw-attribute-name"><?php echo esc_html( $attribute->attribute_label ); ?></span>
										</div>
									</td>
									<td class="paiw-col-preview">
										<?php if ( ! empty( $icon_preview ) ) : ?>
											<span class="paiw-icon-preview-inline <?php echo esc_attr( $icon_preview ); ?>" aria-hidden="true" title="<?php echo esc_attr( $attribute->attribute_label ); ?>"></span>
										<?php endif; ?>
									</td>
									<td class="paiw-col-icon">
										<div class="paiw-icon-cell">
											<input
												type="text"
												name="<?php echo esc_attr( $this->option_name ); ?>[attributes][<?php echo esc_attr( $attribute_slug ); ?>][icon]"
												value="<?php echo esc_attr( $icon ); ?>"
												class="paiw-input paiw-input-compact paiw-icon-input"
												placeholder="<?php esc_attr_e( 'dashicons dashicons-menu', 'product-attribute-icons-display' ); ?>"
											/>
										</div>
									</td>
								</tr>
								<tr class="paiw-attribute-details" style="display: none;">
									<td colspan="4">
										<div style="margin-bottom: 12px;">
											<label style="display: flex; align-items: center; gap: 8px; font-weight: 500;">
												<input
													type="checkbox"
													name="<?php echo esc_attr( $this->option_name ); ?>[attributes][<?php echo esc_attr( $attribute_slug ); ?>][use_custom_colors]"
													value="1"
													<?php checked( $use_custom_colors, true ); ?>
													class="paiw-use-custom-colors"
												/>
												<span><?php esc_html_e( 'Use custom colors for this attribute', 'product-attribute-icons-display' ); ?></span>
											</label>
											<p class="description" style="margin-top: 4px; margin-left: 28px;">
												<?php esc_html_e( 'When unchecked, default colors will be used. When checked, you can set custom colors below.', 'product-attribute-icons-display' ); ?>
											</p>
										</div>
										<table class="paiw-details-table paiw-color-controls"<?php echo $use_custom_colors ? '' : ' style="opacity: 0.5; pointer-events: none;"'; ?>>
											<thead>
												<tr>
													<th><?php esc_html_e( 'Background', 'product-attribute-icons-display' ); ?></th>
													<th><?php esc_html_e( 'Text & Icon', 'product-attribute-icons-display' ); ?></th>
													<th><?php esc_html_e( 'Border', 'product-attribute-icons-display' ); ?></th>
												</tr>
											</thead>
											<tbody>
												<tr>
													<td>
														<input
															type="text"
															name="<?php echo esc_attr( $this->option_name ); ?>[attributes][<?php echo esc_attr( $attribute_slug ); ?>][bg_color]"
															value="<?php echo esc_attr( $bg_color ); ?>"
															class="paiw-color-picker paiw-attr-color wp-color-picker"
															data-default-color="<?php echo esc_attr( isset( $settings['bg_color'] ) && ! empty( $settings['bg_color'] ) ? $settings['bg_color'] : '#f5f5f5' ); ?>"
															<?php echo $use_custom_colors ? '' : 'disabled'; ?>
														/>
														<p class="paiw-color-desc"><?php esc_html_e( 'Badge background', 'product-attribute-icons-display' ); ?></p>
													</td>
													<td>
														<input
															type="text"
															name="<?php echo esc_attr( $this->option_name ); ?>[attributes][<?php echo esc_attr( $attribute_slug ); ?>][text_color]"
															value="<?php echo esc_attr( $text_color ); ?>"
															class="paiw-color-picker paiw-attr-color wp-color-picker"
															data-default-color="<?php echo esc_attr( isset( $settings['text_color'] ) && ! empty( $settings['text_color'] ) ? $settings['text_color'] : '#333333' ); ?>"
															<?php echo $use_custom_colors ? '' : 'disabled'; ?>
														/>
														<p class="paiw-color-desc"><?php esc_html_e( 'Text & icon color', 'product-attribute-icons-display' ); ?></p>
													</td>
													<td>
														<input
															type="text"
															name="<?php echo esc_attr( $this->option_name ); ?>[attributes][<?php echo esc_attr( $attribute_slug ); ?>][border_color]"
															value="<?php echo esc_attr( $border_color ); ?>"
															class="paiw-color-picker paiw-attr-color wp-color-picker"
															data-default-color="<?php echo esc_attr( isset( $settings['border_color'] ) && ! empty( $settings['border_color'] ) ? $settings['border_color'] : '#dddddd' ); ?>"
															<?php echo $use_custom_colors ? '' : 'disabled'; ?>
														/>
														<p class="paiw-color-desc"><?php esc_html_e( 'Border color', 'product-attribute-icons-display' ); ?></p>
													</td>
												</tr>
											</tbody>
										</table>
									</td>
								</tr>
							<?php endforeach; ?>
							</tbody>
						</table>
					<?php endif; ?>
				</div>

				<?php submit_button(); ?>
			</form>
		</div>
		<?php
	}

	// Note: Integrity functions removed as CDN loading is not used.
	// Icon libraries must be loaded by theme or other plugins.
}

