/**
 * Admin JavaScript for Product Attribute Icons Display
 *
 * @package Product_Attribute_Icons_WooCommerce
 */

(function($) {
	'use strict';

	$(document).ready(function() {
		// Wait for wp-color-picker to be available
		if (typeof $.fn.wpColorPicker === 'undefined') {
			// WordPress color picker not loaded, abort initialization.
			return;
		}

	/**
	 * Initialize a single color picker input.
	 *
	 * @param {jQuery} $input The input element to initialize.
	 * @param {jQuery} $row Optional row element for icon preview updates.
	 */
	function initSingleColorPicker($input, $row) {
		if ($input.data('wp-color-picker-initialized') || !$input.is('input[type="text"]') || $input.prop('disabled')) {
			return;
		}

		try {
			var originalInput = $input;
			var $container = null;
			
			$input.wpColorPicker({
				change: function(event, ui) {
					var color = ui.color.toString();
					originalInput.val(color);
					
					// Update hidden input if exists
					if ($container && $container.length > 0) {
						var $hiddenInput = $container.find('input[type="hidden"]');
						if ($hiddenInput.length > 0) {
							$hiddenInput.val(color);
						}
					}
					
					// Update icon preview if row is provided
					if ($row && $row.length > 0) {
						updateIconPreview($row.find('.paiw-icon-input'));
					}
				},
				clear: function() {
					originalInput.val('');
					if ($container && $container.length > 0) {
						var $hiddenInput = $container.find('input[type="hidden"]');
						if ($hiddenInput.length > 0) {
							$hiddenInput.val('');
						}
					}
				}
			});
			
			// Get container after initialization
			$container = originalInput.closest('.wp-picker-container');
			
			// Handle color picker open/close for z-index management
			if ($container.length > 0) {
				// When color picker opens, add active class and remove from others
				$container.on('click', '.wp-color-result', function(e) {
					e.stopPropagation();
					// Remove active class from all other pickers
					$('.wp-picker-container').not($container).removeClass('active');
					// Add active class to this picker
					setTimeout(function() {
						$container.addClass('active');
					}, 10);
				});
			}
			
			// Ensure color picker UI is interactive
			setTimeout(function() {
				if ($container && $container.length > 0) {
					var $picker = $container.find('.iris-picker');
					if ($picker.length > 0) {
						$picker.css({
							'pointer-events': 'auto'
						});
						$picker.find('.iris-square, .iris-slider, .iris-slider-offset, .iris-square-value').css({
							'pointer-events': 'auto',
							'cursor': 'pointer'
						});
					}
				}
			}, 150);
			
			$input.data('wp-color-picker-initialized', true);
		} catch (e) {
			// Color picker initialization failed, skip this input.
			return;
		}
	}

		/**
		 * Initialize all color pickers on the page.
		 */
		function initColorPickers() {
			$('.paiw-color-picker').each(function() {
				var $input = $(this);
				var $row = null;
				
				// Find parent row for icon preview updates
				var $parentRow = $input.closest('tr');
				if ($parentRow.length === 0) {
					$parentRow = $input.closest('.paiw-attribute-details').prev('.paiw-attribute-row');
				}
				if ($parentRow.length > 0) {
					$row = $parentRow;
				}
				
				initSingleColorPicker($input, $row);
			});
		}
		
		// Initialize main color pickers first (they're always visible)
		// These are the 3 default color pickers in the general settings
		function initMainColorPickers() {
			$('#paiw_bg_color, #paiw_text_color, #paiw_border_color').each(function() {
				var $input = $(this);
				if (!$input.data('wp-color-picker-initialized')) {
					initSingleColorPicker($input, null);
				}
			});
		}

		// Handle clicking outside color pickers to close them
		$(document).on('click', function(e) {
			if (!$(e.target).closest('.wp-picker-container').length) {
				$('.wp-picker-container').removeClass('active');
			}
		});

		// Initialize after WordPress color picker is fully loaded
		// Use multiple attempts to ensure all pickers are initialized
		setTimeout(function() {
			// Initialize main color pickers first
			initMainColorPickers();
			// Then initialize all other color pickers
			initColorPickers();
			// Second attempt for any that might have been missed
			setTimeout(function() {
				initMainColorPickers();
				initColorPickers();
			}, 100);
		}, 300);

		// Expand/Collapse functionality
		$('.paiw-expand-btn').on('click', function() {
			var $btn = $(this);
			var $row = $btn.closest('tr');
			var $details = $row.next('.paiw-attribute-details');
			
			if ($details.is(':visible')) {
				$details.slideUp(200);
				$btn.removeClass('active');
			} else {
				$details.slideDown(200, function() {
					// Initialize color pickers in expanded area
					// Remove existing initialization data first
					$details.find('.paiw-color-picker').removeData('wp-color-picker-initialized');
					// Initialize all color pickers in the expanded area
					setTimeout(function() {
						$details.find('.paiw-color-picker').each(function() {
							var $input = $(this);
							if (!$input.prop('disabled')) {
								initSingleColorPicker($input, $row);
							}
						});
					}, 100);
				});
				$btn.addClass('active');
			}
		});

		// Toggle All button
		$('.paiw-toggle-all').on('click', function() {
			var $btn = $(this);
			var state = $btn.data('state');
			var expand = state === 'expand';
			
			$('.paiw-attribute-row').each(function() {
				var $row = $(this);
				var $details = $row.next('.paiw-attribute-details');
				var $expandBtn = $row.find('.paiw-expand-btn');
				
				if (expand) {
					$details.slideDown(200, function() {
						// Remove existing initialization data first
						$details.find('.paiw-color-picker').removeData('wp-color-picker-initialized');
						// Initialize color pickers in expanded area
						setTimeout(function() {
							$details.find('.paiw-color-picker').each(function() {
								var $input = $(this);
								if (!$input.prop('disabled')) {
									initSingleColorPicker($input, $row);
								}
							});
						}, 100);
					});
					$expandBtn.addClass('active');
				} else {
					$details.slideUp(200);
					$expandBtn.removeClass('active');
				}
			});
			
			$btn.data('state', expand ? 'collapse' : 'expand');
			$btn.text(expand ? 'Collapse All' : 'Expand All');
		});

		// Update icon preview when icon input changes
		$('.paiw-icon-input').on('input', function() {
			updateIconPreview($(this));
		});

		/**
		 * Update icon preview display.
		 *
		 * @param {jQuery} $input The icon input element.
		 */
		function updateIconPreview($input) {
			var $row = $input.closest('tr');
			var iconValue = $input.val().trim();
			var $previewInline = $row.find('.paiw-icon-preview-inline');
			var $previewDetails = $row.next('.paiw-attribute-details').find('.paiw-icon-preview [class*="dashicons"]');
			
			if (!iconValue) {
				$previewInline.removeClass().hide();
				$previewDetails.removeClass().parent().hide();
				return;
			}

			// Ensure dashicons have base class
			var iconParts = iconValue.split(' ');
			if (iconParts[0].match(/^dashicons-/)) {
				if (iconParts.indexOf('dashicons') === -1) {
					iconParts.unshift('dashicons');
					iconValue = iconParts.join(' ');
				}
			}

			// Update inline preview
			$previewInline.attr('class', 'paiw-icon-preview-inline ' + iconValue).show();

			// Update details preview
			if ($previewDetails.length) {
				$previewDetails.attr('class', iconValue);
				$previewDetails.closest('.paiw-icon-preview').show();
			}
		}

		// Initialize previews on page load
		$('.paiw-icon-input').each(function() {
			if ($(this).val()) {
				updateIconPreview($(this));
			}
		});

		// Handle custom colors checkbox toggle
		$(document).on('change', '.paiw-use-custom-colors', function() {
			var $checkbox = $(this);
			var $details = $checkbox.closest('.paiw-attribute-details');
			var $colorControls = $details.find('.paiw-color-controls');
			var $colorInputs = $colorControls.find('.paiw-color-picker');
			var isChecked = $checkbox.is(':checked');
			var $row = $details.prev('.paiw-attribute-row');
			
			if (isChecked) {
				// Enable color controls
				$colorControls.css({
					'opacity': '1',
					'pointer-events': 'auto'
				});
				$colorInputs.prop('disabled', false);
				
				// Re-initialize color pickers that were disabled
				// Remove existing initialization data so they can be re-initialized
				$colorInputs.removeData('wp-color-picker-initialized');
				// Destroy existing color pickers if any
				$colorInputs.each(function() {
					var $input = $(this);
					var $container = $input.closest('.wp-picker-container');
					if ($container.length > 0) {
						$container.remove();
					}
				});
				
				// Initialize color pickers after a short delay
				setTimeout(function() {
					$colorInputs.each(function() {
						initSingleColorPicker($(this), $row);
					});
				}, 200);
			} else {
				// Disable color controls
				$colorControls.css({
					'opacity': '0.5',
					'pointer-events': 'none'
				});
				$colorInputs.prop('disabled', true);
			}
		});

		// Ensure color picker values are synced before form submission
		$('form').on('submit', function(e) {
			// Sync all color picker values before submission
			$('.paiw-color-picker').each(function() {
				var $input = $(this);
				var $container = $input.closest('.wp-picker-container');
				
				if ($container.length > 0) {
					// Get the color from the color picker button
					var $colorButton = $container.find('.wp-color-result');
					if ($colorButton.length > 0) {
						// Check data attribute first (most reliable)
						if ($colorButton.data('current-color')) {
							$input.val($colorButton.data('current-color'));
						} else {
							// Fallback to background-color style
							var bgColor = $colorButton.css('background-color');
							if (bgColor && bgColor !== 'rgba(0, 0, 0, 0)' && bgColor !== 'transparent') {
								var hexColor = rgbToHex(bgColor);
								if (hexColor) {
									$input.val(hexColor);
								}
							}
						}
					}
					
					// Check hidden input if it exists
					var $hiddenInput = $container.find('input[type="hidden"]');
					if ($hiddenInput.length > 0 && $hiddenInput.val()) {
						$input.val($hiddenInput.val());
					}
				}
			});
		});
		
		/**
		 * Convert RGB color to hex format.
		 *
		 * @param {string} rgb RGB color string.
		 * @return {string|null} Hex color or null if invalid.
		 */
		function rgbToHex(rgb) {
			if (!rgb || rgb.indexOf('rgb') === -1) {
				return rgb; // Already hex or invalid
			}
			
			var result = rgb.match(/\d+/g);
			if (!result || result.length < 3) {
				return null;
			}
			
			function componentToHex(c) {
				var hex = parseInt(c, 10).toString(16);
				return hex.length === 1 ? '0' + hex : hex;
			}
			
			return '#' + componentToHex(result[0]) + componentToHex(result[1]) + componentToHex(result[2]);
		}
	});

})(jQuery);
