<?php
/**
 * Plugin Uninstall Handler 
 * 
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

if (!defined('WP_UNINSTALL_PLUGIN')) {
    exit;
}

$delete_data = get_option('pwcp_uninstall_delete_data', 'no');

if ($delete_data === 'yes') {
    pwcp_delete_all_data();
} else {
    pwcp_preserve_data_cleanup();
}

/**
 * Delete all plugin data
 */
function pwcp_delete_all_data() {
    global $wpdb;
    
    // Remove database tables
    $table_suffixes = array(
        'pwcp_calculators',
        'pwcp_calculator_fields',
        'pwcp_calculation_logs',
        'pwcp_calculator_assignments'
    );
    
    foreach ($table_suffixes as $suffix) {
        $table_name = $wpdb->prefix . $suffix;
        
        $safe_table = esc_sql( $table_name );
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name from $wpdb->prefix, DDL statement
        $wpdb->query( "DROP TABLE IF EXISTS `{$safe_table}`" );
    }
    
    // Remove options
    $wpdb->query($wpdb->prepare(
        "DELETE FROM {$wpdb->options} 
         WHERE option_name LIKE %s 
         OR option_name LIKE %s",
        $wpdb->esc_like('pwcp_') . '%',
        $wpdb->esc_like('_pwcp_') . '%'
    ));
    
    // Remove post meta
    $wpdb->query($wpdb->prepare(
        "DELETE FROM {$wpdb->postmeta} 
         WHERE meta_key LIKE %s 
         OR meta_key LIKE %s",
        $wpdb->esc_like('pwcp_') . '%',
        $wpdb->esc_like('_pwcp_') . '%'
    ));
    
    // Remove user meta
    $wpdb->query($wpdb->prepare(
        "DELETE FROM {$wpdb->usermeta} 
         WHERE meta_key LIKE %s",
        $wpdb->esc_like('pwcp_') . '%'
    ));
    
    // Check if termmeta table exists
    $termmeta_exists = $wpdb->get_var($wpdb->prepare(
        "SHOW TABLES LIKE %s",
        $wpdb->esc_like($wpdb->termmeta)
    ));
    
    // Remove term meta
    if ($termmeta_exists === $wpdb->termmeta) {
        $wpdb->query($wpdb->prepare(
            "DELETE FROM {$wpdb->termmeta} 
             WHERE meta_key LIKE %s",
            $wpdb->esc_like('pwcp_') . '%'
        ));
    }
    
    // Remove transients
    $wpdb->query($wpdb->prepare(
        "DELETE FROM {$wpdb->options} 
         WHERE option_name LIKE %s 
         OR option_name LIKE %s",
        $wpdb->esc_like('_transient_pwcp_') . '%',
        $wpdb->esc_like('_transient_timeout_pwcp_') . '%'
    ));
    
    // Remove capabilities
    pwcp_remove_all_capabilities();
    
    // Remove upload directory
    $upload_dir = wp_upload_dir();
    $plugin_dirs = array(
        $upload_dir['basedir'] . '/pricewise-calculator-pro/',
        WP_CONTENT_DIR . '/cache/pwcp/',
    );
    
    foreach ($plugin_dirs as $dir) {
        if (is_dir($dir)) {
            pwcp_force_remove_directory($dir);
        }
    }
    
// Remove export files from uploads root
$export_files = glob($upload_dir['basedir'] . '/pwcp-*.json');
if (is_array($export_files)) {
    foreach ($export_files as $file) {
        if (file_exists($file) && is_file($file)) {
            wp_delete_file($file);
        }
    }
}
    
    wp_cache_flush();
}

/**
 * Preserve data cleanup
 */
function pwcp_preserve_data_cleanup() {
    delete_option('pwcp_uninstall_delete_data');
    delete_option('pwcp_uninstall_settings_exported');
    delete_option('pwcp_uninstall_notice_dismissed');
    update_option('pwcp_data_preserved_on', current_time('timestamp'));
    wp_cache_flush();
}

/**
 * Remove capabilities
 */
function pwcp_remove_all_capabilities() {
    $capabilities = array(
        'manage_pwcp_calculators',
        'edit_pwcp_calculators',
        'delete_pwcp_calculators',
        'view_pwcp_analytics',
        'pwcp_temp_access',
        'pwcp_beta_features'
    );
    
    global $wp_roles;
    
    if (!class_exists('WP_Roles')) {
        return;
    }
    
    if (!isset($wp_roles)) {
        $wp_roles = new WP_Roles();
    }
    
    foreach ($wp_roles->roles as $role_name => $role_info) {
        $role = get_role($role_name);
        if ($role) {
            foreach ($capabilities as $cap) {
                if ($role->has_cap($cap)) {
                    $role->remove_cap($cap);
                }
            }
        }
    }
}


function pwcp_force_remove_directory($dir) {
    if (!file_exists($dir)) {
        return true;
    }
    
    if (!is_dir($dir)) {
  if (file_exists($path) && is_file($path)) {
    wp_delete_file($path);
}
        return false;
    }
    
    // Use WordPress filesystem if available
    if (!function_exists('WP_Filesystem')) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }
    
    WP_Filesystem();
    global $wp_filesystem;
    
    if ($wp_filesystem) {
        return $wp_filesystem->rmdir($dir, true);
    }
    
    $files = scandir($dir);
    
    if ($files === false) {
        return false;
    }
    
    $files = array_diff($files, array('.', '..'));
    
    foreach ($files as $file) {
        $path = $dir . DIRECTORY_SEPARATOR . $file;
        
        if (is_dir($path)) {
            pwcp_force_remove_directory($path);
        } else {
         if (file_exists($path) && is_file($path)) {
    wp_delete_file($path);
}
        }
    }
    
return $wp_filesystem ? $wp_filesystem->rmdir($dir, false) : false;
}