<?php
/**
 * Frontend Calculator
 * 
 * File: public/class-pwcp-frontend-calculator.php
 * 
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Load dependencies
require_once PWCP_PLUGIN_DIR . 'includes/class-pwcp-base-ajax-handler.php';
require_once PWCP_PLUGIN_DIR . 'includes/traits/trait-pwcp-security-validation.php';
require_once PWCP_PLUGIN_DIR . 'includes/class-pwcp-input-sanitizer.php';
require_once PWCP_PLUGIN_DIR . 'includes/class-pwcp-response-factory.php';

/**
 * PWCP Frontend Calculator Class
 * 
 * @since 1.0.0
 */
class PWCP_Frontend_Calculator extends PWCP_Base_Ajax_Handler {
	
	use PWCP_Security_Validation_Trait;
	
	/**
	 * Cache group
	 *
	 * @var string
	 */
	private $cache_group = 'pwcp_frontend';
	
	/**
	 * Admin settings cache
	 *
	 * @var array|null
	 */
	private $general_settings = null;
	
	/**
	 * Design settings cache
	 *
	 * @var array|null
	 */
	private $design_settings = null;
	
	/**
	 * Calculation settings cache
	 *
	 * @var array|null
	 */
	private $calculation_settings = null;
	
	/**
	 * Constructor
	 * 
	 * Initialize frontend calculator functionality
	 */
	public function __construct() {
		// AJAX methods called by PWCP_Ajax_Handler router
		
		// Frontend assets and shortcode
		add_action( 'wp_enqueue_scripts', array( $this, 'pwcp_enqueue_frontend_assets' ) );
		add_shortcode( 'pwcp_calculator', array( $this, 'pwcp_calculator_shortcode' ) );
		
		add_action( 'wp_head', array( $this, 'pwcp_output_dynamic_css' ), 100 );
	}
	
	/**
	 * Check if we're in a WooCommerce context
	 *
	 * @return bool
	 */
	private function pwcp_is_wc_context() {
		return function_exists( 'is_shop' )
			&& ( is_product() || is_shop() || is_product_category() );
	}
	
	/**
	 * Load admin settings with caching
	 */
	private function pwcp_load_admin_settings() {
		if ( null === $this->general_settings ) {
			$this->general_settings = get_option( 'pwcp_general_settings', array() );
		}
		
		if ( null === $this->design_settings ) {
			$this->design_settings = get_option( 'pwcp_design_settings', array() );
		}
		
		if ( null === $this->calculation_settings ) {
			$this->calculation_settings = get_option( 'pwcp_calculation_settings', array() );
		}
	}
	
	/**
	 * Unified setting getter
	 *
	 * @param string $type Setting type.
	 * @param string $key Setting key.
	 * @param mixed  $default Default value.
	 * @return mixed
	 */
	private function get_setting( $type, $key, $default = '' ) {
		$this->pwcp_load_admin_settings();
		
		$settings_map = array(
			'general'     => $this->general_settings,
			'design'      => $this->design_settings,
			'calculation' => $this->calculation_settings,
		);
		
		return isset( $settings_map[ $type ][ $key ] ) ? $settings_map[ $type ][ $key ] : $default;
	}
	
	/**
	 * Output dynamic CSS
	 */
public function pwcp_output_dynamic_css() {
    $post = get_post();
    $has_shortcode = $post && has_shortcode( $post->post_content, 'pwcp_calculator' );
    
    $this->pwcp_load_admin_settings();
    
    // THESE ARE THE DEFAULTS - MUST MATCH YOUR ADMIN SETTINGS
    $header_bg_color = $this->get_setting( 'design', 'header_background_color', '#844CA9' );
    $title_color = $this->get_setting( 'design', 'title_color', '#ffffff' );
    $subtitle_color = $this->get_setting( 'design', 'subtitle_color', '#e5e7eb' );
    $background_color = $this->get_setting( 'design', 'background_color', '#ffffff' );
    $text_color = $this->get_setting( 'design', 'text_color', '#333333' );
    $next_button_color = $this->get_setting( 'design', 'next_button_color', '#844CA9' );
    $previous_button_color = $this->get_setting( 'design', 'previous_button_color', '#6b7280' );
    $validation_bg_color = $this->get_setting( 'design', 'validation_background_color', '#e45949' );
    $validation_text_color = $this->get_setting( 'design', 'validation_text_color', '#ffffff' );
    $price_badge_bg_color = $this->get_setting( 'design', 'price_badge_background_color', '#844CA9' );
    $price_badge_text_color = $this->get_setting( 'design', 'price_badge_text_color', '#ffffff' );
    $selection_indicator_color = $this->get_setting( 'design', 'selection_indicator_color', '#6b7280' );
    $selected_option_bg_color = $this->get_setting( 'design', 'selected_option_background_color', '#844CA9' );
    $footer_bg_color = $this->get_setting( 'design', 'footer_background_color', '#e1ccf0' );
    $footer_price_color = $this->get_setting( 'design', 'footer_price_color', '#844CA9' );
    $price_label_color = $this->get_setting( 'design', 'price_label_color', '#844CA9' );
    $footer_text_color = $this->get_setting( 'design', 'footer_text_color', '#844CA9' );
    $progress_bar_color = $this->get_setting( 'design', 'progress_bar_color', '#844CA9' );
    $border_radius = absint( $this->get_setting( 'design', 'border_radius', 12 ) );
    $box_shadow = $this->get_setting( 'design', 'box_shadow', true );
    $animation_effects = $this->get_setting( 'design', 'animation_effects', true );
    $calculator_width = $this->get_setting( 'design', 'calculator_width', 'auto' );
    
		// Build CSS string
		$css = '/* PriceWise Calculator Pro - Dynamic CSS */';
		
		$css .= '.pwcp-calculator-container.pwcp-modern-design {';
		$css .= 'background-color: ' . esc_attr( $background_color ) . ';';
		$css .= 'color: ' . esc_attr( $text_color ) . ';';
		$css .= 'border-radius: ' . esc_attr( $border_radius ) . 'px;';
		if ( 'auto' !== $calculator_width ) {
			$css .= 'width: ' . esc_attr( $calculator_width ) . ';';
			$css .= 'max-width: 100%;';
		}
		if ( $box_shadow ) {
			$css .= 'box-shadow: 0 10px 25px -5px rgba(0, 0, 0, 0.1), 0 8px 10px -6px rgba(0, 0, 0, 0.1);';
		}
		$css .= '}';
		
		$css .= '.pwcp-calculator-header {';
		$css .= 'background: ' . esc_attr( $header_bg_color ) . ';';
		$css .= 'background: linear-gradient(135deg, ' . esc_attr( $header_bg_color ) . ' 0%, ' . esc_attr( $this->pwcp_adjust_color_brightness( $header_bg_color, -10 ) ) . ' 100%);';
		$css .= 'border-radius: ' . esc_attr( $border_radius ) . 'px ' . esc_attr( $border_radius ) . 'px 0 0;';
		$css .= '}';
		
		$css .= '.pwcp-calculator-header .pwcp-calculator-title {';
		$css .= 'color: ' . esc_attr( $title_color ) . ';';
		$css .= '}';
		
		$css .= '.pwcp-calculator-header .pwcp-calculator-description {';
		$css .= 'color: ' . esc_attr( $subtitle_color ) . ';';
		$css .= '}';
		
		$css .= '.pwcp-calculator-container.pwcp-modern-design .pwcp-next-btn,';
$css .= '.pwcp-calculator-container.pwcp-modern-design button.pwcp-next-btn,';
$css .= '.pwcp-calculator-container.pwcp-modern-design button[type="button"].pwcp-next-btn {';
$css .= 'background: ' . esc_attr( $next_button_color ) . ';';
$css .= 'background-color: ' . esc_attr( $next_button_color ) . ';';
$css .= 'background-image: none;';
$css .= 'color: white;';
$css .= 'border: none;';
$css .= 'border-width: 0;';
$css .= 'border-style: none;';
$css .= 'border-radius: ' . esc_attr( $border_radius ) . 'px;';
if ( $animation_effects ) {
    $css .= 'transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);';
}
$css .= '}';
		
		$css .= '.pwcp-calculator-container.pwcp-modern-design .pwcp-prev-btn,';
$css .= '.pwcp-calculator-container.pwcp-modern-design button.pwcp-prev-btn,';
$css .= '.pwcp-calculator-container.pwcp-modern-design button[type="button"].pwcp-prev-btn {';
$css .= 'background: ' . esc_attr( $previous_button_color ) . ';';
$css .= 'background-color: ' . esc_attr( $previous_button_color ) . ';';
$css .= 'background-image: none;';
$css .= 'color: white;';
$css .= 'border: none;';
$css .= 'border-width: 0;';
$css .= 'border-style: none;';
$css .= 'border-radius: ' . esc_attr( $border_radius ) . 'px;';
if ( $animation_effects ) {
    $css .= 'transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);';
}
$css .= '}';
		
		$css .= '.pwcp-calculator-container.pwcp-modern-design .pwcp-next-btn:hover:not(:disabled),';
$css .= '.pwcp-calculator-container.pwcp-modern-design button.pwcp-next-btn:hover:not(:disabled),';
$css .= '.pwcp-calculator-container.pwcp-modern-design button[type="button"].pwcp-next-btn:hover:not(:disabled) {';
$css .= 'background: ' . esc_attr( $this->pwcp_adjust_color_brightness( $next_button_color, -10 ) ) . ';';
$css .= 'background-color: ' . esc_attr( $this->pwcp_adjust_color_brightness( $next_button_color, -10 ) ) . ';';
$css .= 'background-image: none;';
if ( $animation_effects ) {
    $css .= 'transform: translateY(-2px);';
    $css .= 'box-shadow: 0 8px 25px rgba(' . esc_attr( $this->pwcp_hex_to_rgb( $next_button_color ) ) . ', 0.4);';
}
$css .= '}';
		
		$css .= '.pwcp-calculator-container.pwcp-modern-design .pwcp-prev-btn:hover:not(:disabled),';
$css .= '.pwcp-calculator-container.pwcp-modern-design button.pwcp-prev-btn:hover:not(:disabled),';
$css .= '.pwcp-calculator-container.pwcp-modern-design button[type="button"].pwcp-prev-btn:hover:not(:disabled) {';
$css .= 'background: ' . esc_attr( $this->pwcp_adjust_color_brightness( $previous_button_color, -10 ) ) . ';';
$css .= 'background-color: ' . esc_attr( $this->pwcp_adjust_color_brightness( $previous_button_color, -10 ) ) . ';';
$css .= 'background-image: none;';
if ( $animation_effects ) {
    $css .= 'transform: translateY(-2px);';
    $css .= 'box-shadow: 0 8px 25px rgba(' . esc_attr( $this->pwcp_hex_to_rgb( $previous_button_color ) ) . ', 0.4);';
}
$css .= '}';
		
		$css .= '.pwcp-btn-primary {';
		$css .= 'background: ' . esc_attr( $next_button_color ) . ';';
		$css .= 'border-radius: ' . esc_attr( $border_radius ) . 'px;';
		if ( $animation_effects ) {
			$css .= 'transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);';
		}
		$css .= '}';
		
		$css .= '.pwcp-btn-primary:hover {';
		$css .= 'background: ' . esc_attr( $this->pwcp_adjust_color_brightness( $next_button_color, -10 ) ) . ';';
		if ( $animation_effects ) {
			$css .= 'transform: translateY(-2px);';
			$css .= 'box-shadow: 0 12px 20px -5px rgba(' . esc_attr( $this->pwcp_hex_to_rgb( $next_button_color ) ) . ', 0.4);';
		}
		$css .= '}';
		
		$css .= '.pwcp-number-input {';
		$css .= 'border-radius: ' . esc_attr( $border_radius ) . 'px;';
		$css .= 'color: ' . esc_attr( $text_color ) . ';';
		$css .= '}';
		
		$css .= '.pwcp-step-counter {';
		$css .= 'color: ' . esc_attr( $header_bg_color ) . ';';
		$css .= '}';

		$css .= '.pwcp-calculator-container.pwcp-modern-design .pwcp-prev-btn:disabled,';
		$css .= '.pwcp-calculator-container.pwcp-modern-design button.pwcp-prev-btn:disabled,';
		$css .= '.pwcp-calculator-container.pwcp-modern-design button[type="button"].pwcp-prev-btn:disabled {';
		$css .= 'background: rgba(' . esc_attr( $this->pwcp_hex_to_rgb( $previous_button_color ) ) . ', 0.4);'; 
		$css .= 'background-color: rgba(' . esc_attr( $this->pwcp_hex_to_rgb( $previous_button_color ) ) . ', 0.4);';
		$css .= 'background-image: none;';
		$css .= 'border-color: rgba(' . esc_attr( $this->pwcp_hex_to_rgb( $previous_button_color ) ) . ', 0.3);';  
		$css .= '}';

		$css .= '.pwcp-calculator-container.pwcp-modern-design .pwcp-next-btn:disabled,';
		$css .= '.pwcp-calculator-container.pwcp-modern-design button.pwcp-next-btn:disabled,';
		$css .= '.pwcp-calculator-container.pwcp-modern-design button[type="button"].pwcp-next-btn:disabled {';
		$css .= 'background: rgba(' . esc_attr( $this->pwcp_hex_to_rgb( $next_button_color ) ) . ', 0.4);'; 
		$css .= 'background-color: rgba(' . esc_attr( $this->pwcp_hex_to_rgb( $next_button_color ) ) . ', 0.4);';
		$css .= '}';
		
		$css .= '.pwcp-price-amount, #calculatedPrice {';
		$css .= 'color: ' . esc_attr( $next_button_color ) . ';';
		$css .= 'font-weight: 700;';
		$css .= '}';
		
		$css .= '.pwcp-validation-message.show {';
		$css .= 'background: ' . esc_attr( $validation_bg_color ) . ';';
		$css .= 'background: linear-gradient(135deg, ' . esc_attr( $validation_bg_color ) . ' 0%, ' . esc_attr( $this->pwcp_adjust_color_brightness( $validation_bg_color, -20 ) ) . ' 100%);';
		$css .= 'border-radius: ' . esc_attr( $border_radius ) . 'px;';
		$css .= 'color: ' . esc_attr( $validation_text_color ) . ';';
		$css .= 'text-shadow: 0 1px 2px rgba(0, 0, 0, 0.2);';
		$css .= 'border-left: 4px solid ' . esc_attr( $this->pwcp_adjust_color_brightness( $validation_bg_color, -40 ) ) . ';';
		$css .= 'padding: 12px 16px;';
		$css .= 'margin: 16px 0;';
		if ( $box_shadow ) {
			$css .= 'box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);';
		}
		$css .= '}';
		
		$css .= '.pwcp-validation-message .pwcp-validation-text {';
		$css .= 'display: flex;';
		$css .= 'align-items: center;';
		$css .= 'gap: 8px;';
		$css .= 'font-weight: 500;';
		$css .= 'font-size: 14px;';
		$css .= 'color: ' . esc_attr( $validation_text_color ) . ';';
		$css .= '}';
		
		$css .= '.pwcp-footer-section {';
		$css .= 'background: ' . esc_attr( $footer_bg_color ) . ';';
		$css .= 'border-radius: 0 0 ' . esc_attr( $border_radius ) . 'px ' . esc_attr( $border_radius ) . 'px;';
		$css .= 'padding: 20px;';
		$css .= 'margin-top: 0;';
		if ( $box_shadow ) {
			$css .= 'box-shadow: inset 0 1px 0 0 rgba(255, 255, 255, 0.1), 0 1px 3px rgba(0, 0, 0, 0.05);';
		}
		$css .= '}';
		
		$css .= '.pwcp-footer-section .pwcp-price-amount, .pwcp-footer-section #calculatedPrice {';
		$css .= 'color: ' . esc_attr( $footer_price_color ) . ';';
		$css .= 'font-weight: 700;';
		$css .= '}';
		
		$css .= '.pwcp-price-indicator {';
		$css .= 'color: ' . esc_attr( $price_label_color ) . ';';
		$css .= '}';
		
		$css .= '.pwcp-progress-fill {';
		$css .= 'background: ' . esc_attr( $progress_bar_color ) . ';';
		$css .= '}';
		
		$css .= '.pwcp-progress-text, .pwcp-progress-subtext, .pwcp-completed-fields, .pwcp-required-fields, .pwcp-remaining-fields {';
		$css .= 'color: ' . esc_attr( $footer_text_color ) . ';';
		$css .= '}';
		
		$css .= '.pwcp-progress-text strong {';
		$css .= 'color: ' . esc_attr( $footer_text_color ) . ';';
		$css .= '}';
		
		if ( $animation_effects ) {
			$css .= '.pwcp-field-step {';
			$css .= 'transition: all 0.5s cubic-bezier(0.4, 0, 0.2, 1);';
			$css .= '}';
			
			$css .= '.pwcp-validation-message {';
			$css .= 'transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);';
			$css .= '}';
			
			$css .= '.pwcp-footer-section {';
			$css .= 'transition: background-color 0.3s ease;';
			$css .= '}';
		}
		
		$css .= '.pwcp-calculator-container, .pwcp-field-question, .pwcp-field-description {';
		$css .= 'color: ' . esc_attr( $text_color ) . ';';
		$css .= '}';
		
		$css .= '.pwcp-field-step {';
		$css .= 'background: ' . esc_attr( $background_color ) . ';';
		$css .= '}';
		
		$css .= '@media (max-width: 768px) {';
		$css .= '.pwcp-calculator-container {';
		$css .= 'width: 100%;';
		$css .= 'margin: 0;';
		$css .= '}';
		$css .= '.pwcp-footer-section {';
		$css .= 'border-radius: ' . esc_attr( max( 4, $border_radius - 4 ) ) . 'px;';
		$css .= 'padding: 16px;';
		$css .= '}';
		$css .= '.pwcp-validation-message.show {';
		$css .= 'border-radius: ' . esc_attr( max( 4, $border_radius - 4 ) ) . 'px;';
		$css .= 'padding: 10px 12px;';
		$css .= 'margin: 12px 0;';
		$css .= '}';
		$css .= '}';
	
		$css .= '.pwcp-footer-section {';
		if ( '#f8fafc' === $footer_bg_color ) {
			$css .= 'background: #374151;';
		}
		$css .= '}';
		
		
		wp_register_style( 'pwcp-dynamic-inline', false );
		wp_enqueue_style( 'pwcp-dynamic-inline' );
		wp_add_inline_style( 'pwcp-dynamic-inline', $css );
	}
	
	/**
	 * Adjust color brightness
	 *
	 * @param string $hex_color Hex color.
	 * @param int    $percent Percentage.
	 * @return string
	 */
	private function pwcp_adjust_color_brightness( $hex_color, $percent ) {
		$hex_color = ltrim( $hex_color, '#' );
		
		if ( 6 !== strlen( $hex_color ) ) {
			return $hex_color;
		}
		
		$r = hexdec( substr( $hex_color, 0, 2 ) );
		$g = hexdec( substr( $hex_color, 2, 2 ) );
		$b = hexdec( substr( $hex_color, 4, 2 ) );
		
		$r = max( 0, min( 255, $r + ( $r * $percent / 100 ) ) );
		$g = max( 0, min( 255, $g + ( $g * $percent / 100 ) ) );
		$b = max( 0, min( 255, $b + ( $b * $percent / 100 ) ) );
		
		return '#' . sprintf( '%02x%02x%02x', $r, $g, $b );
	}
	
	/**
	 * Convert hex to RGB values
	 *
	 * @param string $hex_color Hex color.
	 * @return string
	 */
	private function pwcp_hex_to_rgb( $hex_color ) {
		$hex_color = ltrim( $hex_color, '#' );
		
		if ( 6 !== strlen( $hex_color ) ) {
			return '0, 0, 0';
		}
		
		$r = hexdec( substr( $hex_color, 0, 2 ) );
		$g = hexdec( substr( $hex_color, 2, 2 ) );
		$b = hexdec( substr( $hex_color, 4, 2 ) );
		
		return "$r, $g, $b";
	}
	
	/**
	 * AJAX handler for instant calculation
	 * 
	 * Called by PWCP_Ajax_Handler router
	 */
	public function pwcp_ajax_calculate_instant() {
		// Verify nonce using base class helper
		$this->verify_public_ajax_security( 'pwcp_calculator_action', 'pwcp_nonce' );
		
		try {
			$this->pwcp_load_admin_settings();
			$enable_live = $this->get_setting( 'calculation', 'enable_live_calculation', true );
			
			if ( ! $enable_live && ! isset( $_POST['explicit_calculation'] ) ) {
				PWCP_Response_Factory::calculation_success(
					0,
					array(),
					0,
					array(
						'formatted_price' => $this->pwcp_format_price_with_settings( 0 ),
						'live_disabled'   => true,
						'message'         => $this->get_setting( 'general', 'loading_message', __( 'Please click calculate to see pricing', 'pricewise-calculator-pro' ) ),
					)
				);
				wp_die();
			}
			
			$calculator_id = isset( $_POST['calculator_id'] ) ? absint( $_POST['calculator_id'] ) : 0;
			$product_id = isset( $_POST['product_id'] ) ? absint( $_POST['product_id'] ) : 0;
			$field_values_raw = isset( $_POST['field_values'] ) && is_array( $_POST['field_values'] ) ? wp_unslash( $_POST['field_values'] ) : array();
			
			if ( ! $calculator_id ) {
				PWCP_Response_Factory::calculation_success(
					0,
					array(),
					0,
					array(
						'formatted_price' => $this->pwcp_format_price_with_settings( 0 ),
						'is_ready_state'  => true,
						'message'         => __( 'Ready to start - select your options above', 'pricewise-calculator-pro' ),
					)
				);
				wp_die();
			}
			
			$calculator = $this->pwcp_get_calculator( $calculator_id );
			if ( ! $calculator ) {
				PWCP_Response_Factory::calculation_success(
					0,
					array(),
					$calculator_id,
					array(
						'formatted_price' => $this->pwcp_format_price_with_settings( 0 ),
						'is_ready_state'  => true,
						'message'         => __( 'Calculator configuration issue', 'pricewise-calculator-pro' ),
					)
				);
				wp_die();
			}
			
			$fields = $this->pwcp_get_calculator_fields( $calculator_id );
			if ( empty( $fields ) ) {
				PWCP_Response_Factory::calculation_success(
					0,
					array(),
					$calculator_id,
					array(
						'formatted_price' => $this->pwcp_format_price_with_settings( 0 ),
						'is_ready_state'  => true,
						'message'         => __( 'Select your options to see pricing', 'pricewise-calculator-pro' ),
					)
				);
				wp_die();
			}
			
			$sanitized_values = PWCP_Input_Sanitizer::sanitize_calculation_values(
				wp_json_encode( $field_values_raw )
			);
			
			$calculation_result = $this->pwcp_calculate_formula_live( $calculator->formula, $sanitized_values, $fields );
			
			$base_price = 0;
			if ( $product_id && function_exists( 'wc_get_product' ) ) {
				$product = wc_get_product( $product_id );
				if ( $product ) {
					$base_price = floatval( $product->get_price() );
				}
			}
			
			$final_price = max( 0, $calculation_result );
			$formatted_price = $this->pwcp_format_price_with_settings( $final_price );
			
			$selection_count = count( array_filter( $sanitized_values ) );
			$message = $this->pwcp_get_live_calculation_message( $selection_count, $final_price );
			
			$this->pwcp_log_calculation( $calculator_id, $product_id, $sanitized_values, $final_price, $base_price );
			
			PWCP_Response_Factory::calculation_success(
				$final_price,
				$sanitized_values,
				$calculator_id,
				array(
					'formatted_price'        => $formatted_price,
					'base_price'             => $base_price,
					'selection_count'        => $selection_count,
					'is_live_calculation'    => true,
					'calculation_breakdown'  => $this->pwcp_get_calculation_breakdown( $calculator->formula, $sanitized_values ),
					'message'                => $message,
				)
			);
			
		} catch ( Exception $e ) {
			PWCP_Response_Factory::calculation_success(
				0,
				array(),
				0,
				array(
					'formatted_price' => $this->pwcp_format_price_with_settings( 0 ),
					'is_ready_state'  => true,
					'error_handled'   => true,
					'message'         => __( 'Keep selecting options to build your quote', 'pricewise-calculator-pro' ),
				)
			);
		}
		
		wp_die();
	}
	
	/**
	 * Format price using calculation settings
	 *
	 * @param float $price Price value.
	 * @return string
	 */
	private function pwcp_format_price_with_settings( $price ) {
		$currency_position = $this->get_setting( 'calculation', 'currency_position', 'left' );
		$decimal_places = absint( $this->get_setting( 'calculation', 'decimal_places', 2 ) );
		$thousand_separator = $this->get_setting( 'calculation', 'thousand_separator', ',' );
		$decimal_separator = $this->get_setting( 'calculation', 'decimal_separator', '.' );
		
		$currency_symbol = $this->get_setting( 'calculation', 'currency_symbol', '' );
		
		if ( empty( $currency_symbol ) ) {
			if ( function_exists( 'get_woocommerce_currency_symbol' ) ) {
				$currency_symbol = html_entity_decode( get_woocommerce_currency_symbol(), ENT_QUOTES, 'UTF-8' );
			} else {
				$currency_symbol = '$';
			}
		}
		
		$formatted_number = number_format( $price, $decimal_places, $decimal_separator, $thousand_separator );
		
		switch ( $currency_position ) {
			case 'right':
				return $formatted_number . $currency_symbol;
			case 'left_space':
				return $currency_symbol . ' ' . $formatted_number;
			case 'right_space':
				return $formatted_number . ' ' . $currency_symbol;
			case 'left':
			default:
				return $currency_symbol . $formatted_number;
		}
	}
	
	/**
	 * Get message for live calculation
	 *
	 * @param int   $selection_count Selection count.
	 * @param float $price Price.
	 * @return string
	 */
	private function pwcp_get_live_calculation_message( $selection_count, $price ) {
		$loading_message = $this->get_setting( 'general', 'loading_message', __( 'Calculating...', 'pricewise-calculator-pro' ) );
		
		if ( 0 === $selection_count ) {
			return __( 'Ready to start - enter your values above', 'pricewise-calculator-pro' );
		} elseif ( 1 === $selection_count ) {
			return __( 'Great start! Keep adding values', 'pricewise-calculator-pro' );
		} elseif ( 2 === $selection_count ) {
			return $loading_message;
		} elseif ( $price > 0 ) {
			return __( 'Live price updates as you enter values', 'pricewise-calculator-pro' );
		} else {
			return __( 'Add more values to see pricing', 'pricewise-calculator-pro' );
		}
	}
	
	/**
	 * Calculate price handler (legacy method name - calls instant calculation)
	 */
	public function pwcp_ajax_calculate_price() {
		$this->pwcp_ajax_calculate_instant();
	}
	
	/**
	 * Enqueue frontend assets
	 */
public function pwcp_enqueue_frontend_assets() {
    if (!class_exists('WooCommerce')) {
        return;
    }
    
    if (is_admin()) {
        return;
    }
    
    $this->pwcp_load_admin_settings();
		wp_enqueue_style(
			'pwcp-modern-calculator',
			PWCP_ASSETS_URL . 'css/all-frontend-css/frontend-slider.css',
			array(),
			PWCP_PLUGIN_VERSION,
			'all'
		);
		
		
		wp_enqueue_script( 'jquery' );
		
		wp_enqueue_script(
			'pwcp-modern-calculator',
			PWCP_ASSETS_URL . 'js/all-frontend-js/frontend-slider.js',
			array( 'jquery', 'wp-i18n' ),
			PWCP_PLUGIN_VERSION,
			true
		);
		
		// Check if translation function exists
		if ( function_exists( 'wp_set_script_translations' ) ) {
			wp_set_script_translations(
				'pwcp-modern-calculator',
				'pricewise-calculator-pro',
				plugin_dir_path( dirname( __FILE__ ) ) . 'languages'
			);
		}
		
		$currency_symbol = '$';
		if ( function_exists( 'get_woocommerce_currency_symbol' ) ) {
			$currency_symbol = html_entity_decode( get_woocommerce_currency_symbol(), ENT_QUOTES, 'UTF-8' );
		}
		
		wp_localize_script(
			'pwcp-modern-calculator',
			'pwcpFrontend',
			array(
				'ajax_url'                => esc_url( admin_url( 'admin-ajax.php' ) ),
				'nonce'                   => wp_create_nonce( 'pwcp_calculator_action' ),
				'currency_symbol'         => esc_js( $this->get_setting( 'calculation', 'currency_symbol', '$' ) ),
				'currency_position'       => esc_js( $this->get_setting( 'calculation', 'currency_position', 'left' ) ),
				'decimal_separator'       => esc_js( $this->get_setting( 'calculation', 'decimal_separator', '.' ) ),
				'thousand_separator'      => esc_js( $this->get_setting( 'calculation', 'thousand_separator', ',' ) ),
				'decimal_places'          => absint( $this->get_setting( 'calculation', 'decimal_places', 2 ) ),
				'live_calculation'        => (bool) $this->get_setting( 'calculation', 'enable_live_calculation', true ),
				'add_to_cart_requirement' => sanitize_text_field( $this->get_setting( 'general', 'add_to_cart_requirement', 'all_required' ) ),
				'strings'                 => array(
					'calculating'                    => esc_js( $this->get_setting( 'general', 'loading_message', __( 'Calculating...', 'pricewise-calculator-pro' ) ) ),
					'calculate'                      => esc_html__( 'Calculate', 'pricewise-calculator-pro' ),
					'error'                          => esc_js( $this->get_setting( 'general', 'error_message', __( 'Calculation error occurred', 'pricewise-calculator-pro' ) ) ),
					'required_fields'                => esc_html__( 'Please fill in all required fields', 'pricewise-calculator-pro' ),
					'add_to_cart'                    => esc_js( $this->get_setting( 'general', 'add_to_cart_text', __( 'Add to Cart', 'pricewise-calculator-pro' ) ) ),
					'reset'                          => esc_html__( 'Reset', 'pricewise-calculator-pro' ),
					'prev'                           => esc_html__( '← PREV', 'pricewise-calculator-pro' ),
					'next'                           => esc_html__( 'NEXT →', 'pricewise-calculator-pro' ),
					'complete'                       => esc_html__( 'Complete', 'pricewise-calculator-pro' ),
					'validation_message'             => esc_html__( 'We need this to calculate your price', 'pricewise-calculator-pro' ),
					'required_field_indicator'       => esc_js( $this->get_setting( 'general', 'required_field_text', '*' ) ),
					'timeout'                        => esc_html__( 'Calculation timed out. Please try again.', 'pricewise-calculator-pro' ),
					'must_complete_required'         => esc_html__( 'Please complete the required fields before adding to cart.', 'pricewise-calculator-pro' ),
					'must_complete_calculator'       => esc_html__( 'Please complete the calculator to get your price before adding to cart.', 'pricewise-calculator-pro' ),
					'field_required'                 => esc_html__( 'This field is required', 'pricewise-calculator-pro' ),
					'tooltip_always_enabled'         => esc_html__( 'Add to Cart is always enabled', 'pricewise-calculator-pro' ),
					'tooltip_all_required_done'      => esc_html__( 'All required fields complete', 'pricewise-calculator-pro' ),
					/* translators: %d: Number of required fields */
					'tooltip_complete_required_x'    => esc_html__( 'Complete all %d required fields to enable this', 'pricewise-calculator-pro' ),
					'tooltip_complete_required_1'    => esc_html__( 'Complete the required field to enable this', 'pricewise-calculator-pro' ),
					'tooltip_all_fields_done'        => esc_html__( 'All fields complete', 'pricewise-calculator-pro' ),
					/* translators: %d: Number of fields */
					'tooltip_complete_fields_x'      => esc_html__( 'Complete all %d fields to enable this', 'pricewise-calculator-pro' ),
					'tooltip_complete_field_1'       => esc_html__( 'Complete the field to enable this', 'pricewise-calculator-pro' ),
					'tooltip_complete_first_1'       => esc_html__( 'Complete the first field to enable this', 'pricewise-calculator-pro' ),
					/* translators: %d: Number of fields */
					'tooltip_complete_first_x'       => esc_html__( 'Complete first %d fields to enable this', 'pricewise-calculator-pro' ),
					/* translators: %d: Number of fields */
					'tooltip_first_x_done'           => esc_html__( 'First %d field complete', 'pricewise-calculator-pro' ),
					/* translators: %d: Number of fields */
					'tooltip_first_x_plural_done'    => esc_html__( 'First %d fields complete', 'pricewise-calculator-pro' ),
				),
			)
		);
	}
	
	/**
	 * Render calculator for display
	 *
	 * @param object      $calculator Calculator object.
	 * @param object|null $product Product object.
	 */
	public function pwcp_render_calculator( $calculator, $product = null ) {
		if ( ! $calculator ) {
			return;
		}
		
		$this->pwcp_load_admin_settings();
		
		$calculator_id = absint( $calculator->id );
		$product_id = $product ? $product->get_id() : 0;
		
		if ( ! class_exists( 'PWCP_Database' ) ) {
			return;
		}
		
		$fields = PWCP_Database::pwcp_get_calculator_fields( $calculator_id );
		
		if ( empty( $fields ) ) {
			echo '<div class="pwcp-calculator-notice">' . esc_html__( 'This calculator has no fields configured yet.', 'pricewise-calculator-pro' ) . '</div>';
			return;
		}
		
		$settings = array();
		if ( ! empty( $calculator->settings ) ) {
			$decoded_settings = json_decode( $calculator->settings, true );
			if ( is_array( $decoded_settings ) ) {
				$settings = $decoded_settings;
			}
		}
		
		$enable_live = $this->get_setting( 'calculation', 'enable_live_calculation', true );
		$current_step = 1;
		$total_steps = count( $fields );
		
		$required_fields_count = 0;
		$total_fields_count = count( $fields );
		
		foreach ( $fields as $field ) {
			if ( ! empty( $field->field_required ) ) {
				++$required_fields_count;
			}
		}
		
		$required_indicator = $this->get_setting( 'general', 'required_field_text', '*' );
		
		$currency_symbol = $this->get_setting( 'calculation', 'currency_symbol', '' );
		
		if ( empty( $currency_symbol ) ) {
			if ( function_exists( 'get_woocommerce_currency_symbol' ) ) {
				$currency_symbol = html_entity_decode( get_woocommerce_currency_symbol(), ENT_QUOTES, 'UTF-8' );
			} else {
				$currency_symbol = '$';
			}
		}
		
		$initial_progress_width = ( 1 / $total_steps ) * 100;
		
		?>
		<div class="pwcp-calculator-container pwcp-modern-design" 
			 data-calculator-id="<?php echo esc_attr( $calculator_id ); ?>" 
			 data-product-id="<?php echo esc_attr( $product_id ); ?>"
			 data-multi-step="true"
			 data-live-calculation="<?php echo esc_attr( $enable_live ? 'true' : 'false' ); ?>">
			
			<div class="pwcp-calculator-header">
				<div class="pwcp-calculator-title"><?php echo esc_html( $calculator->name ); ?></div>
				<?php if ( ! empty( $calculator->description ) ) : ?>
					<div class="pwcp-calculator-description"><?php echo esc_html( $calculator->description ); ?></div>
				<?php endif; ?>
			</div>
			
			<div class="pwcp-step-progress-bar">
				<div class="pwcp-step-progress-fill" data-initial-width="<?php echo esc_attr( $initial_progress_width ); ?>"></div>
			</div>
			
			<div class="pwcp-nav-header">
				<button type="button" class="pwcp-prev-btn" disabled>
					<?php esc_html_e( '← PREV', 'pricewise-calculator-pro' ); ?>
				</button>
				<div class="pwcp-step-counter">
					<span class="pwcp-current-step">1</span> <?php esc_html_e( 'of', 'pricewise-calculator-pro' ); ?> <span class="pwcp-total-steps"><?php echo esc_html( $total_steps ); ?></span>
				</div>
				<button type="button" class="pwcp-next-btn">
					<?php esc_html_e( 'NEXT →', 'pricewise-calculator-pro' ); ?>
				</button>
			</div>
			
			<div class="pwcp-form-section">
				<form class="pwcp-calculator-form pwcp-modern-form" data-calculator-id="<?php echo esc_attr( $calculator_id ); ?>">
					<?php wp_nonce_field( 'pwcp_calculator_action', 'pwcp_nonce' ); ?>
					<input type="hidden" name="calculator_id" value="<?php echo esc_attr( $calculator_id ); ?>">
					<input type="hidden" name="product_id" value="<?php echo esc_attr( $product_id ); ?>">
					<input type="hidden" id="pwcp-calculated-price-field" name="calculated_price" value="0">
					
					<div class="pwcp-calculator-fields">
						<?php foreach ( $fields as $index => $field ) : ?>
							<div class="pwcp-field-step <?php echo $index > 0 ? 'pwcp-hidden' : ''; ?>" 
								 data-step="<?php echo esc_attr( $index + 1 ); ?>" 
								 data-field-key="<?php echo esc_attr( $field->field_key ?? '' ); ?>">
								<?php $this->pwcp_render_field( $field, $required_indicator ); ?>
							</div>
						<?php endforeach; ?>
					</div>
					
					<div class="pwcp-calculator-error pwcp-hidden" role="alert" aria-live="polite">
						<p class="pwcp-error-message"></p>
					</div>
				</form>
			</div>
			
			<div class="pwcp-footer-section">
				<div class="pwcp-price-indicator">
					<span class="pwcp-price-label">
						<?php echo esc_html__( '• Price:', 'pricewise-calculator-pro' ); ?>
					</span>
					<span class="pwcp-price-amount" id="calculatedPrice">
						<?php echo esc_html( $this->pwcp_format_price_with_settings( 0 ) ); ?>
					</span>
				</div>
				
				<?php if ( $required_fields_count > 0 ) : ?>
					<div class="pwcp-progress-bar">
						<div class="pwcp-progress-fill pwcp-progress-fill--zero" data-initial-width="0"></div>
					</div>
					<div class="pwcp-progress-text">
						<strong>
						<span class="pwcp-completed-required">0</span> 
						<?php
						/* translators: %d: Number of required fields */
						echo esc_html( sprintf( _n( 'of %d required field complete', 'of %d required fields complete', $required_fields_count, 'pricewise-calculator-pro' ), $required_fields_count ) );
						?>
						</strong>
					</div>
				<?php endif; ?>
				
				<div class="pwcp-progress-subtext">
					<span class="pwcp-optional-status">
						<?php
						$optional_count = $total_fields_count - $required_fields_count;
						if ( $optional_count > 0 ) {
							/* translators: %d: Number of optional fields */
							echo esc_html( sprintf( _n( '%d optional field remaining', '%d optional fields remaining', $optional_count, 'pricewise-calculator-pro' ), $optional_count ) );
						}
						?>
					</span>
					<br>
					<?php
					if ( $enable_live ) {
						esc_html_e( 'Updates as you enter values', 'pricewise-calculator-pro' );
					} else {
						esc_html_e( 'Click calculate when ready', 'pricewise-calculator-pro' );
					}
					?>
				</div>
			</div>
			
			<script type="application/json" id="pwcp-calculator-config-<?php echo esc_attr( $calculator_id ); ?>">
			<?php
			echo wp_json_encode(
				array(
					'calculator_id'          => $calculator_id,
					'product_id'             => $product_id,
					'formula'                => $calculator->formula,
					'base_price'             => $product ? floatval( $product->get_price() ) : 0,
					'currency_symbol'        => $currency_symbol,
					'currency_position'      => $this->get_setting( 'calculation', 'currency_position', 'left' ),
					'decimal_places'         => absint( $this->get_setting( 'calculation', 'decimal_places', 2 ) ),
					'decimal_separator'      => $this->get_setting( 'calculation', 'decimal_separator', '.' ),
					'thousand_separator'     => $this->get_setting( 'calculation', 'thousand_separator', ',' ),
					'show_result_immediately' => $enable_live,
					'live_calculation'       => $enable_live,
					'multi_step'             => true,
					'total_steps'            => count( $fields ),
					'ajax_url'               => admin_url( 'admin-ajax.php' ),
					'nonce'                  => wp_create_nonce( 'pwcp_calculator_action' ),
					'total_fields'           => $total_fields_count,
					'required_fields_count'  => $required_fields_count,
					'texts'                  => array(
						'loading_message'    => $this->get_setting( 'general', 'loading_message', __( 'Calculating...', 'pricewise-calculator-pro' ) ),
						'add_to_cart_text'   => $this->get_setting( 'general', 'add_to_cart_text', __( 'Add to Cart', 'pricewise-calculator-pro' ) ),
						'error_message'      => $this->get_setting( 'general', 'error_message', __( 'Please check your inputs and try again.', 'pricewise-calculator-pro' ) ),
						'required_field_text' => $required_indicator,
					),
				)
			);
			?>
			</script>
		</div>
		<?php
	}
	
	/**
	 * Render individual field
	 *
	 * @param object $field Field object.
	 * @param string $required_indicator Required indicator.
	 */
	private function pwcp_render_field( $field, $required_indicator = '*' ) {
		$field_key = sanitize_key( $field->field_key ?? '' );
		$field_label = sanitize_text_field( $field->field_label ?? '' );
		$field_type = sanitize_text_field( $field->field_type ?? 'number' );
		$field_required = (bool) ( $field->field_required ?? false );
		$field_default = sanitize_text_field( $field->field_default ?? '' );
		$field_description = sanitize_text_field( $field->field_description ?? '' );
		
		if ( empty( $field_key ) ) {
			return;
		}
		
		$display_label = $field_label;
		if ( $field_required && ! empty( $required_indicator ) ) {
			$display_label .= ' <span class="pwcp-required-indicator">' . esc_html( $required_indicator ) . '</span>';
		}
		
		$show_validation = $field_required;
		
		?>
		<div class="pwcp-field-question"><?php echo wp_kses( $display_label, array( 'span' => array( 'class' => array() ) ) ); ?></div>
		
		<?php if ( ! empty( $field_description ) ) : ?>
			<div class="pwcp-field-description pwcp-text-center">
				<?php echo esc_html( $field_description ); ?>
			</div>
		<?php endif; ?>
		
		<div class="pwcp-field-container">
			<?php
			// Only render number field type
			$this->pwcp_render_modern_number_field( $field_key, $field_default, $field_required, $field );
			?>
			
			<div class="pwcp-validation-message <?php echo $show_validation ? 'show' : ''; ?>" role="alert" aria-live="polite">
				<div class="pwcp-validation-text">
					<span aria-hidden="true">📝 </span>
					<span><?php esc_html_e( 'We need this to calculate your price', 'pricewise-calculator-pro' ); ?></span>
				</div>
			</div>
		</div>
		<?php
	}
	
	/**
	 * Number field
	 *
	 * @param string $field_key Field key.
	 * @param string $default_value Default value.
	 * @param bool   $required Required flag.
	 * @param object $field Field object.
	 */
	private function pwcp_render_modern_number_field( $field_key, $default_value, $required, $field ) {
		$validation = array();
		if ( ! empty( $field->field_validation ) ) {
			$decoded = json_decode( $field->field_validation, true );
			if ( is_array( $decoded ) ) {
				$validation = $decoded;
			}
		}
		
		$min = isset( $validation['min'] ) ? floatval( $validation['min'] ) : '';
		$max = isset( $validation['max'] ) ? floatval( $validation['max'] ) : '';
		$step = isset( $validation['step'] ) ? floatval( $validation['step'] ) : 1;
		$field_id = 'pwcp_field_' . esc_attr( $field_key );
		
		?>
		<div class="pwcp-number-input-container">
			<input type="number" 
				   id="<?php echo esc_attr( $field_id ); ?>" 
				   name="field_<?php echo esc_attr( $field_key ); ?>" 
				   data-field-key="<?php echo esc_attr( $field_key ); ?>"
				   value="<?php echo esc_attr( $default_value ); ?>" 
				   class="pwcp-number-input"
				   placeholder="<?php esc_attr_e( 'Enter a number...', 'pricewise-calculator-pro' ); ?>"
				   <?php echo '' !== $min ? 'min="' . esc_attr( $min ) . '"' : ''; ?>
				   <?php echo '' !== $max ? 'max="' . esc_attr( $max ) . '"' : ''; ?>
				   step="<?php echo esc_attr( $step ); ?>"
				   <?php echo $required ? 'required aria-required="true"' : ''; ?>>
		</div>
		<?php
	}
	
	/**
	 * Calculator shortcode handler
	 *
	 * @param array $atts Shortcode attributes.
	 * @return string
	 */
	public function pwcp_calculator_shortcode( $atts ) {
		$atts = shortcode_atts(
			array(
				'id'         => 0,
				'product_id' => 0,
				'design'     => 'modern',
			),
			$atts
		);
		
		$calculator_id = absint( $atts['id'] );
		
		if ( ! $calculator_id ) {
			return '<p>' . esc_html__( 'Calculator ID is required.', 'pricewise-calculator-pro' ) . '</p>';
		}
		
		if ( ! class_exists( 'PWCP_Database' ) ) {
			return '<p>' . esc_html__( 'Calculator database not available.', 'pricewise-calculator-pro' ) . '</p>';
		}
		
		$calculator = PWCP_Database::pwcp_get_calculator( $calculator_id );
		if ( ! $calculator ) {
			return '<p>' . esc_html__( 'Calculator not found.', 'pricewise-calculator-pro' ) . '</p>';
		}
		
		$product = null;
		$product_id = absint( $atts['product_id'] );
		if ( $product_id && function_exists( 'wc_get_product' ) ) {
			$product = wc_get_product( $product_id );
		}
		
		ob_start();
		$this->pwcp_render_calculator( $calculator, $product );
		return ob_get_clean();
	}
	
	/**
	 * Get calculator from database
	 *
	 * @param int $calculator_id Calculator ID.
	 * @return object|null
	 */
	private function pwcp_get_calculator( $calculator_id ) {
		if ( class_exists( 'PWCP_Database' ) && method_exists( 'PWCP_Database', 'pwcp_get_calculator' ) ) {
			return PWCP_Database::pwcp_get_calculator( $calculator_id );
		}
		
		global $wpdb;
		$table_name = $wpdb->prefix . 'pwcp_calculators';
		
		$table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table_name ) );
		
		if ( $table_exists !== $table_name ) {
			return null;
		}
		
		$result = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT id, name, description, formula, settings, created_at, updated_at 
				 FROM {$table_name} 
				 WHERE id = %d",
				$calculator_id
			)
		);
		
		return $result;
	}
	
	/**
	 * Get calculator fields from database
	 *
	 * @param int $calculator_id Calculator ID.
	 * @return array
	 */
	private function pwcp_get_calculator_fields( $calculator_id ) {
		if ( class_exists( 'PWCP_Database' ) && method_exists( 'PWCP_Database', 'pwcp_get_calculator_fields' ) ) {
			return PWCP_Database::pwcp_get_calculator_fields( $calculator_id );
		}
		
		global $wpdb;
		$table_name = $wpdb->prefix . 'pwcp_calculator_fields';
		
		$table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table_name ) );
		
		if ( $table_exists !== $table_name ) {
			return array();
		}
		
		$results = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT id, calculator_id, field_key, field_label, field_type, field_required, 
						field_default, field_options, field_description, field_validation, field_order 
				 FROM {$table_name} 
				 WHERE calculator_id = %d 
				 ORDER BY field_order ASC",
				$calculator_id
			)
		);
		
		return $results ? $results : array();
	}
	
	/**
	 * Calculate formula with live updates
	 *
	 * @param string $formula Formula.
	 * @param array  $values Field values.
	 * @param array  $fields Fields array.
	 * @return float
	 */
	private function pwcp_calculate_formula_live( $formula, $values, $fields ) {
		if ( empty( $formula ) ) {
			return $this->pwcp_calculate_sum_fallback( $values, $fields );
		}
		
		try {
			$processed_formula = $formula;
			
			foreach ( $values as $key => $value ) {
				if ( is_array( $value ) ) {
					$numeric_value = 0;
					foreach ( $value as $item ) {
						$item_numeric = $this->pwcp_get_field_numeric_value( $key, $item, $fields );
						$numeric_value += $item_numeric;
					}
				} elseif ( is_numeric( $value ) ) {
					$numeric_value = floatval( $value );
				} else {
					$numeric_value = $this->pwcp_get_field_numeric_value( $key, $value, $fields );
				}
				
				$processed_formula = preg_replace(
					'/\b' . preg_quote( $key, '/' ) . '\b/',
					$numeric_value,
					$processed_formula
				);
			}
			
			preg_match_all( '/\b[a-zA-Z_][a-zA-Z0-9_]*\b/', $processed_formula, $matches );
			$remaining_vars = array_unique( $matches[0] );
			
			if ( ! empty( $remaining_vars ) ) {
				foreach ( $remaining_vars as $var ) {
					$default_value = $this->get_smart_default_value( $var, $processed_formula );
					$processed_formula = preg_replace( '/\b' . preg_quote( $var, '/' ) . '\b/', $default_value, $processed_formula );
				}
			}
			
			// Only allow basic math operators
			if ( ! preg_match( '/^[0-9+\-*\/().\s]+$/', $processed_formula ) ) {
				return $this->pwcp_calculate_sum_fallback( $values, $fields );
			}
			
			$result = $this->pwcp_safe_math_eval( $processed_formula );
			
			if ( false === $result || ! is_numeric( $result ) ) {
				return $this->pwcp_calculate_sum_fallback( $values, $fields );
			}
			
			return floatval( $result );
			
		} catch ( Exception $e ) {
			return $this->pwcp_calculate_sum_fallback( $values, $fields );
		}
	}

	/**
	 * Get smart default value for variable
	 *
	 * @param string $var Variable name.
	 * @param string $formula Formula.
	 * @return string
	 */
	private function get_smart_default_value( $var, $formula ) {
		$pattern = '/([+\-*\/])\s*' . preg_quote( $var, '/' ) . '\b/';
		
		if ( preg_match( $pattern, $formula, $matches ) ) {
			$operator_before = $matches[1];
			
			if ( '*' === $operator_before || '/' === $operator_before ) {
				return '1';
			} else {
				return '0';
			}
		} else {
			return '0';
		}
	}
	
	/**
	 * Calculate sum fallback
	 *
	 * @param array $values Field values.
	 * @param array $fields Fields array.
	 * @return float
	 */
	private function pwcp_calculate_sum_fallback( $values, $fields ) {
		$sum = 0;
		
		foreach ( $values as $key => $value ) {
			if ( is_array( $value ) ) {
				foreach ( $value as $item ) {
					$sum += $this->pwcp_get_field_numeric_value( $key, $item, $fields );
				}
			} else {
				$sum += $this->pwcp_get_field_numeric_value( $key, $value, $fields );
			}
		}
		
		return $sum;
	}
	
	/**
	 * Get field numeric value (simplified for number fields only)
	 *
	 * @param string $field_key Field key.
	 * @param mixed  $value Value.
	 * @param array  $fields Fields array.
	 * @return float
	 */
	private function pwcp_get_field_numeric_value( $field_key, $value, $fields ) {
		// For number fields, the value is already numeric
		if ( is_numeric( $value ) ) {
			return (float) $value;
		}
		
		return 0.0;
	}
	
	/**
	 * Safe math evaluation
	 *
	 * @param string $expression Math expression.
	 * @return float|false
	 */
	private function pwcp_safe_math_eval( $expression ) {
		$expression = preg_replace( '/\s+/', '', $expression );
		
		if ( preg_match( '/\/0(?![0-9])/', $expression ) ) {
			return false;
		}
		
		if ( ! preg_match( '/^[0-9+\-*\/().\s]+$/', $expression ) ) {
			return false;
		}
		
		if ( substr_count( $expression, '(' ) !== substr_count( $expression, ')' ) ) {
			return false;
		}
		
		try {
			$result = $this->pwcp_parse_math_expression( $expression );
			return is_numeric( $result ) ? floatval( $result ) : false;
		} catch ( Exception $e ) {
			return false;
		}
	}
	
	/**
	 * Parse math expression
	 *
	 * @param string $expression Expression.
	 * @return float|false
	 */
	private function pwcp_parse_math_expression( $expression ) {
		$expression = str_replace( ' ', '', $expression );
		
		while ( false !== strpos( $expression, '(' ) ) {
			$start = strrpos( $expression, '(' );
			$end = strpos( $expression, ')', $start );
			
			if ( false === $end ) {
				throw new Exception( 'Unmatched parentheses' );
			}
			
			$inner = substr( $expression, $start + 1, $end - $start - 1 );
			$result = $this->pwcp_calculate_simple_expression( $inner );
			
			if ( false === $result ) {
				return false;
			}
			
			$expression = substr_replace( $expression, $result, $start, $end - $start + 1 );
		}
		
		return $this->pwcp_calculate_simple_expression( $expression );
	}
	
	/**
	 * Calculate simple expression
	 *
	 * @param string $expression Expression.
	 * @return float|false
	 */
	private function pwcp_calculate_simple_expression( $expression ) {
		while ( preg_match( '/(-?\d+(?:\.\d+)?)\s*([*\/])\s*(-?\d+(?:\.\d+)?)/', $expression, $matches ) ) {
			$left = floatval( $matches[1] );
			$operator = $matches[2];
			$right = floatval( $matches[3] );
			
			if ( '*' === $operator ) {
				$result = $left * $right;
			} else {
				if ( 0 === $right ) {
					return false;
				}
				$result = $left / $right;
			}
			
			$expression = str_replace( $matches[0], $result, $expression );
		}
		
		while ( preg_match( '/(-?\d+(?:\.\d+)?)\s*([+\-])\s*(-?\d+(?:\.\d+)?)/', $expression, $matches ) ) {
			$left = floatval( $matches[1] );
			$operator = $matches[2];
			$right = floatval( $matches[3] );
			
			if ( '+' === $operator ) {
				$result = $left + $right;
			} else {
				$result = $left - $right;
			}
			
			$expression = str_replace( $matches[0], $result, $expression );
		}
		
		if ( is_numeric( $expression ) ) {
			return floatval( $expression );
		}
		
		return false;
	}
	
	/**
	 * Log calculation
	 *
	 * @param int   $calculator_id Calculator ID.
	 * @param int   $product_id Product ID.
	 * @param array $values Field values.
	 * @param float $price Price.
	 * @param float $base_price Base price.
	 */
	private function pwcp_log_calculation( $calculator_id, $product_id, $values, $price, $base_price ) {
		if ( class_exists( 'PWCP_Database' ) ) {
			$token = function_exists( 'wp_get_session_token' ) && wp_get_session_token() 
				? wp_get_session_token() 
				: wp_generate_uuid4();
			
			PWCP_Database::pwcp_save_calculation_log(
				array(
					'calculator_id'   => $calculator_id,
					'product_id'      => $product_id,
					'field_values'    => $values,
					'calculated_price' => $price,
					'base_price'      => $base_price,
					'session_id'      => $token,
				)
			);
		}
	}
	
	/**
	 * Get calculation breakdown
	 *
	 * @param string $formula Formula.
	 * @param array  $values Field values.
	 * @return array
	 */
	private function pwcp_get_calculation_breakdown( $formula, $values ) {
		$breakdown = array(
			'original_formula'     => $formula,
			'field_values'         => $values,
			'substituted_formula'  => $formula,
		);
		
		foreach ( $values as $key => $value ) {
			if ( is_array( $value ) ) {
				$value_str = '[' . implode( ', ', $value ) . ']';
			} else {
				$value_str = $value;
			}
			$breakdown['substituted_formula'] = str_replace( $key, $value_str, $breakdown['substituted_formula'] );
		}
		
		return $breakdown;
	}
}

if ( ! is_admin() ) {
	new PWCP_Frontend_Calculator();
}