<?php
/**
 * Plugin Name: PriceWise Calculator Pro
 * Plugin URI: https://devtonicstudios.com
 * Description: Create simple price calculators for WooCommerce with number input fields, custom formulas, and global assignment. Upgrade to Premium for advanced field types, product-specific assignment, and multi-step interface.
 * Version: 1.0.0
 * Author: Devtonic Studios
 * Author URI: https://devtonicstudios.com/about
 * Text Domain: pricewise-calculator-pro
 * Domain Path: /languages
 * Requires at least: 5.8
 * Tested up to: 6.9
 * Requires PHP: 7.4
 * WC requires at least: 5.0
 * WC tested up to: 10.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if (!defined('ABSPATH')) {
    exit;
}

define('PWCP_PLUGIN_FILE', __FILE__);
define('PWCP_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('PWCP_PLUGIN_URL', plugin_dir_url(__FILE__));
define('PWCP_PLUGIN_VERSION', '1.0.0');
define('PWCP_TEXT_DOMAIN', 'pricewise-calculator-pro');
define('PWCP_PLUGIN_BASENAME', plugin_basename(__FILE__));
define('PWCP_ASSETS_URL', PWCP_PLUGIN_URL . 'assets/');
define('PWCP_INCLUDES_DIR', PWCP_PLUGIN_DIR . 'includes/');
define('PWCP_ADMIN_DIR', PWCP_PLUGIN_DIR . 'admin/');
define('PWCP_PUBLIC_DIR', PWCP_PLUGIN_DIR . 'public/');
define('PWCP_TEMPLATES_DIR', PWCP_PLUGIN_DIR . 'templates/');

/**
 * Main plugin class
 * 
 * @since 1.0.0
 */
final class PriceWise_Calculator_Pro {
    
    /**
     * Plugin instance
     *
     * @var PriceWise_Calculator_Pro
     */
    private static $instance = null;
    
    /**
     * Loader instance
     *
     * @var PWCP_Loader
     */
    public $loader;
    
    /**
     * Admin instance
     *
     * @var PWCP_Admin
     */
    public $admin;
    
    /**
     * Frontend calculator instance
     *
     * @var PWCP_Frontend_Calculator
     */
    public $frontend_calculator;
    
    /**
     * WooCommerce integration instance
     *
     * @var PWCP_WooCommerce_Integration
     */
    public $woocommerce_integration;
    
    /**
     * WooCommerce loaded flag
     *
     * @var bool
     */
    private $woocommerce_loaded = false;
    
    /**
     * Get plugin instance
     *
     * @return PriceWise_Calculator_Pro
     */
    public static function pwcp_get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->pwcp_define_constants();
        $this->pwcp_includes();
        $this->pwcp_init_hooks();
    }
    
    /**
     * Define plugin constants
     *
     * @since 1.0.0
     */
    private function pwcp_define_constants() {
        if (!defined('PWCP_MIN_PHP_VERSION')) {
            define('PWCP_MIN_PHP_VERSION', '7.4');
        }
        if (!defined('PWCP_MIN_WP_VERSION')) {
            define('PWCP_MIN_WP_VERSION', '5.8');
        }
        if (!defined('PWCP_MIN_WC_VERSION')) {
            define('PWCP_MIN_WC_VERSION', '5.0');
        }
        if (!defined('PWCP_DEBUG')) {
            define('PWCP_DEBUG', false);
        }
    }
    
    /**
     * Include required files
     *
     * @since 1.0.0
     */
    private function pwcp_includes() {
        if (!is_dir(PWCP_INCLUDES_DIR)) {
            add_action('admin_notices', function() {
                echo '<div class="notice notice-error"><p>' . 
                     esc_html__('PriceWise Calculator Pro: Includes directory not found. Please reinstall the plugin.', 'pricewise-calculator-pro') . 
                     '</p></div>';
            });
            return;
        }

        if (is_admin()) {
            $uninstall_prompt_file = PWCP_ADMIN_DIR . 'class-pwcp-uninstall-prompt.php';
            if (file_exists($uninstall_prompt_file)) {
                require_once $uninstall_prompt_file;
            }
        }
        
        $core_files = array(
            'class-pwcp-system-checker.php',
            'class-pwcp-loader.php',
            'class-pwcp-activator.php',
            'class-pwcp-deactivator.php',
            'class-pwcp-database.php'
        );
        
        foreach ($core_files as $file) {
            $file_path = PWCP_INCLUDES_DIR . $file;
            if (file_exists($file_path)) {
                require_once $file_path;
            } else {
                $this->pwcp_handle_missing_file($file);
                return;
            }
        }
        require_once PWCP_INCLUDES_DIR . 'class-pwcp-premium-helper.php';

        
        if (class_exists('PWCP_System_Checker') && !PWCP_System_Checker::pwcp_check_requirements()) {
            return;
        }
        
        $this->pwcp_load_frontend_files();
        $this->pwcp_load_woocommerce_files();
        
        if (is_admin() && is_dir(PWCP_ADMIN_DIR)) {
            $this->pwcp_load_admin_classes();
        }
    }

    /**
 * Hide all admin notices on our plugin pages only 
 *
 * @since 1.0.0
 */
public function pwcp_block_notices_on_our_pages() {
    if ( ! is_admin() ) {
        return;
    }

    $pwcp_pages = array(
        'pwcp-dashboard',
        'pwcp-calculators',
        'pwcp-add-calculator',
        'pwcp-edit-calculator',
        'pwcp-settings',
        'pwcp-import-export',
         'pwcp-uninstall',
    );

// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- This is only checking which admin page we're on, no action is performed
$current_page = isset($_GET['page']) ? sanitize_key($_GET['page']) : '';
    if ( ! $current_page || ! in_array($current_page, $pwcp_pages, true) ) {
        return; // Not our page → do nothing
    }

    // Kill standard notice hooks ONLY on our pages
    remove_all_actions('admin_notices');
    remove_all_actions('all_admin_notices');
    remove_all_actions('user_admin_notices');
    remove_all_actions('network_admin_notices');
    remove_all_actions('update_nag'); // core update nag
}
    
    /**
     * Load frontend files
     *
     * @since 1.0.0
     */
    private function pwcp_load_frontend_files() {
        $frontend_file = PWCP_PUBLIC_DIR . 'class-pwcp-frontend-calculator.php';
        if (file_exists($frontend_file)) {
            if (!class_exists('PWCP_Frontend_Calculator')) {
                require_once $frontend_file;
                
                if (defined('PWCP_DEBUG') && PWCP_DEBUG) {
                    error_log('PWCP: Frontend calculator loaded from main plugin');
                }
            }
        } else {
            if (defined('PWCP_DEBUG') && PWCP_DEBUG) {
                error_log('PWCP Warning: Frontend calculator file missing');
            }
            
            add_action('admin_notices', function() {
                echo '<div class="notice notice-warning"><p>' . 
                     esc_html__('PriceWise Calculator Pro: Frontend calculator file is missing.', 'pricewise-calculator-pro') . 
                     '</p></div>';
            });
        }
        
        $additional_files = array(
            'class-pwcp-shortcodes.php',
            'class-pwcp-frontend-assets.php'
        );
        
        foreach ($additional_files as $file) {
            $file_path = PWCP_PUBLIC_DIR . $file;
            if (file_exists($file_path)) {
                require_once $file_path;
            }
        }
    }
    
    /**
     * Load WooCommerce integration files
     *
     * @since 1.0.0
     */
    private function pwcp_load_woocommerce_files() {
        if (is_admin() || wp_doing_ajax()) {
            $field_manager_file = PWCP_ADMIN_DIR . 'class-pwcp-field-manager.php';
            if (file_exists($field_manager_file)) {
                require_once $field_manager_file;
            }
        }
        
        $woocommerce_files = array(
            'class-pwcp-ajax-handler.php',
            'class-pwcp-woocommerce-integration.php'
        );
        
        foreach ($woocommerce_files as $file) {
            $file_path = PWCP_INCLUDES_DIR . $file;
            if (file_exists($file_path)) {
                require_once $file_path;
            }
        }
    }
    
    /**
     * Load admin classes
     *
     * @since 1.0.0
     */
    private function pwcp_load_admin_classes() {
        $admin_files = array(
            'class-pwcp-admin.php',
            'class-pwcp-admin-menus.php',
            'class-pwcp-admin-dashboard.php',
            'class-pwcp-field-manager.php',              
            'class-pwcp-admin-calculator-manager.php',  
            'class-pwcp-admin-calculator-form-editor.php',
            'class-pwcp-admin-calculator-list.php',
            'class-pwcp-admin-settings.php',
            'class-pwcp-admin-import-export.php',
            'class-pwcp-category-assignment.php',
            'class-pwcp-enhanced-color-picker.php',
            'class-pwcp-uninstall-notices.php',
        );
        
        foreach ($admin_files as $file) {
            $file_path = PWCP_ADMIN_DIR . $file;
            if (file_exists($file_path)) {
                require_once $file_path;
            }
        }
    }
    
    /**
     * Initialize plugin hooks
     *
     * @since 1.0.0
     */
    private function pwcp_init_hooks() {
        register_activation_hook(PWCP_PLUGIN_FILE, array('PWCP_Activator', 'pwcp_activate'));
        register_deactivation_hook(PWCP_PLUGIN_FILE, array('PWCP_Deactivator', 'pwcp_deactivate'));
        
        add_action('init', array($this, 'pwcp_init'));
        // Translations are automatically loaded by WordPress.org for directory-hosted plugins since WP 4.6
        add_action('plugins_loaded', array($this, 'pwcp_init_components'), 11);
        add_action('wp_loaded', array($this, 'pwcp_init_woocommerce'), 20);
        add_action('before_woocommerce_init', array($this, 'pwcp_declare_woocommerce_compatibility'));
        add_action('admin_notices', array($this, 'pwcp_check_woocommerce'));
        add_action('init', array($this, 'pwcp_init_admin_capabilities'));
        add_action('admin_init', array($this, 'pwcp_check_database_migration'));
        add_action('admin_enqueue_scripts', array($this, 'pwcp_enqueue_admin_styles'));
        add_action('admin_head', array($this, 'pwcp_block_notices_on_our_pages'), 0);

    }
    
    /**
     * Enqueue admin styles
     *
     * @since 1.0.0
     * @param string $hook Current admin page hook
     */
    public function pwcp_enqueue_admin_styles($hook) {
        $screen = get_current_screen();
        
        if (!$screen) {
            return;
        }
        
        // Check if we're on a calculator admin page
        $is_calculator_page = (
            strpos($screen->id, 'pwcp-add-calculator') !== false ||
            strpos($screen->id, 'pwcp-edit-calculator') !== false ||
            strpos($screen->id, 'pwcp-calculators') !== false ||
            strpos($screen->id, 'calculator-pro_page_pwcp') !== false
        );
        
        if ($is_calculator_page) {
            $css_file = PWCP_ASSETS_URL . 'css/all-admin-css/admin-notices.css';
            
            wp_enqueue_style(
                'pwcp-admin-notices',
                $css_file,
                array(),
                PWCP_PLUGIN_VERSION,
                'all'
            );
        }
    }
    
    /**
     * Initialize components
     *
     * @since 1.0.0
     */
    public function pwcp_init_components() {
        if (class_exists('PWCP_Loader')) {
            $this->loader = new PWCP_Loader();
        }
        
        if (is_admin()) {
            $this->pwcp_init_admin_classes();
        }
        
        if (!is_admin() || wp_doing_ajax()) {
            $this->frontend_calculator = $this->pwcp_get_frontend_calculator();
        }
    }
    
    /**
     * Initialize WooCommerce integration
     *
     * @since 1.0.0
     */
    public function pwcp_init_woocommerce() {
        if (class_exists('WooCommerce') && class_exists('PWCP_WooCommerce_Integration') && !$this->woocommerce_loaded) {
            $this->woocommerce_integration = new PWCP_WooCommerce_Integration();
            $this->woocommerce_loaded = true;
            
            if (defined('PWCP_DEBUG') && PWCP_DEBUG) {
                error_log('PWCP: WooCommerce integration initialized');
            }
        }
    }
    
    /**
     * Initialize plugin
     *
     * @since 1.0.0
     */
    public function pwcp_init() {
        if ($this->loader) {
            $this->loader->pwcp_run();
        }
        
        do_action('pwcp_init');
    }
    
    /**
     * Initialize admin classes
     *
     * @since 1.0.0
     */
    private function pwcp_init_admin_classes() {
        if (class_exists('PWCP_Admin')) {
            $this->admin = new PWCP_Admin($this->loader ?? null);
        }
        
        $admin_classes = array(
            'PWCP_Admin_Menus',
            'PWCP_Admin_Dashboard', 
            'PWCP_Admin_Calculator_Manager',
            'PWCP_Admin_Calculator_List',
            'PWCP_Admin_Calculator_Form_Editor',
            'PWCP_Admin_Settings',
            'PWCP_Admin_Import_Export',
            'PWCP_Category_Assignment',
            'PWCP_Enhanced_Color_Picker',
        );
        
        foreach ($admin_classes as $class) {
            if (class_exists($class)) {
                new $class();
            }
        }
    }
    
    /**
     * Handle missing file error
     *
     * @since 1.0.0
     * @param string $filename File name
     */
private function pwcp_handle_missing_file($filename) {
    add_action('admin_notices', function() use ($filename) {
        printf(
            '<div class="notice notice-error"><p>%s</p></div>',
            sprintf(
                /* translators: %s: Name of the missing plugin file */
                esc_html__('PriceWise Calculator Pro: Required file %s is missing. Please reinstall the plugin.', 'pricewise-calculator-pro'),
                '<code>' . esc_html($filename) . '</code>'
            )
        );
    });
}
    
    /**
     * Check and run database migration
     *
     * @since 1.0.0
     */
    public function pwcp_check_database_migration() {
        if (!is_admin()) {
            return;
        }
        
        if (!class_exists('PWCP_Database')) {
            return;
        }
        
        if (PWCP_Database::pwcp_needs_database_update()) {
            PWCP_Database::pwcp_maybe_update_database();
            $this->pwcp_run_column_migration();
        }
    }
    
    /**
     * Run column migration
     *
     * @since 1.0.0
     */
    private function pwcp_run_column_migration() {
        global $wpdb;
        
        $fields_table = $wpdb->prefix . 'pwcp_calculator_fields';
        $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $fields_table));
        
        if (!$table_exists) {
            return;
        }
        
        $migration_completed = get_option('pwcp_column_migration_completed');
        if ($migration_completed) {
            return;
        }
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name from $wpdb->prefix, no user input
        $columns = $wpdb->get_results(
            $wpdb->prepare( "SHOW COLUMNS FROM `" . esc_sql( $fields_table ) . "` WHERE 1=%d", 1 )
        );
        $column_names = wp_list_pluck($columns, 'Field');
        
        $has_changes = false;
        
        if (in_array('is_required', $column_names, true) && !in_array('field_required', $column_names, true)) {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name from $wpdb->prefix, DDL column rename
            $wpdb->query(
                $wpdb->prepare(
                    "ALTER TABLE `" . esc_sql( $fields_table ) . "` CHANGE `is_required` `field_required` tinyint(1) DEFAULT %d",
                    0
                )
            );
            $has_changes = true;
        }
        
        if (in_array('default_value', $column_names, true) && !in_array('field_default', $column_names, true)) {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name from $wpdb->prefix, DDL column rename with no parameterizable values
            $wpdb->query(
                $wpdb->prepare(
                    "ALTER TABLE `" . esc_sql( $fields_table ) . "` CHANGE `default_value` `field_default` varchar(%d)",
                    255
                )
            );
            $has_changes = true;
        }
        
        if (in_array('field_id', $column_names, true) && !in_array('field_key', $column_names, true)) {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name from $wpdb->prefix, DDL column rename with no parameterizable values
            $wpdb->query(
                $wpdb->prepare(
                    "ALTER TABLE `" . esc_sql( $fields_table ) . "` CHANGE `field_id` `field_key` varchar(%d) NOT NULL",
                    100
                )
            );
            $has_changes = true;
        }
        
        if ($has_changes) {
            update_option('pwcp_column_migration_completed', current_time('mysql'));
        }
    }
    
    /**
     * Initialize admin capabilities
     *
     * @since 1.0.0
     */
    public function pwcp_init_admin_capabilities() {
        $capabilities = array(
            'manage_pwcp_calculators',
            'edit_pwcp_calculators',
            'delete_pwcp_calculators',
            'view_pwcp_analytics'
        );
        
        $admin_role = get_role('administrator');
        if ($admin_role) {
            foreach ($capabilities as $cap) {
                $admin_role->add_cap($cap);
            }
        }
        
        if (class_exists('WooCommerce')) {
            $shop_manager_role = get_role('shop_manager');
            if ($shop_manager_role) {
                $shop_manager_role->add_cap('manage_pwcp_calculators');
                $shop_manager_role->add_cap('edit_pwcp_calculators');
            }
        }
    }
    
    
    /**
     * Declare WooCommerce compatibility
     *
     * @since 1.0.0
     */
    public function pwcp_declare_woocommerce_compatibility() {
        if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil')) {
            \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
                'custom_order_tables',
                PWCP_PLUGIN_FILE,
                true
            );
            
            \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
                'cart_checkout_blocks',
                PWCP_PLUGIN_FILE,
                true
            );
        }
    }
    
    /**
     * Check if WooCommerce is active
     *
     * @since 1.0.0
     */
public function pwcp_check_woocommerce() {
    if (!class_exists('WooCommerce')) {
        $class = 'notice notice-warning';
        $message = sprintf(
            /* translators: %s: WooCommerce plugin name */
            esc_html__('PriceWise Calculator Pro requires %s to be installed and active for full functionality.', 'pricewise-calculator-pro'),
            '<strong>WooCommerce</strong>'
        );
        
        printf('<div class="%1$s"><p>%2$s</p></div>', esc_attr($class), wp_kses_post($message));
    }
}
    /**
     * Get plugin version
     *
     * @since 1.0.0
     * @return string
     */
    public function pwcp_get_version() {
        return PWCP_PLUGIN_VERSION;
    }
    
    /**
     * Get admin instance
     *
     * @since 1.0.0
     * @return PWCP_Admin|null
     */
    public function pwcp_get_admin() {
        return $this->admin;
    }
    
    /**
     * Get frontend calculator instance
     *
     * @since 1.0.0
     * @return PWCP_Frontend_Calculator|null
     */
    public function pwcp_get_frontend_calculator() {
        if (!$this->frontend_calculator && class_exists('PWCP_Frontend_Calculator')) {
            static $frontend_instance = null;
            if ($frontend_instance === null) {
                $frontend_instance = new PWCP_Frontend_Calculator();
            }
            $this->frontend_calculator = $frontend_instance;
        }
        return $this->frontend_calculator;
    }
    
    /**
     * Get WooCommerce integration instance
     *
     * @since 1.0.0
     * @return PWCP_WooCommerce_Integration|null
     */
    public function pwcp_get_woocommerce_integration() {
        return $this->woocommerce_integration;
    }
    
    /**
     * Check if frontend is initialized
     *
     * @since 1.0.0
     * @return bool
     */
    public function pwcp_is_frontend_initialized() {
        return class_exists('PWCP_Frontend_Calculator');
    }
    
    /**
     * Get product calculators
     *
     * @since 1.0.0
     * @param int $product_id Product ID
     * @return array
     */
    public function pwcp_get_product_calculators($product_id) {
        if (!class_exists('PWCP_Database')) {
            return array();
        }
        
        return PWCP_Database::pwcp_get_product_calculators($product_id);
    }
    
    /**
     * Prevent cloning
     *
     * @since 1.0.0
     */
    public function __clone() {
        _doing_it_wrong(__FUNCTION__, esc_html__('Cloning is forbidden.', 'pricewise-calculator-pro'), '1.0.0');
    }
    
    /**
     * Prevent unserialization
     *
     * @since 1.0.0
     */
    public function __wakeup() {
        _doing_it_wrong(__FUNCTION__, esc_html__('Unserializing instances is forbidden.', 'pricewise-calculator-pro'), '1.0.0');
    }
}

/**
 * Initialize the plugin
 *
 * @since 1.0.0
 * @return PriceWise_Calculator_Pro
 */
function pwcp_init_plugin() {
    return PriceWise_Calculator_Pro::pwcp_get_instance();
}

/**
 * Get plugin instance
 *
 * @since 1.0.0
 * @return PriceWise_Calculator_Pro
 */
function pwcp() {
    return PriceWise_Calculator_Pro::pwcp_get_instance();
}

/**
 * Check if product has calculators
 *
 * @since 1.0.0
 * @param int $product_id Product ID
 * @return bool
 */
function pwcp_product_has_calculators($product_id) {
    $plugin = pwcp();
    $calculators = $plugin->pwcp_get_product_calculators($product_id);
    return !empty($calculators);
}

/**
 * Get product calculators
 *
 * @since 1.0.0
 * @param int $product_id Product ID
 * @return array
 */
function pwcp_get_product_calculators($product_id) {
    $plugin = pwcp();
    return $plugin->pwcp_get_product_calculators($product_id);
}

pwcp_init_plugin();