<?php
/**
 * Security Validation Trait
 * 
 * Provides reusable security validation methods for admin operations
 * 
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Security Validation Trait
 * 
 * Reusable security methods for nonce and capability verification
 * 
 * @since 1.0.0
 */
trait PWCP_Security_Validation_Trait {
    
    /**
     * Check if the nonce is valid for calculator actions
     * 
     * @param string $nonce_action Nonce action to verify
     * @return bool
     */
    protected function verify_calculator_nonce($nonce_action = 'pwcp_calculator_action') {
        return isset($_POST['pwcp_nonce']) && 
               wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['pwcp_nonce'])), $nonce_action);
    }
    
    /**
     * Check if user has the required admin permission
     * 
     * @param string $capability Required capability
     * @return bool
     */
    protected function verify_admin_capability($capability = 'manage_options') {
        return current_user_can($capability);
    }
    
    /**
     * Run both nonce and permission checks at once
     * 
     * @param string $capability Required capability
     * @param string $nonce_action Nonce action
     * @return bool
     */
    protected function verify_security($capability = 'manage_options', $nonce_action = 'pwcp_calculator_action') {
        $nonce_valid = isset($_POST['pwcp_nonce']) && 
                      wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['pwcp_nonce'])), $nonce_action);
        
        return $nonce_valid && current_user_can($capability);
    }
    
    /**
     * Check security and bail with error if it fails
     * 
     * @param string $capability Required capability
     * @param string $nonce_action Nonce action
     * @return void Exits with error if validation fails
     */
    protected function require_security($capability = 'manage_options', $nonce_action = 'pwcp_calculator_action') {
        if (!$this->verify_security($capability, $nonce_action)) {
            if (!$this->verify_calculator_nonce($nonce_action)) {
                wp_send_json_error(array('message' => esc_html__('Security verification failed.', 'pricewise-calculator-pro')));
            }
            
            if (!$this->verify_admin_capability($capability)) {
                wp_send_json_error(array('message' => esc_html__('Insufficient permissions.', 'pricewise-calculator-pro')));
            }
        }
    }
    
    /**
     * Verify standard form submissions
     * 
     * @param string $nonce_field Nonce field name
     * @param string $nonce_action Nonce action
     * @param string $capability Required capability
     * @return bool
     */
    protected function verify_form_security($nonce_field, $nonce_action, $capability = 'manage_options') {
        return isset($_POST[$nonce_field]) && 
               wp_verify_nonce(sanitize_text_field(wp_unslash($_POST[$nonce_field])), $nonce_action) &&
               current_user_can($capability);
    }
    
    /**
     * Verify AJAX request from plugin
     * 
     * @return bool
     */
    protected function is_valid_ajax_request() {
        return defined('DOING_AJAX') && DOING_AJAX && 
               isset($_POST['action']) && 
               strpos(sanitize_text_field(wp_unslash($_POST['action'])), 'pwcp_') === 0;
    }
    
    /**
     * Check if we're in the admin context
     * 
     * @return bool
     */
    protected function is_admin_context() {
        return is_admin() && !defined('DOING_CRON');
    }
    
    /**
     * Check field editing permissions
     * 
     * @return bool
     */
    protected function can_edit_fields() {
        return current_user_can('manage_options') || current_user_can('edit_pwcp_fields');
    }
    
    /**
     * Sanitize calculator ID from user input
     * 
     * @param mixed $calculator_id Input to sanitize
     * @return int Sanitized calculator ID
     */
    protected function sanitize_calculator_id($calculator_id) {
        return absint($calculator_id);
    }
    
    /**
     * Sanitize field ID from user input
     * 
     * @param mixed $field_id Input to sanitize
     * @return int Sanitized field ID
     */
    protected function sanitize_field_id($field_id) {
        return absint($field_id);
    }
    
    /**
     * Verify action is allowed
     * 
     * @param string $action Action to validate
     * @param array $allowed_actions Allowed actions array
     * @return bool
     */
    protected function is_action_allowed($action, $allowed_actions) {
        return in_array(sanitize_key($action), $allowed_actions, true);
    }
}