<?php
/**
 * WooCommerce Integration - ENVATO COMPLIANT VERSION (FINAL)
 * 
 * Priority Order: Product Specific > Category Based > Global
 * Within each type, admin priority number determines order
 * File: includes/class-pwcp-woocommerce-integration.php
 * 
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
	exit;
}

/**
 * PWCP WooCommerce Integration Class - ENVATO COMPLIANT
 * 
 * @since 1.0.0
 */
class PWCP_WooCommerce_Integration {
	
	private $cache_group = 'pwcp_woocommerce';
	
	public function __construct() {
		if (!class_exists('WooCommerce')) {
			return;
		}
		
		$this->pwcp_init_hooks();
	}
	
	/**
	 * Initialize all WooCommerce hooks
	 */
	private function pwcp_init_hooks() {
		// Get calculator position from settings
		$general_settings = get_option('pwcp_general_settings', array());
		$calculator_position = isset($general_settings['calculator_position']) ? $general_settings['calculator_position'] : 'before_add_to_cart';
		
		// Map position to WooCommerce hooks and priorities
		switch ($calculator_position) {
			case 'before_add_to_cart':
				add_action('woocommerce_before_add_to_cart_form', array($this, 'pwcp_display_product_calculator'), 10);
				break;
				
			case 'after_add_to_cart':
				add_action('woocommerce_after_add_to_cart_form', array($this, 'pwcp_display_product_calculator'), 10);
				break;
				
			case 'before_product_summary':
				add_action('woocommerce_before_single_product_summary', array($this, 'pwcp_display_product_calculator'), 30);
				break;
				
			case 'after_product_summary':
				add_action('woocommerce_after_single_product_summary', array($this, 'pwcp_display_product_calculator'), 10);
				break;
				
			case 'in_product_tabs':
				add_filter('woocommerce_product_tabs', array($this, 'pwcp_add_calculator_tab'));
				break;
				
			default:
				// Fallback to before add to cart
				add_action('woocommerce_before_add_to_cart_form', array($this, 'pwcp_display_product_calculator'), 10);
				break;
		}
		
		// MAIN PRICE HIDING
		add_action('woocommerce_single_product_summary', array($this, 'pwcp_maybe_remove_price_display'), 5);
		add_filter('woocommerce_get_price_html', array($this, 'pwcp_filter_price_html'), 10, 2);
		add_filter('woocommerce_variable_price_html', array($this, 'pwcp_filter_price_html'), 10, 2);
		add_filter('woocommerce_grouped_price_html', array($this, 'pwcp_filter_price_html'), 10, 2);
		
		// ADD TO CART BUTTON TEXT CUSTOMIZATION
		add_filter('woocommerce_product_single_add_to_cart_text', array($this, 'pwcp_custom_add_to_cart_text'), 10, 2);
		add_filter('woocommerce_product_add_to_cart_text', array($this, 'pwcp_custom_add_to_cart_text'), 10, 2);
		
		// Cart integration
		add_filter('woocommerce_add_cart_item_data', array($this, 'pwcp_add_calculator_data_to_cart'), 10, 3);
		add_action('woocommerce_before_calculate_totals', array($this, 'pwcp_update_cart_item_prices'));
		add_filter('woocommerce_get_item_data', array($this, 'pwcp_display_calculator_data_in_cart'), 10, 2);
		add_filter('woocommerce_loop_add_to_cart_link', array($this, 'pwcp_filter_add_to_cart_button'), 10, 3);

		// Order integration
		add_action('woocommerce_checkout_create_order_line_item', array($this, 'pwcp_save_calculator_data_to_order'), 10, 4);
		add_filter('woocommerce_order_item_display_meta_key', array($this, 'pwcp_format_order_meta_key'), 10, 3);
		add_filter('woocommerce_order_item_display_meta_value', array($this, 'pwcp_format_order_meta_value'), 10, 3);
		
		// Product admin
		add_filter('woocommerce_product_data_tabs', array($this, 'pwcp_add_product_data_tab'));
		add_action('woocommerce_product_data_panels', array($this, 'pwcp_product_data_panel'));
		add_action('woocommerce_process_product_meta', array($this, 'pwcp_save_product_calculator_settings'));
		
		// Add hidden fields for cart integration
		add_action('woocommerce_before_add_to_cart_button', array($this, 'pwcp_add_calculated_price_field'));
		
		// Hide quantity selector when enabled
		add_action('woocommerce_before_add_to_cart_button', array($this, 'pwcp_maybe_hide_quantity_selector'), 5);

		// Lock quantity in cart for calculator products
		add_filter('woocommerce_cart_item_quantity', array($this, 'pwcp_lock_cart_quantity'), 10, 3);
		add_filter('woocommerce_is_sold_individually', array($this, 'pwcp_force_sold_individually'), 10, 2);
		add_filter('woocommerce_add_to_cart_validation', array($this, 'pwcp_validate_cart_quantity'), 10, 4);
		
		// Enqueue assets
		add_action('wp_enqueue_scripts', array($this, 'pwcp_enqueue_frontend_styles'));
		add_action('admin_enqueue_scripts', array($this, 'pwcp_enqueue_admin_styles'));
		
	}
	

/**
 * Enqueue frontend styles for WooCommerce pages
 * @since 1.0.0
 * @return void
 */
public function pwcp_enqueue_frontend_styles() {
    if (!class_exists('WooCommerce')) {
        return;
    }
    
    if (is_admin()) {
        return;
    }
    
    $dependencies = array();
    if (wp_style_is('woocommerce-general', 'registered')) {
        $dependencies[] = 'woocommerce-general';
    }
    
    wp_enqueue_style(
        'pwcp-woocommerce-integration',
        plugin_dir_url(dirname(__FILE__)) . 'assets/css/all-frontend-css/woocommerce-integration.css',
        $dependencies,
        PWCP_PLUGIN_VERSION, 
        'all'
    );
}
	
	/**
	 * Enqueue admin styles - FIXED: Correct path
	 */
	public function pwcp_enqueue_admin_styles($hook) {
		// Only on product edit pages
		if ($hook !== 'post.php' && $hook !== 'post-new.php') {
			return;
		}
		
		$screen = get_current_screen();
		if ($screen && $screen->post_type === 'product') {
			// FIXED: Correct path to your CSS structure
			wp_enqueue_style(
				'pwcp-woocommerce-admin',
				plugin_dir_url(dirname(__FILE__)) . 'assets/css/all-admin-css/woocommerce-admin.css',
				array(),
				'1.0.0'
			);
		}
	}
	
	/**
	 * Customize Add to Cart button text based on admin settings
	 */
	public function pwcp_custom_add_to_cart_text($text, $product) {
		// Don't change text in admin
		if (is_admin() && !(function_exists('wp_doing_ajax') ? wp_doing_ajax() : (defined('DOING_AJAX') && DOING_AJAX))) {
    return $text;
}
		
		if (!$product) {
			return $text;
		}
		
		$product_id = $product->get_id();
		$calculator = $this->pwcp_get_highest_priority_calculator($product_id);
		
		// Only change button text if calculator is active for this product
		if (!$calculator) {
			return $text;
		}
		
		// Get custom button text from settings
		$general_settings = get_option('pwcp_general_settings', array());
		$custom_text = isset($general_settings['add_to_cart_text']) ? trim($general_settings['add_to_cart_text']) : '';
		
		// Return custom text if set, otherwise return original
		if (!empty($custom_text)) {
			return esc_html($custom_text);
		}
		
		return $text;
	}

	/**
	 * Prevent quantity changes for calculator products with locked quantity
	 */
	public function pwcp_validate_cart_quantity($passed, $product_id, $quantity, $variation_id = 0) {
		$calculator = $this->pwcp_get_highest_priority_calculator($product_id);
		
		if ($calculator && !empty($calculator->hide_quantity_selector) && $quantity > 1) {
			wc_add_notice(
				__('This product can only be purchased with quantity 1 due to custom calculator settings.', 'pricewise-calculator-pro'),
				'error'
			);
			return false;
		}
		
		return $passed;
	}

	/**
	 * Lock quantity in cart when hide_quantity_selector is enabled
	 */
	public function pwcp_lock_cart_quantity($product_quantity, $cart_item_key, $cart_item) {
		// Only apply to items with calculator data
		if (!isset($cart_item['pwcp_calculator_data'])) {
			return $product_quantity;
		}
		
		$calculator_data = $cart_item['pwcp_calculator_data'];
		$calculator_id = isset($calculator_data['calculator_id']) ? absint($calculator_data['calculator_id']) : 0;
		
		if (!$calculator_id) {
			return $product_quantity;
		}
		
		// Get the calculator
		if (class_exists('PWCP_Database')) {
			$calculator = PWCP_Database::pwcp_get_calculator($calculator_id);
		} else {
			global $wpdb;
			$table = $wpdb->prefix . 'pwcp_calculators';
			$calculator = $wpdb->get_row(
				$wpdb->prepare(
					"SELECT id, name, status, hide_quantity_selector FROM `{$table}` WHERE id = %d",
					$calculator_id
				)
			);
		}
		
		// If hide_quantity_selector is enabled, lock quantity to 1
		if ($calculator && !empty($calculator->hide_quantity_selector)) {
			$quantity = $cart_item['quantity'];
			
			// Return plain text quantity (not editable)
			return sprintf(
				'<div class="quantity pwcp-locked-quantity-wrapper">
					<input type="hidden" name="cart[%s][qty]" value="%s" />
					<span class="pwcp-locked-quantity">%s</span>
					<small class="pwcp-locked-notice">%s</small>
				</div>',
				esc_attr($cart_item_key),
				esc_attr($quantity),
				esc_html($quantity),
				esc_html__('Quantity locked for this calculator', 'pricewise-calculator-pro')
			);
		}
		
		return $product_quantity;
	}

	/**
	 * Force sold individually for products with hide_quantity_selector
	 */
	public function pwcp_force_sold_individually($sold_individually, $product) {
		if (is_admin() && !defined('DOING_AJAX')) {
			return $sold_individually;
		}
		
		$product_id = $product->get_id();
		$calculator = $this->pwcp_get_highest_priority_calculator($product_id);
		
		// Force sold individually if calculator has hide_quantity_selector enabled
		if ($calculator && !empty($calculator->hide_quantity_selector)) {
			return true;
		}
		
		return $sold_individually;
	}

	/**
	 * Add calculator as a product tab
	 */
	public function pwcp_add_calculator_tab($tabs) {
		global $product;
		
		if (!$product) {
			return $tabs;
		}
		
		$calculator = $this->pwcp_get_highest_priority_calculator($product->get_id());
		
		if (!$calculator) {
			return $tabs;
		}
		
		$tabs['pwcp_calculator'] = array(
			'title'    => esc_html__('Price Calculator', 'pricewise-calculator-pro'),
			'priority' => 50,
			'callback' => array($this, 'pwcp_display_product_calculator')
		);
		
		return $tabs;
	}
	
/**
 * Hide quantity selector when calculator setting is enabled
 * FIXED: Now prevents rendering instead of CSS hiding
 */
public function pwcp_maybe_hide_quantity_selector() {
    global $product;
    
    if (!$product || !is_object($product) || !($product instanceof WC_Product) || !is_product()) {
        return;
    }
    
    if (!method_exists($product, 'get_id')) {
        return;
    }
    
    $product_id = $product->get_id();
    
    if (!$product_id) {
        return;
    }
    
    $calculator = $this->pwcp_get_highest_priority_calculator($product_id);
    
    if (!$calculator || empty($calculator->hide_quantity_selector)) {
        return;
    }
    
    // FIXED: Remove WooCommerce's quantity input rendering
    remove_action('woocommerce_before_add_to_cart_button', 'woocommerce_quantity_input', 10);
    
    // Add hidden input with quantity=1 instead
    add_action('woocommerce_before_add_to_cart_button', array($this, 'pwcp_add_hidden_quantity_input'), 10);
}

/**
 * Add hidden quantity input set to 1
 */
public function pwcp_add_hidden_quantity_input() {
    echo '<input type="hidden" name="quantity" value="1" class="pwcp-hidden-quantity" />';
}

	/**
	 * Force quantity to 1 when quantity selector is hidden
	 */
	public function pwcp_force_quantity_one($args, $product) {
		if (!$product) {
			return $args;
		}
		
		$product_id = $product->get_id();
		$calculator = $this->pwcp_get_highest_priority_calculator($product_id);
		
		if ($calculator && !empty($calculator->hide_quantity_selector)) {
			$args['input_value'] = 1;
			$args['min_value'] = 1;
			$args['max_value'] = 1;
			$args['step'] = 1;
		}
		
		return $args;
	}
	
	/**
	 * Remove price display when calculator is active
	 */
	public function pwcp_maybe_remove_price_display() {
		// Don't affect admin area
		if (is_admin()) {
			return;
		}
		
		global $product;
		
		if (!$product || !is_a($product, 'WC_Product') || !is_product()) {
			return;
		}
		
		$product_id = $product->get_id();
		
		if ($this->pwcp_should_hide_price($product_id)) {
			// Remove WooCommerce price display
			remove_action('woocommerce_single_product_summary', 'woocommerce_template_single_price', 10);
		}
	}

	/**
	 * Filter price HTML for all contexts
	 */
	public function pwcp_filter_price_html($price_html, $product) {
if (is_admin() && !(function_exists('wp_doing_ajax') ? wp_doing_ajax() : (defined('DOING_AJAX') && DOING_AJAX))) {
    return $price_html;
}
    
    if (!$product || !is_object($product) || !($product instanceof WC_Product)) {
        return $price_html;
    }

    $product_id = $product->get_id();
    
    if (!$product_id) {
        return $price_html;
    }

    if ($this->pwcp_should_hide_price($product_id)) {
        $main_product_id = (int) get_queried_object_id();

        if (is_product() && $product_id === $main_product_id) {
            return '';
        } else {
            return $this->pwcp_get_shop_button($product_id);
        }
    }

    return $price_html;
}
	
	/**
	 * Check if price should be hidden
	 */
	private function pwcp_should_hide_price($product_id) {
		$calculator = $this->pwcp_get_highest_priority_calculator($product_id);
		
		return ($calculator && $calculator->status === 'active');
	}
	
	/**
 * Get shop page button - Uses admin settings for text and color
 */
private function pwcp_get_shop_button($product_id) {
    $product_url = esc_url(get_permalink($product_id));
    
    $general_settings = get_option('pwcp_general_settings', array());
    $design_settings = get_option('pwcp_design_settings', array());
    
    $button_text = isset($general_settings['shop_quote_button_text']) && !empty($general_settings['shop_quote_button_text'])
        ? esc_html($general_settings['shop_quote_button_text'])
        : esc_html__('Get Custom Quote', 'pricewise-calculator-pro');
    
    $button_color = isset($design_settings['shop_quote_button_color']) && !empty($design_settings['shop_quote_button_color'])
        ? sanitize_hex_color($design_settings['shop_quote_button_color'])
        : '#844CA9 ';
    
    $button_color_light = $this->pwcp_lighten_color($button_color, 10);
    
    return sprintf(
        '<div class="pwcp-shop-quote-button" style="background: linear-gradient(135deg, %s 0%%, %s 100%%); box-shadow: 0 2px 4px %s;">
            <a href="%s" class="pwcp-quote-link">
                <svg class="pwcp-quote-icon" width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                    <rect x="4" y="2" width="16" height="20" rx="2" fill="none" stroke="currentColor" stroke-width="1"/>
                    <rect x="6" y="4" width="12" height="3" rx="1" fill="currentColor"/>
                    <circle cx="8" cy="11" r="1.5" fill="currentColor"/>
                    <circle cx="12" cy="11" r="1.5" fill="currentColor"/>
                    <circle cx="16" cy="11" r="1.5" fill="currentColor"/>
                    <circle cx="8" cy="15" r="1.5" fill="currentColor"/>
                    <circle cx="12" cy="15" r="1.5" fill="currentColor"/>
                    <circle cx="16" cy="15" r="1.5" fill="currentColor"/>
                    <circle cx="8" cy="19" r="1.5" fill="currentColor"/>
                    <circle cx="12" cy="19" r="1.5" fill="currentColor"/>
                    <circle cx="16" cy="19" r="1.5" fill="currentColor"/>
                </svg>
                %s
            </a>
        </div>',
        esc_attr($button_color),
        esc_attr($button_color_light),
        esc_attr($button_color . '4d'), 
        $product_url,
        $button_text
    );
}

/**
 * Lighten a hex color
 * 
 * @param string $hex Hex color code
 * @param int $percent Percentage to lighten (0-100)
 * @return string Lightened hex color
 */
private function pwcp_lighten_color($hex, $percent) {
    // Remove # if present
    $hex = str_replace('#', '', $hex);
    
    if (strlen($hex) === 3) {
        $r = hexdec(substr($hex, 0, 1) . substr($hex, 0, 1));
        $g = hexdec(substr($hex, 1, 1) . substr($hex, 1, 1));
        $b = hexdec(substr($hex, 2, 1) . substr($hex, 2, 1));
    } else {
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));
    }
    
    $r = min(255, $r + ((255 - $r) * $percent / 100));
    $g = min(255, $g + ((255 - $g) * $percent / 100));
    $b = min(255, $b + ((255 - $b) * $percent / 100));
    
    return '#' . str_pad(dechex(round($r)), 2, '0', STR_PAD_LEFT)
                . str_pad(dechex(round($g)), 2, '0', STR_PAD_LEFT)
                . str_pad(dechex(round($b)), 2, '0', STR_PAD_LEFT);
}
	
	/**
	 * Get all calculators for a product with cascading priority system
	 */
	private function pwcp_get_product_calculators($product_id) {
		$cache_key = 'product_calculators_' . $product_id;
		$cached = wp_cache_get($cache_key, $this->cache_group);
		
		if ($cached !== false) {
			return $cached;
		}
		
		$all_calculators = array();
		
		$product_calculators = $this->pwcp_get_product_specific_calculators($product_id);
		
		$category_calculators = $this->pwcp_get_category_assigned_calculators($product_id);
		
		$global_calculators = $this->pwcp_get_global_calculators();
		
		$all_calculators = array_merge(
			$this->pwcp_add_type_priority($product_calculators, 'product'),
			$this->pwcp_add_type_priority($category_calculators, 'category'), 
			$this->pwcp_add_type_priority($global_calculators, 'global')
		);
		
		if (!empty($all_calculators)) {
			usort($all_calculators, function($a, $b) {
				if ($a->type_priority !== $b->type_priority) {
					return $b->type_priority - $a->type_priority;
				}
				
				$priority_a = isset($a->priority) ? absint($a->priority) : 1;
				$priority_b = isset($b->priority) ? absint($b->priority) : 1;
				return $priority_b - $priority_a;
			});
		}
		
		wp_cache_set($cache_key, $all_calculators, $this->cache_group, HOUR_IN_SECONDS);
		return $all_calculators;
	}
	
	/**
	 * Get the highest priority calculator for a product
	 */
	private function pwcp_get_highest_priority_calculator($product_id) {
		$calculators = $this->pwcp_get_product_calculators($product_id);
		
		if (empty($calculators)) {
			return null;
		}
		
		return $calculators[0];
	}
	
	/**
	 * Get product-specific calculators
	 */
	private function pwcp_get_product_specific_calculators($product_id) {
		$assigned_calculators = get_post_meta($product_id, '_pwcp_assigned_calculators', true);
		$calculators = array();
		
		if (is_array($assigned_calculators) && !empty($assigned_calculators)) {
			if (class_exists('PWCP_Database')) {
				foreach ($assigned_calculators as $calc_id) {
					$calculator = PWCP_Database::pwcp_get_calculator(absint($calc_id));
					if ($calculator && $calculator->status === 'active' && $calculator->assignment_type === 'product') {
						$calculators[] = $calculator;
					}
				}
			}
		}
		
		return $calculators;
	}
	
	/**
 * Get global calculators - FIXED SQL with description
 */
private function pwcp_get_global_calculators() {
    $calculators = array();
    
    if (class_exists('PWCP_Database')) {
        global $wpdb;
        $table = $wpdb->prefix . 'pwcp_calculators';
        
        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT id, name, status, assignment_type, priority, hide_quantity_selector, description FROM `{$table}` WHERE status = %s AND assignment_type = %s ORDER BY priority DESC",
                'active',
                'global'
            )
        );
        
        if ($results) {
            $calculators = $results;
        }
    }
    
    return $calculators;
}
/**
 * Get calculators assigned to product categories - FIXED SQL with description
 */
private function pwcp_get_category_assigned_calculators($product_id) {
    $product_categories = wp_get_post_terms($product_id, 'product_cat', array('fields' => 'ids'));
    
    if (is_wp_error($product_categories) || empty($product_categories)) {
        return array();
    }
    
    $calculators = array();
    
    if (class_exists('PWCP_Database')) {
        global $wpdb;
        $calculators_table = $wpdb->prefix . 'pwcp_calculators';
        
        $category_calculators = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT id, name, status, assignment_type, priority, hide_quantity_selector, description FROM `{$calculators_table}` WHERE status = %s AND assignment_type = %s ORDER BY priority DESC",
                'active',
                'category'
            )
        );
        
        if (!empty($category_calculators)) {
            foreach ($category_calculators as $calculator) {
                $assigned_categories = $this->pwcp_get_calculator_assigned_categories($calculator->id);
                
                if (!empty($assigned_categories)) {
                    $assigned_category_ids = array_column($assigned_categories, 'id');
                    $matching_categories = array_intersect($product_categories, $assigned_category_ids);
                    
                    if (!empty($matching_categories)) {
                        $calculators[] = $calculator;
                    }
                }
            }
        }
    }
    
    return $calculators;
}
	
	/**
	 * Add type priority to calculators
	 */
	private function pwcp_add_type_priority($calculators, $type) {
		$type_priorities = array(
			'product' => 300,
			'category' => 200,
			'global' => 100
		);
		
		foreach ($calculators as $calculator) {
			$calculator->type_priority = $type_priorities[$type];
			$calculator->assignment_source = $type;
		}
		
		return $calculators;
	}
	
	/**
	 * Get categories assigned to a calculator
	 */
	private function pwcp_get_calculator_assigned_categories($calculator_id) {
		$assignments = get_option('pwcp_category_assignments_' . $calculator_id, array());
		
		if (!is_array($assignments)) {
			$assignments = array();
		}
		
		$sanitized_assignments = array();
		foreach ($assignments as $assignment) {
			if (isset($assignment['id']) && isset($assignment['name'])) {
				$sanitized_assignments[] = array(
					'id' => absint($assignment['id']),
					'name' => sanitize_text_field($assignment['name'])
				);
			}
		}
		
		return $sanitized_assignments;
	}
	
	/**
	 * Display product calculator
	 */
	public function pwcp_display_product_calculator() {
    global $product;
    
    if (!$product || !is_object($product) || !($product instanceof WC_Product)) {
        $post = get_queried_object();
        if ($post && isset($post->ID)) {
            $product = wc_get_product($post->ID);
        }
    }
    
    if (!$product || !is_object($product) || !method_exists($product, 'get_id')) {
        return;
    }
    
    $product_id = $product->get_id();
    $calculator = $this->pwcp_get_highest_priority_calculator($product_id);
    
    if (!$calculator) {
        return;
    }
    
    if (class_exists('PWCP_Frontend_Calculator')) {
        $frontend_calculator = new PWCP_Frontend_Calculator();
        $frontend_calculator->pwcp_render_calculator($calculator, $product);
    }
}
	
	/**
	 * Add hidden field to store calculated price
	 */
	public function pwcp_add_calculated_price_field() {
		global $product;
		
		if (!$product) {
			return;
		}
		
		$calculator = $this->pwcp_get_highest_priority_calculator($product->get_id());
		if (!$calculator) {
			return;
		}
		
		?>
		<input type="hidden" id="pwcp_calculated_price_input" name="pwcp_calculated_price" value="0">
		<input type="hidden" id="pwcp_calculator_id_input" name="pwcp_calculator_id" value="<?php echo esc_attr($calculator->id); ?>">
		<input type="hidden" id="pwcp_field_values_input" name="pwcp_field_values" value="">
		<input type="hidden" id="pwcp_nonce_input" name="pwcp_nonce" value="<?php echo esc_attr(wp_create_nonce('pwcp_calculator_action')); ?>">
		<?php
	}

	/**
	 * Filter add to cart button on shop pages
	 */
	public function pwcp_filter_add_to_cart_button($button, $product, $args) {
	if (is_admin() && !(function_exists('wp_doing_ajax') ? wp_doing_ajax() : (defined('DOING_AJAX') && DOING_AJAX))) {
    return $button;
}

		if (!$product instanceof WC_Product) {
			return $button;
		}

		$product_id = $product->get_id();

		if ($this->pwcp_should_hide_price($product_id)) {
			return '';
		}

		return $button;
	}
	
	/**
	 * Add calculator data to cart - FIXED: Correct calculator_id assignment
	 */
	public function pwcp_add_calculator_data_to_cart($cart_item_data, $product_id, $variation_id) {
		if (!isset($_POST['pwcp_calculated_price']) || !isset($_POST['pwcp_calculator_id'])) {
			return $cart_item_data;
		}
		
		// FIXED: Read calculator_id from correct POST field
		$calculator_id = isset($_POST['pwcp_calculator_id']) ? absint($_POST['pwcp_calculator_id']) : 0;
		$calculated_price = floatval($_POST['pwcp_calculated_price']);
		
		// FIXED: Proper JSON handling without corrupting data
$field_values_json = isset($_POST['pwcp_field_values']) ? sanitize_text_field(wp_unslash($_POST['pwcp_field_values'])) : '';
		$nonce = isset($_POST['pwcp_nonce']) ? sanitize_text_field(wp_unslash($_POST['pwcp_nonce'])) : '';
		
		if (!wp_verify_nonce($nonce, 'pwcp_calculator_action')) {
			return $cart_item_data;
		}
		
		if ($calculator_id <= 0 || $calculated_price <= 0) {
			return $cart_item_data;
		}
		
		$field_values = array();
		if (!empty($field_values_json)) {
			$decoded_values = json_decode($field_values_json, true);
			
			// Validate JSON decode succeeded
			if (json_last_error() === JSON_ERROR_NONE && is_array($decoded_values)) {
				foreach ($decoded_values as $key => $value) {
					$sanitized_key = sanitize_key($key);
					if (is_array($value)) {
						$field_values[$sanitized_key] = array_map('sanitize_text_field', $value);
					} else {
						$field_values[$sanitized_key] = sanitize_text_field($value);
					}
				}
			}
		}
		
		if (class_exists('PWCP_Database')) {
			$calculator = PWCP_Database::pwcp_get_calculator($calculator_id);
			if (!$calculator) {
				return $cart_item_data;
			}
		} else {
			return $cart_item_data;
		}
		
		$cart_item_data['pwcp_calculator_data'] = array(
			'calculator_id' => $calculator_id,
			'calculator_name' => sanitize_text_field($calculator->name ?? ''),
			'calculated_price' => $calculated_price,
			'field_values' => $field_values,
			'original_price' => floatval(get_post_meta($product_id, '_price', true)),
			'calculation_time' => current_time('mysql')
		);
		
		$cart_item_data['pwcp_unique_key'] = md5(wp_json_encode($field_values) . time());
		
		return $cart_item_data;
	}
	
	/**
	 * Update cart item prices
	 */
	public function pwcp_update_cart_item_prices($cart) {
		if (is_admin() && !defined('DOING_AJAX')) {
			return;
		}
		
		foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
			if (isset($cart_item['pwcp_calculator_data'])) {
				$calculator_data = $cart_item['pwcp_calculator_data'];
				$calculated_price = floatval($calculator_data['calculated_price']);
				
				if ($calculated_price > 0) {
					$cart_item['data']->set_price($calculated_price);
					$cart_item['data']->set_regular_price($calculated_price);
					$cart_item['data']->set_sale_price('');
				}
			}
		}
	}
	
	/**
	 * Display calculator data in cart
	 */
	public function pwcp_display_calculator_data_in_cart($item_data, $cart_item) {
		if (!isset($cart_item['pwcp_calculator_data'])) {
			return $item_data;
		}
		
		$calculator_data = $cart_item['pwcp_calculator_data'];
		
		if (!empty($calculator_data['calculator_name'])) {
			$item_data[] = array(
				'key' => esc_html__('Calculator', 'pricewise-calculator-pro'),
				'value' => sanitize_text_field($calculator_data['calculator_name']),
				'display' => ''
			);
		}
		
		$item_data[] = array(
			'key' => esc_html__('Calculated Price', 'pricewise-calculator-pro'),
			'value' => wc_price($calculator_data['calculated_price']),
			'display' => ''
		);
		
		if (!empty($calculator_data['field_values'])) {
			$fields = $this->pwcp_get_calculator_fields($calculator_data['calculator_id']);
			$field_labels = array();
			
			foreach ($fields as $field) {
				$field_labels[sanitize_key($field->field_key)] = sanitize_text_field($field->field_label);
			}
			
			foreach ($calculator_data['field_values'] as $key => $value) {
				$label = isset($field_labels[$key]) ? $field_labels[$key] : ucwords(str_replace('_', ' ', $key));
				
				if (is_array($value)) {
					$value = implode(', ', array_map('sanitize_text_field', $value));
				} else {
					$value = sanitize_text_field($value);
				}
				
				$item_data[] = array(
					'key' => $label,
					'value' => $value,
					'display' => ''
				);
			}
		}
		
		return $item_data;
	}
	
	/**
	 * Save calculator data to order
	 */
	public function pwcp_save_calculator_data_to_order($item, $cart_item_key, $values, $order) {
		if (!isset($values['pwcp_calculator_data'])) {
			return;
		}
		
		$calculator_data = $values['pwcp_calculator_data'];
		
		$item->add_meta_data('_pwcp_calculator_id', $calculator_data['calculator_id'], true);
		$item->add_meta_data('_pwcp_calculator_name', $calculator_data['calculator_name'], true);
		$item->add_meta_data('_pwcp_calculated_price', $calculator_data['calculated_price'], true);
		$item->add_meta_data('_pwcp_calculation_time', $calculator_data['calculation_time'], true);
		
		if (!empty($calculator_data['field_values'])) {
			$fields = $this->pwcp_get_calculator_fields($calculator_data['calculator_id']);
			$field_labels = array();
			
			foreach ($fields as $field) {
				$field_labels[sanitize_key($field->field_key)] = sanitize_text_field($field->field_label);
			}
			
			foreach ($calculator_data['field_values'] as $key => $value) {
				$label = isset($field_labels[$key]) ? $field_labels[$key] : ucwords(str_replace('_', ' ', $key));
				
				if (is_array($value)) {
					$value = implode(', ', array_map('sanitize_text_field', $value));
				} else {
					$value = sanitize_text_field($value);
				}
				
				$item->add_meta_data($label, $value, true);
			}
		}
		
		$this->pwcp_log_order_calculation($order->get_id(), $item->get_id(), $calculator_data);
	}
	
	/**
	 * Format order meta key display
	 */
	public function pwcp_format_order_meta_key($display_key, $meta, $item) {
		$hidden_keys = array(
			'_pwcp_calculator_id',
			'_pwcp_calculated_price',
			'_pwcp_calculation_time'
		);
		
		if (in_array($meta->key, $hidden_keys, true)) {
			return null;
		}
		
		if ($meta->key === '_pwcp_calculator_name') {
			return esc_html__('Calculator Used', 'pricewise-calculator-pro');
		}
		
		return $display_key;
	}
	
	/**
	 * Format order meta value display
	 */
	public function pwcp_format_order_meta_value($display_value, $meta, $item) {
		if ($meta->key === '_pwcp_calculator_name' && is_admin()) {
			$calculator_id = $item->get_meta('_pwcp_calculator_id');
			if ($calculator_id) {
				$edit_url = admin_url('admin.php?page=pwcp-add-calculator&action=edit&calculator_id=' . $calculator_id);
				return '<a href="' . esc_url($edit_url) . '">' . esc_html($display_value) . '</a>';
			}
		}
		
		return $display_value;
	}
	
	/**
	 * Get calculator fields
	 */
	private function pwcp_get_calculator_fields($calculator_id) {
		if (!class_exists('PWCP_Database')) {
			return array();
		}
		
		return PWCP_Database::pwcp_get_calculator_fields($calculator_id);
	}
	
	/**
	 * Log order calculation
	 */
	private function pwcp_log_order_calculation($order_id, $item_id, $calculator_data) {
		if (!class_exists('PWCP_Database')) {
			return;
		}
		
		$log_data = array(
			'calculator_id' => $calculator_data['calculator_id'],
			'product_id' => 0,
			'field_values' => $calculator_data['field_values'],
			'calculated_price' => $calculator_data['calculated_price'],
			'session_id' => 'order_' . $order_id . '_item_' . $item_id
		);
		
		PWCP_Database::pwcp_save_calculation_log($log_data);
	}
	
	/**
	 * Add product data tab
	 */
	public function pwcp_add_product_data_tab($tabs) {
		$tabs['pwcp_calculator'] = array(
			'label' => esc_html__('Calculator', 'pricewise-calculator-pro'),
			'target' => 'pwcp_calculator_product_data',
			'class' => array('show_if_simple', 'show_if_variable'),
			'priority' => 80
		);
		
		return $tabs;
	}
	
	/**
	 * Display product data panel - FIXED: No inline styles
	 */
	public function pwcp_product_data_panel() {
		global $post;
		
		if (!$post) {
			return;
		}
		
		$product_id = $post->ID;
		$assigned_calculators = get_post_meta($product_id, '_pwcp_assigned_calculators', true);
		
		if (!is_array($assigned_calculators)) {
			$assigned_calculators = array();
		}
		
		global $wpdb;
		$table = $wpdb->prefix . 'pwcp_calculators';
		
		$calculators = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT id, name, status, assignment_type FROM `{$table}` WHERE status = %s AND assignment_type = %s ORDER BY name ASC",
				'active',
				'product'
			)
		);
		
		?>
		<div id="pwcp_calculator_product_data" class="panel woocommerce_options_panel">
			<div class="options_group">
				<p class="form-field">
					<label for="pwcp_assigned_calculators"><?php esc_html_e('Assigned Product-Specific Calculators', 'pricewise-calculator-pro'); ?></label>
					<select id="pwcp_assigned_calculators" name="pwcp_assigned_calculators[]" class="wc-enhanced-select" multiple="multiple">
						<?php foreach ($calculators as $calculator): ?>
							<option value="<?php echo absint($calculator->id); ?>" <?php selected(in_array($calculator->id, $assigned_calculators)); ?>>
								<?php echo esc_html($calculator->name); ?>
							</option>
						<?php endforeach; ?>
					</select>
				</p>
				<p class="form-field pwcp-calculator-description">
					<span class="description">
						<?php esc_html_e('Select product-specific calculators for this product. Note: Global and category calculators are applied automatically.', 'pricewise-calculator-pro'); ?>
					</span>
				</p>
				<div class="pwcp-calculator-info">
					<h4><?php esc_html_e('Calculator Priority System', 'pricewise-calculator-pro'); ?></h4>
					<p><?php esc_html_e('This product may have calculators from multiple sources:', 'pricewise-calculator-pro'); ?></p>
					<ol>
						<li><strong><?php esc_html_e('Product-Specific', 'pricewise-calculator-pro'); ?></strong>: <?php esc_html_e('Calculators assigned above (highest priority)', 'pricewise-calculator-pro'); ?></li>
						<li><strong><?php esc_html_e('Category-Based', 'pricewise-calculator-pro'); ?></strong>: <?php esc_html_e('Calculators assigned to this product\'s categories', 'pricewise-calculator-pro'); ?></li>
						<li><strong><?php esc_html_e('Global', 'pricewise-calculator-pro'); ?></strong>: <?php esc_html_e('Calculators set to appear on all products (lowest priority)', 'pricewise-calculator-pro'); ?></li>
					</ol>
					<p><em><?php esc_html_e('The highest priority calculator will be displayed. Within each type, admin priority numbers determine order.', 'pricewise-calculator-pro'); ?></em></p>
				</div>
			</div>
		</div>
		<?php
	}
	
	/**
	 * Save product calculator settings - FIXED: Added nonce verification
	 */
	public function pwcp_save_product_calculator_settings($post_id) {
		// Verify nonce for CSRF protection
if (!isset($_POST['woocommerce_meta_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['woocommerce_meta_nonce'])), 'woocommerce_save_data')) {
				return;
		}
		
		// Check user capabilities
		if (!current_user_can('edit_post', $post_id)) {
			return;
		}
		
		// Check if this is an autosave
		if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
			return;
		}
		
		$assigned_calculators = isset($_POST['pwcp_assigned_calculators']) ? 
			array_map('absint', (array) $_POST['pwcp_assigned_calculators']) : array();
		update_post_meta($post_id, '_pwcp_assigned_calculators', $assigned_calculators);

		
		// Clear cache when settings change
		wp_cache_delete('product_calculators_' . $post_id, $this->cache_group);
	}
}

// Initialize WooCommerce integration
if (class_exists('WooCommerce')) {
	new PWCP_WooCommerce_Integration();
}