<?php
/**
 * System Requirements Checker
 * 
 * Checks if the system meets minimum requirements for PriceWise Calculator Pro
 * File: includes/class-pwcp-system-checker.php
 * 
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * PWCP System Checker Class
 * 
 * @since 1.0.0
 */
class PWCP_System_Checker {
    
    /**
     * Check if system meets all requirements
     * 
     * @return bool
     * @since 1.0.0
     */
    public static function pwcp_check_requirements() {
        $errors = array();
        
        if (!self::pwcp_check_php_version()) {
            $errors[] = sprintf(
                /* translators: %1$s: Required PHP version, %2$s: Current PHP version */
                __('PriceWise Calculator Pro requires PHP version %1$s or higher. You are running version %2$s.', 'pricewise-calculator-pro'),
                esc_html(PWCP_MIN_PHP_VERSION),
                esc_html(PHP_VERSION)
            );
        }
        
        if (!self::pwcp_check_wp_version()) {
            $errors[] = sprintf(
                /* translators: %1$s: Required WordPress version, %2$s: Current WordPress version */
                __('PriceWise Calculator Pro requires WordPress version %1$s or higher. You are running version %2$s.', 'pricewise-calculator-pro'),
                esc_html(PWCP_MIN_WP_VERSION),
                esc_html(get_bloginfo('version'))
            );
        }
        
        $missing_extensions = self::pwcp_check_php_extensions();
        if (!empty($missing_extensions)) {
            $errors[] = sprintf(
                /* translators: %s: List of missing PHP extensions */
                __('PriceWise Calculator Pro requires the following PHP extensions: %s', 'pricewise-calculator-pro'),
                esc_html(implode(', ', $missing_extensions))
            );
        }
        
        if (!empty($errors)) {
            self::pwcp_display_error_notice($errors);
            return false;
        }
        
        return true;
    }
    
    /**
     * Check PHP version
     * 
     * @return bool
     * @since 1.0.0
     */
    private static function pwcp_check_php_version() {
        return version_compare(PHP_VERSION, PWCP_MIN_PHP_VERSION, '>=');
    }
    
    /**
     * Check WordPress version
     * 
     * @return bool
     * @since 1.0.0
     */
    private static function pwcp_check_wp_version() {
        global $wp_version;
        return version_compare($wp_version, PWCP_MIN_WP_VERSION, '>=');
    }
    
    /**
     * Check required PHP extensions
     * 
     * @return array Missing extensions
     * @since 1.0.0
     */
    private static function pwcp_check_php_extensions() {
        $required_extensions = array(
            'json',
            'mbstring',
            'curl'
        );
        
        $missing_extensions = array();
        
        foreach ($required_extensions as $extension) {
            if (!extension_loaded($extension)) {
                $missing_extensions[] = $extension;
            }
        }
        
        return $missing_extensions;
    }
    
    /**
     * Display error notice
     * 
     * @param array $errors Array of error messages
     * @since 1.0.0
     */
    private static function pwcp_display_error_notice($errors) {
        if (is_admin()) {
            add_action('admin_notices', function() use ($errors) {
                $class = 'notice notice-error';
                $title = esc_html__('PriceWise Calculator Pro - System Requirements Not Met', 'pricewise-calculator-pro');
                
                printf('<div class="%1$s"><h3>%2$s</h3>', esc_attr($class), esc_html($title));
                
                foreach ($errors as $error) {
                    printf('<p>%s</p>', wp_kses_post($error));
                }
                
                echo '</div>';
            });
            
            add_action('admin_init', function() {
                if (current_user_can('activate_plugins')) {
                    deactivate_plugins(PWCP_PLUGIN_BASENAME);
                }
            });
        }
    }
    
    /**
     * Check if WooCommerce is active and compatible
     * 
     * @return bool
     * @since 1.0.0
     */
    public static function pwcp_check_woocommerce() {
        if (!class_exists('WooCommerce')) {
            return false;
        }
        
        if (defined('WC_VERSION')) {
            return version_compare(WC_VERSION, PWCP_MIN_WC_VERSION, '>=');
        }
        
        return true;
    }
    
    /**
     * Get system information for debugging
     * 
     * @return array
     * @since 1.0.0
     */
    public static function pwcp_get_system_info() {
        global $wp_version;
        
        return array(
            'php_version' => PHP_VERSION,
            'wp_version' => $wp_version,
            'wc_version' => defined('WC_VERSION') ? WC_VERSION : esc_html__('Not installed', 'pricewise-calculator-pro'),
            'plugin_version' => PWCP_PLUGIN_VERSION,
            'php_extensions' => get_loaded_extensions(),
            'memory_limit' => ini_get('memory_limit'),
            'max_execution_time' => ini_get('max_execution_time'),
            'upload_max_filesize' => ini_get('upload_max_filesize'),
            'active_plugins' => get_option('active_plugins', array()),
            'active_theme' => wp_get_theme()->get('Name')
        );
    }
}