<?php
/**
 * Response Factory
 * 
 * Standardizes all AJAX and form responses
 * 
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Response Factory Class
 * 
 * Provides standardized response methods for AJAX and form operations
 * 
 * @since 1.0.0
 */
class PWCP_Response_Factory {
    
    /**
     * Send success response for calculator operations
     * 
     * @param string $action Action performed (save, update, delete, etc.)
     * @param int $calculator_id Calculator ID
     * @param array $additional_data Additional data to include
     * @return void
     */
    public static function calculator_success($action, $calculator_id, $additional_data = array()) {
        $messages = array(
            'save'      => __('Calculator saved successfully!', 'pricewise-calculator-pro'),
            'update'    => __('Calculator updated successfully!', 'pricewise-calculator-pro'),
            'delete'    => __('Calculator deleted successfully!', 'pricewise-calculator-pro'),
            'duplicate' => __('Calculator duplicated successfully!', 'pricewise-calculator-pro'),
        );
        
        $message = isset($messages[$action])
            ? $messages[$action]
            : __('Operation completed successfully!', 'pricewise-calculator-pro');
        
        $response = array_merge(array(
            'message'       => esc_html($message),
            'calculator_id' => absint($calculator_id),
            'action'        => sanitize_key($action),
            'timestamp'     => current_time('timestamp'),
        ), $additional_data);
        
        wp_send_json_success($response);
    }
    
    /**
     * Send success response for field operations
     * 
     * @param string $action Action performed
     * @param int $field_id Field ID
     * @param array $additional_data Additional data
     * @return void
     */
    public static function field_success($action, $field_id = 0, $additional_data = array()) {
        $messages = array(
            'save'    => __('Field saved successfully!', 'pricewise-calculator-pro'),
            'update'  => __('Field updated successfully!', 'pricewise-calculator-pro'),
            'delete'  => __('Field deleted successfully!', 'pricewise-calculator-pro'),
            'reorder' => __('Field order updated successfully!', 'pricewise-calculator-pro'),
        );
        
        $message = isset($messages[$action])
            ? $messages[$action]
            : __('Field operation completed successfully!', 'pricewise-calculator-pro');
        
        $response = array_merge(array(
            'message'   => esc_html($message),
            'action'    => sanitize_key($action),
            'timestamp' => current_time('timestamp'),
        ), $additional_data);
        
        if ($field_id > 0) {
            $response['field_id'] = absint($field_id);
        }
        
        wp_send_json_success($response);
    }
    
    /**
     * Send auto-save success response
     * 
     * @param int $calculator_id Calculator ID
     * @param array $additional_data Additional data
     * @return void
     */
    public static function auto_save_success($calculator_id, $additional_data = array()) {
        $response = array_merge(array(
            'message'       => esc_html__('Auto-saved successfully!', 'pricewise-calculator-pro'),
            'calculator_id' => absint($calculator_id),
            'is_auto_save'  => true,
            'timestamp'     => current_time('timestamp'),
        ), $additional_data);
        
        wp_send_json_success($response);
    }
    
    /**
     * Send security error response
     * 
     * @param string $custom_message Custom error message (optional)
     * @return void
     */
    public static function security_error($custom_message = '') {
        $message = $custom_message !== ''
            ? esc_html($custom_message)
            : esc_html__('Security verification failed.', 'pricewise-calculator-pro');
        
        wp_send_json_error(array(
            'message'    => $message,
            'error_type' => 'security',
            'timestamp'  => current_time('timestamp'),
        ));
    }
    
    /**
     * Send permission error response
     * 
     * @param string $custom_message Custom error message (optional)
     * @return void
     */
    public static function permission_error($custom_message = '') {
        $message = $custom_message !== ''
            ? esc_html($custom_message)
            : esc_html__('Insufficient permissions.', 'pricewise-calculator-pro');
        
        wp_send_json_error(array(
            'message'    => $message,
            'error_type' => 'permission',
            'timestamp'  => current_time('timestamp'),
        ));
    }
    
    /**
     * Send validation error response
     * 
     * @param string|array $errors Error message(s)
     * @param string $field_name Optional field name for field-specific errors
     * @return void
     */
    public static function validation_error($errors, $field_name = '') {
        $message = is_array($errors) ? implode(' ', $errors) : $errors;
        
        $response = array(
            'message'    => esc_html($message),
            'error_type' => 'validation',
            'timestamp'  => current_time('timestamp'),
        );
        
        if (!empty($field_name)) {
            $response['field'] = sanitize_key($field_name);
        }
        
        wp_send_json_error($response);
    }
    
    /**
     * Send not found error response
     * 
     * @param string $resource_type Type of resource (calculator, field, etc.)
     * @param int $resource_id Resource ID
     * @return void
     */
    public static function not_found_error($resource_type, $resource_id = 0) {
        $messages = array(
            'calculator' => __('Calculator not found.', 'pricewise-calculator-pro'),
            'field'      => __('Field not found.', 'pricewise-calculator-pro'),
        );
        
    if (isset($messages[$resource_type])) {
    $message = $messages[$resource_type];
} else {
    $clean_type = sanitize_text_field($resource_type);
    $message = sprintf(
        /* translators: %s: Resource type (e.g., Calculator, Field) */
        __('%s not found.', 'pricewise-calculator-pro'),
        $clean_type
    );
}
        
        $response = array(
            'message'       => esc_html($message),
            'error_type'    => 'not_found',
            'resource_type' => sanitize_key($resource_type),
            'timestamp'     => current_time('timestamp'),
        );
        
        if ($resource_id > 0) {
            $response['resource_id'] = absint($resource_id);
        }
        
        wp_send_json_error($response);
    }
    
    /**
     * Send database error response
     * 
     * @param string $operation Operation that failed
     * @param string $custom_message Custom error message
     * @return void
     */
    public static function database_error($operation = '', $custom_message = '') {
        $operation_messages = array(
            'create_field'       => __('Failed to create field in database.', 'pricewise-calculator-pro'),
            'update_field'       => __('Failed to update field in database.', 'pricewise-calculator-pro'),
            'delete_field'       => __('Failed to delete field from database.', 'pricewise-calculator-pro'),
            'reorder_fields'     => __('Failed to reorder fields in database.', 'pricewise-calculator-pro'),
            'create_calculator'  => __('Failed to create calculator in database.', 'pricewise-calculator-pro'),
            'update_calculator'  => __('Failed to update calculator in database.', 'pricewise-calculator-pro'),
            'delete_calculator'  => __('Failed to delete calculator from database.', 'pricewise-calculator-pro'),
        );
        
        if ($custom_message !== '') {
            $message = esc_html($custom_message);
        } elseif (!empty($operation) && isset($operation_messages[$operation])) {
            $message = $operation_messages[$operation];
        } else {
            $message = esc_html__('Database error occurred.', 'pricewise-calculator-pro');
        }
        
        $response = array(
            'message'    => $message,
            'error_type' => 'database',
            'timestamp'  => current_time('timestamp'),
        );
        
        if (!empty($operation)) {
            $response['operation'] = sanitize_key($operation);
        }
        
        wp_send_json_error($response);
    }
    
    /**
     * Send generic error response
     * 
     * @param string $message Error message
     * @param string $error_type Error type
     * @param array $additional_data Additional error data
     * @return void
     */
    public static function generic_error($message, $error_type = 'general', $additional_data = array()) {
        $response = array_merge(array(
            'message'    => esc_html($message),
            'error_type' => sanitize_key($error_type),
            'timestamp'  => current_time('timestamp'),
        ), $additional_data);
        
        wp_send_json_error($response);
    }
    
    /**
     * Send form redirect response
     * 
     * @param string $url Redirect URL
     * @param string $message Success message
     * @param string $message_type Message type (success, error, warning)
     * @return void
     */
    public static function form_redirect($url, $message, $message_type = 'success') {
        $redirect_url = add_query_arg(array(
            'pwcp_message' => sanitize_key($message_type),
            'pwcp_text'    => rawurlencode($message),
        ), $url);
        
        wp_safe_redirect($redirect_url);
        exit;
    }
    
    /**
     * Send calculation result response
     * 
     * @param float $calculated_price Calculated price
     * @param array $field_values Field values used
     * @param int $calculator_id Calculator ID
     * @param array $additional_data Additional data
     * @return void
     */
    public static function calculation_success($calculated_price, $field_values, $calculator_id, $additional_data = array()) {
        $response = array_merge(array(
            'calculated_price' => floatval($calculated_price),
            'formatted_price'  => function_exists('wc_price') ? wc_price($calculated_price) : '$' . number_format($calculated_price, 2),
            'field_values'     => $field_values,
            'calculator_id'    => absint($calculator_id),
            'calculation_time' => current_time('mysql'),
            'timestamp'        => current_time('timestamp'),
        ), $additional_data);
        
        wp_send_json_success($response);
    }
}