<?php
/**
 * Premium Feature Helper
 * 
 * Manages premium feature checks and upgrade prompts
 * 
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class PWCP_Premium_Helper {
    
    const UPGRADE_URL = 'https://devtonicstudios.com';
    
    /**
     * Check if feature is available in free version
     * 
     * @param string $feature Feature name
     * @return bool
     */
    public static function is_feature_available($feature) {
        $free_features = array(
            'global_assignment',
            'number_field',
            'basic_formula'
        );
        
        return in_array($feature, $free_features, true);
    }
    
    /**
     * Check if field type is available in free version
     * 
     * @param string $field_type Field type
     * @return bool
     */
    public static function is_field_type_available($field_type) {
        $free_field_types = array('number');
        return in_array($field_type, $free_field_types, true);
    }
    
    /**
     * Check if assignment type is available in free version
     * 
     * @param string $assignment_type Assignment type
     * @return bool
     */
    public static function is_assignment_type_available($assignment_type) {
        $free_assignment_types = array('global');
        return in_array($assignment_type, $free_assignment_types, true);
    }
    
    /**
     * Get upgrade message
     * 
     * @param string $feature Feature name
     * @return string
     */
    public static function get_upgrade_message($feature = '') {
        $message = sprintf(
            /* translators: %s: URL to the premium version purchase page */
            __('This is a premium feature. <a href="%s" target="_blank" class="pwcp-upgrade-link">Purchase our premium version</a> to unlock this feature.', 'pricewise-calculator-pro'),
            esc_url(self::UPGRADE_URL)
        );
        
        return $message;
    }
    
    /**
     * Get upgrade button HTML
     * 
     * @return string
     */
    public static function get_upgrade_button() {
        return sprintf(
            '<a href="%s" target="_blank" class="button button-primary pwcp-upgrade-btn">%s</a>',
            esc_url(self::UPGRADE_URL),
            esc_html__('Upgrade to Premium', 'pricewise-calculator-pro')
        );
    }
    
    /**
     * Render premium badge
     * 
     * @return string
     */
    public static function render_premium_badge() {
        return '<span class="pwcp-premium-badge">' . esc_html__('Premium', 'pricewise-calculator-pro') . '</span>';
    }
    
    /**
     * Render locked overlay for premium features
     * 
     * @param string $feature Feature name
     * @return string
     */
    public static function render_locked_overlay($feature = '') {
        $message = self::get_upgrade_message($feature);
        $button = self::get_upgrade_button();
        
        return sprintf(
            '<div class="pwcp-locked-overlay">
                <div class="pwcp-locked-content">
                    <svg class="pwcp-lock-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <rect x="3" y="11" width="18" height="11" rx="2" ry="2"/>
                        <path d="M7 11V7a5 5 0 0 1 10 0v4"/>
                    </svg>
                    <h3>%s</h3>
                    <p>%s</p>
                    %s
                </div>
            </div>',
            esc_html__('Premium Feature', 'pricewise-calculator-pro'),
            wp_kses_post($message),
            $button
        );
    }
}