<?php
/**
 * Plugin Loader
 * 
 * Manages all hooks and filters for PriceWise Calculator Pro
 * File: includes/class-pwcp-loader.php
 * 
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * PWCP Loader Class
 * 
 * Maintains arrays of all hooks that are registered throughout the plugin,
 * and registers them with the WordPress API.
 * 
 * @since 1.0.0
 */
class PWCP_Loader {
    
    /**
     * Array of actions registered with WordPress
     * 
     * @var array
     * @since 1.0.0
     */
    protected $pwcp_actions;
    
    /**
     * Array of filters registered with WordPress
     * 
     * @var array
     * @since 1.0.0
     */
    protected $pwcp_filters;
    
    /**
     * Array of shortcodes registered with WordPress
     * 
     * @var array
     * @since 1.0.0
     */
    protected $pwcp_shortcodes;
    
    /**
     * Initialize the collections used to maintain the actions and filters
     * 
     * @since 1.0.0
     */
    public function __construct() {
        $this->pwcp_actions = array();
        $this->pwcp_filters = array();
        $this->pwcp_shortcodes = array();
    }
    
    /**
     * Add a new action to the collection to be registered with WordPress
     * 
     * @param string $hook The name of the WordPress action
     * @param object $component A reference to the instance of the object
     * @param string $callback The name of the function definition on the $component
     * @param int $priority Optional. Default 10. The priority at which the function should be fired
     * @param int $accepted_args Optional. Default 1. The number of arguments that should be passed to the $callback
     * @since 1.0.0
     */
    public function pwcp_add_action($hook, $component, $callback, $priority = 10, $accepted_args = 1) {
        $this->pwcp_actions = $this->pwcp_add($this->pwcp_actions, $hook, $component, $callback, $priority, $accepted_args);
    }
    
    /**
     * Add a new filter to the collection to be registered with WordPress
     * 
     * @param string $hook The name of the WordPress filter
     * @param object $component A reference to the instance of the object
     * @param string $callback The name of the function definition on the $component
     * @param int $priority Optional. Default 10. The priority at which the function should be fired
     * @param int $accepted_args Optional. Default 1. The number of arguments that should be passed to the $callback
     * @since 1.0.0
     */
    public function pwcp_add_filter($hook, $component, $callback, $priority = 10, $accepted_args = 1) {
        $this->pwcp_filters = $this->pwcp_add($this->pwcp_filters, $hook, $component, $callback, $priority, $accepted_args);
    }
    
    /**
     * Add a new shortcode to the collection to be registered with WordPress
     * 
     * @param string $tag The name of the new shortcode
     * @param object $component A reference to the instance of the object
     * @param string $callback The name of the function definition on the $component
     * @since 1.0.0
     */
    public function pwcp_add_shortcode($tag, $component, $callback) {
        $this->pwcp_shortcodes = $this->pwcp_add($this->pwcp_shortcodes, $tag, $component, $callback);
    }
    
    /**
     * A utility function that is used to register the actions and hooks into a single
     * collection
     * 
     * @param array $hooks The collection of hooks that is being registered
     * @param string $hook The name of the WordPress filter or action
     * @param object $component A reference to the instance of the object
     * @param string $callback The name of the function definition on the $component
     * @param int $priority The priority at which the function should be fired
     * @param int $accepted_args The number of arguments that should be passed to the $callback
     * @return array The collection of actions and filters registered with WordPress
     * @since 1.0.0
     */
    private function pwcp_add($hooks, $hook, $component, $callback, $priority = 10, $accepted_args = 1) {
        $hooks[] = array(
            'hook' => $hook,
            'component' => $component,
            'callback' => $callback,
            'priority' => $priority,
            'accepted_args' => $accepted_args
        );
        
        return $hooks;
    }
    
    /**
     * Register the filters and actions with WordPress
     * 
     * @since 1.0.0
     */
    public function pwcp_run() {
        // Register actions
        foreach ($this->pwcp_actions as $hook) {
            add_action(
                $hook['hook'],
                array($hook['component'], $hook['callback']),
                $hook['priority'],
                $hook['accepted_args']
            );
        }
        
        // Register filters
        foreach ($this->pwcp_filters as $hook) {
            add_filter(
                $hook['hook'],
                array($hook['component'], $hook['callback']),
                $hook['priority'],
                $hook['accepted_args']
            );
        }
        
        // Register shortcodes
        foreach ($this->pwcp_shortcodes as $shortcode) {
            add_shortcode(
                $shortcode['hook'],
                array($shortcode['component'], $shortcode['callback'])
            );
        }
    }
    
    /**
     * Remove a specific action from the collection
     * 
     * @param string $hook The name of the WordPress action
     * @param object $component A reference to the instance of the object
     * @param string $callback The name of the function definition on the $component
     * @since 1.0.0
     */
    public function pwcp_remove_action($hook, $component, $callback) {
        $this->pwcp_actions = $this->pwcp_remove($this->pwcp_actions, $hook, $component, $callback);
    }
    
    /**
     * Remove a specific filter from the collection
     * 
     * @param string $hook The name of the WordPress filter
     * @param object $component A reference to the instance of the object
     * @param string $callback The name of the function definition on the $component
     * @since 1.0.0
     */
    public function pwcp_remove_filter($hook, $component, $callback) {
        $this->pwcp_filters = $this->pwcp_remove($this->pwcp_filters, $hook, $component, $callback);
    }
    
    /**
     * Remove a specific shortcode from the collection
     * 
     * @param string $tag The name of the shortcode
     * @param object $component A reference to the instance of the object
     * @param string $callback The name of the function definition on the $component
     * @since 1.0.0
     */
    public function pwcp_remove_shortcode($tag, $component, $callback) {
        $this->pwcp_shortcodes = $this->pwcp_remove($this->pwcp_shortcodes, $tag, $component, $callback);
    }
    
    /**
     * Utility function to remove hooks from collections
     * 
     * @param array $hooks The collection of hooks
     * @param string $hook The name of the WordPress filter or action
     * @param object $component A reference to the instance of the object
     * @param string $callback The name of the function definition on the $component
     * @return array The updated collection
     * @since 1.0.0
     */
    private function pwcp_remove($hooks, $hook, $component, $callback) {
        foreach ($hooks as $key => $registered_hook) {
            if ($registered_hook['hook'] === $hook && 
                $registered_hook['component'] === $component && 
                $registered_hook['callback'] === $callback) {
                unset($hooks[$key]);
            }
        }
        
        return array_values($hooks); // Re-index array
    }
    
    /**
     * Get all registered actions
     * 
     * @return array
     * @since 1.0.0
     */
    public function pwcp_get_actions() {
        return $this->pwcp_actions;
    }
    
    /**
     * Get all registered filters
     * 
     * @return array
     * @since 1.0.0
     */
    public function pwcp_get_filters() {
        return $this->pwcp_filters;
    }
    
    /**
     * Get all registered shortcodes
     * 
     * @return array
     * @since 1.0.0
     */
    public function pwcp_get_shortcodes() {
        return $this->pwcp_shortcodes;
    }
}