<?php
/**
 * Input Sanitizer
 * 
 * Centralized input sanitization and validation
 * 
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Input Sanitizer Class
 * 
 * Handles input sanitization for calculator data
 * 
 * @since 1.0.0
 */
class PWCP_Input_Sanitizer {
    
    /**
     * Sanitize calculator data from POST
     * 
     * @param array $raw_data Raw POST data
     * @return array Sanitized calculator data
     */
    public static function sanitize_calculator_data($raw_data) {
        return array(
            'id' => isset($raw_data['calculator_id']) ? absint($raw_data['calculator_id']) : 0,
            'name' => isset($raw_data['calculator_name']) ? sanitize_text_field($raw_data['calculator_name']) : '',
            'description' => isset($raw_data['calculator_description']) ? sanitize_textarea_field($raw_data['calculator_description']) : '',
            'formula' => isset($raw_data['calculator_formula']) ? sanitize_textarea_field($raw_data['calculator_formula']) : '',
            'status' => isset($raw_data['calculator_status']) ? sanitize_text_field($raw_data['calculator_status']) : 'active',
            'assignment_type' => isset($raw_data['assignment_type']) ? sanitize_text_field($raw_data['assignment_type']) : 'global',
            'priority' => isset($raw_data['calculator_priority']) ? absint($raw_data['calculator_priority']) : 1,
            'hide_quantity_selector' => isset($raw_data['hide_quantity_selector']) ? 1 : 0,
            'assignment_ids' => isset($raw_data['assignment_ids']) ? self::sanitize_assignment_ids($raw_data['assignment_ids']) : array()
        );
    }
    
    /**
     * Sanitize field data from POST
     * 
     * @param array $raw_data Raw field data
     * @return array Sanitized field data
     */
    public static function sanitize_field_data($raw_data) {
        return array(
            'id' => isset($raw_data['id']) ? absint($raw_data['id']) : 0,
            'key' => isset($raw_data['key']) ? sanitize_key($raw_data['key']) : '',
            'label' => isset($raw_data['label']) ? sanitize_text_field($raw_data['label']) : '',
            'type' => isset($raw_data['type']) ? sanitize_text_field($raw_data['type']) : 'text',
            'required' => !empty($raw_data['required']) ? 1 : 0,
            'default_value' => isset($raw_data['default_value']) ? sanitize_text_field($raw_data['default_value']) : '',
            'options' => isset($raw_data['options']) ? sanitize_textarea_field($raw_data['options']) : '',
            'order' => isset($raw_data['order']) ? absint($raw_data['order']) : 0,
            'validation_rules' => self::sanitize_validation_rules($raw_data)
        );
    }
    
    /**
     * Sanitize validation rules
     * 
     * @param array $raw_data Raw data containing validation rules
     * @return array Sanitized validation rules
     */
    private static function sanitize_validation_rules($raw_data) {
        $rules = array();
        
        if (isset($raw_data['min']) && $raw_data['min'] !== '') {
            $rules['min'] = floatval($raw_data['min']);
        }
        
        if (isset($raw_data['max']) && $raw_data['max'] !== '') {
            $rules['max'] = floatval($raw_data['max']);
        }
        
        if (isset($raw_data['step']) && $raw_data['step'] !== '') {
            $rules['step'] = floatval($raw_data['step']);
        }
        
        return $rules;
    }
    
    /**
     * Sanitize assignment IDs
     * 
     * @param mixed $assignment_ids Assignment IDs to sanitize
     * @return array Sanitized assignment IDs
     */
    private static function sanitize_assignment_ids($assignment_ids) {
        if (is_string($assignment_ids)) {
            $assignment_ids = json_decode($assignment_ids, true);
        }
        
        if (!is_array($assignment_ids)) {
            return array();
        }
        
        return array_map('absint', $assignment_ids);
    }
    
    /**
     * Get sanitized POST value
     * 
     * @param string $key POST key
     * @param string $type Sanitization type
     * @param mixed $default Default value
     * @return mixed Sanitized value
     */
    public static function get_post_value($key, $type = 'text', $default = '') {
        if (!isset($_POST[$key])) {
            return $default;
        }
        
        $value = wp_unslash($_POST[$key]);
        
        switch ($type) {
            case 'int':
                return absint($value);
            
            case 'float':
                return floatval($value);
            
            case 'array':
                return is_array($value) ? array_map('sanitize_text_field', $value) : array();
            
          case 'json':
    $sanitized_value = sanitize_text_field($value);
    $decoded = json_decode($sanitized_value, true);
    return is_array($decoded) ? self::sanitize_array_recursive($decoded) : array();
            
            case 'textarea':
                return sanitize_textarea_field($value);
            
            case 'key':
                return sanitize_key($value);
            
            case 'email':
                return sanitize_email($value);
            
            case 'url':
                return esc_url_raw($value);
            
            case 'text':
            default:
                return sanitize_text_field($value);
        }
    }
    
    /**
     * Recursively sanitize array
     * 
     * @param array $array Array to sanitize
     * @return array Sanitized array
     */
    private static function sanitize_array_recursive($array) {
        $sanitized = array();
        
        foreach ($array as $key => $value) {
            $clean_key = sanitize_key($key);
            
            if (is_array($value)) {
                $sanitized[$clean_key] = self::sanitize_array_recursive($value);
            } else {
                $sanitized[$clean_key] = sanitize_text_field($value);
            }
        }
        
        return $sanitized;
    }
    
    /**
     * Validate required fields
     * 
     * @param array $required_fields Required field names
     * @param array $data Data to validate (defaults to $_POST)
     * @return array Array with 'valid' boolean and 'missing' array
     */
    public static function validate_required($required_fields, $data = null) {
        if ($data === null) {
            $data = $_POST;
        }
        
        $missing = array();
        
        foreach ($required_fields as $field) {
            if (!isset($data[$field]) || empty($data[$field])) {
                $missing[] = $field;
            }
        }
        
        return array(
            'valid' => empty($missing),
            'missing' => $missing
        );
    }
    
    /**
     * Sanitize calculation field values
     * 
     * @param string $field_values_json JSON string of field values
     * @return array Sanitized field values
     */
    public static function sanitize_calculation_values($field_values_json) {
        if (empty($field_values_json)) {
            return array();
        }
        
        $field_values = json_decode(stripslashes($field_values_json), true);
        
        if (!is_array($field_values)) {
            return array();
        }
        
        $sanitized = array();
        
        foreach ($field_values as $key => $value) {
            $sanitized_key = sanitize_key($key);
            
            if (is_array($value)) {
                $sanitized[$sanitized_key] = array_map('sanitize_text_field', $value);
            } else {
                $sanitized[$sanitized_key] = sanitize_text_field($value);
            }
        }
        
        return $sanitized;
    }
    
    /**
     * Sanitize category assignments
     * 
     * @param string $categories_json JSON string of categories
     * @return array Sanitized categories
     */
    public static function sanitize_category_assignments($categories_json) {
        if (empty($categories_json)) {
            return array();
        }
        
        $categories = json_decode($categories_json, true);
        
        if (!is_array($categories)) {
            return array();
        }
        
        $sanitized = array();
        
        foreach ($categories as $category) {
            if (isset($category['id']) && isset($category['name'])) {
                $sanitized[] = array(
                    'id' => absint($category['id']),
                    'name' => sanitize_text_field($category['name'])
                );
            }
        }
        
        return $sanitized;
    }
}