<?php
/**
 * Database Management (FREE VERSION)
 * 
 * Handles database operations for PriceWise Calculator Pro
 * FREE VERSION: Only global calculators supported
 * 
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
	exit;
}

class PWCP_Database {
	
	const PWCP_DB_VERSION = '1.0.0';
	
	/**
	 * Validate allowed tables for security
	 */
	private static function pwcp_is_allowed_table($table) {
		global $wpdb;
		$allowed = array(
			$wpdb->prefix . 'pwcp_calculators',
			$wpdb->prefix . 'pwcp_calculator_fields',
			$wpdb->prefix . 'pwcp_calculation_logs',
			$wpdb->prefix . 'pwcp_calculator_assignments',
		);
		return in_array($table, $allowed, true);
	}
	
	/**
	 * Create plugin tables
	 */
	public static function pwcp_create_tables() {
		global $wpdb;
		
		$charset_collate = $wpdb->get_charset_collate();
		
		self::pwcp_create_calculators_table($charset_collate);
		self::pwcp_create_calculator_fields_table($charset_collate);
		self::pwcp_create_calculation_logs_table($charset_collate);
		self::pwcp_create_calculator_assignments_table($charset_collate);
		
		update_option('pwcp_db_version', self::PWCP_DB_VERSION);
		do_action('pwcp_tables_created');
	}
	
	/**
	 * Create calculators table
	 */
	private static function pwcp_create_calculators_table($charset_collate) {
		global $wpdb;
		
		$table_name = $wpdb->prefix . 'pwcp_calculators';
		
		$sql = "CREATE TABLE {$table_name} (
			id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
			name varchar(255) NOT NULL,
			description text,
			formula text NOT NULL,
			settings text,
			status varchar(20) DEFAULT 'active',
			assignment_type varchar(20) DEFAULT 'global',
			assignment_ids text,
			priority int(11) DEFAULT 1,
			hide_quantity_selector tinyint(1) DEFAULT 0,
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			PRIMARY KEY (id),
			KEY status (status),
			KEY assignment_type (assignment_type),
			KEY priority (priority)
		) {$charset_collate};";
		
		if (!function_exists('dbDelta')) {
			require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		}
		
		dbDelta($sql);
		
		if (defined('WP_DEBUG') && WP_DEBUG && defined('PWCP_DEBUG') && PWCP_DEBUG && !empty($wpdb->last_error)) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging for development only
			error_log('PWCP: calculators table error - ' . $wpdb->last_error);
		}
	}
	
	/**
	 * Create calculator fields table
	 */
	private static function pwcp_create_calculator_fields_table($charset_collate) {
		global $wpdb;
		
		$table_name = $wpdb->prefix . 'pwcp_calculator_fields';
		
		$sql = "CREATE TABLE {$table_name} (
			id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
			calculator_id bigint(20) UNSIGNED NOT NULL,
			field_key varchar(100) NOT NULL,
			field_label varchar(255) NOT NULL,
			field_type varchar(50) NOT NULL,
			field_options text,
			field_validation text,
			field_default varchar(255),
			field_required tinyint(1) DEFAULT 0,
			field_order int(11) DEFAULT 0,
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			PRIMARY KEY (id),
			KEY calculator_id (calculator_id),
			KEY field_order (field_order)
		) {$charset_collate};";
		
		if (!function_exists('dbDelta')) {
			require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		}
		
		dbDelta($sql);
		
		if (defined('WP_DEBUG') && WP_DEBUG && defined('PWCP_DEBUG') && PWCP_DEBUG && !empty($wpdb->last_error)) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging for development only
			error_log('PWCP: calculator_fields table error - ' . $wpdb->last_error);
		}
	}
	
	/**
	 * Create calculation logs table
	 */
	private static function pwcp_create_calculation_logs_table($charset_collate) {
		global $wpdb;
		
		$table_name = $wpdb->prefix . 'pwcp_calculation_logs';
		
		$sql = "CREATE TABLE {$table_name} (
			id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
			calculator_id bigint(20) UNSIGNED NOT NULL,
			user_id bigint(20) UNSIGNED,
			session_id varchar(100),
			input_data text,
			result_data text,
			product_id bigint(20) UNSIGNED,
			ip_address varchar(45),
			user_agent text,
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY (id),
			KEY calculator_id (calculator_id),
			KEY user_id (user_id),
			KEY session_id (session_id),
			KEY product_id (product_id),
			KEY created_at (created_at)
		) {$charset_collate};";
		
		if (!function_exists('dbDelta')) {
			require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		}
		
		dbDelta($sql);
		
		if (defined('WP_DEBUG') && WP_DEBUG && defined('PWCP_DEBUG') && PWCP_DEBUG && !empty($wpdb->last_error)) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging for development only
			error_log('PWCP: calculation_logs table error - ' . $wpdb->last_error);
		}
	}
	
	/**
	 * Create calculator assignments table
	 * Table exists for compatibility but not used in free version
	 */
	private static function pwcp_create_calculator_assignments_table($charset_collate) {
		global $wpdb;
		
		$table_name = $wpdb->prefix . 'pwcp_calculator_assignments';
		
		$sql = "CREATE TABLE {$table_name} (
			id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
			calculator_id bigint(20) UNSIGNED NOT NULL,
			assignment_type varchar(20) NOT NULL,
			assignment_id bigint(20) UNSIGNED NOT NULL,
			priority int(11) DEFAULT 1,
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY (id),
			KEY calculator_id (calculator_id),
			KEY assignment_type (assignment_type),
			KEY assignment_id (assignment_id),
			KEY priority (priority),
			UNIQUE KEY unique_assignment (calculator_id, assignment_type, assignment_id)
		) {$charset_collate};";
		
		if (!function_exists('dbDelta')) {
			require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		}
		
		dbDelta($sql);
		
		if (defined('WP_DEBUG') && WP_DEBUG && defined('PWCP_DEBUG') && PWCP_DEBUG && !empty($wpdb->last_error)) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging for development only
			error_log('PWCP: assignments table error - ' . $wpdb->last_error);
		}
	}
	
	/**
	 * Get all calculators
	 */
	public static function pwcp_get_calculators() {
		global $wpdb;
		
		$table_name = $wpdb->prefix . 'pwcp_calculators';
		
		if (!self::pwcp_is_allowed_table($table_name)) {
			return array();
		}
		
		$results = $wpdb->get_results(
			$wpdb->prepare(
				// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is properly escaped via prefix
				"SELECT id, name, description, formula, settings, status, assignment_type, assignment_ids, priority, hide_quantity_selector, created_at, updated_at 
				 FROM {$table_name} 
				 WHERE 1=%d 
				 ORDER BY created_at DESC",
				1
			)
		);
		
		return $results ? $results : array();
	}
	
	/**
	 * Get single calculator
	 */
	public static function pwcp_get_calculator($calculator_id) {
		global $wpdb;
		
		$calculator_id = absint($calculator_id);
		if (!$calculator_id) {
			return null;
		}
		
		$table_name = $wpdb->prefix . 'pwcp_calculators';
		
		return $wpdb->get_row(
			$wpdb->prepare(
				// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is properly escaped via prefix
				"SELECT id, name, description, formula, settings, status, assignment_type, assignment_ids, priority, hide_quantity_selector, created_at, updated_at 
				 FROM {$table_name} 
				 WHERE id = %d",
				$calculator_id
			)
		);
	}
	
	/**
	 * Get calculator fields
	 * 
	 * @param int $calculator_id Calculator ID
	 * @return array Array of field objects
	 */
	public static function pwcp_get_calculator_fields($calculator_id) {
		global $wpdb;
		
		$calculator_id = absint($calculator_id);
		if (!$calculator_id) {
			return array();
		}
		
		$table_name = $wpdb->prefix . 'pwcp_calculator_fields';
		
		$results = $wpdb->get_results(
			$wpdb->prepare(
				// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is properly escaped via prefix
				"SELECT id, calculator_id, field_key, field_label, field_type, field_options, 
				        field_validation, field_default, field_required, field_order, 
				        created_at, updated_at 
				 FROM {$table_name} 
				 WHERE calculator_id = %d 
				 ORDER BY field_order ASC",
				$calculator_id
			)
		);
		
		return $results ?: array();
	}
	
	/**
	 * Get calculators for a product
	 * FREE VERSION: Always returns only global calculators
	 */
	public static function pwcp_get_product_calculators($product_id) {
		global $wpdb;
		
		// FREE VERSION: Only return global calculators
		$calculators_table = $wpdb->prefix . 'pwcp_calculators';
		
		$results = $wpdb->get_results(
			$wpdb->prepare(
				// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is properly escaped via prefix
				"SELECT id, name, description, formula, settings, status, assignment_type, assignment_ids, priority, hide_quantity_selector, created_at, updated_at 
				 FROM {$calculators_table} 
				 WHERE status = %s 
				 AND assignment_type = %s
				 ORDER BY priority DESC, id ASC",
				'active',
				'global'
			)
		);
		
		return $results ? $results : array();
	}
	
	/**
	 * Get product calculators using new assignment system
	 * FREE VERSION: Disabled - returns empty
	 */
	private static function pwcp_get_product_calculators_new_system($product_id) {
		// Premium feature - disabled in free version
		return array();
	}
	
	/**
	 * Get product calculators using legacy assignment system
	 * FREE VERSION: Only returns global calculators
	 */
	private static function pwcp_get_product_calculators_legacy_system($product_id) {
		global $wpdb;
		
		$calculators_table = $wpdb->prefix . 'pwcp_calculators';
		
		// FREE VERSION: Only return global calculators
		$results = $wpdb->get_results(
			$wpdb->prepare(
				// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is properly escaped via prefix
				"SELECT id, name, description, formula, settings, status, assignment_type, assignment_ids, priority, hide_quantity_selector, created_at, updated_at 
				 FROM {$calculators_table} 
				 WHERE status = %s 
				 AND assignment_type = %s
				 ORDER BY priority DESC, id ASC",
				'active',
				'global'
			)
		);
		
		return $results ? $results : array();
	}
	
	/**
	 * Save calculation log
	 */
	public static function pwcp_save_calculation_log($log_data) {
		global $wpdb;
		
		if (!is_array($log_data) || empty($log_data['calculator_id'])) {
			return false;
		}
		
		$enable_logging = get_option('pwcp_enable_calculation_logging', '0');
		if ($enable_logging !== '1') {
			return true;
		}
		
		$table_name = $wpdb->prefix . 'pwcp_calculation_logs';
		
		$result = $wpdb->insert(
			$table_name,
			array(
				'calculator_id' => absint($log_data['calculator_id']),
				'user_id'       => isset($log_data['user_id']) ? absint($log_data['user_id']) : null,
				'session_id'    => isset($log_data['session_id']) ? sanitize_text_field($log_data['session_id']) : '',
				'input_data'    => isset($log_data['input_data']) ? wp_json_encode($log_data['input_data']) : '',
				'result_data'   => isset($log_data['result_data']) ? wp_json_encode($log_data['result_data']) : '',
				'product_id'    => isset($log_data['product_id']) ? absint($log_data['product_id']) : null,
				'ip_address'    => isset($log_data['ip_address']) ? sanitize_text_field($log_data['ip_address']) : '',
				'user_agent'    => isset($log_data['user_agent']) ? sanitize_text_field($log_data['user_agent']) : '',
				'created_at'    => current_time('mysql'),
			),
			array('%d', '%d', '%s', '%s', '%s', '%d', '%s', '%s', '%s')
		);
		
		return $result !== false;
	}
	
	/**
	 * Get calculation logs
	 */
	public static function pwcp_get_calculation_logs($args = array()) {
		global $wpdb;
		
		$defaults = array(
			'calculator_id' => 0,
			'user_id'       => 0,
			'product_id'    => 0,
			'limit'         => 50,
			'offset'        => 0,
			'orderby'       => 'created_at',
			'order'         => 'DESC',
		);
		
		$args = wp_parse_args($args, $defaults);
		
		$table_name = $wpdb->prefix . 'pwcp_calculation_logs';
		
		$where = array('1=1');
		$prepare_values = array();
		
		if ($args['calculator_id']) {
			$where[] = 'calculator_id = %d';
			$prepare_values[] = absint($args['calculator_id']);
		}
		
		if ($args['user_id']) {
			$where[] = 'user_id = %d';
			$prepare_values[] = absint($args['user_id']);
		}
		
		if ($args['product_id']) {
			$where[] = 'product_id = %d';
			$prepare_values[] = absint($args['product_id']);
		}
		
		$where_clause = implode(' AND ', $where);
		
		$orderby = in_array($args['orderby'], array('id', 'calculator_id', 'created_at'), true) ? $args['orderby'] : 'created_at';
		$order = in_array(strtoupper($args['order']), array('ASC', 'DESC'), true) ? strtoupper($args['order']) : 'DESC';
		
		$safe_table = esc_sql( $table_name );
		$prepare_values[] = absint($args['limit']);
		$prepare_values[] = absint($args['offset']);
		
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name from $wpdb->prefix, orderby/order from whitelist
		$base_query = "SELECT * FROM `{$safe_table}` WHERE {$where_clause} ORDER BY {$orderby} {$order} LIMIT %d OFFSET %d";
		
		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query is dynamically built with whitelisted columns and prepare() placeholders
		$results = $wpdb->get_results( $wpdb->prepare( $base_query, $prepare_values ) );
		
		return $results ? $results : array();
	}
	
	/**
	 * Delete calculation logs
	 */
	public static function pwcp_delete_calculation_logs($calculator_id = 0) {
		global $wpdb;
		
		$table_name = $wpdb->prefix . 'pwcp_calculation_logs';
		
		if ($calculator_id) {
			return $wpdb->delete(
				$table_name,
				array('calculator_id' => absint($calculator_id)),
				array('%d')
			);
		}
		
		$safe_table = esc_sql( $table_name );
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name from $wpdb->prefix, DDL statement
		return $wpdb->query( "TRUNCATE TABLE `{$safe_table}`" );
	}
	
	/**
	 * Maybe update database
	 */
	public static function pwcp_maybe_update_database() {
		$current_version = get_option('pwcp_db_version', '0.0.0');
		
		if (version_compare($current_version, self::PWCP_DB_VERSION, '<')) {
			self::pwcp_create_tables();
		}
	}
	
	/**
	 * Check if database needs update
	 */
	public static function pwcp_needs_database_update() {
		$current_version = get_option('pwcp_db_version', '0.0.0');
		return version_compare($current_version, self::PWCP_DB_VERSION, '<');
	}
	
	/**
	 * Drop all plugin tables
	 */
	public static function pwcp_drop_tables() {
		global $wpdb;
		
		$tables = array(
			$wpdb->prefix . 'pwcp_calculators',
			$wpdb->prefix . 'pwcp_calculator_fields',
			$wpdb->prefix . 'pwcp_calculation_logs',
			$wpdb->prefix . 'pwcp_calculator_assignments',
		);
		
		foreach ($tables as $table) {
			$safe_table = esc_sql( $table );
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name from $wpdb->prefix, DDL statement
			$wpdb->query( "DROP TABLE IF EXISTS `{$safe_table}`" );
		}
		
		delete_option('pwcp_db_version');
	}
	
	/**
	 * Clean up orphaned records
	 */
	public static function pwcp_cleanup_database() {
		global $wpdb;
		
		$fields_table = $wpdb->prefix . 'pwcp_calculator_fields';
		$calculators_table = $wpdb->prefix . 'pwcp_calculators';
		$logs_table = $wpdb->prefix . 'pwcp_calculation_logs';
		$assignments_table = $wpdb->prefix . 'pwcp_calculator_assignments';
		
		$safe_fields = esc_sql( $fields_table );
		$safe_calcs = esc_sql( $calculators_table );
		$safe_logs = esc_sql( $logs_table );
		$safe_assigns = esc_sql( $assignments_table );
		
		// Delete orphaned calculator fields
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table names from $wpdb->prefix
		$wpdb->query(
			$wpdb->prepare(
				"DELETE f FROM `{$safe_fields}` f 
				 LEFT JOIN `{$safe_calcs}` c ON f.calculator_id = c.id 
				 WHERE c.id IS NULL AND 1=%d",
				1
			)
		);
		
		// Delete orphaned calculation logs
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table names from $wpdb->prefix
		$wpdb->query(
			$wpdb->prepare(
				"DELETE l FROM `{$safe_logs}` l 
				 LEFT JOIN `{$safe_calcs}` c ON l.calculator_id = c.id 
				 WHERE c.id IS NULL AND 1=%d",
				1
			)
		);
		
		// Delete orphaned assignments
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table names from $wpdb->prefix
		$wpdb->query(
			$wpdb->prepare(
				"DELETE a FROM `{$safe_assigns}` a 
				 LEFT JOIN `{$safe_calcs}` c ON a.calculator_id = c.id 
				 WHERE c.id IS NULL AND 1=%d",
				1
			)
		);
		
		return true;
	}
	
	/**
	 * Get database statistics
	 */
	public static function pwcp_get_database_stats() {
		global $wpdb;
		
		$calculators_table = $wpdb->prefix . 'pwcp_calculators';
		$fields_table = $wpdb->prefix . 'pwcp_calculator_fields';
		$logs_table = $wpdb->prefix . 'pwcp_calculation_logs';
		$assignments_table = $wpdb->prefix . 'pwcp_calculator_assignments';
		
		$safe_calcs = esc_sql( $calculators_table );
		$safe_fields = esc_sql( $fields_table );
		$safe_logs = esc_sql( $logs_table );
		$safe_assigns = esc_sql( $assignments_table );
		
		$stats = array(
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name from $wpdb->prefix
			'calculators' => $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM `{$safe_calcs}` WHERE 1=%d", 1 ) ),
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name from $wpdb->prefix
			'fields' => $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM `{$safe_fields}` WHERE 1=%d", 1 ) ),
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name from $wpdb->prefix
			'logs' => $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM `{$safe_logs}` WHERE 1=%d", 1 ) ),
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name from $wpdb->prefix
			'assignments' => $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM `{$safe_assigns}` WHERE 1=%d", 1 ) ),
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name from $wpdb->prefix
			'active_calculators' => $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM `{$safe_calcs}` WHERE status = %s", 'active' ) ),
		);
		
		return $stats;
	}
}