<?php
/**
 * Base AJAX Handler
 * 
 * Abstract base class for AJAX request handling
 * 
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Base AJAX Handler Class
 * 
 * Provides common security checks and response formatting for AJAX handlers
 * 
 * @since 1.0.0
 */
abstract class PWCP_Base_Ajax_Handler {
    
    /**
     * Verify nonce for public-facing AJAX
     * 
     * @param string $nonce_action Nonce action
     * @param string $nonce_field Nonce field name
     * @return void Exits with error if validation fails
     */
    protected function verify_public_ajax_security($nonce_action = 'pwcp_calculator_action', $nonce_field = 'pwcp_nonce') {
        check_ajax_referer($nonce_action, $nonce_field);
    }
    
    /**
     * Verify nonce and permissions for admin AJAX requests
     * 
     * @param string $capability Required capability
     * @param string $nonce_action Nonce action
     * @param string $nonce_field Nonce field name
     * @return void Exits with error if validation fails
     */
    protected function verify_admin_ajax_security($capability = 'manage_options', $nonce_action = 'pwcp_calculator_action', $nonce_field = 'pwcp_nonce') {
        check_ajax_referer($nonce_action, $nonce_field);
        
        if (!$this->can_manage_calculators($capability)) {
            $this->send_error_response(__('Insufficient permissions.', 'pricewise-calculator-pro'));
        }
    }
    
    /**
     * Alias for verify_admin_ajax_security
     */
    protected function verify_ajax_security($capability = 'manage_options', $nonce_action = 'pwcp_calculator_action', $nonce_field = 'pwcp_nonce') {
        $this->verify_admin_ajax_security($capability, $nonce_action, $nonce_field);
    }
    
    /**
     * Get and sanitize POST data
     * 
     * @param string $key POST key
     * @param string $type Data type
     * @param mixed $default Default value
     * @return mixed Sanitized value
     */
    protected function get_post_data($key, $type = 'text', $default = '') {
        if (!isset($_POST[$key])) {
            return $default;
        }
        
        $value = wp_unslash($_POST[$key]);
        
        switch ($type) {
            case 'int':
                return absint($value);
            case 'float':
                return floatval($value);
            case 'array':
                return is_array($value) ? $this->sanitize_array_with_field_options($value) : array();
          case 'json':
    $sanitized_value = sanitize_text_field($value);
    $decoded = json_decode($sanitized_value, true);
    return is_array($decoded) ? $this->sanitize_array_recursive($decoded) : array();
            case 'textarea':
                return sanitize_textarea_field($value);
            case 'key':
                return sanitize_key($value);
            case 'text':
            default:
                return sanitize_text_field($value);
        }
    }
    
    /**
     * Sanitize array with field options handling
     * 
     * @param array $array Array to sanitize
     * @return array Sanitized array
     */
    private function sanitize_array_with_field_options($array) {
        $sanitized = array();
        
        foreach ($array as $key => $value) {
            $clean_key = is_string($key) ? sanitize_key($key) : $key;
            
            if (is_array($value)) {
                $sanitized[$clean_key] = $this->sanitize_array_with_field_options($value);
            } else {
                if ($key === 'options' || (is_string($key) && strpos($key, 'options') !== false)) {
                    $sanitized[$clean_key] = $this->sanitize_field_options($value);
                } else {
                    $sanitized[$clean_key] = sanitize_text_field($value);
                }
            }
        }
        
        return $sanitized;
    }
    
    /**
     * Sanitize field options preserving line breaks
     * 
     * @param string $options Raw options string
     * @return string Sanitized options with preserved line breaks
     */
    private function sanitize_field_options($options) {
        if (empty($options)) {
            return '';
        }
        
        $lines = preg_split('/\r\n|\r|\n/', $options);
        $sanitized_lines = array();
        
        foreach ($lines as $line) {
            $trimmed = trim($line);
            if (!empty($trimmed)) {
                $sanitized_lines[] = sanitize_text_field($trimmed);
            }
        }
        
        return implode("\n", $sanitized_lines);
    }
    
    /**
     * Validate required fields
     * 
     * @param array $required_fields Array of field names
     * @param array $data Data to validate
     * @return void Exits with error if validation fails
     */
    protected function validate_required_fields($required_fields, $data = null) {
        if ($data === null) {
            $data = $_POST;
        }
        
        $missing_fields = array();
        
        foreach ($required_fields as $field) {
            if (!isset($data[$field]) || (empty($data[$field]) && $data[$field] !== '0' && $data[$field] !== 0)) {
                $missing_fields[] = $field;
            }
        }
        
  if (!empty($missing_fields)) {
    $message = sprintf(
        /* translators: %s: Comma-separated list of missing field names */
        __('Missing required fields: %s', 'pricewise-calculator-pro'),
        implode(', ', $missing_fields)
    );
    $this->send_error_response($message);
}
    }
    
    /**
     * Send success response
     * 
     * @param string $message Success message
     * @param array $data Additional data
     * @return void
     */
    protected function send_success_response($message, $data = array()) {
        $response = array_merge(array('message' => esc_html($message)), $data);
        wp_send_json_success($response);
        exit;
    }
    
    /**
     * Send error response
     * 
     * @param string $message Error message
     * @param array $data Additional data
     * @return void
     */
    protected function send_error_response($message, $data = array()) {
        $response = array_merge(array('message' => esc_html($message)), $data);
        wp_send_json_error($response);
        exit;
    }
    
    /**
     * Recursively sanitize array
     * 
     * @param array $array Array to sanitize
     * @return array Sanitized array
     */
    private function sanitize_array_recursive($array) {
        $sanitized = array();
        
        foreach ($array as $key => $value) {
            $clean_key = sanitize_key($key);
            
            if (is_array($value)) {
                $sanitized[$clean_key] = $this->sanitize_array_recursive($value);
            } else {
                $sanitized[$clean_key] = sanitize_text_field($value);
            }
        }
        
        return $sanitized;
    }
    
    /**
     * Get calculator ID from POST
     * 
     * @param bool $required Whether calculator ID is required
     * @return int Calculator ID
     */
    protected function get_calculator_id($required = true) {
        $calculator_id = $this->get_post_data('calculator_id', 'int', 0);
        
        if ($required && $calculator_id <= 0) {
            $this->send_error_response(__('Valid calculator ID is required.', 'pricewise-calculator-pro'));
        }
        
        return $calculator_id;
    }
    
    /**
     * Get field ID from POST
     * 
     * @param bool $required Whether field ID is required
     * @return int Field ID
     */
    protected function get_field_id($required = true) {
        $field_id = $this->get_post_data('field_id', 'int', 0);
        
        if ($required && $field_id <= 0) {
            $this->send_error_response(__('Valid field ID is required.', 'pricewise-calculator-pro'));
        }
        
        return $field_id;
    }
    
    /**
     * Validate calculator access
     * 
     * @param int $calculator_id Calculator ID
     * @return object Calculator object
     */
    protected function validate_calculator_access($calculator_id) {
        if (!class_exists('PWCP_Admin_Calculator_Manager')) {
            $this->send_error_response(__('Calculator service not available.', 'pricewise-calculator-pro'));
        }
        
        $manager = new PWCP_Admin_Calculator_Manager();
        $calculator = $manager->pwcp_get_calculator($calculator_id);
        
        if (!$calculator) {
            $this->send_error_response(__('Calculator not found.', 'pricewise-calculator-pro'));
        }
        
        return $calculator;
    }
    
    /**
     * Log AJAX operation for debugging
     * 
     * @param string $action Action name
     * @param array $data Optional data to log
     * @return void
     */
    protected function log_ajax_operation($action, $data = array()) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log(sprintf(
                'PWCP AJAX: %s - User: %d - Data: %s',
                $action,
                get_current_user_id(),
                wp_json_encode($data)
            ));
        }
    }
    
    /**
     * Check if action is allowed
     * 
     * @param string $action Action to check
     * @param array $allowed_actions Array of allowed actions
     * @return bool True if allowed
     */
    protected function is_action_allowed($action, $allowed_actions) {
        return in_array($action, $allowed_actions, true);
    }
    
    /**
     * Check calculator management capability
     * 
     * Allows customization of who can manage calculators.
     * Example: To allow Shop Managers:
     * add_filter('pwcp_manage_calculators_cap', function() { return 'manage_woocommerce'; });
     * 
     * @param string $default_cap Default capability to check
     * @return bool True if user can manage calculators
     */
    protected function can_manage_calculators($default_cap = 'manage_options') {
        /**
         * Filter the capability required to manage calculators.
         *
         * @since 1.0.0
         * @param string $capability The capability to check. Default 'manage_options'.
         */
        $capability = apply_filters('pwcp_manage_calculators_cap', $default_cap);
        return current_user_can($capability);
    }
}