<?php
/**
 * AJAX Handler
 * 
 * Centralized AJAX request handling for the plugin
 * 
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

// Load dependencies
require_once plugin_dir_path(__FILE__) . 'class-pwcp-base-ajax-handler.php';
require_once plugin_dir_path(__FILE__) . 'traits/trait-pwcp-security-validation.php';
require_once plugin_dir_path(__FILE__) . 'class-pwcp-input-sanitizer.php';
require_once plugin_dir_path(__FILE__) . 'class-pwcp-response-factory.php';

/**
 * AJAX Handler Class
 * 
 * Handles all AJAX requests for the plugin
 * 
 * @since 1.0.0
 */
class PWCP_Ajax_Handler extends PWCP_Base_Ajax_Handler {
    
    use PWCP_Security_Validation_Trait;
    
    /**
     * Field Manager instance
     */
    private $field_manager;
    
    /**
     * Actions that don't need admin permissions
     */
    private $allowed_public_actions = array(
        'pwcp_calculate_price',
        'pwcp_calculate_instant'
    );
    
    /**
     * Actions that require admin access
     */
    private $allowed_admin_actions = array(
        'pwcp_save_calculator',
        'pwcp_auto_save_calculator',
        'pwcp_validate_formula',
        'pwcp_save_calculator_field',
        'pwcp_update_calculator_field',
        'pwcp_delete_calculator_field',
        'pwcp_reorder_calculator_fields',
        'pwcp_validate_field_key',
        'pwcp_analyze_formula',
        'pwcp_auto_fix_requirements',
        'pwcp_generate_fields_from_formula',
        'pwcp_get_category_assignments',
        'pwcp_save_category_assignments',
        'pwcp_get_woocommerce_categories'
    );
    
    /**
     * Constructor
     */
    public function __construct() {
        if (class_exists('PWCP_Field_Manager')) {
            $this->field_manager = new PWCP_Field_Manager();
        }
        
        $this->register_ajax_hooks();
    }
    
    /**
     * Register all AJAX endpoints
     *
     * @since 1.0.0
     * @return void
     */
    private function register_ajax_hooks() {
        // Public calculator operations
        add_action('wp_ajax_pwcp_calculate_price', array($this, 'handle_calculate_price'));
        add_action('wp_ajax_nopriv_pwcp_calculate_price', array($this, 'handle_calculate_price'));
        add_action('wp_ajax_pwcp_calculate_instant', array($this, 'handle_calculate_instant'));
        add_action('wp_ajax_nopriv_pwcp_calculate_instant', array($this, 'handle_calculate_instant'));
        
        // Admin operations - Field related
        add_action('wp_ajax_pwcp_save_calculator_field', array($this, 'ajax_save_calculator_field'));
        add_action('wp_ajax_pwcp_update_calculator_field', array($this, 'ajax_update_calculator_field'));
        add_action('wp_ajax_pwcp_delete_calculator_field', array($this, 'ajax_delete_calculator_field'));
        add_action('wp_ajax_pwcp_reorder_calculator_fields', array($this, 'ajax_reorder_calculator_fields'));
        add_action('wp_ajax_pwcp_validate_field_key', array($this, 'ajax_validate_field_key'));
        
        // Admin operations - Calculator & Form Editor
        add_action('wp_ajax_pwcp_save_calculator', array($this, 'handle_admin_request'));
        add_action('wp_ajax_pwcp_auto_save_calculator', array($this, 'handle_admin_request'));
        add_action('wp_ajax_pwcp_validate_formula', array($this, 'handle_admin_request'));
        add_action('wp_ajax_pwcp_analyze_formula', array($this, 'handle_admin_request'));
        add_action('wp_ajax_pwcp_auto_fix_requirements', array($this, 'handle_admin_request'));
        add_action('wp_ajax_pwcp_generate_fields_from_formula', array($this, 'handle_admin_request'));
        add_action('wp_ajax_pwcp_get_category_assignments', array($this, 'handle_admin_request'));
        add_action('wp_ajax_pwcp_save_category_assignments', array($this, 'handle_admin_request'));
        add_action('wp_ajax_pwcp_get_woocommerce_categories', array($this, 'ajax_get_woocommerce_categories'));
    }
    
    /**
     * Save a new calculator field
     *
     * @since 1.0.0
     * @return void
     */
    public function ajax_save_calculator_field() {
        $this->verify_admin_ajax_security();
        
        $calculator_id = $this->get_calculator_id();
        $field_data_raw = $this->get_post_data('field_data', 'array');
        
        if (empty($field_data_raw)) {
            PWCP_Response_Factory::validation_error(__('Field data is required.', 'pricewise-calculator-pro'));
        }
        
        if (!$this->field_manager) {
            PWCP_Response_Factory::generic_error(__('Field Manager not available.', 'pricewise-calculator-pro'));
        }
        
        $field_data = PWCP_Input_Sanitizer::sanitize_field_data($field_data_raw);
        $validation = $this->field_manager->validate_field_data($field_data, $calculator_id);
        
        if (!$validation['is_valid']) {
            PWCP_Response_Factory::validation_error($validation['errors']);
        }
        
        $field_id = $this->field_manager->create_field($calculator_id, $field_data);
        
        if (!$field_id) {
            PWCP_Response_Factory::database_error('create_field');
        }
        
        PWCP_Response_Factory::field_success('save', $field_id, array(
            'field_data' => array_merge($field_data, array('id' => $field_id))
        ));
    }
    
    /**
     * Update an existing field
     *
     * @since 1.0.0
     * @return void
     */
    public function ajax_update_calculator_field() {
        $this->verify_admin_ajax_security();
        
        $field_id = $this->get_field_id();
        $calculator_id = $this->get_calculator_id();
        $field_data_raw = $this->get_post_data('field_data', 'array');
        
        if (empty($field_data_raw)) {
            PWCP_Response_Factory::validation_error(__('Field data is required.', 'pricewise-calculator-pro'));
        }
        
        if (!$this->field_manager) {
            PWCP_Response_Factory::generic_error(__('Field Manager not available.', 'pricewise-calculator-pro'));
        }
        
        $field_data = PWCP_Input_Sanitizer::sanitize_field_data($field_data_raw);
        $validation = $this->field_manager->validate_field_data($field_data, $calculator_id, $field_id);
        
        if (!$validation['is_valid']) {
            PWCP_Response_Factory::validation_error($validation['errors']);
        }
        
        $result = $this->field_manager->update_field($field_id, $calculator_id, $field_data);
        
        if (!$result) {
            PWCP_Response_Factory::database_error('update_field');
        }
        
        PWCP_Response_Factory::field_success('update', $field_id, array(
            'field_data' => array_merge($field_data, array('id' => $field_id))
        ));
    }
    
    /**
     * Delete a field
     *
     * @since 1.0.0
     * @return void
     */
    public function ajax_delete_calculator_field() {
        $this->verify_admin_ajax_security();
        
        $field_id = $this->get_field_id();
        $calculator_id = $this->get_calculator_id();
        
        if (!$this->field_manager) {
            PWCP_Response_Factory::generic_error(__('Field Manager not available.', 'pricewise-calculator-pro'));
        }
        
        $result = $this->field_manager->delete_field($field_id, $calculator_id);
        
        if (!$result) {
            PWCP_Response_Factory::database_error('delete_field');
        }
        
        PWCP_Response_Factory::field_success('delete', $field_id);
    }
    
    /**
     * Update field order
     *
     * @since 1.0.0
     * @return void
     */
    public function ajax_reorder_calculator_fields() {
        $this->verify_admin_ajax_security();
        
        $field_order = $this->get_post_data('field_order', 'array');
        
        if (empty($field_order)) {
            PWCP_Response_Factory::validation_error(__('No field order provided.', 'pricewise-calculator-pro'));
        }
        
        if (!$this->field_manager) {
            PWCP_Response_Factory::generic_error(__('Field Manager not available.', 'pricewise-calculator-pro'));
        }
        
        $field_order = array_map('absint', $field_order);
        $result = $this->field_manager->reorder_fields($field_order);
        
        if (!$result) {
            PWCP_Response_Factory::database_error('reorder_fields');
        }
        
        PWCP_Response_Factory::field_success('reorder');
    }
    
    /**
     * Validate field key uniqueness
     *
     * @since 1.0.0
     * @return void
     */
    public function ajax_validate_field_key() {
        $this->verify_admin_ajax_security();
        
        $calculator_id = $this->get_calculator_id();
        $field_key = $this->get_post_data('field_key', 'key');
        $field_id = $this->get_post_data('field_id', 'int', 0);
        
        if (empty($field_key)) {
            PWCP_Response_Factory::validation_error(__('Field key is required.', 'pricewise-calculator-pro'));
        }
        
        if (!$this->field_manager) {
            PWCP_Response_Factory::generic_error(__('Field Manager not available.', 'pricewise-calculator-pro'));
        }
        
        // Create a temporary field structure for validation
        $field_data = array(
            'key' => $field_key,
            'label' => 'temp_label',
            'type' => 'text',
            'required' => 0,
            'default_value' => '',
            'options' => '',
            'validation_rules' => array()
        );
        
        $validation = $this->field_manager->validate_field_data($field_data, $calculator_id, $field_id ?: null);
        
        if (!$validation['is_valid']) {
            PWCP_Response_Factory::validation_error($validation['errors']);
        }
        
        $this->send_success_response(__('Field key is valid.', 'pricewise-calculator-pro'));
    }
    
    /**
     * Handle price calculation on the frontend
     *
     * @since 1.0.0
     * @return void
     */
    public function handle_calculate_price() {
        $this->verify_public_ajax_security();
        
        $action = $this->get_post_data('action', 'text');
        
        if (!$this->is_action_allowed($action, $this->allowed_public_actions)) {
            PWCP_Response_Factory::generic_error(__('Invalid action specified.', 'pricewise-calculator-pro'));
        }
        
        $plugin = pwcp();
        $frontend_calculator = $plugin->pwcp_get_frontend_calculator();
        
        if (!$frontend_calculator || !method_exists($frontend_calculator, 'pwcp_ajax_calculate_price')) {
            PWCP_Response_Factory::generic_error(__('Calculator service is temporarily unavailable.', 'pricewise-calculator-pro'));
        }
        
        $frontend_calculator->pwcp_ajax_calculate_price();
    }
    
    /**
     * Handle instant calculation as user types
     *
     * @since 1.0.0
     * @return void
     */
    public function handle_calculate_instant() {
        $this->verify_public_ajax_security();
        
        $action = $this->get_post_data('action', 'text');
        
        if (!$this->is_action_allowed($action, $this->allowed_public_actions)) {
            PWCP_Response_Factory::generic_error(__('Invalid action specified.', 'pricewise-calculator-pro'));
        }
        
        $calculator_id = $this->get_post_data('calculator_id', 'int', 0);
        
        if ($calculator_id <= 0) {
            PWCP_Response_Factory::validation_error(__('Invalid calculator specified.', 'pricewise-calculator-pro'));
        }
        
        $plugin = pwcp();
        $frontend_calculator = $plugin->pwcp_get_frontend_calculator();
        
        if (!$frontend_calculator || !method_exists($frontend_calculator, 'pwcp_ajax_calculate_instant')) {
            PWCP_Response_Factory::generic_error(__('Live calculation service is temporarily unavailable.', 'pricewise-calculator-pro'));
        }
        
        $frontend_calculator->pwcp_ajax_calculate_instant();
    }
    
    /**
     * Route admin AJAX requests
     *
     * @since 1.0.0
     * @return void
     */
    public function handle_admin_request() {
        if (!is_user_logged_in()) {
            PWCP_Response_Factory::security_error(__('Authentication required.', 'pricewise-calculator-pro'));
        }
        
        if (!$this->can_manage_calculators()) {
            PWCP_Response_Factory::permission_error();
        }
        
        $this->verify_admin_ajax_security();
        
        $action = $this->get_post_data('action', 'text');
        
        if (!$this->is_action_allowed($action, $this->allowed_admin_actions)) {
            PWCP_Response_Factory::generic_error(__('Invalid admin action specified.', 'pricewise-calculator-pro'));
        }
        
        switch ($action) {
            case 'pwcp_save_calculator':
            case 'pwcp_auto_save_calculator':
            case 'pwcp_validate_formula':
            case 'pwcp_analyze_formula':
            case 'pwcp_auto_fix_requirements':
            case 'pwcp_generate_fields_from_formula':
                $this->handle_form_editor_request($action);
                break;
                
            case 'pwcp_get_category_assignments':
            case 'pwcp_save_category_assignments':
                $this->handle_category_assignment_request($action);
                break;
                
            default:
                PWCP_Response_Factory::generic_error(__('The requested action is not recognized.', 'pricewise-calculator-pro'));
                break;
        }
    }
    
    /**
     * Handle form editor requests
     *
     * @since 1.0.0
     * @param string $action Action to perform
     * @return void
     */
    private function handle_form_editor_request($action) {
        if (!class_exists('PWCP_Admin_Calculator_Form_Editor')) {
            PWCP_Response_Factory::generic_error(__('Form editor service is not available.', 'pricewise-calculator-pro'));
        }
        
        $form_editor = new PWCP_Admin_Calculator_Form_Editor();
        $method_name = 'ajax_' . str_replace('pwcp_', '', $action);
        
        if (!method_exists($form_editor, $method_name)) {
            PWCP_Response_Factory::generic_error(__('The requested form editor function is not available.', 'pricewise-calculator-pro'));
        }
        
        $form_editor->$method_name();
    }
    
    /**
     * Handle category assignment requests
     *
     * @since 1.0.0
     * @param string $action Action to perform
     * @return void
     */
    private function handle_category_assignment_request($action) {
        if (!class_exists('PWCP_Category_Assignment')) {
            PWCP_Response_Factory::generic_error(__('Category management service is not available.', 'pricewise-calculator-pro'));
        }
        
        if (!current_user_can('manage_product_terms')) {
            PWCP_Response_Factory::permission_error(__('Insufficient permissions to manage categories.', 'pricewise-calculator-pro'));
        }
        
        $category_assignment = new PWCP_Category_Assignment();
        $method_name = 'ajax_' . str_replace('pwcp_', '', $action);
        
        if (!method_exists($category_assignment, $method_name)) {
            PWCP_Response_Factory::generic_error(__('The requested category function is not available.', 'pricewise-calculator-pro'));
        }
        
        $category_assignment->$method_name();
    }
    
    /**
     * Get WooCommerce product categories
     *
     * @since 1.0.0
     * @return void
     */
    public function ajax_get_woocommerce_categories() {
        $this->verify_admin_ajax_security('manage_product_terms', 'pwcp_calculator_action', 'pwcp_nonce');
        
        if (!current_user_can('manage_product_terms')) {
            PWCP_Response_Factory::permission_error(__('Insufficient permissions to manage categories.', 'pricewise-calculator-pro'));
        }
        
        if (!class_exists('WooCommerce')) {
            PWCP_Response_Factory::generic_error(__('WooCommerce is not active.', 'pricewise-calculator-pro'));
        }
        
        try {
            $categories = get_terms(array(
                'taxonomy' => 'product_cat',
                'hide_empty' => false,
                'orderby' => 'name',
                'order' => 'ASC',
                'number' => 1000
            ));
            
            if (is_wp_error($categories)) {
                error_log('PWCP Get Categories Error: ' . $categories->get_error_message());
                PWCP_Response_Factory::generic_error(__('Error loading categories. Please try again.', 'pricewise-calculator-pro'));
            }
            
            $formatted_categories = array();
            if (is_array($categories) && !empty($categories)) {
                foreach ($categories as $category) {
                    if (!is_object($category) || !isset($category->term_id)) {
                        continue;
                    }
                    
                    $formatted_categories[] = array(
                        'id' => absint($category->term_id),
                        'name' => sanitize_text_field($category->name ?? ''),
                        'count' => absint($category->count ?? 0),
                        'slug' => sanitize_title($category->slug ?? '')
                    );
                }
            }
            
            $this->send_success_response(__('Categories loaded successfully', 'pricewise-calculator-pro'), array(
                'categories' => $formatted_categories,
                'total' => count($formatted_categories)
            ));
            
        } catch (Exception $e) {
            error_log('PWCP Get Categories Error: ' . $e->getMessage());
            PWCP_Response_Factory::generic_error(__('Error loading categories. Please try again.', 'pricewise-calculator-pro'));
        }
    }
}

// Initialize the AJAX handler
if (!class_exists('PWCP_Ajax_Handler_Initialized')) {
    class PWCP_Ajax_Handler_Initialized {
        private static $instance = null;
        
        public static function init() {
            if (self::$instance === null) {
                self::$instance = new PWCP_Ajax_Handler();
            }
            return self::$instance;
        }
    }
    
    PWCP_Ajax_Handler_Initialized::init();
}