<?php
/**
 * Plugin Activation Handler
 * 
 * Handles all activation procedures for PriceWise Calculator Pro
 * File: includes/class-pwcp-activator.php
 * 
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * PWCP Activator Class
 * 
 * Handles plugin activation tasks such as creating database tables,
 * setting default options, and checking system requirements.
 * 
 * @since 1.0.0
 */
class PWCP_Activator {
    
    /**
     * Activate the plugin
     * 
     * This method is called when the plugin is activated.
     * It creates necessary database tables and sets default options.
     * 
     * @since 1.0.0
     */
    public static function pwcp_activate() {
        // Check system requirements before activation
        if (!self::pwcp_check_activation_requirements()) {
            return;
        }
        
        // Suppress any potential output during activation
        ob_start();
        
        try {
            // Create database tables
            self::pwcp_create_tables();
            
            // Set default options
            self::pwcp_set_default_options();
            
            // Create default calculator if none exists
            self::pwcp_create_default_calculator();
            
            // Set activation timestamp
            update_option('pwcp_activation_time', current_time('timestamp'));
            
            // Set plugin version
            update_option('pwcp_plugin_version', PWCP_PLUGIN_VERSION);
            
            // Clear any existing caches
            wp_cache_flush();
            
            // Fire activation hook for extensions
            do_action('pwcp_activated');
            
    } catch (Exception $e) {
    // Log error but don't output anything
    if (defined('WP_DEBUG') && WP_DEBUG) {
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when WP_DEBUG is enabled
        error_log('PriceWise Calculator Pro activation error: ' . $e->getMessage());
    }
}
        
        // Clean up any unexpected output
        ob_end_clean();
    }
    
    /**
     * Check activation requirements
     * 
     * @return bool
     * @since 1.0.0
     */
    private static function pwcp_check_activation_requirements() {
        // Check if required files exist
        if (!file_exists(PWCP_INCLUDES_DIR . 'class-pwcp-database.php')) {
            // Don't use wp_die during activation - just return false
            return false;
        }
        
        // Include database class
        require_once PWCP_INCLUDES_DIR . 'class-pwcp-database.php';
        
        return true;
    }
    
    /**
     * Create database tables
     * 
     * @since 1.0.0
     */
    private static function pwcp_create_tables() {
        if (class_exists('PWCP_Database')) {
            // Suppress any dbDelta output
            ob_start();
            PWCP_Database::pwcp_create_tables();
            ob_end_clean();
        }
    }
    
    /**
     * Set default plugin options
     * 
     * @since 1.0.0
     */
    private static function pwcp_set_default_options() {
        $default_options = array(
            'pwcp_settings' => array(
                'enable_glassmorphism' => 'yes',
                'primary_color' => '#007cba',
                'secondary_color' => '#50c878',
                'background_opacity' => '0.1',
                'border_radius' => '12',
                'animation_speed' => '300',
                'enable_live_calculation' => 'yes',
                'enable_progress_bar' => 'yes',
                'enable_quantity_selector' => 'yes',
                'currency_position' => 'left',
                'decimal_places' => '2',
                'thousand_separator' => ',',
                'decimal_separator' => '.',
                'enable_debug_mode' => 'no',
                'cache_calculations' => 'yes',
                'ajax_timeout' => '10',
                'enable_calculator_on_shop' => 'no',
                'enable_calculator_on_archive' => 'no'
            ),
       'pwcp_texts' => array(
    'calculator_title' => esc_html__('Price Calculator', 'pricewise-calculator-pro'),
    'calculate_button_text' => esc_html__('Calculate Price', 'pricewise-calculator-pro'),
    'add_to_cart_text' => esc_html__('Add to Cart', 'pricewise-calculator-pro'),
    'total_price_label' => esc_html__('Total Price:', 'pricewise-calculator-pro'),
    'required_field_message' => esc_html__('This field is required', 'pricewise-calculator-pro'),
    'calculation_error_message' => esc_html__('Error calculating price. Please try again.', 'pricewise-calculator-pro'),
    'loading_message' => esc_html__('Calculating...', 'pricewise-calculator-pro'),
    // translators: %current%: Current step number, %total%: Total number of steps
    'step_progress_text' => esc_html__('Step %current% of %total%', 'pricewise-calculator-pro')
),
            'pwcp_advanced' => array(
                'enable_field_optimization' => 'yes',
                'auto_fix_requirements' => 'no',
                'enable_formula_validation' => 'yes',
                'enable_calculation_breakdown' => 'yes',
                'enable_export_import' => 'yes',
                'enable_calculator_analytics' => 'no',
                'min_calculation_amount' => '0',
                'max_calculation_amount' => '999999'
            )
        );
        
        foreach ($default_options as $option_name => $option_value) {
            if (!get_option($option_name)) {
                add_option($option_name, $option_value);
            }
        }
        
        // Set capability permissions
        self::pwcp_set_capabilities();
    }
    
    /**
     * Set user capabilities
     * 
     * @since 1.0.0
     */
    private static function pwcp_set_capabilities() {
        $capabilities = array(
            'manage_pwcp_calculators',
            'edit_pwcp_calculators',
            'delete_pwcp_calculators',
            'view_pwcp_analytics'
        );
        
        // Add capabilities to administrator role
        $admin_role = get_role('administrator');
        if ($admin_role) {
            foreach ($capabilities as $capability) {
                $admin_role->add_cap($capability);
            }
        }
        
        // Add limited capabilities to shop manager if WooCommerce is active
        if (class_exists('WooCommerce')) {
            $shop_manager_role = get_role('shop_manager');
            if ($shop_manager_role) {
                $shop_manager_role->add_cap('manage_pwcp_calculators');
                $shop_manager_role->add_cap('edit_pwcp_calculators');
            }
        }
    }
    
    /**
     * Create a default calculator
     * 
     * @since 1.0.0
     */
    private static function pwcp_create_default_calculator() {
      global $wpdb;

$table_name = $wpdb->prefix . 'pwcp_calculators';

// Check if any calculators exist
// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- One-time activation query, table name from $wpdb->prefix
$existing_calculators = $wpdb->get_var(
    $wpdb->prepare( "SELECT COUNT(*) FROM `" . esc_sql( $table_name ) . "` WHERE 1=%d", 1 )
);
        
        if ($existing_calculators > 0) {
            return;
        }
        
        // Create default calculator
        $default_calculator = array(
            'name' => esc_html__('Default Calculator', 'pricewise-calculator-pro'),
            'description' => esc_html__('Default calculator created during plugin activation', 'pricewise-calculator-pro'),
            'fields' => wp_json_encode(array(
                array(
                    'id' => 'quantity',
                    'type' => 'number',
                    'label' => esc_html__('Quantity', 'pricewise-calculator-pro'),
                    'required' => true,
                    'default_value' => '1',
                    'min_value' => '1',
                    'max_value' => '100',
                    'step' => '1'
                ),
                array(
                    'id' => 'size',
                    'type' => 'dropdown',
                    'label' => esc_html__('Size', 'pricewise-calculator-pro'),
                    'required' => true,
                    'options' => array(
                        array('label' => esc_html__('Small', 'pricewise-calculator-pro'), 'value' => '1', 'price_modifier' => '0'),
                        array('label' => esc_html__('Medium', 'pricewise-calculator-pro'), 'value' => '2', 'price_modifier' => '10'),
                        array('label' => esc_html__('Large', 'pricewise-calculator-pro'), 'value' => '3', 'price_modifier' => '25')
                    )
                )
            )),
            'formula' => 'base_price + size + (quantity * 5)',
            'settings' => wp_json_encode(array(
                'enable_multistep' => false,
                'show_progress' => true,
                'enable_validation' => true,
                'show_breakdown' => true
            )),
            'status' => 'active',
            'assignment_type' => 'global',
            'assignment_ids' => '',
            'priority' => 1,
            'created_at' => current_time('mysql'),
            'updated_at' => current_time('mysql')
        );
        
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- One-time activation insert, direct access required
$wpdb->insert(
    $table_name,
    $default_calculator,
    array('%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%s', '%s')
);
    }
    
    /**
     * Schedule cron events
     * 
     * @since 1.0.0
     */
    private static function pwcp_schedule_events() {
        // Schedule daily cleanup of temporary data
        if (!wp_next_scheduled('pwcp_daily_cleanup')) {
            wp_schedule_event(time(), 'daily', 'pwcp_daily_cleanup');
        }
        
        // Schedule weekly analytics cleanup (if analytics are enabled)
        if (!wp_next_scheduled('pwcp_weekly_analytics_cleanup')) {
            wp_schedule_event(time(), 'weekly', 'pwcp_weekly_analytics_cleanup');
        }
    }
    
    /**
     * Set up rewrite rules
     * 
     * @since 1.0.0
     */
    private static function pwcp_setup_rewrite_rules() {
        // Flush rewrite rules to ensure proper URL handling
        flush_rewrite_rules();
    }
}