/**
 * PriceWise Calculator Pro - Material Design JavaScript
 * Version 1.0.0
 * 
 * @package PriceWise_Calculator_Pro
 */

(function($) {
    'use strict';

    const { __, _n, sprintf } = wp.i18n;

    const DEBOUNCE_DELAY = 200;
    const AJAX_TIMEOUT = 10000;
    const TRANSITION_FAST = 150;
    const TRANSITION_NORMAL = 200;
    const TRANSITION_SLOW = 280;
    const ANIMATION_QUICK = 220;
    const UPDATE_DELAY_IMMEDIATE = 10;
    const UPDATE_DELAY_SHORT = 50;
    const UPDATE_DELAY_NORMAL = 100;
    const INITIAL_CALC_DELAY = 500;

    /**
     * PWCP Material Calculator Constructor
     * 
     * @param {jQuery} $container Calculator container element
     */
    function PWCPMaterialCalculator($container) {
        this.$container = $container;
        this.calculatorId = parseInt($container.data('calculator-id'), 10);
        this.productId = parseInt($container.data('product-id'), 10);
        this.isLiveCalculation = $container.data('live-calculation') === true || $container.data('live-calculation') === 'true';
        
        this.initializeDOMCache();
        this.initializeState();
        this.init();
    }

    /**
     * Initialize DOM cache
     */
    PWCPMaterialCalculator.prototype.initializeDOMCache = function() {
        this.$form = this.$container.find('.pwcp-calculator-form');
        this.$priceDisplay = this.$container.find('#calculatedPrice');
        this.$errorDisplay = this.$container.find('.pwcp-calculator-error');
        this.$progressFill = this.$container.find('.pwcp-progress-fill');
        this.$stepProgressFill = this.$container.find('.pwcp-step-progress-fill');
        this.$prevBtn = this.$container.find('.pwcp-prev-btn');
        this.$nextBtn = this.$container.find('.pwcp-next-btn');
        this.$fieldSteps = this.$container.find('.pwcp-field-step');
        this.$currentStepSpan = this.$container.find('.pwcp-current-step');
        this.$totalStepsSpan = this.$container.find('.pwcp-total-steps');
        this.$completedRequiredSpan = this.$container.find('.pwcp-completed-required');
        this.$optionalStatusSpan = this.$container.find('.pwcp-optional-status');
        this.$progressTextContainer = this.$container.find('.pwcp-progress-text');
        this.$wooCommerceForm = $('form.cart');
        this.$wooCommerceAddToCartBtn = this.findAddToCartButton();
    };

    /**
     * Initialize state variables
     */
    PWCPMaterialCalculator.prototype.initializeState = function() {
        this.config = this.getCalculatorConfig();
        this.isCalculating = false;
        this.isTransitioning = false;
        this.lastCalculation = null;
        this.calculateTimeout = null;
        this.currentStep = 1;
        this.totalSteps = this.$fieldSteps.length;
        this.fieldValues = {};
        this.validationErrors = {};
        this.completedFields = 0;
        this.userInteractedFields = new Set();
        this.requiredFields = this.getRequiredFields();
        this.totalRequiredFields = this.requiredFields.length;
        this.allFields = this.getAllFieldKeys();
        this.totalFields = this.allFields.length;
        this.completedAllFields = 0;
        this.addToCartEnabled = false;
        this.currentCalculationXHR = null;
        
        this.addToCartRequirement = (typeof pwcpFrontend !== 'undefined' && pwcpFrontend.add_to_cart_requirement) ? 
            pwcpFrontend.add_to_cart_requirement : 'all_required';
    };

    /**
     * Find WooCommerce Add to Cart button
     */
    PWCPMaterialCalculator.prototype.findAddToCartButton = function() {
        var selectors = [
            'button[name="add-to-cart"]',
            'input[name="add-to-cart"]', 
            'button.single_add_to_cart_button',
            '.single_add_to_cart_button',
            'form.cart button[type="submit"]',
            'form.cart input[type="submit"]',
            '.woocommerce-variation-add-to-cart button'
        ];
        return $(selectors.join(', ')).first();
    };

    /**
     * Get all field keys
     */
    PWCPMaterialCalculator.prototype.getAllFieldKeys = function() {
        var allFields = [];
        this.$form.find('input[data-field-key]').each(function(index, element) {
            var fieldKey = $(element).data('field-key');
            if (fieldKey && allFields.indexOf(fieldKey) === -1) {
                allFields.push(fieldKey);
            }
        });
        return allFields;
    };

    /**
     * Get required fields
     */
    PWCPMaterialCalculator.prototype.getRequiredFields = function() {
        var requiredFields = [];
        this.$form.find('input[required]').each(function(index, element) {
            var fieldKey = $(element).data('field-key');
            if (fieldKey && requiredFields.indexOf(fieldKey) === -1) {
                requiredFields.push(fieldKey);
            }
        });
        return requiredFields;
    };

    /**
     * Initialize calculator
     */
    PWCPMaterialCalculator.prototype.init = function() {
        this.bindEvents();
        this.initializeMultiStep();
        this.initializeWooCommerceIntegration();
        
        if (this.isLiveCalculation) {
            this.enableLiveCalculation();
        }
        
        this.setDefaultValues();
        this.validateDefaultRequiredFields();
        this.initializeProgressTracking();
        this.updateNavigationState();
        
        if (this.isLiveCalculation) {
            this.performInitialCalculation();
        } else {
            this.showReadyState();
        }
        
        if (!window.pwcp) {
            window.pwcp = {};
        }
        if (!window.pwcp.calculators) {
            window.pwcp.calculators = new Map();
        }
        window.pwcp.calculators.set(this.calculatorId, this);
        
        $(document).trigger('pwcp_calculator_initialized', [this]);

        this.$stepProgressFill.each(function() {
            var initialWidth = $(this).data('initial-width');
            if (initialWidth !== null && initialWidth !== undefined) {
                $(this).css('width', initialWidth + '%');
            }
        });

        this.$progressFill.each(function() {
            var initialWidth = $(this).data('initial-width');
            if (initialWidth !== null && initialWidth !== undefined) {
                $(this).css('width', initialWidth + '%');
            }
        });
    };

    /**
     * Set default values for fields
     */
    PWCPMaterialCalculator.prototype.setDefaultValues = function() {
        var self = this;
        
        this.$form.find('input[data-field-key]').each(function(index, field) {
            var $field = $(field);
            var fieldKey = $field.data('field-key');
            var fieldType = $field.attr('type');
            var defaultValue = $field.data('default');
            var currentValue = $field.val();
            var isRequired = $field.prop('required') || $field.attr('required') === 'required';
            
            if (defaultValue && !currentValue) {
                $field.val(defaultValue);
                currentValue = defaultValue;
            }
            
            // Only handle number inputs
            if (fieldType === 'number' && currentValue && currentValue !== '') {
                self.fieldValues[fieldKey] = currentValue;
                
                if (isRequired) {
                    self.userInteractedFields.add(fieldKey);
                }
            }
        });
        
        this.updateCompletedFieldsCount();
    };

    /**
     * Enable live calculation mode
     */
    PWCPMaterialCalculator.prototype.enableLiveCalculation = function() {
        var self = this;
        
        this.$form.off('input.pwcp-live change.pwcp-live');
        
        this.$form.on('input.pwcp-live change.pwcp-live', 'input', function(e) {
            var $field = $(e.target);
            var fieldKey = $field.data('field-key');
            
            if (fieldKey) {
                self.userInteractedFields.add(fieldKey);
            }
            
            self.handleFieldChange($field);
            self.debouncedCalculate();
        });
    };

    /**
     * Perform initial calculation
     */
    PWCPMaterialCalculator.prototype.performInitialCalculation = function() {
        var self = this;
        var hasDefaults = this.$form.find('input[value!=""]').length > 0;
        
        if (hasDefaults) {
            setTimeout(function() {
                self.performInstantCalculation();
            }, INITIAL_CALC_DELAY);
        } else {
            this.showReadyState();
        }
    };

    /**
     * Show ready state
     */
    PWCPMaterialCalculator.prototype.showReadyState = function() {
        var $priceElement = this.$priceDisplay;
        
        if ($priceElement.length) {
            var formattedPrice = this.formatPriceAmount(0);
            $priceElement.text(formattedPrice);
        }
    };

    /**
     * Initialize progress tracking
     */
    PWCPMaterialCalculator.prototype.initializeProgressTracking = function() {
        this.$totalStepsSpan.text(this.totalSteps);
        
        if (this.totalRequiredFields === 0) {
            this.$progressFill.closest('.pwcp-progress-bar').hide();
            this.$progressTextContainer.hide();
        }
        
        this.updateProgress();
    };

    /**
     * Initialize WooCommerce integration
     */
    PWCPMaterialCalculator.prototype.initializeWooCommerceIntegration = function() {
        var self = this;
        
        if (this.$wooCommerceAddToCartBtn.length === 0) {
            this.$wooCommerceAddToCartBtn = this.findAddToCartButton();
        }
        
        this.$wooCommerceAddToCartBtn.off('click.pwcp');
        
        if (this.$wooCommerceAddToCartBtn.length) {
            this.$wooCommerceAddToCartBtn.on('click.pwcp', function(e) {
                if (!self.areAllRequiredFieldsCompleted()) {
                    e.preventDefault();
                    self.showError(__('Please complete the required fields before adding to cart.', 'pricewise-calculator-pro'));
                    return false;
                }
                
                if (!self.lastCalculation || self.lastCalculation.price <= 0) {
                    e.preventDefault();
                    self.showError(__('Please complete the calculator to get your price before adding to cart.', 'pricewise-calculator-pro'));
                    return false;
                }
                
                self.updateWooCommerceForm();
                return true;
            });
        }
        
        this.updateAddToCartStatus();
    };

    /**
     * Update WooCommerce form with calculation data
     */
    PWCPMaterialCalculator.prototype.updateWooCommerceForm = function() {
        if (!this.lastCalculation) {
            return;
        }
        
        this.updateOrCreateHiddenField('pwcp_calculated_price', this.lastCalculation.price);
        this.updateOrCreateHiddenField('pwcp_calculator_id', this.calculatorId);
        this.updateOrCreateHiddenField('pwcp_field_values', JSON.stringify(this.fieldValues));
        this.updateOrCreateHiddenField('pwcp_nonce', this.config.nonce);
    };

    /**
     * Update or create hidden form field
     */
    PWCPMaterialCalculator.prototype.updateOrCreateHiddenField = function(name, value) {
        var $field = this.$wooCommerceForm.find('input[name="' + name + '"]');
        
        if ($field.length === 0) {
            $field = $('#' + name + '_input');
        }
        
        if ($field.length === 0) {
            $field = $('<input type="hidden" name="' + name + '" />');
            this.$wooCommerceForm.append($field);
        }
        
        $field.val(value);
    };

    /**
     * Check if all required fields are completed
     */
    PWCPMaterialCalculator.prototype.areAllRequiredFieldsCompleted = function() {
        var self = this;
        
        if (this.addToCartRequirement === 'disabled') {
            return true;
        }
        
        if (this.addToCartRequirement === 'all_required') {
            var completedRequiredCount = 0;
            
            this.requiredFields.forEach(function(fieldKey) {
                var value = self.fieldValues[fieldKey];
                var hasInteracted = self.userInteractedFields.has(fieldKey);
                var isCompleted = false;
                
                // Only number inputs
                isCompleted = hasInteracted && value !== undefined && value !== '' && value !== null;
                
                if (isCompleted) {
                    completedRequiredCount++;
                }
            });
            
            return completedRequiredCount === this.totalRequiredFields;
        }
        
        if (this.addToCartRequirement === 'all_fields') {
            var completedAllCount = 0;
            
            this.allFields.forEach(function(fieldKey) {
                var value = self.fieldValues[fieldKey];
                var hasInteracted = self.userInteractedFields.has(fieldKey);
                
                if (hasInteracted && value !== undefined && value !== '' && value !== null) {
                    completedAllCount++;
                }
            });
            
            return completedAllCount === this.totalFields;
        }
        
        if (this.addToCartRequirement.indexOf('at_least_') === 0) {
            var requiredN = parseInt(this.addToCartRequirement.split('_').pop(), 10);
            var actualRequired = Math.min(requiredN, this.totalFields);
            var completedFirstN = 0;
            
            var orderedFields = [];
            this.$fieldSteps.each(function() {
                $(this).find('[data-field-key]').each(function() {
                    var key = $(this).data('field-key');
                    if (key && orderedFields.indexOf(key) === -1) {
                        orderedFields.push(key);
                    }
                });
            });
            
            for (var i = 0; i < Math.min(actualRequired, orderedFields.length); i++) {
                var fieldKey = orderedFields[i];
                var value = self.fieldValues[fieldKey];
                var hasInteracted = self.userInteractedFields.has(fieldKey);
                
                if (hasInteracted && value !== undefined && value !== '' && value !== null) {
                    completedFirstN++;
                }
            }
            
            return completedFirstN >= actualRequired;
        }
        
        return false;
    };

    /**
     * Update Add to Cart button status
     */
    PWCPMaterialCalculator.prototype.updateAddToCartStatus = function() {
        if (this.addToCartRequirement === 'disabled') {
            this.enableWooCommerceAddToCart();
            this.addToCartEnabled = true;
            return;
        }
        
        if (this.addToCartRequirement === 'all_required' && this.totalRequiredFields === 0) {
            this.enableWooCommerceAddToCart();
            this.addToCartEnabled = true;
            return;
        }
        
        if (!this.lastCalculation || this.lastCalculation.price <= 0) {
            this.disableWooCommerceAddToCart();
            this.addToCartEnabled = false;
            return;
        }
        
        var shouldEnable = this.areAllRequiredFieldsCompleted();
        
        if (shouldEnable !== this.addToCartEnabled) {
            if (shouldEnable) {
                this.enableWooCommerceAddToCart();
                this.addToCartEnabled = true;
            } else {
                this.disableWooCommerceAddToCart();
                this.addToCartEnabled = false;
            }
        }
    };

    /**
     * Check if required fields are complete
     */
    PWCPMaterialCalculator.prototype.areRequiredFieldsComplete = function() {
        var self = this;
        var completedRequiredCount = 0;
        
        this.requiredFields.forEach(function(fieldKey) {
            var value = self.fieldValues[fieldKey];
            var hasInteracted = self.userInteractedFields.has(fieldKey);
            
            // Only number inputs
            var isCompleted = hasInteracted && value !== undefined && value !== '' && value !== null;
            
            if (isCompleted) {
                completedRequiredCount++;
            }
        });
        
        return completedRequiredCount === this.totalRequiredFields;
    };

    /**
     * Enable WooCommerce Add to Cart button
     */
    PWCPMaterialCalculator.prototype.enableWooCommerceAddToCart = function() {
        if (this.$wooCommerceAddToCartBtn.length === 0) {
            return;
        }
        
        this.$wooCommerceAddToCartBtn.prop('disabled', false).removeClass('pwcp-disabled');
        this.$wooCommerceAddToCartBtn.removeAttr('title');
        this.removeCustomTooltip();
    };

    /**
     * Remove custom tooltip
     */
    PWCPMaterialCalculator.prototype.removeCustomTooltip = function() {
        this.$wooCommerceAddToCartBtn.find('.pwcp-custom-tooltip').remove();
        $('.pwcp-custom-tooltip').remove();
        this.$wooCommerceAddToCartBtn.off('mouseenter.pwcp-tooltip mouseleave.pwcp-tooltip');
    };

    /**
     * Disable WooCommerce Add to Cart button
     */
    PWCPMaterialCalculator.prototype.disableWooCommerceAddToCart = function() {
        if (this.$wooCommerceAddToCartBtn.length === 0) {
            return;
        }
        
        this.$wooCommerceAddToCartBtn.prop('disabled', true).addClass('pwcp-disabled');
        this.$wooCommerceAddToCartBtn.addClass('pwcp-tooltip-anchor');
        this.createCustomTooltip();
    };

    /**
     * Create custom tooltip for disabled button
     */
    PWCPMaterialCalculator.prototype.createCustomTooltip = function() {
        var self = this;
        
        $('.pwcp-custom-tooltip').remove();
        
        var tooltipMessage = this.getTooltipMessage();
        
        var $tooltip = $('<div class="pwcp-custom-tooltip" role="tooltip"></div>').text(tooltipMessage);
        var $arrow = $('<div class="pwcp-tooltip-arrow" aria-hidden="true"></div>');
        
        $tooltip.append($arrow);
        this.$wooCommerceAddToCartBtn.append($tooltip);
        
        this.$wooCommerceAddToCartBtn.off('mouseenter.pwcp-tooltip mouseleave.pwcp-tooltip');
        
        this.$wooCommerceAddToCartBtn.on('mouseenter.pwcp-tooltip', function() {
            if ($(this).prop('disabled')) {
                $tooltip.addClass('is-visible');
            }
        });
        
        this.$wooCommerceAddToCartBtn.on('mouseleave.pwcp-tooltip', function() {
            $tooltip.removeClass('is-visible');
        });
    };

    /**
     * Get tooltip message based on completion status
     */
    PWCPMaterialCalculator.prototype.getTooltipMessage = function() {
        var requirement = this.addToCartRequirement;
        var completedRequired = 0;
        var completedAll = 0;
        var self = this;
        
        this.requiredFields.forEach(function(fieldKey) {
            var value = self.fieldValues[fieldKey];
            var hasInteracted = self.userInteractedFields.has(fieldKey);
            
            if (hasInteracted && value !== undefined && value !== '' && value !== null) {
                completedRequired++;
            }
        });
        
        this.allFields.forEach(function(fieldKey) {
            var value = self.fieldValues[fieldKey];
            var hasInteracted = self.userInteractedFields.has(fieldKey);
            
            if (hasInteracted && value !== undefined && value !== '' && value !== null) {
                completedAll++;
            }
        });
        
        switch (requirement) {
            case 'disabled':
                return __('Add to Cart is always enabled', 'pricewise-calculator-pro');
                
            case 'all_required':
                if (completedRequired === this.totalRequiredFields) {
                    return __('All required fields complete', 'pricewise-calculator-pro');
                } else if (this.totalRequiredFields === 1) {
                    return __('Complete the required field to enable this', 'pricewise-calculator-pro');
                } else {
                    return sprintf(__('Complete all %d required fields to enable this', 'pricewise-calculator-pro'), this.totalRequiredFields);
                }
                
            case 'all_fields':
                if (completedAll === this.totalFields) {
                    return __('All fields complete', 'pricewise-calculator-pro');
                } else if (this.totalFields === 1) {
                    return __('Complete the field to enable this', 'pricewise-calculator-pro');
                } else {
                    return sprintf(__('Complete all %d fields to enable this', 'pricewise-calculator-pro'), this.totalFields);
                }
                
            case 'at_least_1':
            case 'at_least_2':
            case 'at_least_3':
            case 'at_least_4':
                var requiredN = parseInt(requirement.split('_').pop(), 10);
                var actualRequired = Math.min(requiredN, this.totalFields);
                
                var orderedFields = [];
                this.$fieldSteps.each(function() {
                    $(this).find('[data-field-key]').each(function() {
                        var key = $(this).data('field-key');
                        if (key && orderedFields.indexOf(key) === -1) {
                            orderedFields.push(key);
                        }
                    });
                });
                
                var completedFirstN = 0;
                for (var i = 0; i < Math.min(actualRequired, orderedFields.length); i++) {
                    var fieldKey = orderedFields[i];
                    var value = self.fieldValues[fieldKey];
                    var hasInteracted = self.userInteractedFields.has(fieldKey);
                    
                    if (hasInteracted && value !== undefined && value !== '' && value !== null) {
                        completedFirstN++;
                    }
                }
                
                if (completedFirstN >= actualRequired) {
                    return actualRequired === 1 ? 
                        sprintf(_n('First %d field complete', 'First %d fields complete', actualRequired, 'pricewise-calculator-pro'), actualRequired) : 
                        sprintf(_n('First %d field complete', 'First %d fields complete', actualRequired, 'pricewise-calculator-pro'), actualRequired);
                } else if (actualRequired === 1) {
                    return __('Complete the first field to enable this', 'pricewise-calculator-pro');
                } else {
                    return sprintf(__('Complete first %d fields to enable this', 'pricewise-calculator-pro'), actualRequired);
                }
                
            default:
                return __('Please complete the required fields before adding to cart.', 'pricewise-calculator-pro');
        }
    };

    /**
     * Get calculator configuration
     */
    PWCPMaterialCalculator.prototype.getCalculatorConfig = function() {
        var configScript = document.getElementById('pwcp-calculator-config-' + this.calculatorId);
        if (configScript) {
            try {
                return JSON.parse(configScript.textContent.trim());
            } catch (e) {
                return this.getDefaultConfig();
            }
        }
        
        return this.getDefaultConfig();
    };

    /**
     * Get default configuration
     */
    PWCPMaterialCalculator.prototype.getDefaultConfig = function() {
        return {
            calculator_id: this.calculatorId,
            product_id: this.productId,
            formula: '',
            base_price: 0,
            currency_symbol: '$',
            decimal_places: 2,
            show_result_immediately: this.isLiveCalculation,
            multi_step: true,
            total_steps: this.totalSteps,
            ajax_url: (typeof pwcpFrontend !== 'undefined') ? pwcpFrontend.ajax_url : '/wp-admin/admin-ajax.php',
            nonce: (typeof pwcpFrontend !== 'undefined') ? pwcpFrontend.nonce : ''
        };
    };

    /**
     * Bind events
     */
    PWCPMaterialCalculator.prototype.bindEvents = function() {
        var self = this;
        
        this.$form.off('submit.pwcp input.pwcp change.pwcp keydown.pwcp blur.pwcp');
        this.$nextBtn.off('click.pwcp');
        this.$prevBtn.off('click.pwcp');
        
        this.$form.on('submit.pwcp', function(e) {
            e.preventDefault();
            self.handleFormSubmission();
        });
        
        this.$nextBtn.on('click.pwcp', function(e) {
            e.preventDefault();
            self.nextStep();
        });
        
        this.$prevBtn.on('click.pwcp', function(e) {
            e.preventDefault();
            self.prevStep();
        });
        
        this.$form.on('input.pwcp', 'input[type="number"]', function(e) {
            var $input = $(this);
            var min = parseFloat($input.attr('min'));
            var max = parseFloat($input.attr('max'));
            var value = parseFloat($input.val());
            
            if (!isNaN(min) && value < min) {
                $input.val(min);
            }
            if (!isNaN(max) && value > max) {
                $input.val(max);
            }
        });
        
        this.$form.on('blur.pwcp', 'input[type="number"]', function(e) {
            var $input = $(this);
            var min = parseFloat($input.attr('min'));
            var max = parseFloat($input.attr('max'));
            var value = parseFloat($input.val());
            
            if (!isNaN(value)) {
                if (!isNaN(min) && value < min) {
                    $input.val(min);
                }
                if (!isNaN(max) && value > max) {
                    $input.val(max);
                }
            } else if (!isNaN(min)) {
                $input.val(min);
            }
        });
        
        this.$form.on('input.pwcp change.pwcp', 'input', function(e) {
            var $field = $(e.target);
            var fieldKey = $field.data('field-key');
            
            if (fieldKey) {
                self.userInteractedFields.add(fieldKey);
            }
            
            self.handleFieldChange($field);
            
            if (fieldKey && self.requiredFields.indexOf(fieldKey) !== -1) {
                self.updateNavigationState();
            }
            
            if (self.isLiveCalculation) {
                self.debouncedCalculate();
            }
        });
        
        this.$form.on('keydown.pwcp', function(e) {
            if (self.config && self.config.multi_step === true) {
                if (e.key === 'Enter' && !e.shiftKey) {
                    e.preventDefault();
                    self.nextStep();
                } else if (e.key === 'Escape') {
                    self.prevStep();
                }
            }
        });
    };

    /**
     * Validate and mark required fields with default values as complete
     */
    PWCPMaterialCalculator.prototype.validateDefaultRequiredFields = function() {
        var self = this;
        
        this.requiredFields.forEach(function(fieldKey) {
            var $field = self.$form.find('[data-field-key="' + fieldKey + '"]').first();
            
            if ($field.length === 0) {
                return;
            }
            
            var currentValue = $field.val();
            
            if (currentValue && currentValue !== '' && currentValue !== null) {
                self.userInteractedFields.add(fieldKey);
                self.fieldValues[fieldKey] = currentValue;
            }
        });
        
        this.updateCompletedFieldsCount();
        this.updateAddToCartStatus();
    };

    /**
     * Update field validation
     */
    PWCPMaterialCalculator.prototype.updateFieldValidation = function($field) {
        var $container = $field.closest('.pwcp-field-container');
        var $validationMessage = $container.find('.pwcp-validation-message');
        var fieldKey = $field.data('field-key');
        var isRequired = $field.prop('required');
        
        if (!isRequired) {
            $container.removeClass('pwcp-error');
            $validationMessage.removeClass('show');
            delete this.validationErrors[fieldKey];
            return;
        }
        
        $validationMessage.addClass('show');
        
        // Only number inputs
        var hasValue = $field.val() && $field.val().trim() !== '';
        
        if (hasValue) {
            $container.removeClass('pwcp-error');
            delete this.validationErrors[fieldKey];
        } else {
            $container.addClass('pwcp-error');
            this.validationErrors[fieldKey] = pwcpFrontend.strings.field_required;
        }
    };

    /**
     * Initialize multi-step
     */
    PWCPMaterialCalculator.prototype.initializeMultiStep = function() {
        this.$fieldSteps.hide();
        if (this.$fieldSteps.length > 0) {
            this.$fieldSteps.first().show();
        }
        
        this.updateNavigationState();
        this.updateProgress();
    };

    /**
     * Go to next step
     */
    PWCPMaterialCalculator.prototype.nextStep = function() {
        this.saveCurrentStepValues();
        this.hideError();
        
        if (this.currentStep >= this.totalSteps) {
            if (this.isLiveCalculation) {
                this.$container.trigger('pwcp_calculator_completed', [this.lastCalculation]);
                return;
            } else {
                this.performInstantCalculation(false);
                return;
            }
        }
        
        this.goToStep(this.currentStep + 1);
    };

    /**
     * Go to previous step
     */
    PWCPMaterialCalculator.prototype.prevStep = function() {
        if (this.currentStep > 1) {
            this.goToStep(this.currentStep - 1);
        }
    };

    /**
     * Go to specific step
     */
    PWCPMaterialCalculator.prototype.goToStep = function(stepNumber) {
        if (stepNumber < 1 || stepNumber > this.totalSteps) {
            return;
        }
        
        var self = this;
        var $currentStep = this.$fieldSteps.eq(this.currentStep - 1);
        var $targetStep = this.$fieldSteps.eq(stepNumber - 1);
        
        this.isTransitioning = true;
        
        // Get the actual width of the form to ensure proper text wrapping
        var formWidth = this.$form.width();
        
        // Clone and measure with proper width constraints
        var $clone = $targetStep.clone()
            .addClass('pwcp-height-measure')
            .css({
                'width': formWidth + 'px',
                'position': 'absolute',
                'visibility': 'hidden',
                'display': 'block',
                'left': '-9999px',
                'top': '0'
            })
            .appendTo(this.$form);
        
        // Small delay to ensure rendering is complete
        setTimeout(function() {
            var targetHeight = $clone.outerHeight();
            $clone.remove();
            
            $currentStep.fadeOut(TRANSITION_NORMAL);
            
            self.$form.stop(true, false).animate({
                height: targetHeight + 'px'
            }, {
                duration: TRANSITION_SLOW,
                easing: 'swing',
                complete: function() {
                    $targetStep.fadeIn(ANIMATION_QUICK, function() {
                        self.$form.css('height', 'auto');
                        self.isTransitioning = false;
                    });
                }
            });
            
        }, 10);
        
        this.currentStep = stepNumber;
        this.updateNavigationState();
        this.updateProgress();
    };

    /**
     * Update navigation state
     */
    PWCPMaterialCalculator.prototype.updateNavigationState = function() {
        this.$currentStepSpan.text(this.currentStep);
        this.$totalStepsSpan.text(this.totalSteps);
        
        var stepProgressPercentage = (this.currentStep / this.totalSteps) * 100;
        this.$stepProgressFill.css('width', stepProgressPercentage + '%');
        
        if (this.currentStep > 1) {
            this.$prevBtn.prop('disabled', false);
        } else {
            this.$prevBtn.prop('disabled', true);
        }
        
        if (this.currentStep < this.totalSteps) {
            this.$nextBtn.text(__('NEXT →', 'pricewise-calculator-pro'));
            this.$nextBtn.prop('disabled', false);
        } else {
            if (this.isLiveCalculation) {
                this.$nextBtn.text(__('Complete', 'pricewise-calculator-pro'));
                this.$nextBtn.prop('disabled', true);
            } else {
                this.$nextBtn.text(__('Calculate', 'pricewise-calculator-pro'));
                this.$nextBtn.prop('disabled', !this.areRequiredFieldsComplete());
            }
        }
    };

    /**
     * Update progress bar
     */
    PWCPMaterialCalculator.prototype.updateProgress = function() {
        var optionalFieldsCount = this.totalFields - this.totalRequiredFields;
        var completedOptional = this.completedAllFields - this.completedFields;
        var remainingOptional = optionalFieldsCount - completedOptional;
        
        if (this.totalRequiredFields > 0) {
            var progressPercentage = (this.completedFields / this.totalRequiredFields) * 100;
            this.$progressFill.css('width', progressPercentage + '%');
            this.$completedRequiredSpan.text(this.completedFields);
            
            if (this.completedFields === this.totalRequiredFields) {
                this.$progressTextContainer.html(
                    '<strong class="pwcp-success-message">' + 
                    __('All required fields filled', 'pricewise-calculator-pro') + 
                    '</strong>'
                );
            } else {
                var requiredText = this.totalRequiredFields === 1 ? 
                    __('of 1 required field complete', 'pricewise-calculator-pro') :
                    sprintf(__('of %d required fields complete', 'pricewise-calculator-pro'), this.totalRequiredFields);
                
                this.$progressTextContainer.html(
                    '<strong><span class="pwcp-completed-required">' + 
                    this.completedFields + 
                    '</span> ' + requiredText + '</strong>'
                );
            }
        }
        
        if (optionalFieldsCount > 0) {
            if (remainingOptional === 0) {
                this.$optionalStatusSpan.html(
                    '<span class="pwcp-success-message">' + 
                    __('All optional fields filled', 'pricewise-calculator-pro') + 
                    '</span>'
                );
            } else if (remainingOptional === 1) {
                this.$optionalStatusSpan.text(__('1 optional field remaining', 'pricewise-calculator-pro'));
            } else {
                this.$optionalStatusSpan.text(
                    sprintf(__('%d optional fields remaining', 'pricewise-calculator-pro'), remainingOptional)
                );
            }
        } else {
            if (this.completedFields === this.totalRequiredFields && this.totalRequiredFields > 0) {
                this.$optionalStatusSpan.html(
                    '<span class="pwcp-success-message">' + 
                    __('All fields complete', 'pricewise-calculator-pro') + 
                    '</span>'
                );
            } else {
                this.$optionalStatusSpan.text('');
            }
        }
    };

    /**
     * Handle field change
     */
    PWCPMaterialCalculator.prototype.handleFieldChange = function($field) {
        var fieldKey = $field.data('field-key');
        var self = this;
        
        this.saveCurrentFieldValue($field);
        this.updateFieldValidation($field);
        
        if (this.currentStep >= this.totalSteps && this.requiredFields.indexOf(fieldKey) !== -1) {
            this.updateNavigationState();
        }
        
        if (this.requiredFields.indexOf(fieldKey) !== -1) {
            setTimeout(function() {
                self.updateAddToCartStatus();
            }, UPDATE_DELAY_SHORT);
        }
        
        if (this.isLiveCalculation) {
            this.debouncedCalculate();
        }
    };

    /**
     * Save current field value
     */
    PWCPMaterialCalculator.prototype.saveCurrentFieldValue = function($field) {
        var fieldKey = $field.data('field-key');
        if (!fieldKey) {
            return;
        }
        
        // Only number inputs
        var fieldValue = $field.val();
        this.fieldValues[fieldKey] = fieldValue;
        
        this.updateCompletedFieldsCount();
    };

    /**
     * Save current step values
     */
    PWCPMaterialCalculator.prototype.saveCurrentStepValues = function() {
        var self = this;
        var $currentStep = this.$fieldSteps.eq(this.currentStep - 1);
        var $fields = $currentStep.find('input[data-field-key]');
        
        $fields.each(function(index, field) {
            self.saveCurrentFieldValue($(field));
        });
    };

    /**
     * Update completed fields count
     */
    PWCPMaterialCalculator.prototype.updateCompletedFieldsCount = function() {
        var completedRequiredCount = 0;
        var completedAllFieldsCount = 0;
        var self = this;
        
        this.requiredFields.forEach(function(fieldKey) {
            var value = self.fieldValues[fieldKey];
            var hasInteracted = self.userInteractedFields.has(fieldKey);
            
            if (hasInteracted && value !== undefined && value !== '' && value !== null) {
                completedRequiredCount++;
            }
        });
        
        this.allFields.forEach(function(fieldKey) {
            var value = self.fieldValues[fieldKey];
            var hasInteracted = self.userInteractedFields.has(fieldKey);
            
            if (hasInteracted && value !== undefined && value !== '' && value !== null) {
                completedAllFieldsCount++;
            }
        });
        
        this.completedFields = completedRequiredCount;
        this.completedAllFields = completedAllFieldsCount;
        this.updateProgress();
    };

    /**
     * Debounced calculate function
     */
    PWCPMaterialCalculator.prototype.debouncedCalculate = function() {
        if (!this.isLiveCalculation) {
            return;
        }
        
        if (this.isTransitioning) {
            return;
        }
        
        var self = this;
        clearTimeout(this.calculateTimeout);
        this.calculateTimeout = setTimeout(function() {
            self.performInstantCalculation(true);
        }, DEBOUNCE_DELAY);
    };

    /**
     * Handle form submission
     */
    PWCPMaterialCalculator.prototype.handleFormSubmission = function() {
        if (this.currentStep < this.totalSteps) {
            this.nextStep();
        } else {
            if (this.isLiveCalculation) {
                this.$container.trigger('pwcp_calculator_completed', [this.lastCalculation]);
            } else {
                this.performInstantCalculation(false);
            }
        }
    };

    /**
     * Perform instant calculation
     */
    PWCPMaterialCalculator.prototype.performInstantCalculation = function(isInstant) {
        if (typeof isInstant === 'undefined') {
            isInstant = this.isLiveCalculation;
        }
        
        if (this.isCalculating) {
            return;
        }
        
        this.isCalculating = true;
        this.setCalculatingState(true);
        
        var fieldValues = this.getAllFieldValues();
        var action = 'pwcp_calculate_instant';
        var self = this;
        
        var requestData = {
            action: action,
            calculator_id: this.calculatorId,
            product_id: this.productId,
            field_values: fieldValues,
            pwcp_nonce: this.config.nonce
        };
        
        if (!isInstant) {
            requestData.explicit_calculation = true;
        }
        
        if (this.currentCalculationXHR && this.currentCalculationXHR.readyState !== 4) {
            this.currentCalculationXHR.abort();
        }
        
        this.currentCalculationXHR = $.ajax({
            url: this.config.ajax_url,
            type: 'POST',
            data: requestData,
            timeout: AJAX_TIMEOUT,
            success: function(response) {
                self.handleCalculationResponse(response, isInstant);
            },
            error: function(xhr, status, error) {
                if (status === 'abort') {
                    return;
                }
                
                if (!isInstant || status === 'timeout') {
                    self.handleCalculationError(xhr, status, error);
                }
            },
            complete: function() {
                self.isCalculating = false;
                self.setCalculatingState(false);
                self.currentCalculationXHR = null;
            }
        });
    };

    /**
     * Get all field values
     */
    PWCPMaterialCalculator.prototype.getAllFieldValues = function() {
        return this.fieldValues;
    };

    /**
     * Handle calculation response
     */
    PWCPMaterialCalculator.prototype.handleCalculationResponse = function(response, isInstant) {
        if (response.success && response.data) {
            var data = response.data;
            
            this.updatePriceDisplay(data.calculated_price, data.formatted_price);
            
            this.lastCalculation = {
                price: data.calculated_price,
                formatted_price: data.formatted_price,
                field_values: data.field_values
            };
            
            this.$form.find('#pwcp-calculated-price-field').val(data.calculated_price);
            
            if (isInstant) {
                var progressPercent = this.completedFields === this.totalRequiredFields ? 100 : 
                    (this.completedFields / this.totalRequiredFields) * 100;
                this.$progressFill.css('width', progressPercent + '%');
            }
            
            setTimeout(() => {
                this.updateAddToCartStatus();
            }, UPDATE_DELAY_NORMAL);
            
            this.hideError();
            this.$container.trigger('pwcp_calculation_complete', [data]);
            
        } else {
            if (!isInstant) {
                var message = response.data && response.data.message ? 
                    response.data.message : __('Calculation error occurred', 'pricewise-calculator-pro');
                
                this.showError(message);
            }
            
            this.disableWooCommerceAddToCart();
            this.addToCartEnabled = false;
        }
    };
    
    /**
     * Handle calculation error
     */
    PWCPMaterialCalculator.prototype.handleCalculationError = function(xhr, status, error) {
        var errorMessage = __('Calculation error occurred', 'pricewise-calculator-pro');
        
        if (status === 'timeout') {
            errorMessage = __('Calculation timed out. Please try again.', 'pricewise-calculator-pro');
        } else if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
            errorMessage = xhr.responseJSON.data.message;
        }
        
        this.showError(errorMessage);
    };

    /**
     * Update price display
     */
    PWCPMaterialCalculator.prototype.updatePriceDisplay = function(price, formattedPrice) {
        var self = this;
        var priceAmount = formattedPrice || this.formatPriceAmount(price);
        
        this.$priceDisplay.addClass('pwcp-updating');
        
        setTimeout(function() {
            self.$priceDisplay.text(priceAmount);
            self.$priceDisplay.removeClass('pwcp-updating');
        }, TRANSITION_FAST);
    };

    /**
     * Format price amount
     */
    PWCPMaterialCalculator.prototype.formatPriceAmount = function(price) {
        var decimals = parseInt(this.config.decimal_places, 10) || 2;
        var formattedPrice = parseFloat(price).toFixed(decimals);
        
        var currencySymbol = this.config.currency_symbol || '$';
        
        if (currencySymbol.indexOf('&#8360;') !== -1) {
            currencySymbol = '₨';
        } else if (currencySymbol.indexOf('&#8377;') !== -1) {
            currencySymbol = '₹';
        } else if (currencySymbol.indexOf('&#8364;') !== -1) {
            currencySymbol = '€';
        } else if (currencySymbol.indexOf('&#163;') !== -1) {
            currencySymbol = '£';
        } else if (currencySymbol.indexOf('&#') !== -1) {
            var temp = document.createElement('div');
            temp.innerHTML = currencySymbol;
            currencySymbol = temp.textContent || temp.innerText || '$';
        }
        
        return currencySymbol + formattedPrice;
    };

    /**
     * Set calculating state
     */
    PWCPMaterialCalculator.prototype.setCalculatingState = function(isCalculating) {
        if (isCalculating) {
            this.$priceDisplay.addClass('pwcp-calculating');
        } else {
            this.$priceDisplay.removeClass('pwcp-calculating');
            this.updateNavigationState();
        }
    };

    /**
     * Show error message
     */
    PWCPMaterialCalculator.prototype.showError = function(message) {
        this.$errorDisplay.find('.pwcp-error-message').text(message);
        this.$errorDisplay.slideDown(DEBOUNCE_DELAY);
    };

    /**
     * Hide error message
     */
    PWCPMaterialCalculator.prototype.hideError = function() {
        this.$errorDisplay.slideUp(DEBOUNCE_DELAY);
    };

    /**
     * Initialize on document ready
     */
    $(document).ready(function() {
        $('.pwcp-calculator-container.pwcp-modern-design').each(function() {
            if (!$(this).data('pwcp-initialized')) {
                new PWCPMaterialCalculator($(this));
                $(this).data('pwcp-initialized', true);
            }
        });
        
        $(document).on('pwcp_calculator_loaded', '.pwcp-calculator-container.pwcp-modern-design', function() {
            if (!$(this).data('pwcp-initialized')) {
                new PWCPMaterialCalculator($(this));
                $(this).data('pwcp-initialized', true);
            }
        });
    });

/**
 * Re-initialize calculators in quick view popups
 */
function pwcpInitQuickViewCalculators() {
    $('.pwcp-calculator-container.pwcp-modern-design').each(function() {
        if (!$(this).data('pwcp-initialized')) {
            new PWCPMaterialCalculator($(this));
            $(this).data('pwcp-initialized', true);
        }
    });
}

// Hook into common quick view plugins
$(document).on('qv_loader_stop', pwcpInitQuickViewCalculators); // Quick View for WooCommerce
$(document).on('quick-view-displayed', pwcpInitQuickViewCalculators); // YITH Quick View
$(document).on('elementor/popup/show', pwcpInitQuickViewCalculators); // Elementor
$(document).on('wc_quick_view_loaded', pwcpInitQuickViewCalculators); // WooCommerce Quick View
$(document).on('jet-popup/show-event/after-show', pwcpInitQuickViewCalculators); // JetPopup

// Generic fallback for any AJAX content
$(document).ajaxComplete(function(event, xhr, settings) {
    // Only reinit if the AJAX response contains calculator HTML
    if (settings.url && settings.url.indexOf('product') !== -1) {
        setTimeout(pwcpInitQuickViewCalculators, 100);
    }
});

// MutationObserver fallback for any dynamic content
if (window.MutationObserver) {
    var observer = new MutationObserver(function(mutations) {
        mutations.forEach(function(mutation) {
            if (mutation.addedNodes.length) {
                $(mutation.addedNodes).find('.pwcp-calculator-container.pwcp-modern-design').each(function() {
                    if (!$(this).data('pwcp-initialized')) {
                        new PWCPMaterialCalculator($(this));
                        $(this).data('pwcp-initialized', true);
                    }
                });
            }
        });
    });
    
    observer.observe(document.body, {
        childList: true,
        subtree: true
    });
}

})(jQuery);