/**
 * PriceWise Calculator Pro - Admin Settings JavaScript
 * Version: 1.0.0
 */

(function($, window, document, undefined) {
    'use strict';

    var PWCP_AdminSettings = {
        
        activeTab: null,
        initialized: false,
        strings: {},
        
        init: function() {
            if (this.initialized) return;
            
            this.strings = (typeof pwcpSettings !== 'undefined' && pwcpSettings.strings) ? pwcpSettings.strings : {};
            
            this.activeTab = $('.pwcp-tab-panel.pwcp-tab-active').data('panel') || 'general';
            
            this.bindEvents();
            this.initializeColorPickers();
            
            this.initialized = true;
        },
        
        bindEvents: function() {
            var self = this;
            
            $(document).on('click', '.pwcp-tab-btn', function(e) {
                e.preventDefault();
                var tabId = $(this).data('tab');
                if (tabId && tabId !== self.activeTab) {
                    self.switchTab(tabId);
                }
            });
            
            $(document).on('submit', '#pwcp-settings-form', function(e) {
                e.preventDefault();
                self.saveSettingsAjax($(this));
            });
            
            $(document).on('click', '#pwcp-reset-settings', function(e) {
                e.preventDefault();
                var tab = $(this).data('tab');
                self.resetSettings(tab);
            });
            
            $(document).on('change', '.pwcp-visual-picker', function() {
                var color = $(this).val();
                var targetId = $(this).data('target');
                if (targetId) {
                    $('#' + targetId).val(color).trigger('change');
                }
                self.updateColorPreview($(this).closest('.pwcp-enhanced-color-picker'));
            });
            
            $(document).on('input change', '.pwcp-hex-input', function() {
                var color = $(this).val();
                var $container = $(this).closest('.pwcp-enhanced-color-picker');
                var $visualPicker = $container.find('.pwcp-visual-picker');
                
                if (self.isValidHexColor(color)) {
                    $visualPicker.val(color);
                    self.updateColorPreview($container);
                    self.showColorValidation($container, true, self.strings.valid_hex || 'Valid hex color');
                } else {
                    self.showColorValidation($container, false, self.strings.invalid_hex || 'Invalid hex color format');
                }
            });
            
            $(document).on('click', '.pwcp-color-reset', function() {
                var target = $(this).data('target');
                var defaultColor = $(this).data('default');
                if (target && defaultColor) {
                    $('#' + target + '_text').val(defaultColor).trigger('change');
                    $('#' + target + '_visual').val(defaultColor);
                    self.updateColorPreview($(this).closest('.pwcp-enhanced-color-picker'));
                }
            });
            
            $(window).on('popstate', function() {
                window.location.reload();
            });
        },
        
        switchTab: function(tabId) {
            if (this.activeTab === tabId) return;
            
            $('.pwcp-tab-btn').removeClass('pwcp-tab-active');
            $('.pwcp-tab-btn[data-tab="' + tabId + '"]').addClass('pwcp-tab-active');
            
            $('.pwcp-tab-panel').hide().removeClass('pwcp-tab-active');
            $('.pwcp-tab-panel[data-panel="' + tabId + '"]').show().addClass('pwcp-tab-active');
            
            this.activeTab = tabId;
            $('#pwcp-reset-settings').data('tab', tabId);
            
            this.updatePanelHeader(tabId);
            
            var url = new URL(window.location);
            url.searchParams.set('tab', tabId);
            window.history.pushState({}, '', url.toString());
        },
        
        saveSettingsAjax: function($form) {
            var self = this;
            var $submitBtn = $form.find('#pwcp-save-settings');
            var originalText = $submitBtn.val() || $submitBtn.text();
            
            if (typeof pwcpSettings === 'undefined') {
                self.showNotification(self.strings.settings_data_unavailable || 'Settings data not available', 'error');
                return;
            }
            
            var formData = $form.serialize();
            var formNonce = $form.find('#pwcp_settings_nonce').val();
            var nonceToUse = formNonce || pwcpSettings.nonce;
            
            $.ajax({
                url: pwcpSettings.ajax_url,
                type: 'POST',
                data: formData + '&action=pwcp_save_settings&tab=' + this.activeTab + '&pwcp_settings_nonce=' + nonceToUse,
                beforeSend: function() {
                    $submitBtn.prop('disabled', true).addClass('pwcp-saving');
                    var savingText = self.strings.saving || 'Saving...';
                    if ($submitBtn.is('input')) {
                        $submitBtn.val(savingText);
                    } else {
                        $submitBtn.text(savingText);
                    }
                },
                success: function(response) {
                    if (response.success) {
                        var successMsg = response.data.message || self.strings.settings_saved || 'Settings saved successfully!';
                        self.showNotification(successMsg, 'success');
                        
                        if (response.data.new_nonce) {
                            $form.find('#pwcp_settings_nonce').val(response.data.new_nonce);
                            pwcpSettings.nonce = response.data.new_nonce;
                        }
                    } else {
                        var errorMsg = response.data.message || self.strings.save_failed || 'Failed to save settings';
                        self.showNotification(errorMsg, 'error');
                        
                        if (response.data.error_code === 'nonce_failed') {
                            setTimeout(function() {
                                window.location.reload();
                            }, 2000);
                        }
                    }
                },
                error: function() {
                    self.showNotification(self.strings.network_error || 'Network error: Failed to save settings.', 'error');
                },
                complete: function() {
                    $submitBtn.prop('disabled', false).removeClass('pwcp-saving');
                    if ($submitBtn.is('input')) {
                        $submitBtn.val(originalText);
                    } else {
                        $submitBtn.text(originalText);
                    }
                }
            });
        },
        
        initializeColorPickers: function() {
            var self = this;
            $('.pwcp-enhanced-color-picker').each(function() {
                self.updateColorPreview($(this));
            });
        },
        
        updateColorPreview: function($container) {
            var color = $container.find('.pwcp-hex-input').val() || '#000000';
            $container.find('.pwcp-color-preview').css('background-color', color);
        },
        
        showColorValidation: function($container, isValid, message) {
            var $validation = $container.siblings('.pwcp-color-validation');
            if ($validation.length === 0) return;
            
            $validation.removeClass('pwcp-valid pwcp-invalid');
            $validation.addClass(isValid ? 'pwcp-valid' : 'pwcp-invalid');
            $validation.text(message);
        },
        
        isValidHexColor: function(color) {
            return /^#[A-Fa-f0-9]{6}$/.test(color);
        },
        
        resetSettings: function(tab) {
            var self = this;
            var $button = $('#pwcp-reset-settings');
            var originalText = $button.text();
            
            var confirmMsg = self.strings.reset_confirm || 'Are you sure you want to reset these settings to defaults?';
            if (!confirm(confirmMsg)) {
                return;
            }
            
            if (typeof pwcpSettings === 'undefined') {
                self.showNotification(self.strings.settings_data_unavailable || 'Settings data not available', 'error');
                return;
            }
            
            $.ajax({
                url: pwcpSettings.ajax_url,
                type: 'POST',
                data: {
                    action: 'pwcp_reset_settings',
                    tab: tab,
                    pwcp_settings_nonce: pwcpSettings.nonce
                },
                beforeSend: function() {
                    $button.prop('disabled', true).text(self.strings.resetting || 'Resetting...');
                },
                success: function(response) {
                    if (response.success) {
                        self.showNotification(response.data.message, 'success');
                        self.resetFormFieldsToDefaults(tab, response.data.defaults);
                    } else {
                        var errorMsg = response.data.message || self.strings.reset_failed || 'Reset failed';
                        self.showNotification(errorMsg, 'error');
                    }
                },
                error: function() {
                    self.showNotification(self.strings.network_error || 'Network error occurred', 'error');
                },
                complete: function() {
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },
        
        resetFormFieldsToDefaults: function(tab, defaults) {
            var self = this;
            
            $.each(defaults, function(fieldName, defaultValue) {
                var inputName = 'pwcp_' + tab + '_settings[' + fieldName + ']';
                var $field = $('[name="' + inputName + '"]');
                
                if ($field.length > 0) {
                    var fieldType = $field.attr('type');
                    
                    if (fieldType === 'checkbox') {
                        $field.prop('checked', defaultValue === '1' || defaultValue === true);
                    } else if ($field.is('select')) {
                        $field.val(defaultValue).trigger('change');
                    } else if (fieldType === 'number') {
                        $field.val(defaultValue);
                    } else if (fieldType === 'text' || $field.hasClass('pwcp-hex-input')) {
                        $field.val(defaultValue);
                        
                        var $container = $field.closest('.pwcp-enhanced-color-picker');
                        if ($container.length) {
                            var $visualPicker = $container.find('.pwcp-visual-picker');
                            if ($visualPicker.length) {
                                $visualPicker.val(defaultValue);
                            }
                            
                            var $preview = $container.find('.pwcp-color-preview');
                            if ($preview.length) {
                                $preview.css('background-color', defaultValue);
                            }
                            
                            var $hexInput = $container.find('.pwcp-hex-input');
                            if ($hexInput.length) {
                                $hexInput.val(defaultValue);
                            }
                            
                            self.showColorValidation($container, true, self.strings.valid_hex || 'Valid hex color');
                        }
                    } else {
                        $field.val(defaultValue);
                    }
                } else {
                    $('.pwcp-enhanced-color-picker').each(function() {
                        var $container = $(this);
                        var $input = $container.find('input[name="' + inputName + '"]');
                        
                        if ($input.length) {
                            $container.find('.pwcp-visual-picker').val(defaultValue);
                            $container.find('.pwcp-hex-input').val(defaultValue);
                            $container.find('.pwcp-color-preview').css('background-color', defaultValue);
                            $input.val(defaultValue);
                            
                            self.showColorValidation($container, true, self.strings.valid_hex || 'Valid hex color');
                        }
                    });
                }
            });
            
            self.initializeColorPickers();
        },
        
        updatePanelHeader: function(tabId) {
            var self = this;
            
            var defaultTitles = {
                'general': 'General Settings',
                'design': 'Design & Styling',
                'calculation': 'Calculation Settings'
            };
            
            var defaultDescriptions = {
                'general': 'Configure basic calculator behavior, text labels, and messages shown to users.',
                'design': 'Customize the visual appearance of your calculators to match your site design',
                'calculation': 'Configure how calculations are performed and displayed to users.'
            };
            
            var title = (self.strings.panel_titles && self.strings.panel_titles[tabId]) 
                ? self.strings.panel_titles[tabId] 
                : defaultTitles[tabId];
            
            var description = (self.strings.panel_descriptions && self.strings.panel_descriptions[tabId])
                ? self.strings.panel_descriptions[tabId]
                : defaultDescriptions[tabId];
            
            $('#pwcp-panel-title').text(title);
            $('#pwcp-panel-description').text(description);
        },
        
        showNotification: function(message, type) {
            type = type || 'info';
            $('.pwcp-notification').remove();
            
            var iconMap = {
                'success': '✓',
                'error': '✕',
                'warning': '⚠',
                'info': 'ℹ'
            };
            
            var notification = $('<div class="pwcp-notification pwcp-notification-' + type + '">' + 
                                 '<div class="pwcp-notification-icon">' + (iconMap[type] || 'ℹ') + '</div>' +
                                 '<p>' + message + '</p>' +
                                 '<button type="button" class="pwcp-notification-close">&times;</button>' +
                                 '</div>');
            
            $('body').append(notification);
            
            setTimeout(function() {
                notification.addClass('pwcp-notification-show');
            }, 100);
            
            var autoHideTimeout = setTimeout(function() {
                notification.removeClass('pwcp-notification-show');
                setTimeout(function() {
                    notification.remove();
                }, 300);
            }, 4000);
            
            notification.on('click', '.pwcp-notification-close', function() {
                clearTimeout(autoHideTimeout);
                notification.removeClass('pwcp-notification-show');
                setTimeout(function() {
                    notification.remove();
                }, 300);
            });
        }
    };
    
    $(document).ready(function() {
        PWCP_AdminSettings.init();
    });
    
    window.PWCP_AdminSettings = PWCP_AdminSettings;
    
})(jQuery, window, document);