/**
 * Premium Features Lock Handler
 * Handles clicks on locked premium features (Product-Specific, Category-Based & Field Types)
 * 
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

(function($, window, document, undefined) {
    'use strict';

    // Import WordPress i18n functions
    const { __, sprintf } = wp.i18n;

    /**
     * Premium Lock Manager
     * Integrates with existing modal system
     */
    var PWCPPremiumLock = {
        
        /**
         * Configuration
         */
        config: {
            selectors: {
                premiumModal: '#pwcp-premium-upgrade-modal',
                premiumModalClose: '#pwcp-premium-modal-close',
                premiumModalOverlay: '#pwcp-premium-upgrade-modal .pwcp-modal-overlay',
                premiumLockedLabel: '.pwcp-premium-locked',
                premiumLockedCard: '.pwcp-strategy-locked',
                premiumLockedInput: '.pwcp-strategy-locked input[type="radio"]',
                featureName: '#pwcp-premium-feature-name',
                featureDescription: '#pwcp-premium-feature-description',
                
                // Field type related
                fieldTypeSelect: '#modal-field-type',
                premiumFieldTypeHint: '.pwcp-premium-field-hint'
            },
            classes: {
                modalOpen: 'pwcp-modal-open',
                modalActive: 'pwcp-modal-active'
            }
        },
        
        /**
         * Initialize
         */
        init: function() {
            this.bindEvents();
            this.addPremiumFieldHint();
        },
        
        /**
         * Add visual hint for premium field types
         * NO INLINE STYLES - all styling in CSS
         */
        addPremiumFieldHint: function() {
            var $selectWrapper = $(this.config.selectors.fieldTypeSelect).closest('.pwcp-select-wrapper');
            
            if ($selectWrapper.length && !$('.pwcp-premium-field-hint').length) {
                var hintHtml = 
                    '<div class="pwcp-premium-field-hint">' +
                        '<a href="#" class="pwcp-premium-field-link">' +
                            '<span class="pwcp-premium-lock-emoji">🔒</span>' +
                            '<span>' + __('Unlock Premium Field Types', 'pricewise-calculator-pro') + '</span>' +
                            '<svg class="pwcp-premium-arrow" fill="none" stroke="currentColor" viewBox="0 0 24 24">' +
                                '<path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"/>' +
                            '</svg>' +
                        '</a>' +
                    '</div>';
                
                $selectWrapper.after(hintHtml);
            }
        },
        
        /**
         * Bind event handlers
         * ALL EVENTS NAMESPACED with .pwcpPremium
         */
        bindEvents: function() {
            var self = this;
            
            // Handle clicks on locked strategy card labels
            $(document).on('click.pwcpPremium', this.config.selectors.premiumLockedLabel, function(e) {
                e.preventDefault();
                e.stopPropagation();
                
                var featureType = $(this).data('premium');
                self.showPremiumModal(featureType);
            });
            
            // Handle clicks on locked strategy card inputs
            $(document).on('click.pwcpPremium', this.config.selectors.premiumLockedInput, function(e) {
                e.preventDefault();
                e.stopPropagation();
                
                var $label = $(this).closest(self.config.selectors.premiumLockedCard)
                                    .find(self.config.selectors.premiumLockedLabel);
                var featureType = $label.data('premium');
                
                self.showPremiumModal(featureType);
            });
            
            // Handle clicks on entire locked strategy cards
            $(document).on('click.pwcpPremium', this.config.selectors.premiumLockedCard, function(e) {
                var $target = $(e.target);
                
                // Don't trigger if clicking on input or label (already handled above)
                if ($target.is('input') || $target.closest('label').length) {
                    return;
                }
                
                e.preventDefault();
                e.stopPropagation();
                
                var $label = $(this).find(self.config.selectors.premiumLockedLabel);
                var featureType = $label.data('premium');
                
                self.showPremiumModal(featureType);
            });
            
            // Handle clicks on premium field type hint/link
            $(document).on('click.pwcpPremium', '.pwcp-premium-field-link', function(e) {
                e.preventDefault();
                self.showPremiumModal('field-types');
            });
            
            // Handle clicks on the field type select when trying to access premium options
            $(document).on('mousedown.pwcpPremium click.pwcpPremium', this.config.selectors.fieldTypeSelect, function(e) {
                var $select = $(this);
                
                // Check if select is opening (not already open)
                if (!$select.data('is-open')) {
                    $select.data('is-open', true);
                    
                    // Reset after a delay
                    setTimeout(function() {
                        $select.data('is-open', false);
                    }, 300);
                }
            });
            
            // Detect when user tries to select a premium field type
            $(document).on('focus.pwcpPremium', this.config.selectors.fieldTypeSelect, function(e) {
                var $select = $(this);
                var currentValue = $select.val();
                
                // Store the current valid value
                $select.data('last-valid-value', currentValue);
            });
            
            // Check if premium option was somehow selected (safety check)
            $(document).on('change.pwcpPremium', this.config.selectors.fieldTypeSelect, function(e) {
                var $select = $(this);
                var selectedOption = $select.find('option:selected');
                
                // Check if selected option has premium attribute
                if (selectedOption.is('[data-premium-field]')) {
                    e.preventDefault();
                    e.stopPropagation();
                    
                    // Revert to last valid value
                    var lastValid = $select.data('last-valid-value') || 'number';
                    $select.val(lastValid);
                    
                    // Show premium modal
                    self.showPremiumModal('field-types');
                    
                    return false;
                }
                
                // Update last valid value
                $select.data('last-valid-value', $select.val());
            });
            
            // Close modal on overlay click
            $(document).on('click.pwcpPremium', this.config.selectors.premiumModalOverlay, function(e) {
                e.preventDefault();
                self.hidePremiumModal();
            });
            
            // Close modal on close button click
            $(document).on('click.pwcpPremium', this.config.selectors.premiumModalClose, function(e) {
                e.preventDefault();
                self.hidePremiumModal();
            });
            
            // Close modal on ESC key - NAMESPACED
            $(document).on('keydown.pwcpPremiumModal', function(e) {
                var $modal = $(self.config.selectors.premiumModal);
                if (e.key === 'Escape' && $modal.hasClass(self.config.classes.modalOpen)) {
                    e.preventDefault();
                    self.hidePremiumModal();
                }
            });
            
            // Prevent clicks inside modal container from closing
            $(document).on('click.pwcpPremium', this.config.selectors.premiumModal + ' .pwcp-modal-container', function(e) {
                e.stopPropagation();
            });
        },
        
        /**
         * Show premium upgrade modal
         * 
         * @param {string} featureType - Type of premium feature (product/category/field-types)
         */
        showPremiumModal: function(featureType) {
            var $modal = $(this.config.selectors.premiumModal);
            var $featureName = $(this.config.selectors.featureName);
            var $featureDescription = $(this.config.selectors.featureDescription);
            var $body = $('body');
            
            if (!$modal.length) {
                if (window.console && window.console.error) {
                    window.console.error(__('Premium modal not found in DOM', 'pricewise-calculator-pro'));
                }
                return;
            }
            
            // Customize content based on feature type
            if (featureType === 'product') {
                $featureName.text(__('Product-Specific Calculators', 'pricewise-calculator-pro'));
                $featureDescription.text(
                    __('Assign calculators to specific products for targeted pricing and calculations. Create unique experiences for each product in your store.', 'pricewise-calculator-pro')
                );
            } else if (featureType === 'category') {
                $featureName.text(__('Category-Based Assignment', 'pricewise-calculator-pro'));
                $featureDescription.text(
                    __('Automatically assign calculators to entire product categories. Perfect for applying the same calculator across multiple related products.', 'pricewise-calculator-pro')
                );
            } else if (featureType === 'field-types') {
                $featureName.text(__('Advanced Field Types', 'pricewise-calculator-pro'));
                $featureDescription.text(
                    __('Unlock powerful field types including Text, Dropdown, Radio Buttons, Checkboxes, Slider, and File Upload. Create interactive and dynamic calculators with advanced input options.', 'pricewise-calculator-pro')
                );
            }
            
            // Remove hidden class and add open class (using existing modal system)
            $modal.removeClass('pwcp-hidden');
            
            // Use setTimeout to ensure CSS transition works
            setTimeout(function() {
                $modal.addClass('pwcp-modal-open');
                $body.addClass('pwcp-modal-active');
            }, 10);
            
            // Focus management for accessibility
            setTimeout(function() {
                $(this.config.selectors.premiumModalClose).focus();
            }.bind(this), 300);
        },
        
        /**
         * Hide premium upgrade modal
         */
        hidePremiumModal: function() {
            var $modal = $(this.config.selectors.premiumModal);
            var $body = $('body');
            
            // Remove open class (triggers CSS transition)
            $modal.removeClass(this.config.classes.modalOpen);
            $body.removeClass(this.config.classes.modalActive);
            
            // Add hidden class after transition completes
            setTimeout(function() {
                $modal.addClass('pwcp-hidden');
            }, 300);
        },
        
        /**
         * Teardown - clean unbind of all events
         * Call this if you need to destroy the instance
         */
        teardown: function() {
            $(document).off('.pwcpPremium');
            $(document).off('.pwcpPremiumModal');
            $('.pwcp-premium-field-hint').remove();
        }
    };
    
    // Expose to global scope with LOWERCASE camelCase (Envato standard)
    window.pwcpPremiumLock = PWCPPremiumLock;
    
    /**
     * Initialize on document ready
     */
    $(document).ready(function() {
        PWCPPremiumLock.init();
    });
    
})(jQuery, window, document);