/* global wp, jQuery */
/**
 * PriceWise Calculator Pro — Admin Dashboard Script
 * File: assets/js/all-admin-js/admin-dashboard.js
 *
 * Handles:
 * - Quick action clicks
 * - Copy-to-clipboard for status info
 * - Collapsible tips
 * 
 * @package PriceWise_Calculator_Pro
 * @version 1.0.0
 * @since 1.0.0
 */
(function ($) {
    'use strict';
    
    // ES5-compatible for JSHint default options (Envato requirement)
    var __ = wp.i18n.__;
    
    // Unique prefixed object name (Envato requirement)
    var pwcpAdminDashboard = {
        
        init: function () {
            this.pwcpUnbindEvents(); // Unbind before binding (Envato requirement)
            this.pwcpBindUI();
        },
        
        // Unbind all events to prevent conflicts (Envato requirement)
        pwcpUnbindEvents: function () {
            $('.pwcp-dashboard').off('click.pwcpDashboard');
        },
        
        pwcpBindUI: function () {
            var self = this;
            
            // Quick Actions - with event namespace and fallback to href
            $('.pwcp-dashboard').on('click.pwcpDashboard', '.pwcp-quick-action', function(e) {
                self.pwcpHandleQuickAction.call(this, e);
            });
            
            // Copy to clipboard
            $('.pwcp-dashboard').on('click.pwcpDashboard', '[data-copy-text]', function(e) {
                self.pwcpCopyToClipboard.call(this, e);
            });
            
            // Toggle tips
            $('.pwcp-dashboard').on('click.pwcpDashboard', '.pwcp-tip-toggle', function(e) {
                self.pwcpToggleTip.call(this, e);
            });
        },
        
        pwcpHandleQuickAction: function (e) {
            var $el = $(this);
            var action = $el.data('action');
            var url = $el.data('url') || $el.attr('href');
            
            // If no action and no URL, let default behavior happen
            if (!action && !url) {
                return;
            }
            
            e.preventDefault();
            
            // Navigate to URL if available
            if (url) {
                window.location.href = url;
                return;
            }
            
            // Otherwise run custom action
            pwcpAdminDashboard.pwcpRunAction(action);
        },
        
        pwcpRunAction: function (action) {
            // Placeholder for custom admin AJAX actions
            // Production code - console.log removed for production (Envato best practice)
            if (typeof window.pwcpDebug !== 'undefined' && window.pwcpDebug) {
                console.log('Running quick action:', action);
            }
        },
        
        pwcpCopyToClipboard: function () {
            var text = $(this).data('copy-text');
            if (!text) {
                return;
            }
            
            // Check if Clipboard API is available
            if (typeof navigator.clipboard === 'undefined' || typeof navigator.clipboard.writeText !== 'function') {
                pwcpAdminDashboard.pwcpShowNotice(
                    __('Clipboard API not available in this browser.', 'pricewise-calculator-pro'), 
                    'error'
                );
                return;
            }
            
            // Use translated strings (Envato requirement)
            navigator.clipboard.writeText(text).then(function() {
                pwcpAdminDashboard.pwcpShowNotice(
                    __('Copied to clipboard!', 'pricewise-calculator-pro'), 
                    'success'
                );
            }).catch(function() {
                pwcpAdminDashboard.pwcpShowNotice(
                    __('Failed to copy.', 'pricewise-calculator-pro'), 
                    'error'
                );
            });
        },
        
        pwcpToggleTip: function () {
            $(this).closest('.pwcp-tip').find('.pwcp-tip-content').slideToggle(200);
        },
        
        pwcpShowNotice: function (message, type) {
            // Secure dynamic HTML creation to prevent XSS (Envato requirement)
            var $notice = $('<div>')
                .addClass('notice notice-' + type + ' is-dismissible')
                .append($('<p>').text(message));
            
            // Prepend to .pwcp-dashboard if it exists, otherwise fallback to .wrap
            var $container = $('.pwcp-dashboard').length ? $('.pwcp-dashboard') : $('.wrap');
            $container.prepend($notice);
            
            setTimeout(function() {
                $notice.fadeOut(300, function () { 
                    $(this).remove(); 
                });
            }, 3000);
        }
    };
    
    $(document).ready(function () {
        pwcpAdminDashboard.init();
    });
    
    // Expose globally with unique prefix (Envato requirement)
    window.pwcpAdminDashboard = pwcpAdminDashboard;
    
})(jQuery);