/**
 * Calculator List Page JavaScript
 * @package PriceWise_Calculator_Pro
 * @version 1.0.0
 * @since 1.0.0
 */

(function($, window, document, undefined) {
    'use strict';

    /**
     * Calculator List Manager
     * 
     * Uses server-side counts and proper localization
     * @since 1.0.0
     */
    var CalculatorListManager = {
        
        // Configuration
        config: {
            selectors: {
                // Main containers
                calculatorGrid: '#pwcp-calculator-grid',
                calculatorCard: '.pwcp-calculator-card',
                loadingOverlay: '#pwcp-loading-overlay',
                
                // Stats counters
                totalCount: '#pwcp-total-count',
                activeCount: '#pwcp-active-count',
                recentCount: '#pwcp-recent-count',
                
                // View controls
                viewToggle: '.pwcp-view-btn',
                viewToggleContainer: '.pwcp-view-toggle[data-mobile-behavior="hide"]',
                searchInput: '#pwcp-calculator-search',
                statusFilter: '#pwcp-status-filter',
                typeFilter: '#pwcp-type-filter',

                // Bulk actions
                bulkSelect: '#pwcp-bulk-action',
                applyBulk: '#pwcp-apply-bulk',
                calculatorCheckbox: '.pwcp-calculator-checkbox',
                selectAll: '#pwcp-select-all-calculators',
                
                // Card actions
                deleteBtn: '.pwcp-delete-calculator',
                duplicateBtn: '.pwcp-duplicate-calculator',
                dropdown: '.pwcp-dropdown',
                dropdownToggle: '.pwcp-dropdown-toggle',
                dropdownMenu: '.pwcp-dropdown-menu'
            },
            
            classes: {
                hidden: 'pwcp-hidden',
                viewGrid: 'pwcp-view-grid',
                viewList: 'pwcp-view-list',
                viewActive: 'pwcp-view-active',
                dropdownOpen: 'pwcp-dropdown-open',
                loading: 'pwcp-loading',
                selected: 'pwcp-selected',
                fadeOut: 'pwcp-fade-out',
                slideInTop: 'pwcp-slide-in-top'
            },
            
            animations: {
                fast: 300,
                normal: 500
            },
            
            // Responsive breakpoint (matches CSS)
            mobileBreakpoint: '(max-width: 47.99em)'
        },
        
        // State management
        state: {
            currentView: 'grid',
            isInitialized: false,
            searchTerm: '',
            statusFilter: '',
            typeFilter: '',
            selectedItems: [],
            totalItems: 0,
            filteredItems: 0,
            isMobile: false,
            isProcessing: false
        },
        
        /**
         * Initialize the list manager
         * 
         * @since 1.0.0
         */
        init: function() {
            if (this.state.isInitialized) {
                return;
            }
            
            // Only initialize on calculator list page
            if (!$('.pwcp-page-calculator-list').length) {
                return;
            }
            
            this._bindEvents();
            this._initializeResponsive();
            this._initializeView();
            this._initializeFilters();
            this._initializeBulkActions();
            this._initializeDropdowns();
            
            // Set initial totals from DOM
            this.state.totalItems = $(this.config.selectors.calculatorCard).length;
            
            this.state.isInitialized = true;
            
            if (window.console && window.console.log) {
                window.console.log('Calculator List Manager initialized with server-side counts');
            }
        },
        
        /**
         * Initialize responsive behavior
         * 
         * @since 1.0.0
         * @private
         */
        _initializeResponsive: function() {
            this.state.isMobile = this._isMobileView();
            this._handleResponsiveView();
        },
        
        /**
         * Check if current view is mobile
         * 
         * @since 1.0.0
         * @private
         * @returns {boolean}
         */
        _isMobileView: function() {
            if (window.matchMedia) {
                return window.matchMedia(this.config.mobileBreakpoint).matches;
            }
            return window.innerWidth <= 767;
        },
        
        /**
         * Handle responsive view changes
         * 
         * @since 1.0.0
         * @private
         */
        _handleResponsiveView: function() {
            var $grid = $(this.config.selectors.calculatorGrid);
            
            if (this.state.isMobile) {
                // Force grid view on mobile
                this.state.currentView = 'grid';
                $grid.removeClass(this.config.classes.viewList)
                     .addClass(this.config.classes.viewGrid);
            }
        },
        
        /**
         * Bind all event handlers with namespacing
         * 
         * @since 1.0.0
         * @private
         */
        _bindEvents: function() {
            var self = this;
            
            // Prevent duplicate handlers on re-init
            $(document).off('.pwcp');
            $(window).off('.pwcp');
            
            // View toggle - only works on desktop/tablet
            $(document).on('click.pwcp', this.config.selectors.viewToggle, function(e) {
                e.preventDefault();
                if (!self._isMobileView()) {
                    var view = $(this).data('view');
                    self._switchView(view);
                }
            });
            
            // Responsive handler using matchMedia
            if (window.matchMedia) {
                var mediaQuery = window.matchMedia(this.config.mobileBreakpoint);
                
                var handleMediaChange = function() {
                    self._handleMediaQueryChange();
                };
                
                if (mediaQuery.addEventListener) {
                    mediaQuery.addEventListener('change', handleMediaChange);
                } else if (mediaQuery.addListener) {
                    mediaQuery.addListener(handleMediaChange);
                }
            } else {
                $(window).on('resize.pwcp', self._debounce(function() {
                    self._handleMediaQueryChange();
                }, 150));
            }
            
            // Search functionality
            $(document).on('input.pwcp', this.config.selectors.searchInput, function() {
                if (self._searchTimeout) {
                    clearTimeout(self._searchTimeout);
                }
                self._searchTimeout = setTimeout(function() {
                    self.state.searchTerm = $(self.config.selectors.searchInput).val().toLowerCase();
                    self._applyFilters();
                }, 300);
            });
            
            // Filter changes
            $(document).on('change.pwcp', this.config.selectors.statusFilter + ', ' + this.config.selectors.typeFilter, function() {
                self.state.statusFilter = $(self.config.selectors.statusFilter).val();
                self.state.typeFilter = $(self.config.selectors.typeFilter).val();
                self._applyFilters();
            });
            
            // Bulk actions
            $(document).on('click.pwcp', this.config.selectors.applyBulk, function(e) {
                e.preventDefault();
                if (!self.state.isProcessing) {
                    self._handleBulkAction();
                }
            });
            
            // Checkbox selection
            $(document).on('change.pwcp', this.config.selectors.calculatorCheckbox, function() {
                var $card = $(this).closest('.pwcp-calculator-card');
                
                if ($(this).is(':checked')) {
                    $card.addClass('pwcp-selected');
                } else {
                    $card.removeClass('pwcp-selected');
                }
                
                self._updateSelection();
            });
            
            // Select all functionality
            $(document).on('change.pwcp', this.config.selectors.selectAll, function() {
                var isChecked = $(this).is(':checked');
                $(self.config.selectors.calculatorCheckbox + ':visible').each(function() {
                    $(this).prop('checked', isChecked);
                    var $card = $(this).closest('.pwcp-calculator-card');
                    
                    if (isChecked) {
                        $card.addClass('pwcp-selected');
                    } else {
                        $card.removeClass('pwcp-selected');
                    }
                });
                self._updateSelection();
            });
            
            // Card actions - AJAX Delete
            $(document).on('click.pwcp', this.config.selectors.deleteBtn, function(e) {
                e.preventDefault();
                if (!self.state.isProcessing) {
                    var calculatorId = $(this).data('calculator-id');
                    self._deleteCalculator(calculatorId);
                }
            });
            
            // Card actions - AJAX Duplicate
            $(document).on('click.pwcp', this.config.selectors.duplicateBtn, function(e) {
                e.preventDefault();
                if (!self.state.isProcessing) {
                    var calculatorId = $(this).data('calculator-id');
                    self._duplicateCalculator(calculatorId);
                }
            });
            
            // Dropdown management
            $(document).on('click.pwcp', this.config.selectors.dropdownToggle, function(e) {
                e.preventDefault();
                e.stopPropagation();
                self._toggleDropdown($(this).closest(self.config.selectors.dropdown));
            });
            
            // Close dropdowns when clicking outside
            $(document).on('click.pwcp', function(e) {
                if (!$(e.target).closest(self.config.selectors.dropdown).length) {
                    self._closeAllDropdowns();
                }
            });
            
            // Keyboard navigation
            $(document).on('keydown.pwcp', function(e) {
                if (e.key === 'Escape') {
                    self._closeAllDropdowns();
                }
            });
        },
        
        /**
         * Handle media query changes
         * 
         * @since 1.0.0
         * @private
         */
        _handleMediaQueryChange: function() {
            var wasMobile = this.state.isMobile;
            this.state.isMobile = this._isMobileView();
            
            if (!wasMobile && this.state.isMobile) {
                this._handleResponsiveView();
                this._updateViewState();
            } else if (wasMobile && !this.state.isMobile) {
                var savedView = this._getSavedView();
                this.state.currentView = savedView;
                this._updateViewState();
            }
        },
        
        /**
         * Initialize view settings
         * 
         * @since 1.0.0
         * @private
         */
        _initializeView: function() {
            var initialView = 'grid';
            
            if (this.state.isMobile) {
                initialView = 'grid';
            } else {
                initialView = this._getSavedView();
            }
            
            this.state.currentView = initialView;
            this._updateViewState();
                 
            if (window.console && window.console.log) {
                window.console.log('View initialized:', initialView, '(Mobile:', this.state.isMobile + ')');
            }
        },
        
        /**
         * Get saved view preference
         * 
         * @since 1.0.0
         * @private
         * @returns {string}
         */
        _getSavedView: function() {
            var savedView = 'grid';
            
            if (window.pwcpCalculatorList && window.pwcpCalculatorList.initial_view) {
                savedView = window.pwcpCalculatorList.initial_view;
            } else {
                var $grid = $(this.config.selectors.calculatorGrid);
                if ($grid.data('initial-view')) {
                    savedView = $grid.data('initial-view');
                } else if (window.localStorage) {
                    savedView = window.localStorage.getItem('pwcp_calculator_view') || 'grid';
                }
            }
            
            return savedView;
        },
        
        /**
         * Update view state and UI
         * 
         * @since 1.0.0
         * @private
         */
        _updateViewState: function() {
            var $grid = $(this.config.selectors.calculatorGrid);
            var $viewBtns = $(this.config.selectors.viewToggle);
            
            $grid.removeClass(this.config.classes.viewGrid + ' ' + this.config.classes.viewList)
                 .addClass('pwcp-view-' + this.state.currentView);
            
            if ($viewBtns.is(':visible')) {
                $viewBtns.removeClass(this.config.classes.viewActive);
                $viewBtns.filter('[data-view="' + this.state.currentView + '"]')
                         .addClass(this.config.classes.viewActive);
            }
        },
        
        /**
         * Initialize filter functionality
         * 
         * @since 1.0.0
         * @private
         */
        _initializeFilters: function() {
            this.state.statusFilter = $(this.config.selectors.statusFilter).val() || '';
            this.state.typeFilter = $(this.config.selectors.typeFilter).val() || '';
            this.state.searchTerm = $(this.config.selectors.searchInput).val().toLowerCase() || '';
            
            this._applyFilters();
        },
        
        /**
         * Initialize bulk actions
         * 
         * @since 1.0.0
         * @private
         */
        _initializeBulkActions: function() {
            this.state.selectedItems = [];
            this._updateBulkActionsState();
        },
        
        /**
         * Initialize dropdown functionality
         * 
         * @since 1.0.0
         * @private
         */
        _initializeDropdowns: function() {
            this._closeAllDropdowns();
        },
        
        /**
         * Switch between grid and list views
         * 
         * @param {string} view View type ('grid' or 'list')
         * @since 1.0.0
         * @private
         */
        _switchView: function(view) {
            if (this.state.isMobile) {
                if (window.console && window.console.log) {
                    window.console.log('View switching disabled on mobile');
                }
                return;
            }
            
            if (view === this.state.currentView) {
                return;
            }
            
            this.state.currentView = view;
            this._updateViewState();
            
            if (window.localStorage) {
                window.localStorage.setItem('pwcp_calculator_view', view);
            }
            this._saveViewPreference(view);
            
            $(document).trigger('pwcp:view:changed', [view]);
            
            if (window.console && window.console.log) {
                window.console.log('View switched to:', view);
            }
        },
        
        /**
         * Save view preference to server
         * 
         * @param {string} view View type
         * @since 1.0.0
         * @private
         */
        _saveViewPreference: function(view) {
            if (!window.pwcpCalculatorList || !window.pwcpCalculatorList.ajax_url) {
                this._logError(this._getString('ajax_config_missing'));
                return;
            }
            
            $.post(window.pwcpCalculatorList.ajax_url, {
                action: 'pwcp_save_view_preference',
                view: view,
                nonce: window.pwcpCalculatorList.nonce
            }).done(function(response) {
                if (response.success) {
                    if (window.console && window.console.log) {
                        window.console.log('View preference saved to server:', view);
                    }
                } else {
                    if (window.console && window.console.warn) {
                        window.console.warn('Failed to save view preference:', response.data ? response.data.message : 'Unknown error');
                    }
                }
            }).fail(function(xhr, status, error) {
                if (window.console && window.console.warn) {
                    window.console.warn('AJAX request failed:', error);
                }
            });
        },
        
        /**
         * Apply search and filter criteria
         * 
         * @since 1.0.0
         * @private
         */
        _applyFilters: function() {
            var self = this;
            var $cards = $(this.config.selectors.calculatorCard);
            var visibleCount = 0;
            
            // Recalculate total items
            this.state.totalItems = $cards.length;
            
            $cards.each(function() {
                var $card = $(this);
                var name = $card.data('name') || '';
                var status = $card.data('status') || '';
                var type = $card.data('type') || '';
                
                var matchesSearch = !self.state.searchTerm || name.indexOf(self.state.searchTerm) !== -1;
                var matchesStatus = !self.state.statusFilter || status === self.state.statusFilter;
                var matchesType = !self.state.typeFilter || type === self.state.typeFilter;
                
                var isVisible = matchesSearch && matchesStatus && matchesType;
                
                if (isVisible) {
                    $card.show();
                    visibleCount++;
                } else {
                    $card.hide();
                }
            });
            
            this.state.filteredItems = visibleCount;
            this._showEmptyStateIfNeeded();
        },
        
        /**
         * Show empty state if no results
         * 
         * @since 1.0.0
         * @private
         */
        _showEmptyStateIfNeeded: function() {
            var self = this;
            var $grid = $(this.config.selectors.calculatorGrid);
            var $emptyState = $grid.find('.pwcp-empty-search-results');
            
            if (this.state.filteredItems === 0 && this.state.totalItems > 0) {
                if (!$emptyState.length) {
                    var emptyHtml = $('<div>', {
                        class: 'pwcp-empty-search-results pwcp-empty-state pwcp-empty-centered',
                        html: '<div class="pwcp-empty-illustration">' +
                              '<svg class="pwcp-empty-icon" viewBox="0 0 200 150">' +
                              '<circle cx="100" cy="75" r="50" fill="none" stroke="#e5e7eb" stroke-width="4"/>' +
                              '<path d="m125 100 20 20" stroke="#e5e7eb" stroke-width="4" stroke-linecap="round"/>' +
                              '</svg>' +
                              '</div>' +
                              '<h3>' + self._escapeHtml(self._getString('no_results_title')) + '</h3>' +
                              '<p>' + self._escapeHtml(self._getString('no_results_body')) + '</p>' +
                              '<button type="button" class="pwcp-btn pwcp-btn-ghost pwcp-clear-filters">' + self._escapeHtml(self._getString('clear_filters')) + '</button>'
                    });
                    
                    $grid.append(emptyHtml);
                    
                    emptyHtml.find('.pwcp-clear-filters').on('click', function() {
                        $(self.config.selectors.searchInput).val('');
                        $(self.config.selectors.statusFilter).val('');
                        $(self.config.selectors.typeFilter).val('');
                        self.state.searchTerm = '';
                        self.state.statusFilter = '';
                        self.state.typeFilter = '';
                        self._applyFilters();
                        if (window.console && window.console.log) {
                            window.console.log('Filters cleared');
                        }
                    });
                    
                    emptyHtml.show();
                } else {
                    $emptyState.show();
                }
            } else {
                $emptyState.hide();
            }
        },
        
        /**
         * Handle bulk actions
         * 
         * @since 1.0.0
         * @private
         */
        _handleBulkAction: function() {
            var action = $(this.config.selectors.bulkSelect).val();
            var selectedIds = this.state.selectedItems;
            
            if (!action) {
                this._showNotification(this._getString('no_action'), 'warning');
                return;
            }
            
            if (selectedIds.length === 0) {
                this._showNotification(this._getString('no_selection'), 'warning');
                return;
            }
            
            // Optional confirmation (configurable via localized settings)
            if (window.pwcpCalculatorList && window.pwcpCalculatorList.confirmations !== false) {
                var confirmKey = 'confirm_bulk_' + action;
                var confirmMessage = this._getString(confirmKey);
                if (confirmMessage && confirmMessage !== 'Unknown') {
                    if (!window.confirm(confirmMessage)) {
                        return;
                    }
                }
            }
            
            // Execute action
            this._performBulkAction(action, selectedIds);
        },
        
        /**
         * Perform bulk action with server-side count updates
         * 
         * @param {string} action Action to perform
         * @param {array} calculatorIds Calculator IDs
         * @since 1.0.0
         * @private
         */
        _performBulkAction: function(action, calculatorIds) {
            if (!window.pwcpCalculatorList || !window.pwcpCalculatorList.ajax_url) {
                this._showNotification(this._getString('ajax_config_missing'), 'error');
                return;
            }
            
            var self = this;
            this.state.isProcessing = true;
                
            $.post(window.pwcpCalculatorList.ajax_url, {
                action: 'pwcp_bulk_action_calculators',
                action_type: action,
                calculator_ids: calculatorIds,
                nonce: window.pwcpCalculatorList.bulk_nonce
            })
            .done(function(response) {
                if (response.success) {
                    self._showNotification(response.data.message, 'success');
                    
                    // Handle different actions
                    if (action === 'delete') {
                        // Remove deleted items from DOM with animation
                        var remaining = calculatorIds.length;
                        calculatorIds.forEach(function(id) {
                            var $card = $('[data-id="' + id + '"]');
                            $card.addClass('pwcp-fade-out');
                            
                            setTimeout(function() {
                                $card.remove();
                                if (--remaining === 0) {
                                    self.state.totalItems = $(self.config.selectors.calculatorCard).length;
                                    if (self.state.totalItems === 0) {
                                        self._showCompleteEmptyState();
                                    } else {
                                        self._applyFilters();
                                    }
                                }
                            }, 300);
                        });
                    } else if (action === 'duplicate' && response.data.new_cards_html) {
                        // Add new duplicated cards at the TOP of the list
                        var $grid = $(self.config.selectors.calculatorGrid);
                        
                        response.data.new_cards_html.forEach(function(cardData, index) {
                            var $newCard = $(cardData.html);
                            
                            // Start invisible
                            $newCard.addClass('pwcp-invisible');
                            $grid.prepend($newCard);
                            
                            // Stagger animation for multiple cards
                            setTimeout(function() {
                                $newCard.addClass(self.config.classes.slideInTop);
                            }, index * 50 + 10);
                            
                            // Remove animation class after it completes
                            setTimeout(function() {
                                $newCard.removeClass(self.config.classes.slideInTop + ' pwcp-invisible');
                            }, index * 50 + 450);
                        });
                        
                        self.state.totalItems = $(self.config.selectors.calculatorCard).length;
                        self._applyFilters();
                    } else if (action === 'activate' || action === 'deactivate') {
                        // Update status for affected cards
                        var newStatus = action === 'activate' ? 'active' : 'inactive';
                        var statusClass = newStatus === 'active' ? 'pwcp-status-active' : 'pwcp-status-inactive';
                        var statusText = newStatus === 'active'
                            ? self._getString('status_active') || 'Active'
                            : self._getString('status_inactive') || 'Inactive';
                        
                        calculatorIds.forEach(function(id) {
                            var $card = $('[data-id="' + id + '"]');
                            var $statusIndicator = $card.find('.pwcp-status-indicator');
                            
                            // Add pulse effect
                            $card.addClass('pwcp-pulse-once');
                            setTimeout(function() {
                                $card.removeClass('pwcp-pulse-once');
                            }, 600);
                            
                            // Update card data
                            $card.attr('data-status', newStatus);
                            
                            // Update visual status indicators
                            $statusIndicator.removeClass('pwcp-status-active pwcp-status-inactive')
                                           .addClass(statusClass)
                                           .text(statusText);
                        });
                    }
                    
                    // Get fresh counts from server after operation
                    self._refreshCountsFromServer();
                    self._clearSelection();
                } else {
                    self._showNotification(response.data.message || self._getString('error'), 'error');
                }
            })
            .fail(function() {
                self._showNotification(self._getString('request_failed'), 'error');
            })
            .always(function() {
                self.state.isProcessing = false;
            });
        },
        
        /**
         * Show complete empty state when no calculators exist
         * 
         * @since 1.0.0
         * @private
         */
        _showCompleteEmptyState: function() {
            var $grid = $(this.config.selectors.calculatorGrid);
            var adminUrl = this._getAdminUrl('admin.php?page=pwcp-add-calculator');
            
            var $emptyState = $('<div>', {
                class: 'pwcp-empty-state pwcp-empty-calculators pwcp-empty-centered',
                html: '<div class="pwcp-empty-illustration">' +
                    '<svg class="pwcp-empty-icon" viewBox="0 0 200 150">' +
                        '<defs>' +
                            '<linearGradient id="calc-gradient" x1="0%" y1="0%" x2="100%" y2="100%">' +
                                '<stop offset="0%" style="stop-color:#6366f1;stop-opacity:0.1"/>' +
                                '<stop offset="100%" style="stop-color:#a855f7;stop-opacity:0.1"/>' +
                            '</linearGradient>' +
                        '</defs>' +
                        '<rect x="40" y="30" width="120" height="90" rx="8" fill="url(#calc-gradient)" stroke="#6366f1" stroke-width="2"/>' +
                        '<rect x="50" y="45" width="25" height="15" rx="2" fill="#6366f1" opacity="0.3"/>' +
                        '<rect x="80" y="45" width="25" height="15" rx="2" fill="#8b5cf6" opacity="0.3"/>' +
                        '<rect x="110" y="45" width="25" height="15" rx="2" fill="#a855f7" opacity="0.3"/>' +
                        '<rect x="50" y="65" width="25" height="15" rx="2" fill="#6366f1" opacity="0.3"/>' +
                        '<rect x="80" y="65" width="25" height="15" rx="2" fill="#8b5cf6" opacity="0.3"/>' +
                        '<rect x="110" y="65" width="25" height="15" rx="2" fill="#a855f7" opacity="0.3"/>' +
                        '<rect x="50" y="85" width="85" height="20" rx="3" fill="#6366f1" opacity="0.2"/>' +
                        '<circle cx="180" cy="40" r="8" fill="#6366f1" opacity="0.2"/>' +
                        '<circle cx="20" cy="80" r="6" fill="#8b5cf6" opacity="0.2"/>' +
                        '<circle cx="25" cy="45" r="4" fill="#a855f7" opacity="0.2"/>' +
                    '</svg>' +
                '</div>' +
                '<h3>' + this._escapeHtml(this._getString('empty_state_title')) + '</h3>' +
                '<p>' + this._escapeHtml(this._getString('empty_state_description')) + '</p>' +
                '<a href="' + this._escapeHtml(adminUrl) + '" class="pwcp-btn pwcp-btn-primary pwcp-btn-icon">' +
                    '<svg class="pwcp-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">' +
                        '<line x1="12" y1="5" x2="12" y2="19"/>' +
                        '<line x1="5" y1="12" x2="19" y2="12"/>' +
                    '</svg>' +
                    this._escapeHtml(this._getString('create_first_calculator')) +
                '</a>'
            });
            
            $grid.html($emptyState);
        },
        
        /**
         * Get admin URL
         * 
         * @param {string} path Path to append
         * @since 1.0.0
         * @private
         * @returns {string}
         */
        _getAdminUrl: function(path) {
            if (window.ajaxurl) {
                return window.ajaxurl.replace('admin-ajax.php', path);
            }
            return '/wp-admin/' + path;
        },
        
        /**
         * Update selection state
         * 
         * @since 1.0.0
         * @private
         */
        _updateSelection: function() {
            var selectedIds = [];
            
            $(this.config.selectors.calculatorCheckbox + ':checked').each(function() {
                selectedIds.push($(this).val());
            });
            
            this.state.selectedItems = selectedIds;
            this._updateBulkActionsState();
            
            // Update select all checkbox
            var totalVisible = $(this.config.selectors.calculatorCheckbox + ':visible').length;
            var totalSelected = selectedIds.length;
            var $selectAll = $(this.config.selectors.selectAll);
            
            if (totalSelected === totalVisible && totalVisible > 0) {
                $selectAll.prop('indeterminate', false).prop('checked', true);
            } else {
                $selectAll.prop('indeterminate', false).prop('checked', false);
            }
        },
        
        /**
         * Clear all selections
         * 
         * @since 1.0.0
         * @private
         */
        _clearSelection: function() {
            $(this.config.selectors.calculatorCheckbox).prop('checked', false);
            $(this.config.selectors.selectAll).prop('checked', false).prop('indeterminate', false);
            $('.pwcp-calculator-card').removeClass('pwcp-selected');
            this.state.selectedItems = [];
            this._updateBulkActionsState();
        },
        
        /**
         * Update bulk actions state
         * 
         * @since 1.0.0
         * @private
         */
        _updateBulkActionsState: function() {
            var hasSelection = this.state.selectedItems.length > 0;
            $(this.config.selectors.applyBulk).prop('disabled', !hasSelection);
            
            if (!hasSelection) {
                $(this.config.selectors.bulkSelect).val('');
            }
        },
        
        /**
         * Delete single calculator
         * 
         * @param {number} calculatorId Calculator ID
         * @since 1.0.0
         * @private
         */
        _deleteCalculator: function(calculatorId) {
            // Optional confirmation
            if (window.pwcpCalculatorList && window.pwcpCalculatorList.confirmations !== false) {
                var confirmMessage = this._getString('confirm_delete');
                if (confirmMessage && confirmMessage !== 'Unknown') {
                    if (!window.confirm(confirmMessage)) {
                        return;
                    }
                }
            }
            
            // Use bulk handler with single ID
            this._performBulkAction('delete', [calculatorId]);
        },
        
        /**
         * Duplicate single calculator
         * 
         * @param {number} calculatorId Calculator ID
         * @since 1.0.0
         * @private
         */
        _duplicateCalculator: function(calculatorId) {
            // Optional confirmation
            if (window.pwcpCalculatorList && window.pwcpCalculatorList.confirmations !== false) {
                var confirmMessage = this._getString('confirm_duplicate');
                if (confirmMessage && confirmMessage !== 'Unknown') {
                    if (!window.confirm(confirmMessage)) {
                        return;
                    }
                }
            }
            
            // Use bulk handler with single ID
            this._performBulkAction('duplicate', [calculatorId]);
        },
        
        /**
         * Get fresh counts from server
         * 
         * @since 1.0.0
         * @private
         */
        _refreshCountsFromServer: function() {
            if (!window.pwcpCalculatorList || !window.pwcpCalculatorList.ajax_url) {
                return;
            }
            
            var self = this;
            
            $.post(window.pwcpCalculatorList.ajax_url, {
                action: 'pwcp_get_calculator_counts',
                nonce: window.pwcpCalculatorList.nonce
            })
            .done(function(response) {
                if (response.success && response.data.counts) {
                    var counts = response.data.counts;
                    $(self.config.selectors.totalCount).text(counts.total.toString());
                    $(self.config.selectors.activeCount).text(counts.active.toString());
                    $(self.config.selectors.recentCount).text(counts.recent.toString());
                    if (window.console && window.console.log) {
                        window.console.log('Counts refreshed from server:', counts);
                    }
                } else {
                    self._logError(self._getString('counts_invalid'));
                }
            })
            .fail(function() {
                self._logError(self._getString('request_failed'));
            });
        },
        
        /**
         * Toggle dropdown menu
         * 
         * @param {jQuery} $dropdown Dropdown element
         * @since 1.0.0
         * @private
         */
        _toggleDropdown: function($dropdown) {
            var isOpen = $dropdown.hasClass(this.config.classes.dropdownOpen);
            
            this._closeAllDropdowns();
            
            if (!isOpen) {
                $dropdown.addClass(this.config.classes.dropdownOpen);
            }
        },
        
        /**
         * Close all dropdown menus
         * 
         * @since 1.0.0
         * @private
         */
        _closeAllDropdowns: function() {
            $(this.config.selectors.dropdown).removeClass(this.config.classes.dropdownOpen);
        },
        
        /**
         * Show notification message with accessibility support
         * 
         * @param {string} message Notification message
         * @param {string} type Notification type
         * @since 1.0.0
         * @private
         */
        _showNotification: function(message, type) {
            type = type || 'info';
            var escapedMessage = this._escapeHtml(message);
            
            var $notification = $('<div>', {
                class: 'pwcp-notification pwcp-notification-' + type,
                html: '<p>' + escapedMessage + '</p>'
            });
            
            $('body').append($notification);
            
            // Also announce to screen readers
            var $ariaLive = $('.pwcp-aria-live');
            if ($ariaLive.length) {
                $ariaLive.text(message);
                // Clear after a moment to prevent stacking
                setTimeout(function() {
                    $ariaLive.text('');
                }, 1000);
            }
            
            setTimeout(function() {
                $notification.addClass('pwcp-notification-show');
            }, 10);
            
            setTimeout(function() {
                $notification.removeClass('pwcp-notification-show');
                setTimeout(function() {
                    $notification.remove();
                }, 300);
            }, type === 'error' ? 5000 : 3000);
        },
        
        /**
         * Log error messages
         * 
         * @param {string} message Error message
         * @since 1.0.0
         * @private
         */
        _logError: function(message) {
            if (window.console && window.console.warn) {
                window.console.warn('PWCP Calculator List:', message);
            }
        },
        
        /**
         * Get localized string
         * 
         * @param {string} key String key
         * @since 1.0.0
         * @private
         */
        _getString: function(key) {
            if (window.pwcpCalculatorList && window.pwcpCalculatorList.strings && window.pwcpCalculatorList.strings[key]) {
                return window.pwcpCalculatorList.strings[key];
            }
            
            var fallbacks = {
                'ajax_config_missing': 'AJAX configuration not found.',
                'request_failed': 'Request failed. Please try again.',
                'counts_invalid': 'Invalid count response from server.',
                'confirm_delete': 'Are you sure you want to delete this calculator?',
                'confirm_bulk_delete': 'Are you sure you want to delete the selected calculators?',
                'confirm_duplicate': 'Create a duplicate of this calculator?',
                'confirm_bulk_duplicate': 'Create duplicates of the selected calculators?',
                'no_selection': 'Please select calculators first',
                'no_action': 'Please select an action',
                'processing': 'Processing...',
                'error': 'An error occurred',
                'no_results_title': 'No Results Found',
                'no_results_body': 'Try adjusting your search or filter criteria to find what you\'re looking for.',
                'clear_filters': 'Clear Filters',
                'status_active': 'Active',
                'status_inactive': 'Inactive',
                'empty_state_title': 'No Calculators Found',
                'empty_state_description': 'Start building powerful calculators for your users with our easy-to-use form builder.',
                'create_first_calculator': 'Create Your First Calculator'
            };
            
            return fallbacks[key] || 'Unknown';
        },
        
        /**
         * Escape HTML to prevent XSS
         * 
         * @param {string} text Text to escape
         * @since 1.0.0
         * @private
         */
        _escapeHtml: function(text) {
            var map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };
            
            return String(text).replace(/[&<>"']/g, function(match) {
                return map[match];
            });
        },
        
        /**
         * Create debounced function
         * 
         * @param {function} func Function to debounce
         * @param {number} wait Wait time in ms
         * @since 1.0.0
         * @private
         */
        _debounce: function(func, wait) {
            var timeout;
            return function() {
                var context = this;
                var args = arguments;
                if (timeout) {
                    clearTimeout(timeout);
                }
                timeout = setTimeout(function() {
                    func.apply(context, args);
                }, wait);
            };
        },
        
        /**
         * Destroy functionality for cleanup
         * 
         * @since 1.0.0
         */
        destroy: function() {
            $(document).off('.pwcp');
            $(window).off('.pwcp');
            
            if (this._searchTimeout) {
                clearTimeout(this._searchTimeout);
            }
            
            this.state.isInitialized = false;
            
            if (window.console && window.console.log) {
                window.console.log('Calculator List Manager destroyed');
            }
        }
    };
    
    // Expose to global scope with proper prefix
    window.PWCP_CalculatorListManager = CalculatorListManager;
    
    // Auto-initialize when DOM is ready
    $(document).ready(function() {
        window.PWCP_CalculatorListManager.init();
    });
    
})(jQuery, window, document);