/**
 * Calculator Form Core Class
 * Core functionality for the Modern Calculator Form Manager
 * 
 * @package PriceWise_Calculator_Pro
 * @version 1.1.1
 * @since 1.0.0
 */

(function($, window, document, undefined) {
    'use strict';

    const { __, sprintf } = wp.i18n;

    /**
     * Modern Calculator Form Manager Core Class
     */
    var PWCPModernCalculatorFormCore = {
        
        config: {
            selectors: {
                tabNavigation: '.pwcp-tab-navigation',
                tabBtn: '.pwcp-tab-btn',
                tabPanel: '.pwcp-tab-panel',
                
                progressTracker: '.pwcp-progress-tracker',
                progressStep: '.pwcp-progress-step',
                
                form: '#pwcp-calculator-form',
                calculatorName: '#calculator_name',
                calculatorFormula: '#calculator_formula',
                
                addFieldBtn: '.pwcp-add-field',
                fieldsContainer: '#pwcp-fields-container',
                modernField: '.pwcp-modern-field',
                fieldHeader: '.pwcp-field-header',
                fieldContent: '.pwcp-field-content',
                toggleFieldBtn: '.pwcp-toggle-field',
                removeFieldBtn: '.pwcp-remove-field',
                fieldTemplate: '#pwcp-field-template',
                
                fieldLabelInput: '.pwcp-field-label-input',
                fieldKeyInput: '.pwcp-field-key-input',
                fieldTypeSelect: '.pwcp-field-type-select',
                numberOptions: '.pwcp-number-options',
                selectOptions: '.pwcp-select-options',
                
                formulaInput: '.pwcp-formula-input',
                operatorBtn: '.pwcp-operator-btn',
                exampleCard: '.pwcp-example-card',
                formulaStatus: '#formula-status',
                
                modal: '#pwcp-field-modal',
                modalOverlay: '.pwcp-modal-overlay',
                modalContainer: '.pwcp-modal-container',
                modalClose: '#pwcp-modal-close',
                modalCancel: '#pwcp-modal-cancel',
                modalSave: '#pwcp-modal-save',
                modalForm: '#pwcp-field-modal-form',
                
                modalFieldLabel: '#modal-field-label',
                modalFieldKey: '#modal-field-key',
                modalFieldType: '#modal-field-type',
                modalFieldRequired: '#modal-field-required',
                modalFieldDefault: '#modal-field-default',
                modalFieldMin: '#modal-field-min',
                modalFieldMax: '#modal-field-max',
                modalFieldStep: '#modal-field-step',
                modalFieldOptions: '#modal-field-options',
                modalNumberOptions: '#modal-number-options',
                modalSelectOptions: '#modal-select-options',
                
                actionBar: '.pwcp-action-bar',
                saveStatus: '.pwcp-save-status',
                cancelBtn: '#pwcp-cancel-btn',
                submitBtn: 'button[type="submit"], .pwcp-submit-btn',
                
                strategyInput: '.pwcp-strategy-input',
                strategyCard: '.pwcp-strategy-card',
                
                saveFirstMessage: '#save-calculator-first-message',
                emptyState: '.pwcp-empty-state'
            },
            
            classes: {
                active: 'pwcp-tab-active',
                stepActive: 'pwcp-step-active',
                expanded: 'pwcp-field-expanded',
                collapsed: 'pwcp-field-collapsed',
                hidden: 'pwcp-hidden',
                saving: 'pwcp-saving',
                saved: 'pwcp-saved',
                disabled: 'pwcp-disabled',
                mbMd: 'pwcp-mb-md',
                show: 'pwcp-show',
                error: 'pwcp-field-error',
                valid: 'pwcp-field-valid',
                modalOpen: 'pwcp-modal-open',
                modalActive: 'pwcp-modal-active',
                modalLoading: 'pwcp-modal-loading'
            },
            
            animations: {
                fast: 300,
                normal: 500
            }
        },
        
        state: {
            activeTab: 'basic',
            fieldIndex: 1000,
            isInitialized: false,
            isSaving: false,
            calculatorId: 0,
            isEditMode: false,
            lastSaveAttempt: 0,
            saveInProgress: false,
            fieldIdMapping: new Map(),
            modalData: {
                isOpen: false,
                editingField: null
            }
        },
        
        /**
         * Initialize the modern form
         */
        init: function() {
            if (this.state.isInitialized) {
                return;
            }
            
            if (!$('.pwcp-modern-admin').length) {
                return;
            }
            
            if (window.pwcpCalculatorForm) {
                this.state.calculatorId = parseInt(window.pwcpCalculatorForm.calculator_id, 10) || 0;
                this.state.isEditMode = window.pwcpCalculatorForm.is_edit_mode || false;
            }
            
            this.bindEvents();
            this.bindModalEvents();
            this.initializeTabs();
            this.initializeProgress();
            this.initializeFields();
            this.initializeFormula();
            this.resetButtonStates();
            this.initializeDynamicUI();
            this.updateButtonText();
            this.initializeCategoryAssignment();
            
            this.state.isInitialized = true;
        },
        
        /**
         * Initialize dynamic UI components
         */
        initializeDynamicUI: function() {
            this.updateUIBasedOnCalculatorState();
        },
        
        /**
         * Update UI based on calculator state
         */
        updateUIBasedOnCalculatorState: function() {
            var hasCalculatorId = this.state.calculatorId > 0;
            
            if (hasCalculatorId) {
                this.enableFieldsTabCompletely();
                this.hideAllSaveFirstMessages();
            } else {
                this.showSaveFirstMessage();
                this.disableFieldsTabTemporarily();
            }
        },
        
        /**
         * Enable fields tab completely
         */
        enableFieldsTabCompletely: function() {
            this.hideAllSaveFirstMessages();
            $(this.config.selectors.addFieldBtn).show().prop('disabled', false);
            $('.pwcp-tab-btn[data-tab="fields"]').removeClass('pwcp-tab-disabled');
            this.updateFieldsContainerState();
        },
      
        /**
         * Hide all save first messages
         */
        hideAllSaveFirstMessages: function() {
            $(this.config.selectors.saveFirstMessage).fadeOut(300);
            $('.pwcp-save-calculator-first').fadeOut(300);
        },
        
        /**
         * Show save first message
         */
        showSaveFirstMessage: function() {
            var $container = $(this.config.selectors.fieldsContainer);
            
            if (!$(this.config.selectors.saveFirstMessage).length && $container.length) {
                var saveFirstHtml = $('<div>').attr('id', 'save-calculator-first-message').addClass('pwcp-info-card').html(
                    '<div class="pwcp-info-icon">' +
                        '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">' +
                            '<circle cx="12" cy="12" r="10"/>' +
                            '<path d="M12 16v-4"/>' +
                            '<path d="M12 8h.01"/>' +
                        '</svg>' +
                    '</div>' +
                    '<div class="pwcp-info-content">' +
                        '<h4>' + __('Save Calculator First', 'pricewise-calculator-pro') + '</h4>' +
                        '<p>' + __('Please save the calculator in the Basic Information tab to start adding input fields.', 'pricewise-calculator-pro') + '</p>' +
                    '</div>'
                );
                
                $container.html(saveFirstHtml);
            }
        },
        
        /**
         * Disable fields tab temporarily
         */
        disableFieldsTabTemporarily: function() {
            $(this.config.selectors.addFieldBtn).hide();
            $('.pwcp-tab-btn[data-tab="fields"]').addClass('pwcp-tab-disabled');
        },
        
        /**
         * Reset button states
         */
        resetButtonStates: function() {
            $(this.config.selectors.submitBtn)
                .prop('disabled', false)
                .removeClass(this.config.classes.saving)
                .removeClass(this.config.classes.disabled);
                
            this.state.isSaving = false;
            this.state.saveInProgress = false;
        },
        
        /**
         * Update button text based on calculator state
         */
        updateButtonText: function() {
            var $submitBtn = $('.pwcp-submit-btn, button[type="submit"]');
            var isEditMode = this.state.calculatorId > 0;
            var buttonText = isEditMode 
                ? __('Update Calculator', 'pricewise-calculator-pro') 
                : __('Save Calculator', 'pricewise-calculator-pro');
            
            var $icon = $submitBtn.find('svg');
            if ($icon.length > 0) {
                $submitBtn.html($icon[0].outerHTML + ' ' + buttonText);
            } else {
                $submitBtn.text(buttonText);
            }
        },
        
        /**
         * Bind all event handlers
         */
        bindEvents: function() {
            var self = this;
            
            $(document)
                .off('click', this.config.selectors.tabBtn)
                .on('click', this.config.selectors.tabBtn, function(e) {
                    e.preventDefault();
                    var tabId = $(this).data('tab');
                    self.switchTab(tabId);
                });
            
            $(document)
                .off('click', this.config.selectors.addFieldBtn)
                .on('click', this.config.selectors.addFieldBtn, function(e) {
                    e.preventDefault();
                    self.showFieldModal();
                });
            
            $(document)
                .off('click', this.config.selectors.toggleFieldBtn)
                .on('click', this.config.selectors.toggleFieldBtn, function(e) {
                    e.preventDefault();
                    self.editFieldInModal($(this));
                });
            
            $(document)
                .off('click', this.config.selectors.removeFieldBtn)
                .on('click', this.config.selectors.removeFieldBtn, function(e) {
                    e.preventDefault();
                    self.removeField($(this));
                });
            
            $(document)
                .off('click', this.config.selectors.operatorBtn)
                .on('click', this.config.selectors.operatorBtn, function(e) {
                    e.preventDefault();
                    self.insertOperator($(this));
                });
            
            $(document)
                .off('click', this.config.selectors.exampleCard)
                .on('click', this.config.selectors.exampleCard, function(e) {
                    e.preventDefault();
                    self.insertExample($(this));
                });
            
            $(document)
                .off('input', this.config.selectors.formulaInput)
                .on('input', this.config.selectors.formulaInput, function() {
                    self.validateFormula();
                });

            $(document)
                .off('click', this.config.selectors.cancelBtn)
                .on('click', this.config.selectors.cancelBtn, function(e) {
                    e.preventDefault();
                    var url = $(this).data('target');
                    if (url) {
                        window.location.href = url;
                    }
                });
            
            $(this.config.selectors.form)
                .off('submit')
                .on('submit', function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    self.handleAjaxSubmit();
                    return false;
                });
            
            $(document)
                .off('click', this.config.selectors.submitBtn)
                .on('click', this.config.selectors.submitBtn, function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    
                    if (self.state.saveInProgress) {
                        return false;
                    }
                    
                    self.handleAjaxSubmit();
                    return false;
                });
            
            $(document)
                .off('change', this.config.selectors.strategyInput)
                .on('change', this.config.selectors.strategyInput, function() {
                    self.updateStrategySelection($(this));
                });
            
            $(document)
                .off('input', this.config.selectors.calculatorName)
                .on('input', this.config.selectors.calculatorName, function() {
                    self.updatePageTitle();
                });
        },
        
        /**
         * Update page title based on calculator name
         */
        updatePageTitle: function() {
            var name = $(this.config.selectors.calculatorName).val().trim();
            if (name) {
                $('.pwcp-hero-title').text(name);
                document.title = name + __(' - Calculator Editor', 'pricewise-calculator-pro');
            }
        },
        
        /**
         * Initialize tab system
         */
        initializeTabs: function() {
            this.switchTab(this.state.activeTab);
        },
        
        /**
         * Initialize progress tracker
         */
        initializeProgress: function() {
            this.updateProgress();
        },
        
        /**
         * Initialize existing fields
         */
        initializeFields: function() {
            var self = this;
            
            $(this.config.selectors.modernField).each(function() {
                var $field = $(this);
                var fieldType = $field.find(self.config.selectors.fieldTypeSelect).val();
                self.updateFieldOptions($field.find(self.config.selectors.fieldTypeSelect));
                self.updateFieldTitle($field.find(self.config.selectors.fieldLabelInput));
                
                var fieldId = $field.data('field-id');
                if (fieldId && !self.isTemporaryId(fieldId)) {
                    self.state.fieldIdMapping.set(fieldId, fieldId);
                }
            });
            
            var existingFields = $(this.config.selectors.modernField);
            if (existingFields.length > 0) {
                this.state.fieldIndex = existingFields.length + 1000;
            }
            
            this.updateFieldCount();
        },
        
        /**
         * Initialize formula editor
         */
        initializeFormula: function() {
            this.updateLineNumbers();
            this.validateFormula();
            
            var self = this;
            $(this.config.selectors.formulaInput)
                .off('scroll')
                .on('scroll', function() {
                    self.syncLineNumbers();
                });
        },
     
        /**
         * Initialize category assignment functionality
         */
        initializeCategoryAssignment: function() {
            var self = this;
            this.categoryData = {
                allCategories: [],
                selectedCategories: [],
                isLoaded: false
            };
            
            $(document)
                .off('change', this.config.selectors.strategyInput)
                .on('change', this.config.selectors.strategyInput, function() {
                    var selectedType = $(this).val();
                    var $wrapper = $('#pwcp-category-assignment-wrapper');
                    
                    if (selectedType === 'category') {
                        $wrapper.slideDown(300);
                        self.loadCategories();
                    } else {
                        $wrapper.slideUp(300);
                    }
                });
            
            $(document)
                .off('click', '.pwcp-category-card')
                .on('click', '.pwcp-category-card', function() {
                    self.toggleCategorySelection($(this));
                });
            
            var selectedType = $('input[name="assignment_type"]:checked').val();
            if (selectedType === 'category') {
                $('#pwcp-category-assignment-wrapper').show();
                this.loadCategories();
            }
        },
        
        /**
         * Load categories via AJAX
         */
        loadCategories: function() {
            if (this.categoryData.isLoaded) {
                this.displayCategories(this.categoryData.allCategories);
                return;
            }
            
            var self = this;
            
            $('#pwcp-categories-loading').show();
            $('#pwcp-categories-grid').hide();
            $('#pwcp-categories-error').hide();
            
            $.ajax({
                url: window.pwcpCalculatorForm.ajax_url,
                type: 'POST',
                data: {
                    action: 'pwcp_get_woocommerce_categories',
                    pwcp_nonce: window.pwcpCalculatorForm.nonce
                },
                success: function(response) {
                    if (response.success && response.data && response.data.categories) {
                        self.categoryData.allCategories = response.data.categories;
                        self.categoryData.isLoaded = true;
                        self.loadExistingSelections();
                        self.displayCategories(self.categoryData.allCategories);
                    } else {
                        self.showCategoryError();
                    }
                },
                error: function() {
                    self.showCategoryError();
                },
                complete: function() {
                    $('#pwcp-categories-loading').hide();
                }
            });
        },
        
        /**
         * Load existing category selections
         */
        loadExistingSelections: function() {
            try {
                var existingData = $('#assigned_categories').val();
                if (existingData) {
                    this.categoryData.selectedCategories = JSON.parse(existingData);
                }
            } catch (e) {
                this.categoryData.selectedCategories = [];
            }
        },
        
        /**
         * Display categories
         */
        displayCategories: function(categories) {
            var $grid = $('#pwcp-categories-grid');
            var html = '';
            var self = this;
            
            if (categories.length === 0) {
                html = '<div class="pwcp-info-card">' +
                    '<div class="pwcp-info-content">' +
                    '<h4>' + __('No Categories Found', 'pricewise-calculator-pro') + '</h4>' +
                    '<p>' + __('No WooCommerce product categories were found. Please create some categories first.', 'pricewise-calculator-pro') + '</p>' +
                    '</div></div>';
            } else {
                categories.forEach(function(category) {
                    var isSelected = self.isCategorySelected(category.id);
                    var productsText = sprintf(
                        __('%d products', 'pricewise-calculator-pro'),
                        category.count
                    );
                    
                    html += '<div class="pwcp-category-card' + (isSelected ? ' pwcp-selected' : '') + '" ' +
                        'data-category-id="' + category.id + '" ' +
                        'data-category-name="' + self.escapeHtml(category.name) + '">' +
                        '<div class="pwcp-category-check"></div>' +
                        '<div class="pwcp-category-info">' +
                        '<h4>' + self.escapeHtml(category.name) + '</h4>' +
                        '<p class="pwcp-category-meta">' + productsText + '</p>' +
                        '</div>' +
                        '</div>';
                });
            }
            
            $grid.html(html).show();
        },
        
        /**
         * Check if category is selected
         */
        isCategorySelected: function(categoryId) {
            return this.categoryData.selectedCategories.some(function(selected) {
                return parseInt(selected.id, 10) === parseInt(categoryId, 10);
            });
        },
        
        /**
         * Toggle category selection
         */
        toggleCategorySelection: function($card) {
            var categoryId = parseInt($card.data('category-id'), 10);
            var categoryName = $card.data('category-name');
            var isSelected = $card.hasClass('pwcp-selected');
            
            if (isSelected) {
                this.categoryData.selectedCategories = this.categoryData.selectedCategories.filter(function(cat) {
                    return parseInt(cat.id, 10) !== categoryId;
                });
                $card.removeClass('pwcp-selected');
            } else {
                this.categoryData.selectedCategories.push({
                    id: categoryId,
                    name: categoryName
                });
                $card.addClass('pwcp-selected');
            }
            
            this.updateCategoriesInput();
        },
        
        /**
         * Update hidden categories input
         */
        updateCategoriesInput: function() {
            $('#assigned_categories').val(JSON.stringify(this.categoryData.selectedCategories));
        },
        
        /**
         * Show category error
         */
        showCategoryError: function() {
            $('#pwcp-categories-grid').hide();
            $('#pwcp-categories-error').show();
        },
        
        /**
         * Escape HTML to prevent XSS
         */
        escapeHtml: function(text) {
            var map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };
            
            return String(text).replace(/[&<>"']/g, function(match) {
                return map[match];
            });
        },
        
        /**
         * Show notification
         */
        showNotification: function(message, type) {
            type = type || 'info';
            
            $('.pwcp-notification.pwcp-notification-show').each(function() {
                var $existing = $(this);
                $existing.removeClass('pwcp-notification-show');
                setTimeout(function() { 
                    $existing.remove(); 
                }, 300);
            });
            
            var icons = {
                'success': '✓',
                'error': '✕',
                'warning': '!',
                'info': 'i'
            };
            
            var iconContent = icons[type] || 'i';
            
            var $notification = $('<div>', {
                class: 'pwcp-notification pwcp-notification-' + type,
                html: '<span class="pwcp-notification-icon">' + iconContent + '</span>' +
                      '<p>' + this.escapeHtml(message) + '</p>' +
                      '<button class="pwcp-notification-close" aria-label="' + 
                      __('Close notification', 'pricewise-calculator-pro') + '">&times;</button>'
            });
            
            $('body').append($notification);
            
            $notification.find('.pwcp-notification-close').on('click', function() {
                $notification.removeClass('pwcp-notification-show');
                setTimeout(function() { 
                    $notification.remove(); 
                }, 300);
            });
            
            setTimeout(function() { 
                $notification.addClass('pwcp-notification-show'); 
            }, 50);
            
            var autoHideDelay = type === 'error' ? 6000 : 4000;
            setTimeout(function() {
                if ($notification.hasClass('pwcp-notification-show')) {
                    $notification.removeClass('pwcp-notification-show');
                    setTimeout(function() { 
                        $notification.remove(); 
                    }, 300);
                }
            }, autoHideDelay);
        },
     
        /**
         * Update progress tracker
         */
        updateProgress: function() {
            var stepMap = {
                'basic': 1,
                'formula': 2,
                'fields': 3,
                'settings': 4
            };
            
            var currentStep = stepMap[this.state.activeTab] || 1;
            
            $(this.config.selectors.progressStep).removeClass(this.config.classes.stepActive);
            $(this.config.selectors.progressStep + '[data-step="' + currentStep + '"]').addClass(this.config.classes.stepActive);
        },
        
        /**
         * Update line numbers for formula editor
         */
        updateLineNumbers: function() {
            var $textarea = $(this.config.selectors.formulaInput);
            var $lineNumbers = $('.pwcp-formula-line-numbers');
            
            if (!$lineNumbers.length || !$textarea.length) {
                return;
            }
            
            var lines = $textarea.val().split('\n').length;
            var lineNumbersHtml = '';
            
            for (var i = 1; i <= Math.max(lines, 8); i++) {
                lineNumbersHtml += i + '\n';
            }
            
            $lineNumbers.text(lineNumbersHtml);
        },
        
        /**
         * Sync line numbers with textarea scroll
         */
        syncLineNumbers: function() {
            var $textarea = $(this.config.selectors.formulaInput);
            var $lineNumbers = $('.pwcp-formula-line-numbers');
            
            if ($lineNumbers.length) {
                $lineNumbers[0].scrollTop = $textarea[0].scrollTop;
            }
        },
        
        /**
         * Validate formula
         */
        validateFormula: function() {
            var $textarea = $(this.config.selectors.formulaInput);
            var value = $textarea.val().trim();
            
            this.updateLineNumbers();
            
            if (!value) {
                this.updateFormulaStatus('pending', __('Formula not entered (optional)', 'pricewise-calculator-pro'));
                return true;
            }
            
            var openParens = (value.match(/\(/g) || []).length;
            var closeParens = (value.match(/\)/g) || []).length;
            
            if (openParens !== closeParens) {
                this.updateFormulaStatus('invalid', __('Unbalanced parentheses', 'pricewise-calculator-pro'));
                return false;
            }
            
            var validChars = /^[a-zA-Z0-9_+\-*/().\s]+$/;
            if (!validChars.test(value)) {
                this.updateFormulaStatus('invalid', __('Invalid characters in formula', 'pricewise-calculator-pro'));
                return false;
            }
            
            this.updateFormulaStatus('valid', __('Formula looks good', 'pricewise-calculator-pro'));
            return true;
        },
        
        /**
         * Update formula status indicator
         */
        updateFormulaStatus: function(status, message) {
            var $statusEl = $(this.config.selectors.formulaStatus);
            if (!$statusEl.length) {
                return;
            }
            
            var $indicator = $statusEl.find('.pwcp-status-indicator');
            $indicator.removeClass('pwcp-status-valid pwcp-status-invalid pwcp-status-pending')
                     .addClass('pwcp-status-' + status);
            
            $indicator.find('span').text(message);
        },
        
        /**
         * Update field title
         */
        updateFieldTitle: function($input, defaultTitle) {
            var $field = $input.closest(this.config.selectors.modernField);
            var $title = $field.find('.pwcp-field-title');
            
            var label = $input.val() || defaultTitle || __('New Field', 'pricewise-calculator-pro');
            $title.text(label);
        },
        
        /**
         * Generate field key from label
         */
        generateFieldKey: function($labelInput) {
            var $field = $labelInput.closest(this.config.selectors.modernField);
            var $keyInput = $field.find(this.config.selectors.fieldKeyInput);
            
            if ($keyInput.val().trim()) {
                return;
            }
            
            var label = $labelInput.val().trim();
            if (!label) {
                return;
            }
            
            var key = label.toLowerCase()
                           .replace(/[^a-z0-9\s]/g, '')
                           .replace(/\s+/g, '_')
                           .substring(0, 50);
            
            $keyInput.val(key);
            this.validateFieldKey($keyInput);
        },
        
        /**
         * Validate field key
         */
        validateFieldKey: function($input) {
            var value = $input.val().trim();
            var isValid = /^[a-z0-9_]+$/.test(value) && value.length > 0;
            
            $input.removeClass('pwcp-field-valid pwcp-field-error');
            
            if (!isValid) {
                $input.addClass('pwcp-field-error');
                return false;
            }
            
            var duplicateCount = 0;
            $(this.config.selectors.fieldKeyInput).each(function() {
                if ($(this).val().trim() === value) {
                    duplicateCount++;
                }
            });
            
            if (duplicateCount > 1) {
                $input.addClass('pwcp-field-error');
                return false;
            }
            
            $input.addClass('pwcp-field-valid');
            return true;
        },
        
        /**
         * Update field options based on type
         */
        updateFieldOptions: function($select) {
            var $field = $select.closest(this.config.selectors.modernField);
            var fieldType = $select.val();
            
            var $numberOptions = $field.find(this.config.selectors.numberOptions);
            var $selectOptions = $field.find(this.config.selectors.selectOptions);
            
            $numberOptions.addClass(this.config.classes.hidden);
            $selectOptions.addClass(this.config.classes.hidden);
            
            switch (fieldType) {
                case 'number':
                case 'slider':
                    $numberOptions.removeClass(this.config.classes.hidden);
                    break;
                case 'dropdown':
                case 'radio':
                case 'checkbox':
                    $selectOptions.removeClass(this.config.classes.hidden);
                    break;
            }
            
            this.updateFieldTypeBadge($field, fieldType);
        },
        
        /**
         * Update field type badge
         */
        updateFieldTypeBadge: function($field, fieldType) {
            var $badge = $field.find('.pwcp-field-type-badge');
            if ($badge.length) {
                $badge.removeClass(function(index, className) {
                    return (className.match(/(^|\s)pwcp-type-\S+/g) || []).join(' ');
                });
                
                $badge.addClass('pwcp-type-' + fieldType);
                
                var coreTypeLabels = {
                    'number': __('Number', 'pricewise-calculator-pro'),
                    'text': __('Text', 'pricewise-calculator-pro'),
                    'slider': __('Slider', 'pricewise-calculator-pro'),
                    'dropdown': __('Dropdown', 'pricewise-calculator-pro'),
                    'radio': __('Radio', 'pricewise-calculator-pro'),
                    'checkbox': __('Checkbox', 'pricewise-calculator-pro'),
                    'textarea': __('Textarea', 'pricewise-calculator-pro'),
                    'email': __('Email', 'pricewise-calculator-pro'),
                    'url': __('URL', 'pricewise-calculator-pro'),
                    'date': __('Date', 'pricewise-calculator-pro'),
                    'time': __('Time', 'pricewise-calculator-pro'),
                    'file': __('File', 'pricewise-calculator-pro'),
                    'color': __('Color', 'pricewise-calculator-pro'),
                    'range': __('Range', 'pricewise-calculator-pro'),
                    'select': __('Select', 'pricewise-calculator-pro')
                };
                
                var typeLabels = window.fieldTypeLabels || coreTypeLabels;
                var typeLabel = typeLabels[fieldType] || coreTypeLabels[fieldType];
                
                if (!typeLabel) {
                    typeLabel = __('Field', 'pricewise-calculator-pro');
                }
                
                $badge.text(typeLabel);
            }
        },
        
        /**
         * Toggle field expansion
         */
        toggleField: function($button) {
            var $field = $button.closest(this.config.selectors.modernField);
            var $content = $field.find(this.config.selectors.fieldContent);
            
            if ($content.hasClass(this.config.classes.expanded)) {
                $content.removeClass(this.config.classes.expanded).addClass(this.config.classes.collapsed);
            } else {
                $content.removeClass(this.config.classes.collapsed).addClass(this.config.classes.expanded);
            }
        },
        
        /**
         * Update strategy selection
         */
        updateStrategySelection: function($input) {
            var value = $input.val();
            
            $(this.config.selectors.strategyCard).removeClass('pwcp-strategy-active');
            $input.closest(this.config.selectors.strategyCard).addClass('pwcp-strategy-active');
        },
        
        /**
         * Insert operator into formula
         */
        insertOperator: function($button) {
            var operator = $button.data('operator');
            var $textarea = $(this.config.selectors.formulaInput);
            var cursorPos = $textarea[0].selectionStart;
            var textBefore = $textarea.val().substring(0, cursorPos);
            var textAfter = $textarea.val().substring(cursorPos);
            
            $textarea.val(textBefore + operator + textAfter);
            $textarea[0].selectionStart = cursorPos + operator.length;
            $textarea[0].selectionEnd = cursorPos + operator.length;
            $textarea.focus();
            
            this.validateFormula();
        },
        
        /**
         * Insert example formula
         */
        insertExample: function($card) {
            var formula = $card.data('formula');
            $(this.config.selectors.formulaInput).val(formula);
            this.validateFormula();
        },
        
        /**
         * Update field count
         */
        updateFieldCount: function() {
            var count = $(this.config.selectors.modernField).length;
            $('.pwcp-field-count').text(count);
        },
        
        /**
         * Update field order
         */
        updateFieldOrder: function() {
            $(this.config.selectors.modernField).each(function(index) {
                $(this).find('.pwcp-field-order').val(index + 1);
            });
        },
       
        /**
         * Update fields container state
         */
        updateFieldsContainerState: function() {
            var hasFields = $(this.config.selectors.modernField).length > 0;
            
            if (!hasFields && this.state.calculatorId > 0) {
                this.showEmptyFieldsState();
            }
        },
        
        /**
         * Check if ID is temporary
         */
        isTemporaryId: function(id) {
            return typeof id === 'string' && (id.startsWith('new_') || id === 'TEMPLATE_INDEX');
        },
        
        /**
         * Switch to different tab
         */
        switchTab: function(tabId) {
            var $tabBtn = $('.pwcp-tab-btn[data-tab="' + tabId + '"]');
            var $tabPanel = $('.pwcp-tab-panel[data-panel="' + tabId + '"]');
            
            if (!$tabBtn.length || !$tabPanel.length) {
                return;
            }
            
            if (tabId === 'fields' && $tabBtn.hasClass('pwcp-tab-disabled')) {
                this.showNotification(
                    __('Please save the calculator first before managing fields.', 'pricewise-calculator-pro'), 
                    'warning'
                );
                return;
            }
            
            this.state.activeTab = tabId;
            
            $('.pwcp-tab-btn').removeClass('pwcp-tab-active');
            $('.pwcp-tab-panel').removeClass('pwcp-tab-active');
            
            $tabBtn.addClass('pwcp-tab-active');
            $tabPanel.addClass('pwcp-tab-active');
            
            this.updateProgress();
        },
        
        /**
         * Bind modal event handlers
         * Extended in main implementation file
         */
        bindModalEvents: function() {
        },
        
        /**
         * Show field modal
         * Extended in main implementation file
         */
        showFieldModal: function() {
        },
        
        /**
         * Edit field in modal
         * Extended in main implementation file
         */
        editFieldInModal: function($button) {
        },
        
        /**
         * Remove field
         * Extended in main implementation file
         */
        removeField: function($button) {
        },
        
        /**
         * Handle AJAX submit
         * Extended in main implementation file
         */
        handleAjaxSubmit: function() {
        }
    };
    
    window.PWCPModernCalculatorFormCore = PWCPModernCalculatorFormCore;
    
})(jQuery, window, document);


