/**
 * Calculator Form JavaScript
 * Admin stays on current tab after page refresh via URL parameters
 * @package PriceWise_Calculator_Pro
 * @version 1.0.0
 * @since 1.0.0
 */
(function($, window, document, undefined) {
    'use strict';

    // Import WordPress i18n functions
    const { __, sprintf } = wp.i18n;

    // Debug mode - set to false for production
    const DEBUG_MODE = false;

    // Debug logger helper function
    function debugLog(message, data) {
        if (DEBUG_MODE && window.console) {
            if (data !== undefined) {
                console.log(message, data);
            } else {
                console.log(message);
            }
        }
    }

    function debugError(message, data) {
        if (DEBUG_MODE && window.console) {
            if (data !== undefined) {
                console.error(message, data);
            } else {
                console.error(message);
            }
        }
    }

    // Ensure core is loaded
    if (typeof window.PWCPModernCalculatorFormCore === 'undefined') {
        debugError(__('PWCPModernCalculatorFormCore not found. Please ensure PWCPModernCalculatorFormCore.js is loaded first.', 'pricewise-calculator-pro'));
        return;
    }

    /**
     * Field Type Translation Map
     * Maps field type keys to translatable labels
     */
    var fieldTypeLabels = {
        'number': __('Number', 'pricewise-calculator-pro'),
        'text': __('Text', 'pricewise-calculator-pro'),
        'slider': __('Slider', 'pricewise-calculator-pro'),
        'file': __('File Upload', 'pricewise-calculator-pro'),
        'dropdown': __('Dropdown', 'pricewise-calculator-pro'),
        'radio': __('Radio', 'pricewise-calculator-pro'),
        'checkbox': __('Checkbox', 'pricewise-calculator-pro'),
        'textarea': __('Textarea', 'pricewise-calculator-pro'),
        'email': __('Email', 'pricewise-calculator-pro'),
        'url': __('URL', 'pricewise-calculator-pro'),
        'date': __('Date', 'pricewise-calculator-pro'),
        'time': __('Time', 'pricewise-calculator-pro'),
        'file': __('File', 'pricewise-calculator-pro'),
        'color': __('Color', 'pricewise-calculator-pro'),
        'range': __('Range', 'pricewise-calculator-pro'),
        'select': __('Select', 'pricewise-calculator-pro')
    };

    /**
     * Modern Calculator Form Manager
     */
    var PWCPModernCalculatorForm = $.extend({}, window.PWCPModernCalculatorFormCore, {

        /**
         * Field Data Manager - Handles consistent data storage/retrieval
         */
        fieldDataManager: {
            setFieldData: function(fieldElement, data) {
                const $field = $(fieldElement);
                const normalizedData = this.normalizeFieldData(data);
                $field.data('field-data', normalizedData);
                this.updateJsonScript($field, normalizedData);
                debugLog(__('Field data stored consistently:', 'pricewise-calculator-pro'), normalizedData);
            },

            getFieldData: function(fieldElement) {
                const $field = $(fieldElement);
                let data = $field.data('field-data');

                if (this.isValidFieldData(data)) {
                    return data;
                }

                data = this.parseJsonScript($field);
                if (this.isValidFieldData(data)) {
                    $field.data('field-data', data);
                    return data;
                }

                return this.extractFromDOM($field);
            },

            normalizeFieldData: function(data) {
                return {
                    id: data.id || 0,
                    label: data.label || '',
                    key: data.key || '',
                    type: data.type || 'number',
                    required: data.required ? 1 : 0,
                    default_value: data.default_value || '',
                    order: data.order || 0,
                    options: data.options || '',
                    min: data.min || '',
                    max: data.max || '',
                    step: data.step || ''
                };
            },

            updateJsonScript: function($field, data) {
                let $jsonScript = $field.find('.pwcp-field-data-json');
                if (!$jsonScript.length) {
                    $jsonScript = $('<script type="application/json" class="pwcp-field-data-json"></script>');
                    $field.find('.pwcp-field-content').prepend($jsonScript);
                }
                $jsonScript.text(JSON.stringify(data));
            },

            parseJsonScript: function($field) {
                const $jsonScript = $field.find('.pwcp-field-data-json');
                if ($jsonScript.length) {
                    try {
                        return JSON.parse($jsonScript.text());
                    } catch (e) {
                        debugError(__('Error parsing field JSON:', 'pricewise-calculator-pro'), e);
                    }
                }
                return null;
            },

            extractFromDOM: function($field) {
                const fieldTitle = $field.find('.pwcp-field-title').text().trim();
                const fieldRequired = $field.find('.pwcp-field-required-badge').length > 0;
                let fieldType = 'text';

                const $typeBadge = $field.find('.pwcp-field-type-badge');
                if ($typeBadge.length) {
                    const badgeClass = $typeBadge.attr('class');
                    const typeMatch = badgeClass.match(/pwcp-type-(\w+)/);
                    if (typeMatch) {
                        fieldType = typeMatch[1];
                    }
                }

                return this.normalizeFieldData({
                    id: $field.data('field-id') || 0,
                    label: fieldTitle,
                    key: fieldTitle.toLowerCase().replace(/[^a-z0-9\s]/g, '').replace(/\s+/g, '_'),
                    type: fieldType,
                    required: fieldRequired
                });
            },

            isValidFieldData: function(data) {
                return data && typeof data === 'object' && data.label && data.key;
            }
        },

        /**
         * Modal State Manager
         */
        modalState: {
            currentField: null,
            mode: 'create',

            setCreateMode: function() {
                this.currentField = null;
                this.mode = 'create';
            },

            setEditMode: function(fieldElement) {
                this.currentField = fieldElement;
                this.mode = 'edit';
            },

            isEditMode: function() {
                return this.mode === 'edit';
            },

            getCurrentField: function() {
                return this.currentField;
            }
        },

        /**
         * Field Validator
         */
        fieldValidator: {
            validateField: function(fieldData, excludeFieldId) {
                if (excludeFieldId === undefined) {
                    excludeFieldId = null;
                }

                const errors = [];

                if (!fieldData.label || !fieldData.label.trim()) {
                    errors.push({
                        field: 'label',
                        message: __('Field label is required', 'pricewise-calculator-pro')
                    });
                }

                const keyValidation = this.validateKey(fieldData.key, excludeFieldId);
                if (!keyValidation.isValid) {
                    errors.push({
                        field: 'key',
                        message: keyValidation.message
                    });
                }

                return {
                    isValid: errors.length === 0,
                    errors: errors
                };
            },

            validateKey: function(key, excludeFieldId) {
                if (excludeFieldId === undefined) {
                    excludeFieldId = null;
                }

                if (!key || !key.trim()) {
                    return {
                        isValid: false,
                        message: __('Field key is required', 'pricewise-calculator-pro')
                    };
                }

                if (!/^[a-z0-9_]+$/.test(key)) {
                    return {
                        isValid: false,
                        message: __('Only lowercase letters, numbers, and underscores allowed', 'pricewise-calculator-pro')
                    };
                }

                if (this.isDuplicateKey(key, excludeFieldId)) {
                    return {
                        isValid: false,
                        message: __('This field key already exists', 'pricewise-calculator-pro')
                    };
                }

                return {
                    isValid: true,
                    message: __('Key is valid', 'pricewise-calculator-pro')
                };
            },

            isDuplicateKey: function(key, excludeFieldId) {
                if (excludeFieldId === undefined) {
                    excludeFieldId = null;
                }

                let isDuplicate = false;

                $('.pwcp-modern-field').each(function() {
                    const $field = $(this);
                    const fieldId = $field.data('field-id');

                    if (excludeFieldId && fieldId === excludeFieldId) {
                        return true;
                    }

                    const fieldData = PWCPModernCalculatorForm.fieldDataManager.getFieldData($field);
                    if (fieldData && fieldData.key === key) {
                        isDuplicate = true;
                        return false;
                    }
                });

                return isDuplicate;
            }
        },

        /**
         * URL Parameter Helper Functions
         */
        getUrlParameter: function(name) {
            const urlParams = new URLSearchParams(window.location.search);
            return urlParams.get(name);
        },

        updateUrlParameter: function(name, value) {
            const url = new URL(window.location);
            url.searchParams.set(name, value);
            window.history.replaceState({}, '', url.toString());
        },

        /**
         * Initialize with URL parameter persistence
         */
        init: function() {
            if (this.state.isInitialized) {
                return;
            }

            if (!$('.pwcp-modern-admin').length) {
                return;
            }

            if (window.pwcpCalculatorForm) {
                this.state.calculatorId = parseInt(window.pwcpCalculatorForm.calculator_id, 10) || 0;
                this.state.isEditMode = window.pwcpCalculatorForm.is_edit_mode || false;
            }

            this.bindEvents();
            this.bindModalEvents();
            this.initializeTabsWithUrlPersistence();
            this.initializeProgress();
            this.initializeFields();
            this.initializeFormula();
            this.resetButtonStates();
            this.initializeDynamicUI();
            this.updateButtonText();
            this.initializeCategoryAssignment();
            this.setupPopstateListener();

            this.state.isInitialized = true;
            debugLog(__('PWCPModernCalculatorForm initialized with URL parameter persistence', 'pricewise-calculator-pro'));
        },

        initializeTabsWithUrlPersistence: function() {
            var tabFromUrl = this.getUrlParameter('tab');
            var validTabs = ['basic', 'formula', 'fields', 'settings'];

            if (tabFromUrl && validTabs.indexOf(tabFromUrl) !== -1) {
                this.state.activeTab = tabFromUrl;
                debugLog(__('Initialized to URL tab:', 'pricewise-calculator-pro'), tabFromUrl);
            } else {
                var $currentActive = $('.pwcp-tab-btn.pwcp-tab-active');
                var currentTab = $currentActive.data('tab') || 'basic';
                this.state.activeTab = currentTab;
                this.updateUrlParameter('tab', currentTab);
                debugLog(__('Set URL to current tab:', 'pricewise-calculator-pro'), currentTab);
            }

            this.switchTabUI(this.state.activeTab);
            this.updateProgress();
        },

        setupPopstateListener: function() {
            var self = this;
            var validTabs = ['basic', 'formula', 'fields', 'settings'];

            window.addEventListener('popstate', function(event) {
                var newTab = self.getUrlParameter('tab');
                if (newTab && validTabs.indexOf(newTab) !== -1 && newTab !== self.state.activeTab) {
                    self.state.activeTab = newTab;
                    self.switchTabUI(newTab);
                    self.updateProgress();
                    debugLog(__('Browser navigation to tab:', 'pricewise-calculator-pro'), newTab);
                }
            });
        },

        switchTab: function(tabId) {
            var $tabBtn = $('.pwcp-tab-btn[data-tab="' + tabId + '"]');
            var $tabPanel = $('.pwcp-tab-panel[data-panel="' + tabId + '"]');

            if (!$tabBtn.length || !$tabPanel.length) {
                debugError(__('Tab elements not found for:', 'pricewise-calculator-pro'), tabId);
                return;
            }

            if (tabId === 'fields' && $tabBtn.hasClass('pwcp-tab-disabled')) {
                this.showNotification(
                    __('Please save the calculator first before managing fields.', 'pricewise-calculator-pro'),
                    'warning'
                );
                return;
            }

            this.state.activeTab = tabId;
            this.updateUrlParameter('tab', tabId);
            this.switchTabUI(tabId);
            this.updateProgress();

            debugLog(__('Switched to tab:', 'pricewise-calculator-pro'), tabId);
        },

        switchTabUI: function(tabId) {
            $('.pwcp-tab-btn').removeClass('pwcp-tab-active');
            $('.pwcp-tab-panel').removeClass('pwcp-tab-active');

            $('.pwcp-tab-btn[data-tab="' + tabId + '"]').addClass('pwcp-tab-active');
            $('.pwcp-tab-panel[data-panel="' + tabId + '"]').addClass('pwcp-tab-active');
        },

        updateProgress: function() {
            var stepMap = {'basic': 1, 'formula': 2, 'fields': 3, 'settings': 4};
            var currentStep = stepMap[this.state.activeTab] || 1;

            $('.pwcp-progress-step').removeClass('pwcp-step-active');
            $('.pwcp-progress-step[data-step="' + currentStep + '"]').addClass('pwcp-step-active');
        },

        showFieldModal: function() {
            if (!this.state.calculatorId || this.state.calculatorId === 0) {
                this.showNotification(
                    __('Please save the calculator first before adding fields.', 'pricewise-calculator-pro'),
                    'warning'
                );
                this.switchTab('basic');
                return;
            }

            this.modalState.setCreateMode();
            this.resetModalForm();
            this.updateModalUI();
            this.showModal();
        },

        editFieldInModal: function($button) {
            var $field = $button.closest('.pwcp-modern-field');
            var fieldData = this.fieldDataManager.getFieldData($field);

            if (!fieldData || !fieldData.label) {
                this.showNotification(
                    __('Unable to load field data for editing. Please refresh the page.', 'pricewise-calculator-pro'),
                    'error'
                );
                return;
            }

            this.modalState.setEditMode($field);
            this.populateModalForm(fieldData);
            this.updateModalUI();
            this.showModal();
        },

        showModal: function() {
            var $modal = $('#pwcp-field-modal');
            var $body = $('body');

            if ($modal.hasClass('pwcp-modal-open')) {
                return;
            }

            $body.addClass('pwcp-modal-active');
            $modal.removeClass('pwcp-hidden');
            $modal[0].offsetHeight;

            requestAnimationFrame(function() {
                $modal.addClass('pwcp-modal-open');
            });

            setTimeout(function() {
                $('#modal-field-label').focus();
            }, 150);
        },

        hideFieldModal: function() {
            var $modal = $('#pwcp-field-modal');
            var $body = $('body');
            var self = this;

            if (!$modal.hasClass('pwcp-modal-open')) {
                return;
            }

            $modal.removeClass('pwcp-modal-open');

            setTimeout(function() {
                $modal.addClass('pwcp-hidden');
                $body.removeClass('pwcp-modal-active');
                self.modalState.setCreateMode();
            }, 300);
        },

        resetModalForm: function() {
            $('#pwcp-field-modal-form')[0].reset();
            $('#modal-field-type').val('number');
            $('#modal-field-min').val('0');
            $('#modal-field-max').val('1000');
            $('#modal-field-step').val('1');

            this.updateModalFieldOptions();
            this.clearModalValidationErrors();
        },

        populateModalForm: function(fieldData) {
            $('#modal-field-label').val(fieldData.label || '');
            $('#modal-field-key').val(fieldData.key || '');
            $('#modal-field-type').val(fieldData.type || 'text');
            $('#modal-field-required').prop('checked', Boolean(fieldData.required));
            $('#modal-field-default').val(fieldData.default_value || '');
            $('#modal-field-min').val(fieldData.min || '0');
            $('#modal-field-max').val(fieldData.max || '1000');
            $('#modal-field-step').val(fieldData.step || '1');
            $('#modal-field-options').val(fieldData.options || '');
            $('#modal-field-allowed-types').val(fieldData.allowed_types || 'image/*');
$('#modal-field-max-size').val(fieldData.max_size || '5');

            this.updateModalFieldOptions();
            this.clearModalValidationErrors();
        },

        updateModalUI: function() {
            var isEdit = this.modalState.isEditMode();

            $('.pwcp-modal-title').text(
                isEdit
                    ? __('Edit Field', 'pricewise-calculator-pro')
                    : __('Add New Field', 'pricewise-calculator-pro')
            );

            $('#pwcp-modal-save').text(
                isEdit
                    ? __('Update Field', 'pricewise-calculator-pro')
                    : __('Add Field', 'pricewise-calculator-pro')
            );
        },

 updateModalFieldOptions: function() {
    var fieldType = $('#modal-field-type').val();
    var $numberOptions = $('#modal-number-options');
    var $selectOptions = $('#modal-select-options');
    var $fileOptions = $('#modal-file-options');

    $numberOptions.addClass('pwcp-hidden');
    $selectOptions.addClass('pwcp-hidden');
    $fileOptions.addClass('pwcp-hidden');

    switch (fieldType) {
        case 'number':
        case 'slider':
            $numberOptions.removeClass('pwcp-hidden');
            break;
        case 'dropdown':
        case 'radio':
        case 'checkbox':
            $selectOptions.removeClass('pwcp-hidden');
            break;
        case 'file':
            $fileOptions.removeClass('pwcp-hidden');
            break;
    }
},

        generateModalFieldKey: function() {
            var label = $('#modal-field-label').val().trim();
            var $keyInput = $('#modal-field-key');

            if ($keyInput.val().trim() !== '') {
                return;
            }

            if (label) {
                var key = label.toLowerCase()
                    .replace(/[^a-z0-9\s]/g, '')
                    .replace(/\s+/g, '_')
                    .substring(0, 50);

                $keyInput.val(key);
                this.validateModalFieldKey();
            }
        },

        validateModalFieldKey: function() {
            var $keyInput = $('#modal-field-key');
            var value = $keyInput.val().trim();
            var excludeFieldId = this.modalState.isEditMode()
                ? this.modalState.getCurrentField().data('field-id')
                : null;

            $keyInput.removeClass('pwcp-field-valid pwcp-field-error');

            var validation = this.fieldValidator.validateKey(value, excludeFieldId);

            if (validation.isValid) {
                $keyInput.addClass('pwcp-field-valid');
                this.clearModalFieldError($keyInput);
                return true;
            } else {
                $keyInput.addClass('pwcp-field-error');
                this.showModalFieldError($keyInput, validation.message);
                return false;
            }
        },

        validateModalForm: function() {
            var fieldData = this.collectModalFieldData();
            var excludeFieldId = this.modalState.isEditMode()
                ? this.modalState.getCurrentField().data('field-id')
                : null;

            var validation = this.fieldValidator.validateField(fieldData, excludeFieldId);

            this.clearModalValidationErrors();

            if (!validation.isValid) {
                var self = this;
                validation.errors.forEach(function(error) {
                    var $input = $('#modal-field-' + error.field);
                    $input.addClass('pwcp-field-error');
                    self.showModalFieldError($input, error.message);
                });
            }

            return validation.isValid;
        },

        collectModalFieldData: function() {
            var fieldType = $('#modal-field-type').val();
            var fieldData = {
                label: $('#modal-field-label').val().trim(),
                key: $('#modal-field-key').val().trim(),
                type: fieldType,
                required: $('#modal-field-required').is(':checked') ? 1 : 0,
                default_value: $('#modal-field-default').val().trim()
            };

          switch (fieldType) {
    case 'number':
    case 'slider':
        fieldData.min = $('#modal-field-min').val() || '0';
        fieldData.max = $('#modal-field-max').val() || '100';
        fieldData.step = $('#modal-field-step').val() || '1';
        break;
    case 'dropdown':
    case 'radio':
    case 'checkbox':
        fieldData.options = $('#modal-field-options').val().trim();
        break;
    case 'file':
        fieldData.allowed_types = $('#modal-field-allowed-types').val() || 'image/*';
        fieldData.max_size = $('#modal-field-max-size').val() || '5';
        break;
}

            return fieldData;
        },

        saveModalField: function() {
            if (!this.validateModalForm()) {
                return;
            }

            var fieldData = this.collectModalFieldData();
            var $saveBtn = $('#pwcp-modal-save');
            var isEdit = this.modalState.isEditMode();

            $saveBtn.prop('disabled', true).addClass('pwcp-saving');
            var originalText = $saveBtn.text();
            $saveBtn.text(isEdit
                ? __('Updating...', 'pricewise-calculator-pro')
                : __('Saving...', 'pricewise-calculator-pro')
            );

            if (isEdit) {
                this.updateExistingField(fieldData);
                $saveBtn.prop('disabled', false).removeClass('pwcp-saving').text(originalText);
            } else {
                this.createNewField(fieldData, $saveBtn, originalText);
            }
        },

        updateExistingField: function(fieldData) {
            var $field = this.modalState.getCurrentField();
            var fieldId = $field.data('field-id') || 0;

            if (!fieldId || this.isTemporaryId(fieldId) || parseInt(fieldId, 10) <= 0) {
                fieldData.id = fieldId;
                this.fieldDataManager.setFieldData($field, fieldData);
                this.updateFieldVisuals($field, fieldData);
                this.hideFieldModal();
                this.showNotification(
                    __('Field updated successfully!', 'pricewise-calculator-pro'),
                    'success'
                );
                return;
            }

            fieldData.id = fieldId;

            var $saveBtn = $('#pwcp-modal-save');
            var originalText = $saveBtn.text();
            $saveBtn.prop('disabled', true).addClass('pwcp-saving').text(__('Updating...', 'pricewise-calculator-pro'));

            var self = this;

            $.ajax({
                url: window.pwcpCalculatorForm.ajax_url,
                type: 'POST',
                data: {
                    action: 'pwcp_update_calculator_field',
                    field_id: fieldId,
                    calculator_id: this.state.calculatorId,
                    field_data: fieldData,
                    pwcp_nonce: window.pwcpCalculatorForm.nonce
                },
                success: function(response) {
                    if (response.success) {
                        self.fieldDataManager.setFieldData($field, fieldData);
                        self.updateFieldVisuals($field, fieldData);
                        self.hideFieldModal();
                        self.showNotification(
                            __('Field updated and saved successfully!', 'pricewise-calculator-pro'),
                            'success'
                        );
                    } else {
                        self.showNotification(
                            response.data.message || __('Failed to update field', 'pricewise-calculator-pro'),
                            'error'
                        );
                    }
                },
                error: function() {
                    self.showNotification(
                        __('Network error occurred while updating field', 'pricewise-calculator-pro'),
                        'error'
                    );
                },
                complete: function() {
                    $saveBtn.prop('disabled', false).removeClass('pwcp-saving').text(originalText);
                }
            });
        },

        createNewField: function(fieldData, $saveBtn, originalText) {
            var self = this;

            $.ajax({
                url: window.pwcpCalculatorForm.ajax_url,
                type: 'POST',
                data: {
                    action: 'pwcp_save_calculator_field',
                    calculator_id: this.state.calculatorId,
                    field_data: fieldData,
                    pwcp_nonce: window.pwcpCalculatorForm.nonce
                },
                success: function(response) {
                    if (response.success) {
                        self.handleNewFieldSuccess(response.data, fieldData);
                    } else {
                        self.handleFieldError(response.data);
                    }
                },
                error: function() {
                    self.handleFieldError({
                        message: __('Network error occurred', 'pricewise-calculator-pro')
                    });
                },
                complete: function() {
                    $saveBtn.prop('disabled', false).removeClass('pwcp-saving').text(originalText);
                }
            });
        },

        handleNewFieldSuccess: function(serverData, fieldData) {
            fieldData.id = serverData.field_id || Date.now();
            this.addFieldToUI(fieldData);
            this.hideFieldModal();
            this.showNotification(
                __('Field added successfully!', 'pricewise-calculator-pro'),
                'success'
            );
        },

        handleFieldError: function(error) {
            var message = error.message || __('Failed to save field', 'pricewise-calculator-pro');
            this.showNotification(message, 'error');
        },

        addFieldToUI: function(fieldData) {
            var fieldId = fieldData.id;
            var template = $('#pwcp-field-template').html();

            if (!template) {
                debugError(__('Field template not found', 'pricewise-calculator-pro'));
                return;
            }

            var fieldHTML = template.replace(/TEMPLATE_INDEX/g, fieldId);
            var $fieldElement = $(fieldHTML);

            this.fieldDataManager.setFieldData($fieldElement, fieldData);
            this.updateFieldVisuals($fieldElement, fieldData);

            var $container = $('#pwcp-fields-container');
            $container.find('.pwcp-empty-state, .pwcp-info-card').remove();

            var self = this;
            $fieldElement.hide().appendTo($container).slideDown(300, function() {
                self.updateFieldCount();
            });
        },

        updateFieldVisuals: function($field, fieldData) {
            $field.attr('data-field-id', fieldData.id);
            $field.find('.pwcp-field-title').text(fieldData.label);

            var $typeBadge = $field.find('.pwcp-field-type-badge');
            $typeBadge.removeClass().addClass('pwcp-field-type-badge pwcp-type-' + fieldData.type);

            // Use translation map for field type labels
            var typeLabel = fieldTypeLabels[fieldData.type] ||
                            fieldData.type.charAt(0).toUpperCase() + fieldData.type.slice(1);
            $typeBadge.text(typeLabel);

            $field.find('.pwcp-field-required-badge').remove();
            if (fieldData.required) {
                var $requiredBadge = $('<span class="pwcp-field-required-badge">' +
                    __('Required', 'pricewise-calculator-pro') + '</span>');
                $field.find('.pwcp-field-meta').append($requiredBadge);
            }

            $field.find('.pwcp-field-order').val(fieldData.order || 0);
            $field.find('input[name*="[id]"]').val(fieldData.id);
        },

        showModalFieldError: function($field, message) {
            var $wrapper = $field.closest('.pwcp-field-group');
            var $error = $wrapper.find('.pwcp-field-error-message');

            if (!$error.length) {
                $error = $('<div class="pwcp-field-error-message"></div>');
                $wrapper.append($error);
            }

            $error.text(message).show();
        },

        clearModalFieldError: function($field) {
            var $wrapper = $field.closest('.pwcp-field-group');
            $wrapper.find('.pwcp-field-error-message').hide();
        },

        clearModalValidationErrors: function() {
            $('.pwcp-field-error-message').hide();
            $('.pwcp-field-valid, .pwcp-field-error').removeClass('pwcp-field-valid pwcp-field-error');
        },

        removeField: function($button) {
            var confirmMessage = window.pwcpCalculatorForm && window.pwcpCalculatorForm.strings
                ? window.pwcpCalculatorForm.strings.confirm_delete
                : __('Are you sure you want to remove this field?', 'pricewise-calculator-pro');

            if (!confirm(confirmMessage)) {
                return;
            }

            var self = this;
            var $field = $button.closest('.pwcp-modern-field');
            var fieldId = $field.data('field-id');
            var isRealField = fieldId && !this.isTemporaryId(fieldId) && parseInt(fieldId, 10) > 0;

            if (isRealField) {
                this.deleteFieldFromDatabase(fieldId, function(success) {
                    if (success) {
                        self.removeFieldFromUI($field, fieldId);
                    } else {
                        self.showNotification(
                            __('Failed to delete field from database', 'pricewise-calculator-pro'),
                            'error'
                        );
                    }
                });
            } else {
                this.removeFieldFromUI($field, fieldId);
            }
        },

        deleteFieldFromDatabase: function(fieldId, callback) {
            $.ajax({
                url: window.pwcpCalculatorForm.ajax_url,
                type: 'POST',
                data: {
                    action: 'pwcp_delete_calculator_field',
                    field_id: fieldId,
                    calculator_id: this.state.calculatorId,
                    pwcp_nonce: window.pwcpCalculatorForm.nonce
                },
                success: function(response) {
                    callback(response.success);
                },
                error: function() {
                    callback(false);
                }
            });
        },

        removeFieldFromUI: function($field, fieldId) {
            var self = this;

            if (fieldId) {
                this.state.fieldIdMapping.delete(fieldId);
            }

            $field.slideUp(300, function() {
                $field.remove();
                self.updateFieldsContainerState();
                self.updateFieldCount();
                self.updateFieldOrder();
                self.showNotification(
                    __('Field deleted successfully', 'pricewise-calculator-pro'),
                    'success'
                );
            });
        },

        updateFieldCount: function() {
            var count = $(this.config.selectors.modernField).length;
            $('.pwcp-field-count').text(count);
        },

        updateFieldOrder: function() {
            $(this.config.selectors.modernField).each(function(index) {
                $(this).find('.pwcp-field-order').val(index + 1);
            });
        },

        updateFieldsContainerState: function() {
            var hasFields = $(this.config.selectors.modernField).length > 0;
            if (!hasFields && this.state.calculatorId > 0) {
                this.showEmptyFieldsState();
            }
        },

        showFormulaHelp: function() {
            var $modal = $('#pwcp-formula-help-modal');
            var $body = $('body');

            if ($modal.hasClass('pwcp-modal-open')) {
                return;
            }

            var scrollbarWidth = window.innerWidth - document.documentElement.clientWidth;
            $body.addClass('pwcp-modal-active');

            if (scrollbarWidth > 0) {
                $body.css('padding-right', scrollbarWidth + 'px');
            }

            $modal.removeClass('pwcp-hidden');
            $modal[0].offsetHeight;

            requestAnimationFrame(function() {
                $modal.addClass('pwcp-modal-open');
            });
        },

        hideFormulaHelp: function() {
            var $modal = $('#pwcp-formula-help-modal');
            var $body = $('body');

            if (!$modal.hasClass('pwcp-modal-open')) {
                return;
            }

            $modal.removeClass('pwcp-modal-open');

            setTimeout(function() {
                $modal.addClass('pwcp-hidden');
                $body.removeClass('pwcp-modal-active');
                $body.css('padding-right', '');
            }, 300);
        },

        showEmptyFieldsState: function() {
            if (!this.state.calculatorId) {
                return;
            }

            var emptyStateHtml = $('<div>').addClass('pwcp-empty-state').html(
                '<div class="pwcp-empty-illustration">' +
                '<svg class="pwcp-empty-icon" viewBox="0 0 200 150">' +
                '<rect x="30" y="30" width="140" height="20" rx="4" fill="#6366f1" fill-opacity="0.1" stroke="#6366f1"/>' +
                '<rect x="30" y="60" width="100" height="20" rx="4" fill="#8b5cf6" fill-opacity="0.1" stroke="#8b5cf6"/>' +
                '<rect x="30" y="90" width="120" height="20" rx="4" fill="#a855f7" fill-opacity="0.1" stroke="#a855f7"/>' +
                '</svg>' +
                '</div>' +
                '<h3>' + __('No Fields Yet', 'pricewise-calculator-pro') + '</h3>' +
                '<p>' + __('Add input fields to collect data from users for your calculations.', 'pricewise-calculator-pro') + '</p>' +
                '<button type="button" class="pwcp-btn pwcp-btn-primary pwcp-btn-icon pwcp-add-field">' +
                '<svg class="pwcp-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">' +
                '<line x1="12" y1="5" x2="12" y2="19"/>' +
                '<line x1="5" y1="12" x2="19" y2="12"/>' +
                '</svg>' +
                __('Add Your First Field', 'pricewise-calculator-pro') +
                '</button>'
            );

            $('#pwcp-fields-container').html(emptyStateHtml);
        },

        setSaveStatus: function(status, message) {
            var $saveStatus = $(this.config.selectors.saveStatus);

            if (!$saveStatus.length) {
                return;
            }

            $saveStatus.removeClass('pwcp-saving pwcp-saved pwcp-pending pwcp-error')
                .addClass('pwcp-' + status);

            $saveStatus.find('.pwcp-save-text').text(message);
        },

        validateForm: function() {
            var isValid = true;
            var $name = $(this.config.selectors.calculatorName);

            if (!$name.val().trim()) {
                $name.addClass(this.config.classes.error);
                isValid = false;
                if (this.state.activeTab !== 'basic') {
                    this.switchTab('basic');
                }
            } else {
                $name.removeClass(this.config.classes.error).addClass(this.config.classes.valid);
            }

            var formulaValue = $(this.config.selectors.formulaInput).val().trim();
            if (formulaValue && !this.validateFormula()) {
                isValid = false;
                if (this.state.activeTab !== 'formula') {
                    this.switchTab('formula');
                }
            }

            var self = this;
            var hasFieldErrors = false;

            $(this.config.selectors.fieldKeyInput).each(function() {
                if (!self.validateFieldKey($(this))) {
                    isValid = false;
                    hasFieldErrors = true;
                }
            });

            if (hasFieldErrors && this.state.activeTab !== 'fields') {
                this.switchTab('fields');
            }

            return isValid;
        },

        handleAjaxSubmit: function() {
            if (!window.pwcpCalculatorForm || !window.pwcpCalculatorForm.ajax_url) {
                this.showNotification(
                    __('Configuration error - please refresh the page', 'pricewise-calculator-pro'),
                    'error'
                );
                return false;
            }

            if (this.state.saveInProgress) {
                return false;
            }

            if (!this.validateForm()) {
                this.showNotification(
                    __('Please fix validation errors', 'pricewise-calculator-pro'),
                    'error'
                );
                return false;
            }

            this.state.saveInProgress = true;
            this.setSaveStatus('saving', __('Saving calculator...', 'pricewise-calculator-pro'));

            var $submitBtn = $('.pwcp-submit-btn, button[type="submit"]');
            $submitBtn.prop('disabled', true);

            var $icon = $submitBtn.find('svg');
            if ($icon.length > 0) {
                $submitBtn.html($icon[0].outerHTML + ' ' + __('Saving...', 'pricewise-calculator-pro'));
            } else {
                $submitBtn.text(__('Saving...', 'pricewise-calculator-pro'));
            }

            var formData = this.collectFormData();

            if (!formData) {
                this.resetSaveButtonState($submitBtn);
                return false;
            }

            var self = this;
            var startTime = Date.now();

            $.ajax({
                url: window.pwcpCalculatorForm.ajax_url,
                type: 'POST',
                data: formData,
                timeout: 30000,
                success: function(response) {
                    var elapsed = Date.now() - startTime;
                    var minDelay = 500;
                    var remainingDelay = Math.max(0, minDelay - elapsed);

                    setTimeout(function() {
                        self.handleSaveSuccess(response);
                    }, remainingDelay);
                },
                error: function(xhr, status, error) {
                    var elapsed = Date.now() - startTime;
                    var minDelay = 500;
                    var remainingDelay = Math.max(0, minDelay - elapsed);

                    setTimeout(function() {
                        self.handleSaveError(xhr, status, error);
                    }, remainingDelay);
                },
                complete: function() {
                    var elapsed = Date.now() - startTime;
                    var minDelay = 500;
                    var remainingDelay = Math.max(0, minDelay - elapsed);

                    setTimeout(function() {
                        self.resetSaveButtonState($submitBtn);
                    }, remainingDelay);
                }
            });

            return false;
        },

        resetSaveButtonState: function($submitBtn) {
            this.state.saveInProgress = false;
            $submitBtn.prop('disabled', false);

            var isEditMode = this.state.calculatorId > 0;
            var buttonText = isEditMode
                ? __('Update Calculator', 'pricewise-calculator-pro')
                : __('Save Calculator', 'pricewise-calculator-pro');

            var $icon = $submitBtn.find('svg');
            if ($icon.length > 0) {
                $submitBtn.html($icon[0].outerHTML + ' ' + buttonText);
            } else {
                $submitBtn.text(buttonText);
            }
        },

        collectFormData: function() {
            var formData = {
                action: 'pwcp_save_calculator',
                pwcp_nonce: window.pwcpCalculatorForm.nonce,
                calculator_id: $('input[name="calculator_id"]').val() || 0,
                calculator_name: $('#calculator_name').val() || '',
                calculator_description: $('#calculator_description').val() || '',
                calculator_formula: $('#calculator_formula').val() || '',
                calculator_status: $('#calculator_status').val() || 'active',
                calculator_priority: $('#calculator_priority').val() || 1,
                assignment_type: $('input[name="assignment_type"]:checked').val() || 'global',
                assigned_categories: $('#assigned_categories').val() || '[]',
                hide_quantity_selector: $('#hide_quantity_selector').is(':checked') ? 1 : 0
            };

            var fieldsData = [];
            var self = this;

            $('.pwcp-modern-field').each(function(index) {
                var $field = $(this);
                var fieldData = self.fieldDataManager.getFieldData($field);

                if (fieldData && fieldData.label && fieldData.key) {
                    fieldsData.push({
                        id: parseInt(fieldData.id, 10) || 0,
                        label: fieldData.label,
                        key: fieldData.key,
                        type: fieldData.type,
                        required: fieldData.required ? 1 : 0,
                        default: fieldData.default_value || '',
                        order: index + 1,
                        min: fieldData.min || '',
                        max: fieldData.max || '',
                        step: fieldData.step || '',
                        options: fieldData.options || ''
                    });
                }
            });

            formData.fields = fieldsData;

            return formData;
        },

        handleSaveSuccess: function(response) {
    // Safe message extraction with proper fallback
    const message = (response && (response.message || (response.data && response.data.message))) 
        || __('Operation completed successfully', 'pricewise-calculator-pro');
    
    if (response.success) {
        this.setSaveStatus('saved', message);
        this.showNotification(message, 'success');

        // Handle new calculator creation
        if (response.data && response.data.is_new && response.data.edit_url) {
            this.state.calculatorId = response.data.calculator_id;
            $('input[name="calculator_id"]').val(this.state.calculatorId);
            var newUrl = response.data.edit_url + '&tab=' + this.state.activeTab;
            window.history.replaceState({}, '', newUrl);
        }
        
        // Ensure fields functionality is available for BOTH new and existing calculators
        if (this.state.calculatorId > 0) {
            this.state.isEditMode = true;
            this.enableFieldsTabCompletely();
            this.updateButtonText();
        }
    } else {
        this.handleSaveError(null, 'error', message);
    }
},

        handleSaveError: function(xhr, status, error) {
            // Safe message extraction with proper fallback chain
            var message = __('An error occurred while saving', 'pricewise-calculator-pro');
            
            if (xhr && xhr.responseJSON) {
                message = xhr.responseJSON.message || 
                          (xhr.responseJSON.data && xhr.responseJSON.data.message) || 
                          message;
            } else if (typeof error === 'string' && error) {
                message = error;
            }
            
            this.setSaveStatus('error', message);
            this.showNotification(message, 'error');
        },

        isTemporaryId: function(id) {
            return typeof id === 'string' && (id.startsWith('new_') || id === 'TEMPLATE_INDEX');
        },

        bindModalEvents: function() {
            var self = this;

            $(document)
                .off('click', this.config.selectors.modalClose + ', ' + this.config.selectors.modalCancel + ', ' + this.config.selectors.modalOverlay)
                .on('click', this.config.selectors.modalClose + ', ' + this.config.selectors.modalCancel + ', ' + this.config.selectors.modalOverlay, function(e) {
                    e.preventDefault();
                    self.hideFieldModal();
                });

            $(document)
                .off('click', this.config.selectors.modalContainer)
                .on('click', this.config.selectors.modalContainer, function(e) {
                    e.stopPropagation();
                });

            $(document)
                .off('keydown.pwcpModal')
                .on('keydown.pwcpModal', function(e) {
                    if (e.key === 'Escape' && $(self.config.selectors.modal).hasClass(self.config.classes.modalOpen)) {
                        self.hideFieldModal();
                    }
                });

            $(document)
                .off('input', this.config.selectors.modalFieldLabel)
                .on('input', this.config.selectors.modalFieldLabel, function() {
                    self.generateModalFieldKey();
                });

            $(document)
                .off('input', this.config.selectors.modalFieldKey)
                .on('input', this.config.selectors.modalFieldKey, function() {
                    self.validateModalFieldKey();
                });

            $(document)
                .off('change', this.config.selectors.modalFieldType)
                .on('change', this.config.selectors.modalFieldType, function() {
                    self.updateModalFieldOptions();
                });

            $(document)
                .off('click', this.config.selectors.modalSave)
                .on('click', this.config.selectors.modalSave, function(e) {
                    e.preventDefault();
                    self.saveModalField();
                });

            $(document)
                .off('submit', this.config.selectors.modalForm)
                .on('submit', this.config.selectors.modalForm, function(e) {
                    e.preventDefault();
                    self.saveModalField();
                });

            $(document)
                .off('click', '#pwcp-formula-help-close-footer')
                .on('click', '#pwcp-formula-help-close-footer', function(e) {
                    e.preventDefault();
                    self.hideFormulaHelp();
                });

            $(document)
                .off('click', '#formula-help')
                .on('click', '#formula-help', function(e) {
                    e.preventDefault();
                    self.showFormulaHelp();
                });

            $(document)
                .off('click', '#pwcp-formula-help-close')
                .on('click', '#pwcp-formula-help-close', function(e) {
                    e.preventDefault();
                    self.hideFormulaHelp();
                });

            $(document)
                .off('click', '#pwcp-formula-help-modal .pwcp-modal-overlay')
                .on('click', '#pwcp-formula-help-modal .pwcp-modal-overlay', function(e) {
                    e.preventDefault();
                    self.hideFormulaHelp();
                });

            $(document)
                .off('keydown.pwcpFormulaHelp')
                .on('keydown.pwcpFormulaHelp', function(e) {
                    if (e.key === 'Escape' && $('#pwcp-formula-help-modal').hasClass('pwcp-modal-open')) {
                        e.preventDefault();
                        self.hideFormulaHelp();
                    }
                });

            $(document)
                .off('click', '#pwcp-formula-help-modal .pwcp-modal-container')
                .on('click', '#pwcp-formula-help-modal .pwcp-modal-container', function(e) {
                    e.stopPropagation();
                });
        }
    });

    window.PWCPModernCalculatorForm = PWCPModernCalculatorForm;

    $(document).ready(function() {
        debugLog(__('DOM Ready - Initializing with URL parameter persistence', 'pricewise-calculator-pro'));

        setTimeout(function() {
            PWCPModernCalculatorForm.init();
        }, 100);
    });

})(jQuery, window, document);


