<?php
/**
 * Uninstall Confirmation Prompt 
 * 
 * 
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class PWCP_Uninstall_Prompt {
    
    public function __construct() {
        // Hook into plugin action links
        add_filter('plugin_action_links_' . PWCP_PLUGIN_BASENAME, array($this, 'add_uninstall_settings_link'), 10, 4);
        
        // Add admin menu for uninstall settings
        add_action('admin_menu', array($this, 'add_uninstall_settings_page'), 100);
        
        // Handle uninstall preference save
        add_action('admin_init', array($this, 'handle_uninstall_preference'));
        
        // Enqueue styles for uninstall page
        add_action('admin_enqueue_scripts', array($this, 'enqueue_uninstall_styles'));
    }
    
    /**
     * Enqueue styles for uninstall page
     */
    public function enqueue_uninstall_styles($hook) {
        // Only load on our uninstall settings page
        if ($hook !== 'admin_page_pwcp-uninstall-settings') {
            return;
        }
        
        wp_enqueue_style(
            'pwcp-admin-uninstall',
            PWCP_PLUGIN_URL . 'assets/css/all-admin-css/admin-uninstall.css',
            array(),
            PWCP_PLUGIN_VERSION
        );
    }
    
    /**
     * Add uninstall settings link to plugin actions
     */
    public function add_uninstall_settings_link($actions, $plugin_file, $plugin_data, $context) {
        $settings_link = sprintf(
            '<a class="pwcp-uninstall-link" href="%s">%s</a>',
            esc_url(admin_url('admin.php?page=pwcp-uninstall-settings')),
            esc_html__('Uninstall Settings', 'pricewise-calculator-pro')
        );
        
        $actions['pwcp_uninstall'] = $settings_link;
        return $actions;
    }
    
    /**
     * Register uninstall settings page
     */
    public function add_uninstall_settings_page() {
        add_submenu_page(
            null, // Hidden from menu
            __('Uninstall Settings', 'pricewise-calculator-pro'),
            __('Uninstall Settings', 'pricewise-calculator-pro'),
            'manage_options',
            'pwcp-uninstall-settings',
            array($this, 'render_uninstall_settings_page')
        );
    }
    
    /**
     * Render uninstall settings page
     */
    public function render_uninstall_settings_page() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'pricewise-calculator-pro'));
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'pwcp_calculators';
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name from $wpdb->prefix
        $calculator_count = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM `" . esc_sql( $table_name ) . "` WHERE 1=%d", 1 ) );
        $settings_exported = get_option('pwcp_uninstall_settings_exported', 'no');
        
        ?>
        <div class="wrap pwcp-modern-admin pwcp-page-uninstall">
            
            <!-- Hero Header -->
            <div class="pwcp-hero-header">
                <div class="pwcp-hero-background">
                    <div class="pwcp-hero-particles"></div>
                    <div class="pwcp-hero-gradient"></div>
                </div>
                <div class="pwcp-hero-content">
                    <div class="pwcp-hero-left">
                        <div class="pwcp-hero-badge">
                            <?php esc_html_e('Uninstall', 'pricewise-calculator-pro'); ?>
                        </div>
                        <h1 class="pwcp-hero-title">
                            <?php esc_html_e('Before You Go...', 'pricewise-calculator-pro'); ?>
                        </h1>
                        <p class="pwcp-hero-subtitle">
                            <?php esc_html_e('Choose what happens to your data when you delete the plugin', 'pricewise-calculator-pro'); ?>
                        </p>
                    </div>
                </div>
            </div>

            <!-- Main Container -->
            <div class="pwcp-modern-container">
                <div class="pwcp-uninstall-grid">
                    
                    <!-- Main Content -->
                    <div class="pwcp-uninstall-main">
                        
                        <!-- Warning Notice -->
                        <div class="pwcp-notice pwcp-notice-warning">
                            <div class="pwcp-notice-icon">
                                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z" />
                                </svg>
                            </div>
                            <div class="pwcp-notice-content">
                                <h3><?php esc_html_e('Important Decision Required', 'pricewise-calculator-pro'); ?></h3>
                                <p><?php esc_html_e('Please read the following information carefully and make your choice about your data.', 'pricewise-calculator-pro'); ?></p>
                            </div>
                        </div>

                        <!-- Data Overview -->
                        <div class="pwcp-dashboard-section">
                            <div class="pwcp-section-header">
                                <div class="pwcp-section-title">
                                    <h2><?php esc_html_e('What Data Will Be Affected?', 'pricewise-calculator-pro'); ?></h2>
                                    <p><?php esc_html_e('The following data is currently stored in your database', 'pricewise-calculator-pro'); ?></p>
                                </div>
                            </div>
                            <div class="pwcp-section-content">
                                <div class="pwcp-data-items">
                                    <div class="pwcp-data-item">
                                        <div class="pwcp-data-icon pwcp-data-primary">
                                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 7h6m0 10v-3m-3 3h.01M9 17h.01M9 14h.01M12 14h.01M15 11h.01M12 11h.01M9 11h.01M7 21h10a2 2 0 002-2V5a2 2 0 00-2-2H7a2 2 0 00-2 2v14a2 2 0 002 2z" />
                                            </svg>
                                        </div>
                                        <div class="pwcp-data-info">
                                            <div class="pwcp-data-count"><?php echo esc_html($calculator_count); ?></div>
                                            <div class="pwcp-data-label"><?php esc_html_e('Calculator(s)', 'pricewise-calculator-pro'); ?></div>
                                        </div>
                                    </div>
                                    
                                    <div class="pwcp-data-item">
                                        <div class="pwcp-data-icon pwcp-data-success">
                                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 7v10c0 2.21 3.582 4 8 4s8-1.79 8-4V7M4 7c0 2.21 3.582 4 8 4s8-1.79 8-4M4 7c0-2.21 3.582-4 8-4s8 1.79 8 4" />
                                            </svg>
                                        </div>
                                        <div class="pwcp-data-info">
                                            <div class="pwcp-data-count">✓</div>
                                            <div class="pwcp-data-label"><?php esc_html_e('Fields & Formulas', 'pricewise-calculator-pro'); ?></div>
                                        </div>
                                    </div>
                                    
                                    <div class="pwcp-data-item">
                                        <div class="pwcp-data-icon pwcp-data-info">
                                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z" />
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                                            </svg>
                                        </div>
                                        <div class="pwcp-data-info">
                                            <div class="pwcp-data-count">✓</div>
                                            <div class="pwcp-data-label"><?php esc_html_e('Settings & Config', 'pricewise-calculator-pro'); ?></div>
                                        </div>
                                    </div> 
                                </div>
                            </div>
                        </div>

                        <!-- Choice Form -->
                        <form method="post" action="" class="pwcp-uninstall-form">
                            <?php wp_nonce_field('pwcp_uninstall_preference', 'pwcp_uninstall_nonce'); ?>
                            
                            <div class="pwcp-dashboard-section">
                                <div class="pwcp-section-header">
                                    <div class="pwcp-section-title">
                                        <h2><?php esc_html_e('Choose What to Do With Your Data', 'pricewise-calculator-pro'); ?></h2>
                                        <p><?php esc_html_e('Select one option below - this choice will be remembered when you delete the plugin', 'pricewise-calculator-pro'); ?></p>
                                    </div>
                                </div>
                                <div class="pwcp-section-content">
                                    <div class="pwcp-choice-options">
                                        
                                        <!-- Keep Data Option -->
                                        <label class="pwcp-choice-card pwcp-choice-recommended">
                                            <input type="radio" name="uninstall_action" value="keep" required class="pwcp-choice-radio">
                                            <div class="pwcp-choice-header">
                                                <div class="pwcp-choice-icon">
                                                    <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z" />
                                                    </svg>
                                                </div>
                                                <div>
                                                    <h3><?php esc_html_e('Keep My Data', 'pricewise-calculator-pro'); ?></h3>
                                                    <span class="pwcp-choice-badge"><?php esc_html_e('Recommended', 'pricewise-calculator-pro'); ?></span>
                                                </div>
                                            </div>
                                            <p class="pwcp-choice-description">
                                                <?php esc_html_e('Your calculators and settings will be preserved. You can reinstall the plugin anytime and continue where you left off.', 'pricewise-calculator-pro'); ?>
                                            </p>
                                            <div class="pwcp-choice-check">
                                                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7" />
                                                </svg>
                                            </div>
                                        </label>
                                        
                                        <!-- Delete Data Option -->
                                        <label class="pwcp-choice-card pwcp-choice-danger">
                                            <input type="radio" name="uninstall_action" value="delete" required class="pwcp-choice-radio">
                                            <div class="pwcp-choice-header">
                                                <div class="pwcp-choice-icon">
                                                    <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                                                    </svg>
                                                </div>
                                                <div>
                                                    <h3><?php esc_html_e('Delete All Data Permanently', 'pricewise-calculator-pro'); ?></h3>
                                                </div>
                                            </div>
                                            <p class="pwcp-choice-description">
                                                <?php esc_html_e('All calculators, settings, and data will be permanently deleted. This action cannot be undone.', 'pricewise-calculator-pro'); ?>
                                            </p>
                                            <div class="pwcp-choice-check">
                                                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7" />
                                                </svg>
                                            </div>
                                        </label>
                                        
                                    </div>
                                </div>
                                
                                <!-- Form Footer -->
                                <div class="pwcp-section-footer">
                                    <button type="submit" name="save_uninstall_preference" class="pwcp-btn pwcp-btn-primary pwcp-btn-lg">
                                        <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor" class="pwcp-icon">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7" />
                                        </svg>
                                        <?php esc_html_e('Save My Choice & Proceed', 'pricewise-calculator-pro'); ?>
                                    </button>
                                    <a href="<?php echo esc_url(admin_url('plugins.php')); ?>" class="pwcp-btn pwcp-btn-ghost pwcp-btn-lg">
                                        <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor" class="pwcp-icon">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
                                        </svg>
                                        <?php esc_html_e('Cancel', 'pricewise-calculator-pro'); ?>
                                    </a>
                                </div>
                            </div>
                        </form>

                    </div>

                    <!-- Sidebar -->
                    <div class="pwcp-uninstall-sidebar">
                        
                        <!-- Export Widget -->
                        <div class="pwcp-sidebar-widget">
                            <div class="pwcp-widget-header">
                                <h3><?php esc_html_e('Export Your Settings', 'pricewise-calculator-pro'); ?></h3>
                            </div>
                            <div class="pwcp-widget-content">
                                <p><?php esc_html_e('Before uninstalling, export your settings and calculators to restore them later or use on another site.', 'pricewise-calculator-pro'); ?></p>
                                
                                <?php if ($settings_exported === 'yes'): ?>
                                    <div class="pwcp-export-status pwcp-export-success">
                                        <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
                                        </svg>
                                        <span><?php esc_html_e('Settings already exported', 'pricewise-calculator-pro'); ?></span>
                                    </div>
                                <?php endif; ?>
                                
                                <a href="<?php echo esc_url(admin_url('admin.php?page=pwcp-import-export')); ?>" 
                                   class="pwcp-btn pwcp-btn-outline pwcp-btn-full pwcp-btn-center pwcp-mt-16" 
                                   target="_blank"
                                   rel="noopener noreferrer">
                                    <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor" class="pwcp-icon">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-4l-4 4m0 0l-4-4m4 4V4" />
                                    </svg>
                                    <?php esc_html_e('Export Settings', 'pricewise-calculator-pro'); ?>
                                </a>
                            </div>
                        </div>

                        <!-- Info Widget -->
                        <div class="pwcp-sidebar-widget">
                            <div class="pwcp-widget-header">
                                <h3><?php esc_html_e('What Happens Next?', 'pricewise-calculator-pro'); ?></h3>
                            </div>
                            <div class="pwcp-widget-content">
                                <div class="pwcp-info-steps">
                                    <div class="pwcp-info-step">
                                        <div class="pwcp-step-number">1</div>
                                        <p><?php esc_html_e('Save your choice using the button below', 'pricewise-calculator-pro'); ?></p>
                                    </div>
                                    <div class="pwcp-info-step">
                                        <div class="pwcp-step-number">2</div>
                                        <p><?php esc_html_e('Go to the WordPress Plugins page', 'pricewise-calculator-pro'); ?></p>
                                    </div>
                                    <div class="pwcp-info-step">
                                        <div class="pwcp-step-number">3</div>
                                        <p><?php esc_html_e('Delete the plugin - your choice will be honored', 'pricewise-calculator-pro'); ?></p>
                                    </div>
                                </div>
                            </div>
                        </div>

                    </div>
                    
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Handle uninstall preference save
     */
    public function handle_uninstall_preference() {
        if (!isset($_POST['save_uninstall_preference'])) {
            return;
        }
        
       if (!isset($_POST['pwcp_uninstall_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['pwcp_uninstall_nonce'])), 'pwcp_uninstall_preference')) {
    wp_die(esc_html__('Security verification failed.', 'pricewise-calculator-pro'));
}
        
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Insufficient permissions.', 'pricewise-calculator-pro'));
        }
        
        if (!isset($_POST['uninstall_action'])) {
            wp_die(esc_html__('Missing required choice.', 'pricewise-calculator-pro'));
        }
        
        $action = sanitize_text_field(wp_unslash($_POST['uninstall_action']));
        
        if ($action === 'delete') {
            update_option('pwcp_uninstall_delete_data', 'yes');
            $message = __('Your choice has been saved. You can now delete the plugin - all data will be permanently removed.', 'pricewise-calculator-pro');
            $type = 'warning';
        } elseif ($action === 'keep') {
            update_option('pwcp_uninstall_delete_data', 'no');
            $message = __('Your choice has been saved. You can now delete the plugin - your data will be preserved.', 'pricewise-calculator-pro');
            $type = 'success';
        } else {
            wp_die(esc_html__('Invalid choice.', 'pricewise-calculator-pro'));
        }
        
        set_transient('pwcp_uninstall_preference_saved', array(
            'message' => $message,
            'type' => $type
        ), 45);
        
        wp_safe_redirect(admin_url('plugins.php'));
        exit;
    }
}

if (is_admin()) {
    new PWCP_Uninstall_Prompt();
}