<?php
/**
 * Uninstall Reminder Notices
 * 
 * Shows persistent reminders to set uninstall preference
 * Complies with Envato security, asset loading, and WordPress standards
 * 
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class PWCP_Uninstall_Notices {
    
    public function __construct() {
        add_action('admin_notices', array($this, 'show_uninstall_preference_notice'));
        add_action('admin_init', array($this, 'dismiss_notice'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_notice_assets'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_delete_warning'));
        add_action('wp_ajax_pwcp_dismiss_uninstall_notice', array($this, 'ajax_dismiss_notice'));
    }
    
    /**
     * Check if current user can manage plugin settings (multisite compatible)
     */
    private function pwcp_can_manage() {
        return current_user_can(is_network_admin() ? 'manage_network_plugins' : 'manage_options');
    }
    
    /**
     * Get option value (network-aware)
     */
    private function get_flag($key) {
        return is_network_admin() ? get_site_option($key) : get_option($key);
    }
    
    /**
     * Set option value (network-aware)
     */
    private function set_flag($key, $value) {
        return is_network_admin() ? update_site_option($key, $value) : update_option($key, $value);
    }
    
    /**
     * Enqueue admin notice assets (CSS and JS)
     */
    public function enqueue_notice_assets($hook) {
        // Only load on plugins page (including network admin)
        if ($hook !== 'plugins.php' && $hook !== 'plugins-network.php') {
            return;
        }
        
        // Only load if notice will be shown
        if ($this->get_flag('pwcp_uninstall_notice_dismissed')) {
            return;
        }
        
        $preference_set = $this->get_flag('pwcp_uninstall_delete_data');
        if ($preference_set !== false) {
            return;
        }
        
        // Enqueue CSS with proper version constant and correct URL
        wp_enqueue_style(
            'pwcp-admin-notices-css',
            PWCP_PLUGIN_URL . 'assets/css/all-admin-css/admin-notices.css',
            array(),
            PWCP_PLUGIN_VERSION
        );
        
        // Enqueue JS for AJAX dismissal
        wp_enqueue_script(
            'pwcp-admin-notices-js',
            PWCP_PLUGIN_URL . 'assets/js/admin-notices.js',
            array('jquery'),
            PWCP_PLUGIN_VERSION,
            true
        );
        
        // Localize script with AJAX data
        wp_localize_script('pwcp-admin-notices-js', 'pwcpNoticeAjax', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce'   => wp_create_nonce('pwcp_dismiss_uninstall_notice'),
        ));
    }
    
    /**
     * Enqueue delete warning script on plugins page
     * Shows confirmation dialog when user tries to delete the plugin
     * Supports both single-site and network admin
     */
    public function enqueue_delete_warning($hook) {
        // Only load on plugins page (including network admin)
        if ($hook !== 'plugins.php' && $hook !== 'plugins-network.php') {
            return;
        }
        
        // Enqueue delete warning script with correct URL
        wp_enqueue_script(
            'pwcp-delete-warning-js',
            PWCP_PLUGIN_URL . 'assets/js/delete-warning.js',
            array('jquery'),
            PWCP_PLUGIN_VERSION,
            true
        );
        
        // Localize all user-facing strings
        wp_localize_script('pwcp-delete-warning-js', 'pwcpDeleteWarning', array(
            'pluginName'     => __('PriceWise Calculator Pro', 'pricewise-calculator-pro'),
            'warningMessage' => __('Deleting the plugin may remove your data depending on your saved preference.', 'pricewise-calculator-pro'),
            'settingsLink'   => __('Click OK to proceed with deletion, or Cancel to visit Uninstall Settings.', 'pricewise-calculator-pro'),
            'goToSettings'   => __('Open Uninstall Settings?', 'pricewise-calculator-pro'),
            'settingsUrl'    => admin_url('admin.php?page=pwcp-uninstall-settings'),
        ));
    }
    
    /**
     * Show notice if user hasn't set uninstall preference
     * Security: Capability check per requirement
     * Supports multisite network admin
     */
    public function show_uninstall_preference_notice() {
        // Security: Capability check - only show to users who can manage
        if (!$this->pwcp_can_manage()) {
            return;
        }
        
        // Only show on plugins page (including network admin)
        $screen = get_current_screen();
        if (!$screen || !in_array($screen->id, array('plugins', 'plugins-network'), true)) {
            return;
        }
        
        // Check if user dismissed
        if ($this->get_flag('pwcp_uninstall_notice_dismissed')) {
            return;
        }
        
        // Check if preference is already set
        $preference_set = $this->get_flag('pwcp_uninstall_delete_data');
        if ($preference_set !== false) {
            return; // Preference already set
        }
        
        // Show notice (all inline styles removed - now in CSS file)
        ?>
        <div class="notice notice-info is-dismissible pwcp-uninstall-notice">
            <p>
                <strong><?php esc_html_e('PriceWise Calculator Pro:', 'pricewise-calculator-pro'); ?></strong>
                <?php esc_html_e('Before uninstalling, please set your data preference.', 'pricewise-calculator-pro'); ?>
                <a href="<?php echo esc_url(admin_url('admin.php?page=pwcp-uninstall-settings')); ?>" class="button button-primary pwcp-notice-primary-btn">
                    <?php esc_html_e('Set Uninstall Preference', 'pricewise-calculator-pro'); ?>
                </a>
                <a href="<?php echo esc_url(wp_nonce_url(add_query_arg('pwcp_dismiss_uninstall_notice', '1'), 'pwcp_dismiss_notice')); ?>" class="button button-secondary pwcp-notice-secondary-btn">
                    <?php esc_html_e('Dismiss', 'pricewise-calculator-pro'); ?>
                </a>
            </p>
        </div>
        <?php
    }
    
    /**
     * Handle AJAX notice dismissal (for "X" button)
     */
    public function ajax_dismiss_notice() {
        // Security: Ensure this is an AJAX request
        if (!defined('DOING_AJAX') || !DOING_AJAX) {
            wp_send_json_error(array('message' => __('Invalid request', 'pricewise-calculator-pro')));
            return;
        }
        
        // Security: Capability check (multisite compatible)
        if (!$this->pwcp_can_manage()) {
            wp_send_json_error(array('message' => __('Unauthorized', 'pricewise-calculator-pro')));
            return;
        }
        
        // Security: Nonce verification
        check_ajax_referer('pwcp_dismiss_uninstall_notice');
        
        // Save dismissal preference (network-aware)
        $this->set_flag('pwcp_uninstall_notice_dismissed', 'yes');
        
        wp_send_json_success(array('message' => __('Notice dismissed', 'pricewise-calculator-pro')));
    }
    
    /**
     * Handle notice dismissal via URL parameter (for "Dismiss" button)
     * Security: Capability check and input sanitization per requirements
     */
    public function dismiss_notice() {
        // Security: Sanitize input before checking
        $dismiss = isset($_GET['pwcp_dismiss_uninstall_notice']) ? sanitize_text_field(wp_unslash($_GET['pwcp_dismiss_uninstall_notice'])) : '';
        
        if (!$dismiss) {
            return;
        }
        
     // Security: Sanitize nonce before verification (required by WordPress)
$nonce = isset($_GET['_wpnonce']) ? sanitize_text_field(wp_unslash($_GET['_wpnonce'])) : '';

if (!wp_verify_nonce($nonce, 'pwcp_dismiss_notice')) {
    return;
}
        // Security: Capability check - only authorized users can dismiss (multisite compatible)
        if (!$this->pwcp_can_manage()) {
            return;
        }
        
        // Save dismissal preference (network-aware)
        $this->set_flag('pwcp_uninstall_notice_dismissed', 'yes');
        
        wp_safe_redirect(remove_query_arg(array('pwcp_dismiss_uninstall_notice', '_wpnonce')));
        exit;
    }
}

new PWCP_Uninstall_Notices();