<?php
/**
 * Field Manager
 * 
 * Handles field CRUD operations for calculator forms
 * 
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * PWCP Field Manager Class
 * 
 * Centralizes field-related database operations and validation
 * 
 * @since 1.0.0
 */
class PWCP_Field_Manager {
    
    /**
     * Validate field data
     * 
     * @param array $field_data Field data to validate
     * @param int $calculator_id Calculator ID
     * @param int|null $exclude_field_id Field ID to exclude from duplicate check
     * @return array Validation result with is_valid and errors
     */
    public function validate_field_data($field_data, $calculator_id, $exclude_field_id = null) {
        $errors = array();
        
        // Required fields check
        if (empty($field_data['label']) || empty($field_data['key'])) {
            $errors[] = __('Field label and key are required.', 'pricewise-calculator-pro');
        }
        
        // FREE VERSION: Only allow number field type
        $field_type = isset($field_data['type']) ? $field_data['type'] : 'number';
        if ($field_type !== 'number') {
            $errors[] = __('Only Number field type is available in the free version. Visit devtonicstudios.com for premium version.', 'pricewise-calculator-pro');
        }
        
        // Key format validation
        $field_key = sanitize_key($field_data['key']);
        if (!preg_match('/^[a-z0-9_]+$/', $field_key)) {
            $errors[] = __('Field key must contain only lowercase letters, numbers, and underscores.', 'pricewise-calculator-pro');
        }
        
        // Duplicate key check
        if (!empty($field_key) && $this->is_duplicate_key($field_key, $calculator_id, $exclude_field_id)) {
            $errors[] = __('A field with this key already exists.', 'pricewise-calculator-pro');
        }
        
        // Type-specific validation for numeric fields
        if (in_array($field_type, array('number', 'slider'), true)) {
            $min = isset($field_data['validation_rules']['min']) 
                ? (float) $field_data['validation_rules']['min'] 
                : null;
            $max = isset($field_data['validation_rules']['max']) 
                ? (float) $field_data['validation_rules']['max'] 
                : null;
            
            if ($min !== null && $max !== null && $min >= $max) {
                $errors[] = __('Minimum value must be less than maximum value.', 'pricewise-calculator-pro');
            }
        }
        
        return array(
            'is_valid' => empty($errors),
            'errors' => $errors
        );
    }
    
    /**
     * Check for duplicate field key
     * 
     * @param string $field_key Field key to check
     * @param int $calculator_id Calculator ID
     * @param int|null $exclude_field_id Field ID to exclude from check
     * @return bool True if duplicate exists
     */
    private function is_duplicate_key($field_key, $calculator_id, $exclude_field_id = null) {
        global $wpdb;
        $fields_table = $wpdb->prefix . 'pwcp_calculator_fields';
        
        if ($exclude_field_id) {
            $count = $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM `{$fields_table}` WHERE calculator_id = %d AND field_key = %s AND id != %d",
                    $calculator_id,
                    $field_key,
                    $exclude_field_id
                )
            );
        } else {
            $count = $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM `{$fields_table}` WHERE calculator_id = %d AND field_key = %s",
                    $calculator_id,
                    $field_key
                )
            );
        }
        
        return $count > 0;
    }
    
    /**
     * Sanitize field data
     * 
     * @param array $field_data_raw Raw field data from POST
     * @return array Sanitized field data
     */
    public function sanitize_field_data($field_data_raw) {
        // FREE VERSION: Force field type to number
        $field_data = array(
            'key' => sanitize_key($field_data_raw['key'] ?? ''),
            'label' => sanitize_text_field($field_data_raw['label'] ?? ''),
            'type' => 'number', // FORCED TO NUMBER IN FREE VERSION
            'required' => !empty($field_data_raw['required']) ? 1 : 0,
            'default_value' => sanitize_text_field($field_data_raw['default_value'] ?? ''),
            'options' => '' // No options needed for number field
        );
        
        // Build validation rules for number field
        $validation_rules = array();
        if (isset($field_data_raw['min']) && $field_data_raw['min'] !== '') {
            $validation_rules['min'] = floatval($field_data_raw['min']);
        }
        if (isset($field_data_raw['max']) && $field_data_raw['max'] !== '') {
            $validation_rules['max'] = floatval($field_data_raw['max']);
        }
        if (isset($field_data_raw['step']) && $field_data_raw['step'] !== '') {
            $validation_rules['step'] = floatval($field_data_raw['step']);
        }
        
        $field_data['validation_rules'] = $validation_rules;
        
        return $field_data;
    }
    
    /**
     * Create new field in database
     * 
     * @param int $calculator_id Calculator ID
     * @param array $field_data Sanitized field data
     * @return int|false New field ID or false on failure
     */
    public function create_field($calculator_id, $field_data) {
        global $wpdb;
        $fields_table = $wpdb->prefix . 'pwcp_calculator_fields';
        
        // Get next field order
        $field_order = $this->get_next_field_order($calculator_id);
        
        $insert_data = array(
            'calculator_id'   => $calculator_id,
            'field_key'       => $field_data['key'],
            'field_label'     => $field_data['label'],
            'field_type'      => $field_data['type'],
            'field_options'   => $field_data['options'],
            'field_validation' => wp_json_encode($field_data['validation_rules']),
            'field_default'   => $field_data['default_value'],
            'field_required'  => $field_data['required'],
            'field_order'     => $field_order,
            'created_at'      => current_time('mysql'),
            'updated_at'      => current_time('mysql')
        );
        
        $format = array('%d', '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%d', '%s', '%s');
        
        $result = $wpdb->insert($fields_table, $insert_data, $format);
        
        if ($result === false) {
            return false;
        }
        
        return $wpdb->insert_id;
    }
    
    /**
     * Update existing field in database
     * 
     * @param int $field_id Field ID
     * @param int $calculator_id Calculator ID
     * @param array $field_data Sanitized field data
     * @return bool Success status
     */
    public function update_field($field_id, $calculator_id, $field_data) {
        global $wpdb;
        $fields_table = $wpdb->prefix . 'pwcp_calculator_fields';
        
        // Verify field exists and belongs to calculator
        $existing_field = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT id, field_order FROM `{$fields_table}` WHERE id = %d AND calculator_id = %d",
                $field_id,
                $calculator_id
            )
        );
        
        if (!$existing_field) {
            return false;
        }
        
        $update_data = array(
            'field_key'       => $field_data['key'],
            'field_label'     => $field_data['label'],
            'field_type'      => $field_data['type'],
            'field_options'   => $field_data['options'],
            'field_validation' => wp_json_encode($field_data['validation_rules']),
            'field_default'   => $field_data['default_value'],
            'field_required'  => $field_data['required'],
            'updated_at'      => current_time('mysql')
        );
        
        $format = array('%s', '%s', '%s', '%s', '%s', '%s', '%d', '%s');
        
        $result = $wpdb->update(
            $fields_table,
            $update_data,
            array('id' => $field_id, 'calculator_id' => $calculator_id),
            $format,
            array('%d', '%d')
        );
        
        return $result !== false;
    }
    
    /**
     * Delete field from database
     * 
     * @param int $field_id Field ID
     * @param int $calculator_id Calculator ID (for security)
     * @return bool Success status
     */
    public function delete_field($field_id, $calculator_id) {
        global $wpdb;
        $fields_table = $wpdb->prefix . 'pwcp_calculator_fields';
        
        $result = $wpdb->delete(
            $fields_table,
            array(
                'id' => $field_id,
                'calculator_id' => $calculator_id
            ),
            array('%d', '%d')
        );
        
        return $result !== false;
    }
    
    /**
     * Get next field order number
     * 
     * @param int $calculator_id Calculator ID
     * @return int Next order number
     */
    private function get_next_field_order($calculator_id) {
        global $wpdb;
        $fields_table = $wpdb->prefix . 'pwcp_calculator_fields';
        
        $max_order = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT MAX(field_order) FROM `{$fields_table}` WHERE calculator_id = %d",
                $calculator_id
            )
        );
        
        return $max_order ? $max_order + 1 : 1;
    }
    
    /**
     * Get calculator fields
     * 
     * @param int $calculator_id Calculator ID
     * @return array Fields array
     */
    public function get_calculator_fields($calculator_id) {
        global $wpdb;
        $fields_table = $wpdb->prefix . 'pwcp_calculator_fields';
        
        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT id, calculator_id, field_key, field_label, field_type, field_options, 
                        field_validation, field_default, field_required, field_order, 
                        created_at, updated_at 
                 FROM `{$fields_table}` 
                 WHERE calculator_id = %d 
                 ORDER BY field_order ASC",
                $calculator_id
            )
        );
        
        return $results ?: array();
    }
    
    /**
     * Get single field by ID
     * 
     * @param int $field_id Field ID
     * @param int $calculator_id Calculator ID (for security)
     * @return object|null Field object or null
     */
    public function get_field($field_id, $calculator_id = null) {
        global $wpdb;
        $fields_table = $wpdb->prefix . 'pwcp_calculator_fields';
        
        $columns = 'id, calculator_id, field_key, field_label, field_type, field_options, 
                    field_validation, field_default, field_required, field_order, 
                    created_at, updated_at';
        
        if ($calculator_id) {
            return $wpdb->get_row(
                $wpdb->prepare(
                    "SELECT {$columns} FROM `{$fields_table}` WHERE id = %d AND calculator_id = %d",
                    $field_id,
                    $calculator_id
                )
            );
        } else {
            return $wpdb->get_row(
                $wpdb->prepare(
                    "SELECT {$columns} FROM `{$fields_table}` WHERE id = %d",
                    $field_id
                )
            );
        }
    }
    
    /**
     * Bulk save fields for calculator
     * 
     * @param int $calculator_id Calculator ID
     * @param array $fields_data Array of field data
     * @return array Field ID mappings for AJAX response
     */
    public function bulk_save_fields($calculator_id, $fields_data) {
        global $wpdb;
        $fields_table = $wpdb->prefix . 'pwcp_calculator_fields';

        // Don't process if no fields data provided
        if (empty($fields_data) || !is_array($fields_data)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('PWCP Field Manager: No fields data provided to bulk_save_fields');
            }
            return array();
        }

        // Verify we have valid field data
        $has_valid_fields = false;
        foreach ($fields_data as $field_data) {
            if (is_array($field_data) && !empty($field_data['key']) && !empty($field_data['label'])) {
                $has_valid_fields = true;
                break;
            }
        }

        if (!$has_valid_fields) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('PWCP Field Manager: No valid field data found');
            }
            return array();
        }

        // Get existing field IDs
        $existing_field_ids = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT id FROM `{$fields_table}` WHERE calculator_id = %d",
                $calculator_id
            )
        );

        $processed_field_ids = array();
        $field_id_mappings = array();

        foreach ($fields_data as $field_data) {
            if (!is_array($field_data)) {
                continue;
            }

            // Sanitize the raw data
            $sanitized_data = $this->sanitize_field_data($field_data);
            
            // Skip invalid fields
            if (empty($sanitized_data['key']) || empty($sanitized_data['label'])) {
                continue;
            }

            $field_id = absint($field_data['id'] ?? 0);
            $temp_id = isset($field_data['temp_id']) ? sanitize_text_field($field_data['temp_id']) : '';

            // Validate field data
            $validation = $this->validate_field_data($sanitized_data, $calculator_id, $field_id > 0 ? $field_id : null);
            if (!$validation['is_valid']) {
                continue;
            }

            $is_existing_field = $field_id > 0 && in_array($field_id, $existing_field_ids, true);

            if ($is_existing_field) {
                // Update existing field
                $result = $this->update_field($field_id, $calculator_id, $sanitized_data);
                if ($result) {
                    $processed_field_ids[] = $field_id;
                    if (!empty($temp_id)) {
                        $field_id_mappings[] = array(
                            'temp_id' => $temp_id,
                            'real_id' => $field_id
                        );
                    }
                }
            } else {
                // Create new field
                $new_id = $this->create_field($calculator_id, $sanitized_data);
                if ($new_id) {
                    $processed_field_ids[] = $new_id;
                    if (!empty($temp_id)) {
                        $field_id_mappings[] = array(
                            'temp_id' => $temp_id,
                            'real_id' => $new_id
                        );
                    }
                }
            }
        }

        // Clean up removed fields
        if (!empty($processed_field_ids) && !empty($existing_field_ids)) {
            $fields_to_delete = array_diff($existing_field_ids, $processed_field_ids);
            
            if (!empty($fields_to_delete)) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log(sprintf('PWCP: Deleting %d removed fields from calculator %d', 
                        count($fields_to_delete), $calculator_id));
                }
                
                $placeholders = implode(',', array_fill(0, count($fields_to_delete), '%d'));
                $wpdb->query(
                    $wpdb->prepare(
                        "DELETE FROM `{$fields_table}` WHERE id IN ($placeholders)",
                        ...$fields_to_delete
                    )
                );
            }
        }

        return $field_id_mappings;
    }
    
    /**
     * Reorder fields for calculator
     * 
     * @param array $field_order Array of field IDs in order
     * @return bool Success status
     */
    public function reorder_fields($field_order) {
        global $wpdb;
        $fields_table = $wpdb->prefix . 'pwcp_calculator_fields';

        foreach ($field_order as $order => $field_id) {
            $field_id = absint($field_id);
            $order = absint($order) + 1;

            $wpdb->update(
                $fields_table,
                array('field_order' => $order, 'updated_at' => current_time('mysql')),
                array('id' => $field_id),
                array('%d', '%s'),
                array('%d')
            );
        }

        return true;
    }
}