<?php 
/**
 * Enhanced Color Picker System
 * Dual visual/hex input color picker with validation
 * 
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * PWCP Enhanced Color Picker Class
 * 
 * Provides enhanced color picker functionality for settings
 * 
 * @since 1.0.0
 */
class PWCP_Enhanced_Color_Picker {
    
    /**
     * Constructor
     * 
     * @since 1.0.0
     */
    public function __construct() {
        // Color picker CSS is in admin-pages-1.css
        // Color picker JS is in admin-settings.js
        // AJAX validation is handled in PWCP_Admin_Settings class
    }
    
    /**
     * Enhanced color field callback
     * 
     * @param array $args Field arguments
     * @since 1.0.0
     */
    public function pwcp_enhanced_color_field_callback($args) {
        if (!is_array($args) || empty($args['option_name']) || empty($args['field_name'])) {
            return;
        }
        
        $options = get_option(sanitize_key($args['option_name']), array());
        $value = isset($options[$args['field_name']]) ? $options[$args['field_name']] : '#007cba';
        
        $value = sanitize_hex_color($value) ?: '#007cba';
        
        $field_id = sanitize_key($args['option_name'] . '_' . $args['field_name']);
        
        $has_desc = !empty($args['description']);
        $aria_describedby = $has_desc ? sprintf('%s_description', $field_id) : '';
        
        printf(
            '<div class="pwcp-enhanced-color-picker" data-field="%s">
                <input type="color" 
                       id="%s_visual" 
                       value="%s" 
                       class="pwcp-visual-picker"
                       data-target="%s_text"
                       aria-label="%s">
                <input type="text" 
                       name="%s[%s]" 
                       id="%s_text"
                       value="%s" 
                       class="pwcp-hex-input regular-text" 
                       placeholder="#000000" 
                       maxlength="7"
                       pattern="^#([A-Fa-f0-9]{3}|[A-Fa-f0-9]{6})$"%s>
                <span class="pwcp-color-preview" 
                      data-color="%s" 
                      aria-hidden="true"></span>
            </div>',
            esc_attr($field_id),
            esc_attr($field_id),
            esc_attr($value),
            esc_attr($field_id),
            esc_attr(sprintf(
                /* translators: %s: Field label */
                __('Visual color picker for %s', 'pricewise-calculator-pro'),
                $args['description'] ?? $args['field_name']
            )),
            esc_attr($args['option_name']),
            esc_attr($args['field_name']),
            esc_attr($field_id),
            esc_attr($value),
            $has_desc ? ' aria-describedby="' . esc_attr($aria_describedby) . '"' : '',
            esc_attr($value)
        );
        
        if ($has_desc) {
            printf(
                '<p class="pwcp-field-description" id="%s_description">%s</p>',
                esc_attr($field_id),
                esc_html($args['description'])
            );
        }
    }
    
    /**
     * Render color picker group for settings sections
     * 
     * @param array $color_fields Array of color field configurations
     * @param string $section_title Section title
     * @since 1.0.0
     */
    public function pwcp_render_color_picker_section($color_fields, $section_title = '') {
        if (!is_array($color_fields) || empty($color_fields)) {
            return;
        }
        
        if (!empty($section_title)) {
            printf('<h4 class="pwcp-color-section-title">%s</h4>', esc_html($section_title));
        }
        
        echo '<div class="pwcp-color-picker-grid">';
        
        foreach ($color_fields as $field_config) {
            if (!is_array($field_config) || empty($field_config['name']) || empty($field_config['label'])) {
                continue;
            }
            
            $this->pwcp_render_single_color_picker(
                sanitize_key($field_config['name']),
                sanitize_text_field($field_config['label']),
                sanitize_text_field($field_config['description'] ?? ''),
                sanitize_hex_color($field_config['default'] ?? '#007cba'),
                sanitize_key($field_config['option_group'] ?? 'pwcp_design_settings')
            );
        }
        
        echo '</div>';
    }
    
    /**
     * Render single color picker with enhanced validation
     * 
     * @param string $field_name Field name
     * @param string $label Field label
     * @param string $description Field description
     * @param string $default_value Default color value
     * @param string $option_group Option group name
     * @since 1.0.0
     */
    public function pwcp_render_single_color_picker($field_name, $label, $description = '', $default_value = '#007cba', $option_group = 'pwcp_design_settings') {
        $field_name = sanitize_key($field_name);
        $label = sanitize_text_field($label);
        $description = sanitize_text_field($description);
        $default_value = sanitize_hex_color($default_value) ?: '#007cba';
        $option_group = sanitize_key($option_group);
        
        $options = get_option($option_group, array());
        $current_value = isset($options[$field_name]) ? sanitize_hex_color($options[$field_name]) : $default_value;
        
        $field_id = $option_group . '_' . $field_name;
        
        $has_desc = !empty($description);
        
        ?>
        <div class="pwcp-color-picker-field">
            <label for="<?php echo esc_attr($field_id); ?>_text" class="pwcp-color-field-label">
                <?php echo esc_html($label); ?>
                <?php if ($has_desc): ?>
                    <span class="pwcp-color-help" title="<?php echo esc_attr($description); ?>">?</span>
                <?php endif; ?>
            </label>
            
            <div class="pwcp-enhanced-color-picker" data-field="<?php echo esc_attr($field_id); ?>">
           <input type="color" 
       id="<?php echo esc_attr($field_id); ?>_visual" 
       value="<?php echo esc_attr($current_value); ?>" 
       class="pwcp-visual-picker"
       data-target="<?php echo esc_attr($field_id); ?>_text"
       aria-label="<?php echo esc_attr(sprintf(
           /* translators: %s: Field label */
           __('Visual picker for %s', 'pricewise-calculator-pro'), 
           $label
       )); ?>">
                <input type="text" 
                       name="<?php echo esc_attr($option_group); ?>[<?php echo esc_attr($field_name); ?>]" 
                       id="<?php echo esc_attr($field_id); ?>_text"
                       value="<?php echo esc_attr($current_value); ?>" 
                       class="pwcp-hex-input" 
                       placeholder="#000000" 
                       maxlength="7"
                       pattern="^#([A-Fa-f0-9]{3}|[A-Fa-f0-9]{6})$"
                       data-default="<?php echo esc_attr($default_value); ?>"<?php
                       echo $has_desc ? ' aria-describedby="' . esc_attr($field_id) . '_description"' : ''; ?>>
                       
                <span class="pwcp-color-preview" 
                      data-color="<?php echo esc_attr($current_value); ?>" 
                      aria-hidden="true"></span>
                      
         <button type="button" 
        class="pwcp-color-reset" 
        data-target="<?php echo esc_attr($field_id); ?>"
        data-default="<?php echo esc_attr($default_value); ?>"
        title="<?php echo esc_attr__('Reset to default', 'pricewise-calculator-pro'); ?>"
        aria-label="<?php echo esc_attr(sprintf(
            /* translators: %s: Field label */
            __('Reset %s to default color', 'pricewise-calculator-pro'), 
            $label
        )); ?>">
    ↺
</button>
            </div>
            
            <?php if ($has_desc): ?>
                <p class="pwcp-field-description" id="<?php echo esc_attr($field_id); ?>_description">
                    <?php echo esc_html($description); ?>
                </p>
            <?php endif; ?>
            
            <div class="pwcp-color-validation" id="<?php echo esc_attr($field_id); ?>_validation" aria-live="polite"></div>
        </div>
        <?php
    }
    
    /**
     * Get color picker CSS for frontend
     * 
     * @param array $color_settings Color settings array
     * @return string Generated CSS
     * @since 1.0.0
     */
    public function pwcp_generate_color_css($color_settings) {
        if (!is_array($color_settings) || empty($color_settings)) {
            return '';
        }
        
        $css = '';
        $css_rules = array();
        
        foreach ($color_settings as $key => $value) {
            $sanitized_key = sanitize_key($key);
            $sanitized_value = sanitize_hex_color($value);
            
            if (empty($sanitized_value)) {
                continue;
            }
            
            switch ($sanitized_key) {
                case 'primary_color':
                    $css_rules[] = sprintf(
                        '.pwcp-calculator .pwcp-button-primary { background-color: %s; }',
                        esc_attr($sanitized_value)
                    );
                    break;
                    
                case 'header_background_color':
                    $css_rules[] = sprintf(
                        '.pwcp-calculator .pwcp-header { background-color: %s; }',
                        esc_attr($sanitized_value)
                    );
                    break;
                    
                case 'price_color':
                    $css_rules[] = sprintf(
                        '.pwcp-calculator .pwcp-live-price { color: %s; }',
                        esc_attr($sanitized_value)
                    );
                    break;
                    
                case 'calculator_background_color':
                    $css_rules[] = sprintf(
                        '.pwcp-calculator { background-color: %s; }',
                        esc_attr($sanitized_value)
                    );
                    break;
            }
        }
        
        if (!empty($css_rules)) {
            $css = "/* PriceWise Calculator Pro - Generated Color Styles */\n";
            $css .= implode("\n", $css_rules);
            $css .= "\n/* End Generated Styles */";
        }
        
        return $css;
    }
    
    /**
     * Validate hex color format
     * 
     * @param string $color Color value to validate
     * @return array Validation result
     * @since 1.0.0
     */
    public function pwcp_validate_hex_color($color) {
        $color = sanitize_text_field($color);
        
        $result = array(
            'valid' => false,
            'formatted' => '',
            'message' => ''
        );
        
        if (empty($color)) {
            $result['message'] = esc_html__('Color value is required', 'pricewise-calculator-pro');
            return $result;
        }
        
        if (strpos($color, '#') !== 0) {
            $color = '#' . $color;
        }
        
        $validated_color = sanitize_hex_color($color);
        
        if ($validated_color) {
            $result['valid'] = true;
            $result['formatted'] = strtoupper($validated_color);
            $result['message'] = esc_html__('Valid hex color', 'pricewise-calculator-pro');
        } else {
            $result['message'] = esc_html__('Invalid hex color format. Use format: #FFFFFF', 'pricewise-calculator-pro');
        }
        
        return $result;
    }
}