<?php
/**
 * Calculator Form Templates
 * Renders the calculator form interface with tabbed navigation
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class PWCP_Calculator_Form_Templates {

    /**
     * Category assignment manager instance
     * @var PWCP_Category_Assignment|null
     * @since 1.0.0
     */
    private $category_assignment = null;

    /**
     * Constructor
     * 
     * @since 1.0.0
     */
    public function __construct() {
        if (class_exists('PWCP_Category_Assignment')) {
            $this->category_assignment = new PWCP_Category_Assignment();
        }
    }

    /**
     * Get translated field type label
     * 
     * @param string $type Field type
     * @return string Translated label
     * @since 1.0.0
     */
  private function pwcp_type_label($type) {
    $map = array(
        'text'     => __('Text', 'pricewise-calculator-pro'),
        'number'   => __('Number', 'pricewise-calculator-pro'),
        'dropdown' => __('Dropdown', 'pricewise-calculator-pro'),
        'radio'    => __('Radio Buttons', 'pricewise-calculator-pro'),
        'checkbox' => __('Checkboxes', 'pricewise-calculator-pro'),
        'slider'   => __('Slider', 'pricewise-calculator-pro'),
        'file'     => __('File Upload', 'pricewise-calculator-pro'),
    );
    return isset($map[$type]) ? $map[$type] : __('Text', 'pricewise-calculator-pro');
}

    /**
     * Get calculator fields
     * 
     * @param int $calculator_id Calculator ID
     * @return array Fields data
     * @since 1.0.0
     */
    private function pwcp_get_calculator_fields($calculator_id) {
        if (class_exists('PWCP_Field_Manager')) {
            $field_manager = new PWCP_Field_Manager();
            return $field_manager->get_calculator_fields($calculator_id);
        }
        
        if (class_exists('PWCP_Admin_Calculator_Manager')) {
            $manager = new PWCP_Admin_Calculator_Manager();
            if (method_exists($manager, 'pwcp_get_calculator_fields')) {
                return $manager->pwcp_get_calculator_fields($calculator_id);
            }
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'pwcp_calculator_fields';
        $cols = 'id, calculator_id, field_key, field_label, field_type, field_options, field_validation, field_default, field_required, field_order, created_at, updated_at';
        return $wpdb->get_results(
            $wpdb->prepare("SELECT {$cols} FROM `{$table}` WHERE calculator_id = %d ORDER BY field_order ASC", $calculator_id)
        );
    }

   
    /**
     * Get active tab from URL parameter
     * 
     * @return string Active tab name
     * @since 1.0.0
     */
    private function pwcp_get_active_tab_from_request() {
        if (isset($_GET['tab'])) {
            $tab = sanitize_key($_GET['tab']);
            $valid_tabs = array('basic', 'formula', 'fields', 'settings');
            if (in_array($tab, $valid_tabs, true)) {
                return $tab;
            }
        }
        return 'basic';
    }

    /**
     * Display admin notices
     * 
     * @since 1.0.0
     */
    private function pwcp_display_admin_notices() {
        if (!isset($_GET['message'])) {
            return;
        }

        $message = sanitize_text_field(wp_unslash($_GET['message']));
        $notices = array(
            'created' => array('type' => 'success', 'text' => esc_html__('Calculator created successfully!', 'pricewise-calculator-pro')),
            'updated' => array('type' => 'success', 'text' => esc_html__('Calculator updated successfully!', 'pricewise-calculator-pro')),
            'error'   => array('type' => 'error', 'text' => esc_html__('An error occurred. Please try again.', 'pricewise-calculator-pro')),
        );

        if (isset($notices[$message])) {
            $notice = $notices[$message];
         printf(
    '<div class="notice notice-%1$s is-dismissible"><p>%2$s</p></div>',
    esc_attr($notice['type']),
    wp_kses_post($notice['text'])
);
        }
    }

    /**
     * Render calculator form
     * 
     * @param object|null $calculator Calculator object for editing
     * @since 1.0.0
     */
    public function pwcp_render_calculator_form($calculator = null) {
        $is_edit = !empty($calculator);
        $calculator_id = $is_edit && isset($calculator->id) ? absint($calculator->id) : 0;
        $fields = $is_edit ? $this->pwcp_get_calculator_fields($calculator_id) : array();

        $active_tab = $this->pwcp_get_active_tab_from_request();

        ?>
        <div class="wrap pwcp-modern-admin pwcp-page-add-calculator">
            <div class="pwcp-hero-header">
                <div class="pwcp-hero-background">
                    <div class="pwcp-hero-particles"></div>
                    <div class="pwcp-hero-gradient"></div>
                </div>
                
                <div class="pwcp-hero-content">
                    <div class="pwcp-hero-left">
                        <div class="pwcp-hero-badge">
                            <?php echo $is_edit ? esc_html__('Edit Mode', 'pricewise-calculator-pro') : esc_html__('Create Mode', 'pricewise-calculator-pro'); ?>
                        </div>
                        <h1 class="pwcp-hero-title">
                            <?php echo $is_edit && $calculator && isset($calculator->name) ? esc_html($calculator->name) : esc_html__('New Calculator', 'pricewise-calculator-pro'); ?>
                        </h1>
                        <p class="pwcp-hero-subtitle">
                            <?php echo $is_edit ? 
                                esc_html__('Update your calculator settings and formula for enhanced user experience', 'pricewise-calculator-pro') : 
                                esc_html__('Create a powerful calculator with advanced features and smart field management', 'pricewise-calculator-pro'); ?>
                        </p>
                    </div>
                    
                    <div class="pwcp-hero-right">
                        <div class="pwcp-progress-tracker">
                            <div class="pwcp-progress-step <?php echo ($active_tab === 'basic') ? 'pwcp-step-active' : ''; ?>" data-step="1">
                                <div class="pwcp-step-number">1</div>
                                <div class="pwcp-step-label"><?php esc_html_e('Basic Info', 'pricewise-calculator-pro'); ?></div>
                            </div>
                            <div class="pwcp-progress-line"></div>
                            <div class="pwcp-progress-step <?php echo ($active_tab === 'formula') ? 'pwcp-step-active' : ''; ?>" data-step="2">
                                <div class="pwcp-step-number">2</div>
                                <div class="pwcp-step-label"><?php esc_html_e('Formula', 'pricewise-calculator-pro'); ?></div>
                            </div>
                            <div class="pwcp-progress-line"></div>
                            <div class="pwcp-progress-step <?php echo ($active_tab === 'fields') ? 'pwcp-step-active' : ''; ?>" data-step="3">
                                <div class="pwcp-step-number">3</div>
                                <div class="pwcp-step-label"><?php esc_html_e('Fields', 'pricewise-calculator-pro'); ?></div>
                            </div>
                            <div class="pwcp-progress-line"></div>
                            <div class="pwcp-progress-step <?php echo ($active_tab === 'settings') ? 'pwcp-step-active' : ''; ?>" data-step="4">
                                <div class="pwcp-step-number">4</div>
                                <div class="pwcp-step-label"><?php esc_html_e('Settings', 'pricewise-calculator-pro'); ?></div>
                            </div>
                        </div>
                        
                        <div class="pwcp-hero-actions">
                            <a href="<?php echo esc_url(admin_url('admin.php?page=pwcp-calculators')); ?>" 
                               class="pwcp-btn pwcp-btn-outline">
                                <svg class="pwcp-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <path d="M3 3h18v18H3zM21 9H3M9 21V9"/>
                                </svg>
                                <?php esc_html_e('View All', 'pricewise-calculator-pro'); ?>
                            </a>
                            
                            <?php if ($is_edit && $calculator): ?>
                            <a href="<?php echo esc_url(admin_url('admin.php?page=pwcp-add-calculator')); ?>" 
                               class="pwcp-btn pwcp-btn-primary">
                                <svg class="pwcp-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <line x1="12" y1="5" x2="12" y2="19"/>
                                    <line x1="5" y1="12" x2="19" y2="12"/>
                                </svg>
                                <?php esc_html_e('Add New', 'pricewise-calculator-pro'); ?>
                            </a>
                            <?php endif; ?>
                        </div>
                        
                        <div class="pwcp-hero-save-action">
                            <button type="submit" form="pwcp-calculator-form" class="pwcp-btn pwcp-btn-primary pwcp-btn-lg">
                                <svg class="pwcp-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <path d="M19 21H5a2 2 0 01-2-2V5a2 2 0 012-2h11l5 5v11a2 2 0 01-2 2z"/>
                                    <polyline points="17,21 17,13 7,13 7,21"/>
                                    <polyline points="7,3 7,8 15,8"/>
                                </svg>
                                <?php echo $is_edit ? esc_html__('Update Calculator', 'pricewise-calculator-pro') : esc_html__('Save Calculator', 'pricewise-calculator-pro'); ?>
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <?php $this->pwcp_display_admin_notices(); ?>

            <div class="pwcp-modern-container">
                <div class="pwcp-tab-navigation">
                    <button type="button" class="pwcp-tab-btn <?php echo ($active_tab === 'basic') ? 'pwcp-tab-active' : ''; ?>" data-tab="basic">
                        <svg class="pwcp-tab-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M14 2H6a2 2 0 00-2 2v16a2 2 0 002 2h12a2 2 0 002-2V8z"/>
                            <polyline points="14,2 14,8 20,8"/>
                            <line x1="16" y1="13" x2="8" y2="13"/>
                            <line x1="16" y1="17" x2="8" y2="17"/>
                            <polyline points="10,9 9,9 8,9"/>
                        </svg>
                        <?php esc_html_e('Basic Information', 'pricewise-calculator-pro'); ?>
                    </button>
                    
                    <button type="button" class="pwcp-tab-btn <?php echo ($active_tab === 'formula') ? 'pwcp-tab-active' : ''; ?>" data-tab="formula">
                        <svg class="pwcp-tab-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                            <path d="M8 12h8"/>
                            <path d="M12 8v8"/>
                        </svg>
                        <?php esc_html_e('Formula Editor', 'pricewise-calculator-pro'); ?>
                    </button>
                    
                    <button type="button" class="pwcp-tab-btn <?php echo ($active_tab === 'fields') ? 'pwcp-tab-active' : ''; ?>" data-tab="fields">
                        <svg class="pwcp-tab-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                            <line x1="9" y1="12" x2="15" y2="12"/>
                            <line x1="12" y1="9" x2="12" y2="15"/>
                        </svg>
                        <?php esc_html_e('Input Fields', 'pricewise-calculator-pro'); ?>
                        <?php if (!empty($fields)): ?>
                            <span class="pwcp-field-count"><?php echo esc_html((string) absint(count($fields))); ?></span>
                        <?php endif; ?>
                    </button>
                    
                    <button type="button" class="pwcp-tab-btn <?php echo ($active_tab === 'settings') ? 'pwcp-tab-active' : ''; ?>" data-tab="settings">
                        <svg class="pwcp-tab-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <circle cx="12" cy="12" r="3"/>
                            <path d="M19.4 15a1.65 1.65 0 00.33 1.82l.06.06a2 2 0 010 2.83 2 2 0 01-2.83 0l-.06-.06a1.65 1.65 0 00-1.82-.33 1.65 1.65 0 00-1 1.51V21a2 2 0 01-2 2 2 2 0 01-2-2v-.09A1.65 1.65 0 009 19.4a1.65 1.65 0 00-1.82.33l-.06.06a2 2 0 01-2.83 0 2 2 0 010-2.83l.06-.06a1.65 1.65 0 00.33-1.82 1.65 1.65 0 00-1.51-1H3a2 2 0 01-2-2 2 2 0 012-2h.09A1.65 1.65 0 004.6 9a1.65 1.65 0 00-.33-1.82l-.06-.06a2 2 0 010-2.83 2 2 0 012.83 0l.06.06a1.65 1.65 0 001.82.33H9a1.65 1.65 0 001-1.51V3a2 2 0 012-2 2 2 0 012 2v.09a1.65 1.65 0 001 1.51 1.65 1.65 0 001.82-.33l.06-.06a2 2 0 012.83 0 2 2 0 010 2.83l-.06.06a1.65 1.65 0 00-.33 1.82V9a1.65 1.65 0 001.51 1H21a2 2 0 012 2 2 2 0 01-2 2h-.09a1.65 1.65 0 00-1.51 1z"/>
                        </svg>
                        <?php esc_html_e('Settings', 'pricewise-calculator-pro'); ?>
                    </button>
                </div>

                <div class="pwcp-tab-content">
                    <form id="pwcp-calculator-form" class="pwcp-modern-form" data-ajax-save="true">
                        <?php wp_nonce_field('pwcp_calculator_action', 'pwcp_nonce'); ?>
                        <input type="hidden" name="calculator_id" value="<?php echo esc_attr((string) $calculator_id); ?>">

                        <div class="pwcp-tab-panel <?php echo ($active_tab === 'basic') ? 'pwcp-tab-active' : ''; ?>" data-panel="basic">
                            <div class="pwcp-panel-header">
                                <h2 class="pwcp-panel-title"><?php esc_html_e('Basic Information', 'pricewise-calculator-pro'); ?></h2>
                                <p class="pwcp-panel-description">
                                    <?php esc_html_e('Set up the fundamental details and assignment strategy for your calculator', 'pricewise-calculator-pro'); ?>
                                </p>
                            </div>

                            <div class="pwcp-form-grid">
                                <div class="pwcp-form-section">
                                    <div class="pwcp-field-group">
                                        <label for="calculator_name" class="pwcp-field-label pwcp-required">
                                            <?php esc_html_e('Calculator Name', 'pricewise-calculator-pro'); ?>
                                        </label>
                                        <div class="pwcp-input-wrapper">
                                            <input type="text" 
                                                   id="calculator_name" 
                                                   name="calculator_name" 
                                                   value="<?php echo $is_edit && isset($calculator->name) ? esc_attr((string) $calculator->name) : ''; ?>" 
                                                   class="pwcp-input-modern" 
                                                   required
                                                   placeholder="<?php esc_attr_e('e.g., Mortgage Calculator, BMI Calculator', 'pricewise-calculator-pro'); ?>">
                                            <div class="pwcp-input-highlight"></div>
                                        </div>
                                        <div class="pwcp-field-hint">
                                            <?php esc_html_e('A clear, descriptive name that users will see', 'pricewise-calculator-pro'); ?>
                                        </div>
                                    </div>
                                    
                                    <div class="pwcp-field-group">
                                        <label for="calculator_description" class="pwcp-field-label">
                                            <?php esc_html_e('Description', 'pricewise-calculator-pro'); ?>
                                        </label>
                                        <div class="pwcp-textarea-wrapper">
                                            <textarea id="calculator_description" 
                                                      name="calculator_description" 
                                                      rows="4" 
                                                      class="pwcp-textarea-modern"
                                                      placeholder="<?php esc_attr_e('Brief description of what this calculator does...', 'pricewise-calculator-pro'); ?>"><?php echo $is_edit && isset($calculator->description) ? esc_textarea((string) $calculator->description) : ''; ?></textarea>
                                            <div class="pwcp-textarea-highlight"></div>
                                        </div>
                                        <div class="pwcp-field-hint">
                                            <?php esc_html_e('Optional description to help users understand the calculator\'s purpose', 'pricewise-calculator-pro'); ?>
                                        </div>
                                    </div>
                                </div>

                                <div class="pwcp-form-section">
                                    <div class="pwcp-field-group">
                                        <label class="pwcp-field-label"><?php esc_html_e('Assignment Strategy', 'pricewise-calculator-pro'); ?></label>
                                        <div class="pwcp-field-hint pwcp-mb-md">
                                            <?php esc_html_e('Choose how and where this calculator will be displayed', 'pricewise-calculator-pro'); ?>
                                        </div>
                                        
                                       <div class="pwcp-strategy-grid">
    <!-- Global Assignment (FREE) -->
    <div class="pwcp-strategy-card">
        <input type="radio" 
               id="assignment_global" 
               name="assignment_type" 
               value="global" 
               class="pwcp-strategy-input"
               checked>
        <label for="assignment_global" class="pwcp-strategy-label">
            <div class="pwcp-strategy-icon pwcp-icon-global"></div>
            <div class="pwcp-strategy-content">
                <h4 class="pwcp-strategy-title"><?php esc_html_e('Global Calculator', 'pricewise-calculator-pro'); ?></h4>
                <p class="pwcp-strategy-desc"><?php esc_html_e('Available everywhere on your site via shortcode', 'pricewise-calculator-pro'); ?></p>
            </div>
            <div class="pwcp-strategy-check"></div>
        </label>
    </div>

    <!-- Product-Specific (LOCKED) -->
    <div class="pwcp-strategy-card pwcp-strategy-locked">
        <input type="radio" 
               id="assignment_product" 
               name="assignment_type_fake" 
               value="product" 
               class="pwcp-strategy-input"
               disabled>
        <label for="assignment_product" class="pwcp-strategy-label pwcp-premium-locked" data-premium="product">
            <div class="pwcp-strategy-icon pwcp-icon-product"></div>
            <div class="pwcp-strategy-content">
                <h4 class="pwcp-strategy-title">
                    <?php esc_html_e('Product-Specific', 'pricewise-calculator-pro'); ?>
                    <span class="pwcp-lock-badge">🔒</span>
                </h4>
                <p class="pwcp-strategy-desc"><?php esc_html_e('Assign to specific products', 'pricewise-calculator-pro'); ?></p>
            </div>
            <div class="pwcp-strategy-check"></div>
        </label>
    </div>

    <!-- Category-Based (LOCKED) -->
    <div class="pwcp-strategy-card pwcp-strategy-locked">
        <input type="radio" 
               id="assignment_category" 
               name="assignment_type_fake" 
               value="category" 
               class="pwcp-strategy-input"
               disabled>
        <label for="assignment_category" class="pwcp-strategy-label pwcp-premium-locked" data-premium="category">
            <div class="pwcp-strategy-icon pwcp-icon-category"></div>
            <div class="pwcp-strategy-content">
                <h4 class="pwcp-strategy-title">
                    <?php esc_html_e('Category-Based', 'pricewise-calculator-pro'); ?>
                    <span class="pwcp-lock-badge">🔒</span>
                </h4>
                <p class="pwcp-strategy-desc"><?php esc_html_e('Assign to product categories', 'pricewise-calculator-pro'); ?></p>
            </div>
            <div class="pwcp-strategy-check"></div>
        </label>
    </div>
</div>

<!-- Demo category selection (hidden by default) -->
<div id="pwcp-category-demo" style="display: none; margin-top: 20px; position: relative;">
    <div style="padding: 20px; background: #f5f5f5; border: 2px dashed #ddd; border-radius: 8px; position: relative;">
        <div class="pwcp-category-grid">
            <div class="pwcp-category-option">
                <input type="checkbox" disabled>
                <label class="pwcp-category-label">
                    <?php esc_html_e('Electronics', 'pricewise-calculator-pro'); ?>
                    <small><?php esc_html_e('15 products', 'pricewise-calculator-pro'); ?></small>
                </label>
            </div>
            <div class="pwcp-category-option">
                <input type="checkbox" disabled>
                <label class="pwcp-category-label">
                    <?php esc_html_e('Clothing', 'pricewise-calculator-pro'); ?>
                    <small><?php esc_html_e('28 products', 'pricewise-calculator-pro'); ?></small>
                </label>
            </div>
            <div class="pwcp-category-option">
                <input type="checkbox" disabled>
                <label class="pwcp-category-label">
                    <?php esc_html_e('Home & Garden', 'pricewise-calculator-pro'); ?>
                    <small><?php esc_html_e('42 products', 'pricewise-calculator-pro'); ?></small>
                </label>
            </div>
        </div>
        <div style="position: absolute; top: 0; left: 0; right: 0; bottom: 0; background: rgba(255,255,255,0.9); display: flex; align-items: center; justify-content: center; border-radius: 8px; cursor: pointer;" class="pwcp-unlock-overlay">
            <button type="button" class="pwcp-btn pwcp-btn-primary">
                🔒 <?php esc_html_e('Unlock Premium', 'pricewise-calculator-pro'); ?>
            </button>
        </div>
    </div>
</div>
                                    </div>
                                    
                                    <div id="pwcp-category-assignment-wrapper" class="pwcp-field-group pwcp-hidden-initially">
                                        <label class="pwcp-field-label"><?php esc_html_e('Select Categories', 'pricewise-calculator-pro'); ?></label>
                                        <div class="pwcp-field-hint pwcp-mb-md">
                                            <?php esc_html_e('Choose which product categories this calculator will be available for', 'pricewise-calculator-pro'); ?>
                                        </div>
                                        
                                        <div id="pwcp-category-selection-container">
                                            <?php if (class_exists('WooCommerce')): ?>
                                                <div id="pwcp-categories-loading" class="pwcp-info-card">
                                                    <div class="pwcp-info-icon">
                                                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                            <path d="M12 2v4M12 18v4M4.93 4.93l2.83 2.83M16.24 16.24l2.83 2.83M2 12h4M18 12h4M4.93 19.07l2.83-2.83M16.24 7.76l2.83-2.83"/>
                                                        </svg>
                                                    </div>
                                                    <div class="pwcp-info-content">
                                                        <h4><?php esc_html_e('Loading Categories', 'pricewise-calculator-pro'); ?></h4>
                                                        <p><?php esc_html_e('Please wait while we load your WooCommerce categories...', 'pricewise-calculator-pro'); ?></p>
                                                    </div>
                                                </div>
                                                
                                                <div id="pwcp-categories-grid" class="pwcp-categories-grid pwcp-hidden"></div>
                                                
                                                <div id="pwcp-categories-error" class="pwcp-info-card pwcp-hidden">
                                                    <div class="pwcp-info-icon pwcp-error-icon">
                                                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                            <circle cx="12" cy="12" r="10"/>
                                                            <line x1="15" y1="9" x2="9" y2="15"/>
                                                            <line x1="9" y1="9" x2="15" y2="15"/>
                                                        </svg>
                                                    </div>
                                                    <div class="pwcp-info-content">
                                                        <h4><?php esc_html_e('Error Loading Categories', 'pricewise-calculator-pro'); ?></h4>
                                                        <p><?php esc_html_e('Unable to load product categories. Please refresh the page and try again.', 'pricewise-calculator-pro'); ?></p>
                                                    </div>
                                                </div>
                                            <?php else: ?>
                                                <div class="pwcp-info-card">
                                                    <div class="pwcp-info-icon pwcp-warning-icon">
                                                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                            <path d="M12 9v3m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"/>
                                                        </svg>
                                                    </div>
                                                    <div class="pwcp-info-content">
                                                        <h4><?php esc_html_e('WooCommerce Required', 'pricewise-calculator-pro'); ?></h4>
                                                        <p><?php esc_html_e('WooCommerce must be installed and activated to use category assignments.', 'pricewise-calculator-pro'); ?></p>
                                                    </div>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                        <input type="hidden" name="assigned_categories" id="assigned_categories" value="<?php echo esc_attr(wp_json_encode($is_edit && $this->category_assignment ? $this->category_assignment->pwcp_get_assigned_categories($calculator_id) : array())); ?>">
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="pwcp-tab-panel <?php echo ($active_tab === 'formula') ? 'pwcp-tab-active' : ''; ?>" data-panel="formula">
                            <div class="pwcp-panel-header">
                                <h2 class="pwcp-panel-title"><?php esc_html_e('Formula Editor', 'pricewise-calculator-pro'); ?></h2>
                                <p class="pwcp-panel-description">
                                    <?php esc_html_e('Define the calculation logic using field names and mathematical operators with smart field analysis', 'pricewise-calculator-pro'); ?>
                                </p>
                                <div class="pwcp-panel-actions">
                                    <button type="button" class="pwcp-btn pwcp-btn-ghost pwcp-btn-sm" id="formula-help">
                                        <svg class="pwcp-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <circle cx="12" cy="12" r="10"/>
                                            <path d="M9.09 9a3 3 0 015.83 1c0 2-3 3-3 3"/>
                                            <line x1="12" y1="17" x2="12.01" y2="17"/>
                                        </svg>
                                        <?php esc_html_e('Help', 'pricewise-calculator-pro'); ?>
                                    </button>
                                </div>
                            </div>

                            <div class="pwcp-formula-workspace">
                                <div class="pwcp-formula-editor-container">
                                    <div class="pwcp-formula-toolbar">
                                        <div class="pwcp-toolbar-section">
                                            <span class="pwcp-toolbar-label"><?php esc_html_e('Operators:', 'pricewise-calculator-pro'); ?></span>
                                            <div class="pwcp-operator-buttons">
                                                <button type="button" class="pwcp-operator-btn" data-operator=" + " title="<?php esc_attr_e('Addition', 'pricewise-calculator-pro'); ?>">+</button>
                                                <button type="button" class="pwcp-operator-btn" data-operator=" - " title="<?php esc_attr_e('Subtraction', 'pricewise-calculator-pro'); ?>">-</button>
                                                <button type="button" class="pwcp-operator-btn" data-operator=" * " title="<?php esc_attr_e('Multiplication', 'pricewise-calculator-pro'); ?>">×</button>
                                                <button type="button" class="pwcp-operator-btn" data-operator=" / " title="<?php esc_attr_e('Division', 'pricewise-calculator-pro'); ?>">÷</button>
                                                <button type="button" class="pwcp-operator-btn" data-operator="(" title="<?php esc_attr_e('Open Parenthesis', 'pricewise-calculator-pro'); ?>">(</button>
                                                <button type="button" class="pwcp-operator-btn" data-operator=")" title="<?php esc_attr_e('Close Parenthesis', 'pricewise-calculator-pro'); ?>">)</button>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="pwcp-formula-input-container">
                                        <div class="pwcp-formula-line-numbers"></div>
                                        <textarea id="calculator_formula" 
                                                  name="calculator_formula" 
                                                  class="pwcp-formula-input"
                                                  placeholder="<?php esc_attr_e('Enter your formula here... Example: length * width * price_per_sqft', 'pricewise-calculator-pro'); ?>"
                                                  rows="10"><?php echo $is_edit && isset($calculator->formula) ? esc_textarea((string) $calculator->formula) : ''; ?></textarea>
                                    </div>
                                    
                                    <div class="pwcp-formula-status" id="formula-status">
                                        <div class="pwcp-status-indicator pwcp-status-pending">
                                            <svg class="pwcp-status-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <circle cx="12" cy="12" r="10"/>
                                                <path d="M12 6v6l4 2"/>
                                            </svg>
                                            <span><?php esc_html_e('Formula not validated', 'pricewise-calculator-pro'); ?></span>
                                        </div>
                                    </div>
                                </div>
                                <div class="pwcp-examples-section">
                                    <h4><?php esc_html_e('Formula Examples', 'pricewise-calculator-pro'); ?></h4>
                                    <div class="pwcp-examples-grid">
                                        <div class="pwcp-example-card" data-formula="length * width">
                                            <div class="pwcp-example-code">length * width</div>
                                            <p class="pwcp-example-desc"><?php esc_html_e('Area calculation', 'pricewise-calculator-pro'); ?></p>
                                        </div>
                                        <div class="pwcp-example-card" data-formula="principal * (rate / 100) * time">
                                            <div class="pwcp-example-code">principal * (rate / 100) * time</div>
                                            <p class="pwcp-example-desc"><?php esc_html_e('Simple interest', 'pricewise-calculator-pro'); ?></p>
                                        </div>
                                        <div class="pwcp-example-card" data-formula="weight / (height * height)">
                                            <div class="pwcp-example-code">weight / (height * height)</div>
                                            <p class="pwcp-example-desc"><?php esc_html_e('BMI calculation', 'pricewise-calculator-pro'); ?></p>
                                        </div>
                                        <div class="pwcp-example-card" data-formula="price * quantity * (1 + tax_rate / 100)">
                                            <div class="pwcp-example-code">price * quantity * (1 + tax_rate / 100)</div>
                                            <p class="pwcp-example-desc"><?php esc_html_e('Price with tax', 'pricewise-calculator-pro'); ?></p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="pwcp-tab-panel <?php echo ($active_tab === 'fields') ? 'pwcp-tab-active' : ''; ?>" data-panel="fields">
                            <div class="pwcp-panel-header">
                                <h2 class="pwcp-panel-title"><?php esc_html_e('Input Fields', 'pricewise-calculator-pro'); ?></h2>
                                <p class="pwcp-panel-description">
                                    <?php esc_html_e('Configure the input fields that users will interact with. Click "Add Field" to open the field editor.', 'pricewise-calculator-pro'); ?>
                                </p>
                                <div class="pwcp-panel-actions">
                                    <button type="button" class="pwcp-btn pwcp-btn-primary pwcp-add-field <?php echo !$calculator_id ? 'pwcp-hidden' : ''; ?>">
                                        <svg class="pwcp-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <line x1="12" y1="5" x2="12" y2="19"/>
                                            <line x1="5" y1="12" x2="19" y2="12"/>
                                        </svg>
                                        <?php esc_html_e('Add Field', 'pricewise-calculator-pro'); ?>
                                    </button>
                                </div>
                            </div>
                            
                            <div class="pwcp-fields-workspace">
                                <div id="pwcp-fields-container" class="pwcp-fields-container">
                                    <?php if (!$calculator_id): ?>
                                        <div class="pwcp-info-card" id="save-calculator-first-message">
                                            <div class="pwcp-info-icon">
                                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                    <circle cx="12" cy="12" r="10"/>
                                                    <path d="M12 16v-4"/>
                                                    <path d="M12 8h.01"/>
                                                </svg>
                                            </div>
                                            <div class="pwcp-info-content">
                                                <h4><?php esc_html_e('Save Calculator First', 'pricewise-calculator-pro'); ?></h4>
                                                <p><?php esc_html_e('Please save the calculator in the Basic Information tab to start adding input fields.', 'pricewise-calculator-pro'); ?></p>
                                            </div>
                                        </div>
                                    <?php elseif (empty($fields)): ?>
                                        <div class="pwcp-empty-state">
                                            <div class="pwcp-empty-illustration">
                                                <svg class="pwcp-empty-icon" viewBox="0 0 200 150">
                                                    <defs>
                                                        <linearGradient id="field-gradient" x1="0%" y1="0%" x2="100%" y2="100%">
                                                            <stop offset="0%" style="stop-color:#6366f1;stop-opacity:0.1"/>
                                                            <stop offset="100%" style="stop-color:#a855f7;stop-opacity:0.1"/>
                                                        </linearGradient>
                                                    </defs>
                                                    <rect x="30" y="30" width="140" height="20" rx="4" fill="url(#field-gradient)" stroke="#6366f1" stroke-width="1"/>
                                                    <rect x="30" y="60" width="100" height="20" rx="4" fill="url(#field-gradient)" stroke="#8b5cf6" stroke-width="1"/>
                                                    <rect x="30" y="90" width="120" height="20" rx="4" fill="url(#field-gradient)" stroke="#a855f7" stroke-width="1"/>
                                                    <circle cx="180" cy="40" r="8" fill="#6366f1" opacity="0.3"/>
                                                    <circle cx="150" cy="70" r="6" fill="#8b5cf6" opacity="0.3"/>
                                                    <circle cx="170" cy="100" r="7" fill="#a855f7" opacity="0.3"/>
                                                </svg>
                                            </div>
                                            <h3><?php esc_html_e('No Fields Yet', 'pricewise-calculator-pro'); ?></h3>
                                            <p><?php esc_html_e('Add input fields to collect data from users for your calculations.', 'pricewise-calculator-pro'); ?></p>
                                            <button type="button" class="pwcp-btn pwcp-btn-primary pwcp-btn-icon pwcp-add-field">
                                                <svg class="pwcp-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                    <line x1="12" y1="5" x2="12" y2="19"/>
                                                    <line x1="5" y1="12" x2="19" y2="12"/>
                                                </svg>
                                                <?php esc_html_e('Add Your First Field', 'pricewise-calculator-pro'); ?>
                                            </button>
                                        </div>
                                    <?php else: ?>
                                        <?php foreach ($fields as $field) : ?>
                                            <?php $this->pwcp_render_field_editor($field); ?>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>

                        <div class="pwcp-tab-panel <?php echo ($active_tab === 'settings') ? 'pwcp-tab-active' : ''; ?>" data-panel="settings">
                            <div class="pwcp-panel-header">
                                <h2 class="pwcp-panel-title"><?php esc_html_e('Calculator Settings', 'pricewise-calculator-pro'); ?></h2>
                                <p class="pwcp-panel-description">
                                    <?php esc_html_e('Configure publication settings for your calculator', 'pricewise-calculator-pro'); ?>
                                </p>
                            </div>

                            <div class="pwcp-settings-grid">
                                <div class="pwcp-settings-section">
                                    <h3><?php esc_html_e('Publication Settings', 'pricewise-calculator-pro'); ?></h3>
                                    
                                    <div class="pwcp-field-group">
                                        <label for="calculator_status" class="pwcp-field-label"><?php esc_html_e('Status', 'pricewise-calculator-pro'); ?></label>
                                        <div class="pwcp-select-wrapper">
                                            <select name="calculator_status" id="calculator_status" class="pwcp-select-modern">
                                                <option value="active" <?php selected($is_edit && isset($calculator->status) ? $calculator->status : 'active', 'active'); ?>>
                                                    <?php esc_html_e('Active', 'pricewise-calculator-pro'); ?>
                                                </option>
                                                <option value="inactive" <?php selected($is_edit && isset($calculator->status) ? $calculator->status : '', 'inactive'); ?>>
                                                    <?php esc_html_e('Inactive', 'pricewise-calculator-pro'); ?>
                                                </option>
                                            </select>
                                            <div class="pwcp-select-highlight"></div>
                                        </div>
                                        <div class="pwcp-field-hint">
                                            <?php esc_html_e('Only active calculators will be displayed to users', 'pricewise-calculator-pro'); ?>
                                        </div>
                                    </div>

                                    <div class="pwcp-field-group">
                                        <label for="calculator_priority" class="pwcp-field-label"><?php esc_html_e('Priority', 'pricewise-calculator-pro'); ?></label>
                                        <div class="pwcp-input-wrapper">
                                            <input type="number" 
                                                   name="calculator_priority" 
                                                   id="calculator_priority" 
                                                   value="<?php echo esc_attr((string) ($is_edit && isset($calculator->priority) ? absint($calculator->priority) : 1)); ?>" 
                                                   min="1" 
                                                   max="100" 
                                                   class="pwcp-input-modern">
                                            <div class="pwcp-input-highlight"></div>
                                        </div>
                                        <div class="pwcp-field-hint">
                                            <?php esc_html_e('Higher numbers = higher priority when multiple calculators are available', 'pricewise-calculator-pro'); ?>
                                        </div>
                                    </div>

                                    <div class="pwcp-field-group">
                                        <div class="pwcp-checkbox-group">
                                            <label class="pwcp-checkbox-label">
                                                <input type="checkbox" 
                                                       id="hide_quantity_selector" 
                                                       name="hide_quantity_selector" 
                                                       value="1"
                                                       class="pwcp-checkbox-input"
                                                       <?php checked($is_edit && isset($calculator->hide_quantity_selector) ? $calculator->hide_quantity_selector : 0, 1); ?>>
                                                <div class="pwcp-checkbox-custom"></div>
                                                <span class="pwcp-checkbox-text"><?php esc_html_e('Hide Quantity Selector', 'pricewise-calculator-pro'); ?></span>
                                            </label>
                                        </div>
                                        <div class="pwcp-field-hint">
                                            <?php esc_html_e('Hide the quantity selector on product pages where this calculator is active', 'pricewise-calculator-pro'); ?>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>

            <div id="pwcp-field-modal" class="pwcp-modal pwcp-hidden">
                <div class="pwcp-modal-overlay"></div>
                <div class="pwcp-modal-container">
                    <div class="pwcp-modal-header">
                        <h3 class="pwcp-modal-title"><?php esc_html_e('Add New Field', 'pricewise-calculator-pro'); ?></h3>
                        <button type="button" class="pwcp-modal-close" id="pwcp-modal-close">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <line x1="18" y1="6" x2="6" y2="18"/>
                                <line x1="6" y1="6" x2="18" y2="18"/>
                            </svg>
                        </button>
                    </div>
                    
                    <div class="pwcp-modal-content">
                        <form id="pwcp-field-modal-form">
                            <div class="pwcp-modal-grid">
                                <div class="pwcp-modal-section">
                                    <h4><?php esc_html_e('Basic Settings', 'pricewise-calculator-pro'); ?></h4>
                                    
                                    <div class="pwcp-field-group">
                                        <label class="pwcp-field-label pwcp-required">
                                            <?php esc_html_e('Field Label', 'pricewise-calculator-pro'); ?>
                                        </label>
                                        <div class="pwcp-input-wrapper">
                                            <input type="text" 
                                                   id="modal-field-label" 
                                                   class="pwcp-input-modern" 
                                                   placeholder="<?php esc_attr_e('e.g., Product Price, Quantity', 'pricewise-calculator-pro'); ?>"
                                                   required>
                                            <div class="pwcp-input-highlight"></div>
                                        </div>
                                    </div>
                                    
                                    <div class="pwcp-field-group">
                                        <label class="pwcp-field-label pwcp-required">
                                            <?php esc_html_e('Field Key', 'pricewise-calculator-pro'); ?>
                                        </label>
                                        <div class="pwcp-input-wrapper">
                                            <input type="text" 
                                                   id="modal-field-key" 
                                                   class="pwcp-input-modern" 
                                                   placeholder="<?php esc_attr_e('e.g., product_price, quantity', 'pricewise-calculator-pro'); ?>"
                                                   required>
                                            <div class="pwcp-input-highlight"></div>
                                        </div>
                                        <div class="pwcp-field-hint">
                                            <?php esc_html_e('Used in formulas (lowercase, underscores only)', 'pricewise-calculator-pro'); ?>
                                        </div>
                                    </div>
                                    
        <div class="pwcp-field-group">
    <label class="pwcp-field-label">
        <?php esc_html_e('Field Type', 'pricewise-calculator-pro'); ?>
    </label>
    <div class="pwcp-select-wrapper">
     <select id="modal-field-type" class="pwcp-select-modern">
    <option value="number" selected><?php esc_html_e('Number (Quantity)', 'pricewise-calculator-pro'); ?></option>
    <option value="text" disabled data-premium-field="text">🔒 <?php esc_html_e('Text - Premium', 'pricewise-calculator-pro'); ?></option>
    <option value="dropdown" disabled data-premium-field="dropdown">🔒 <?php esc_html_e('Dropdown - Premium', 'pricewise-calculator-pro'); ?></option>
    <option value="radio" disabled data-premium-field="radio">🔒 <?php esc_html_e('Radio - Premium', 'pricewise-calculator-pro'); ?></option>
    <option value="checkbox" disabled data-premium-field="checkbox">🔒 <?php esc_html_e('Checkbox - Premium', 'pricewise-calculator-pro'); ?></option>
    <option value="slider" disabled data-premium-field="slider">🔒 <?php esc_html_e('Slider - Premium', 'pricewise-calculator-pro'); ?></option>
    <option value="file" disabled data-premium-field="file">🔒 <?php esc_html_e('File Upload - Premium', 'pricewise-calculator-pro'); ?></option>
</select>
        <div class="pwcp-select-highlight"></div>
    </div>
    <div class="pwcp-field-hint">
        <?php esc_html_e('Select the type of input field', 'pricewise-calculator-pro'); ?>
    </div>
</div>
                                    
                                    <div class="pwcp-field-group">
                                        <label class="pwcp-field-label">
                                            <?php esc_html_e('Default Value', 'pricewise-calculator-pro'); ?>
                                        </label>
                                        <div class="pwcp-input-wrapper">
                                            <input type="text" 
                                                   id="modal-field-default" 
                                                   class="pwcp-input-modern" 
                                                   placeholder="<?php esc_attr_e('Optional default value', 'pricewise-calculator-pro'); ?>">
                                            <div class="pwcp-input-highlight"></div>
                                        </div>
                                    </div>
                                    
                                    <div class="pwcp-checkbox-group">
                                        <label class="pwcp-checkbox-label">
                                            <input type="checkbox" id="modal-field-required" class="pwcp-checkbox-input">
                                            <div class="pwcp-checkbox-custom"></div>
                                            <span class="pwcp-checkbox-text"><?php esc_html_e('Required Field', 'pricewise-calculator-pro'); ?></span>
                                        </label>
                                    </div>
                                </div>
                                
                                <div class="pwcp-modal-section">
                                    <h4><?php esc_html_e('Field Options', 'pricewise-calculator-pro'); ?></h4>
                                    
                                    <div id="modal-number-options" class="pwcp-field-options">
                                        <div class="pwcp-field-group">
                                            <label class="pwcp-field-label"><?php esc_html_e('Min Value', 'pricewise-calculator-pro'); ?></label>
                                            <div class="pwcp-input-wrapper">
                                                <input type="number" id="modal-field-min" class="pwcp-input-modern" value="0">
                                                <div class="pwcp-input-highlight"></div>
                                            </div>
                                        </div>
                                        <div class="pwcp-field-group">
                                            <label class="pwcp-field-label"><?php esc_html_e('Max Value', 'pricewise-calculator-pro'); ?></label>
                                            <div class="pwcp-input-wrapper">
                                                <input type="number" id="modal-field-max" class="pwcp-input-modern" value="1000">
                                                <div class="pwcp-input-highlight"></div>
                                            </div>
                                        </div>
                                        <div class="pwcp-field-group">
                                            <label class="pwcp-field-label"><?php esc_html_e('Step', 'pricewise-calculator-pro'); ?></label>
                                            <div class="pwcp-input-wrapper">
                                                <input type="number" id="modal-field-step" class="pwcp-input-modern" value="1" step="0.1">
                                                <div class="pwcp-input-highlight"></div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div id="modal-select-options" class="pwcp-field-options pwcp-hidden">
                                        <div class="pwcp-field-group">
                                            <label class="pwcp-field-label"><?php esc_html_e('Options (one per line)', 'pricewise-calculator-pro'); ?></label>
                                            <div class="pwcp-textarea-wrapper">
                                                <textarea id="modal-field-options" 
                                                          rows="6" 
                                                          class="pwcp-textarea-modern"
                                                          placeholder="<?php esc_attr_e('Small|10', 'pricewise-calculator-pro'); ?>&#10;<?php esc_attr_e('Medium|20', 'pricewise-calculator-pro'); ?>&#10;<?php esc_attr_e('Large|30', 'pricewise-calculator-pro'); ?>"></textarea>
                                                <div class="pwcp-textarea-highlight"></div>
                                            </div>
                                            <div class="pwcp-field-hint"><?php esc_html_e('Format: Label|Value (e.g., Small|10)', 'pricewise-calculator-pro'); ?></div>
                                        </div>
                                    </div>

                                 <div id="modal-file-options" class="pwcp-field-options pwcp-hidden">
    <div class="pwcp-field-group">
        <label class="pwcp-field-label"><?php esc_html_e('Allowed File Types', 'pricewise-calculator-pro'); ?></label>
        <div class="pwcp-select-wrapper">
            <select id="modal-field-allowed-types" class="pwcp-select-modern">
                <option value="image/*"><?php esc_html_e('Images Only (JPG, PNG, GIF)', 'pricewise-calculator-pro'); ?></option>
                <option value=".pdf"><?php esc_html_e('PDF Only', 'pricewise-calculator-pro'); ?></option>
                <option value=".pdf,.doc,.docx"><?php esc_html_e('PDF and Documents', 'pricewise-calculator-pro'); ?></option>
                <option value="image/*,.pdf"><?php esc_html_e('Images and PDF', 'pricewise-calculator-pro'); ?></option>
                <option value="*"><?php esc_html_e('All Files', 'pricewise-calculator-pro'); ?></option>
            </select>
            <div class="pwcp-select-highlight"></div>
        </div>
    </div>
    <div class="pwcp-field-group">
        <label class="pwcp-field-label"><?php esc_html_e('Maximum File Size (MB)', 'pricewise-calculator-pro'); ?></label>
        <div class="pwcp-input-wrapper">
            <input type="number" id="modal-field-max-size" class="pwcp-input-modern" value="5" min="1" max="50">
            <div class="pwcp-input-highlight"></div>
        </div>
        <div class="pwcp-field-hint"><?php esc_html_e('Maximum file size in megabytes (1-50 MB)', 'pricewise-calculator-pro'); ?></div>
    </div>
</div>

                                </div>
                            </div>
                        </form>
                    </div>
                    
                    <div class="pwcp-modal-footer">
                        <button type="button" class="pwcp-btn pwcp-btn-outline" id="pwcp-modal-cancel">
                            <?php esc_html_e('Cancel', 'pricewise-calculator-pro'); ?>
                        </button>
                        <button type="button" class="pwcp-btn pwcp-btn-primary" id="pwcp-modal-save">
                            <svg class="pwcp-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <line x1="12" y1="5" x2="12" y2="19"/>
                                <line x1="5" y1="12" x2="19" y2="12"/>
                            </svg>
                            <?php esc_html_e('Add Field', 'pricewise-calculator-pro'); ?>
                        </button>
                    </div>
                </div>
            </div>

            <div id="pwcp-formula-help-modal" class="pwcp-modal pwcp-formula-help-modal pwcp-hidden">
                <div class="pwcp-modal-overlay"></div>
                <div class="pwcp-modal-container">
                    <div class="pwcp-modal-header">
                        <h3 class="pwcp-modal-title"><?php esc_html_e('Formula Editor Help', 'pricewise-calculator-pro'); ?></h3>
                        <button type="button" class="pwcp-modal-close" id="pwcp-formula-help-close">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <line x1="18" y1="6" x2="6" y2="18"/>
                                <line x1="6" y1="6" x2="18" y2="18"/>
                            </svg>
                        </button>
                    </div>
                    
                    <div class="pwcp-modal-content">
                        <div class="pwcp-modal-grid">
                            <div class="pwcp-modal-section">
                                <h4><?php esc_html_e('Basic Formula Syntax', 'pricewise-calculator-pro'); ?></h4>
                                
                                <div class="pwcp-field-group">
                                    <label class="pwcp-field-label"><?php esc_html_e('Field References', 'pricewise-calculator-pro'); ?></label>
                                    <div class="pwcp-field-hint">
                                        <?php esc_html_e('Use exact field keys from your Fields tab:', 'pricewise-calculator-pro'); ?><br>
                                        <code>price</code>, <code>quantity</code>, <code>tax_rate</code><br>
                                        <?php esc_html_e('Must match exactly - case sensitive!', 'pricewise-calculator-pro'); ?>
                                    </div>
                                </div>
                                
                                <div class="pwcp-field-group">
                                    <label class="pwcp-field-label"><?php esc_html_e('Mathematical Operators', 'pricewise-calculator-pro'); ?></label>
                                    <div class="pwcp-field-hint">
                                        <code>+</code> <?php esc_html_e('Addition', 'pricewise-calculator-pro'); ?> &nbsp;&nbsp;
                                        <code>-</code> <?php esc_html_e('Subtraction', 'pricewise-calculator-pro'); ?><br>
                                        <code>*</code> <?php esc_html_e('Multiplication', 'pricewise-calculator-pro'); ?> &nbsp;&nbsp;
                                        <code>/</code> <?php esc_html_e('Division', 'pricewise-calculator-pro'); ?><br>
                                        <code>( )</code> <?php esc_html_e('Parentheses for grouping', 'pricewise-calculator-pro'); ?>
                                    </div>
                                </div>
                                
                                <div class="pwcp-field-group">
                                    <label class="pwcp-field-label"><?php esc_html_e('Numbers & Decimals', 'pricewise-calculator-pro'); ?></label>
                                    <div class="pwcp-field-hint">
                                        <?php esc_html_e('Whole numbers:', 'pricewise-calculator-pro'); ?> <code>10</code>, <code>100</code><br>
                                        <?php esc_html_e('Decimals:', 'pricewise-calculator-pro'); ?> <code>1.5</code>, <code>0.25</code>, <code>12.99</code><br>
                                        <?php esc_html_e('Percentages:', 'pricewise-calculator-pro'); ?> <code>price * 1.15</code> <?php esc_html_e('(15% markup)', 'pricewise-calculator-pro'); ?>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="pwcp-modal-section">
                                <h4><?php esc_html_e('Common Formula Examples', 'pricewise-calculator-pro'); ?></h4>
                                
                                <div class="pwcp-field-group">
                                    <label class="pwcp-field-label"><?php esc_html_e('Basic Calculations', 'pricewise-calculator-pro'); ?></label>
                                    <div class="pwcp-field-hint">
                                        <strong><?php esc_html_e('Simple total:', 'pricewise-calculator-pro'); ?></strong><br>
                                        <code>price * quantity</code><br><br>
                                        <strong><?php esc_html_e('With tax:', 'pricewise-calculator-pro'); ?></strong><br>
                                        <code>price * quantity * (1 + tax_rate / 100)</code><br><br>
                                        <strong><?php esc_html_e('Area pricing:', 'pricewise-calculator-pro'); ?></strong><br>
                                        <code>length * width * price_per_sqm</code>
                                    </div>
                                </div>
                                
                                <div class="pwcp-field-group">
                                    <label class="pwcp-field-label"><?php esc_html_e('Advanced Examples', 'pricewise-calculator-pro'); ?></label>
                                    <div class="pwcp-field-hint">
                                        <strong><?php esc_html_e('Volume calculation:', 'pricewise-calculator-pro'); ?></strong><br>
                                        <code>length * width * height * rate</code><br><br>
                                        <strong><?php esc_html_e('Discount pricing:', 'pricewise-calculator-pro'); ?></strong><br>
                                        <code>price * (1 - discount_percent / 100)</code><br><br>
                                        <strong><?php esc_html_e('Complex formula:', 'pricewise-calculator-pro'); ?></strong><br>
                                        <code>(base_price + extras) * quantity</code>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="pwcp-modal-grid">
                            <div class="pwcp-modal-section">
                                <h4><?php esc_html_e('What\'s NOT Supported', 'pricewise-calculator-pro'); ?></h4>
                                
                                <div class="pwcp-field-group">
                                    <label class="pwcp-field-label"><?php esc_html_e('Mathematical Functions', 'pricewise-calculator-pro'); ?></label>
                                    <div class="pwcp-field-hint">
                                        <?php esc_html_e('Advanced functions are not supported:', 'pricewise-calculator-pro'); ?><br>
                                        <code>sin()</code>, <code>cos()</code>, <code>sqrt()</code><br>
                                        <code>log()</code>, <code>exp()</code>, <code>pow()</code>, <code>abs()</code>
                                    </div>
                                </div>
                                
                                <div class="pwcp-field-group">
                                    <label class="pwcp-field-label"><?php esc_html_e('Conditional Logic', 'pricewise-calculator-pro'); ?></label>
                                    <div class="pwcp-field-hint">
                                        <?php esc_html_e('No if/then statements:', 'pricewise-calculator-pro'); ?><br>
                                        <code>if (price > 100) then ...</code> ❌<br>
                                        <code>price > 50 ? value1 : value2</code> ❌<br>
                                        <?php esc_html_e('Use separate calculators for conditions.', 'pricewise-calculator-pro'); ?>
                                    </div>
                                </div>
                                
                                <div class="pwcp-field-group">
                                    <label class="pwcp-field-label"><?php esc_html_e('Text Operations', 'pricewise-calculator-pro'); ?></label>
                                    <div class="pwcp-field-hint">
                                        <?php esc_html_e('No text concatenation:', 'pricewise-calculator-pro'); ?><br>
                                        <code>"Total: " + price</code> ❌<br>
                                        <?php esc_html_e('Only numerical calculations supported.', 'pricewise-calculator-pro'); ?>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="pwcp-modal-section">
                                <h4><?php esc_html_e('Common Mistakes', 'pricewise-calculator-pro'); ?></h4>
                                
                                <div class="pwcp-field-group">
                                    <label class="pwcp-field-label"><?php esc_html_e('Field Name Errors', 'pricewise-calculator-pro'); ?></label>
                                    <div class="pwcp-field-hint">
                                        <strong><?php esc_html_e('Wrong:', 'pricewise-calculator-pro'); ?></strong> <code>Product Price</code> <?php esc_html_e('(spaces)', 'pricewise-calculator-pro'); ?><br>
                                        <strong><?php esc_html_e('Right:', 'pricewise-calculator-pro'); ?></strong> <code>product_price</code><br><br>
                                        <strong><?php esc_html_e('Wrong:', 'pricewise-calculator-pro'); ?></strong> <code>Price</code> <?php esc_html_e('(wrong case)', 'pricewise-calculator-pro'); ?><br>
                                        <strong><?php esc_html_e('Right:', 'pricewise-calculator-pro'); ?></strong> <code>price</code> <?php esc_html_e('(exact match)', 'pricewise-calculator-pro'); ?>
                                    </div>
                                </div>
                                
                                <div class="pwcp-field-group">
                                    <label class="pwcp-field-label"><?php esc_html_e('Syntax Errors', 'pricewise-calculator-pro'); ?></label>
                                    <div class="pwcp-field-hint">
                                        <strong><?php esc_html_e('Wrong:', 'pricewise-calculator-pro'); ?></strong> <code>price x quantity</code><br>
                                        <strong><?php esc_html_e('Right:', 'pricewise-calculator-pro'); ?></strong> <code>price * quantity</code><br><br>
                                        <strong><?php esc_html_e('Wrong:', 'pricewise-calculator-pro'); ?></strong> <code>((price + tax)</code><br>
                                        <strong><?php esc_html_e('Right:', 'pricewise-calculator-pro'); ?></strong> <code>(price + tax)</code>
                                    </div>
                                </div>
                                
                                <div class="pwcp-field-group">
                                    <label class="pwcp-field-label"><?php esc_html_e('Missing Fields', 'pricewise-calculator-pro'); ?></label>
                                    <div class="pwcp-field-hint">
                                        <?php esc_html_e('Create fields first in Fields tab, then reference them. The editor shows warnings for missing fields.', 'pricewise-calculator-pro'); ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="pwcp-modal-footer">
                        <button type="button" class="pwcp-btn pwcp-btn-primary" id="pwcp-formula-help-close-footer">
                            <svg class="pwcp-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M9 11l3 3L22 4"/>
                                <path d="M21 12v7a2 2 0 01-2 2H5a2 2 0 01-2-2V5a2 2 0 012-2h11"/>
                            </svg>
                            <?php esc_html_e('Got it, Thanks!', 'pricewise-calculator-pro'); ?>
                        </button>
                    </div>
                </div>
            </div>

            <!-- Premium Upgrade Modal -->
        <div id="pwcp-premium-upgrade-modal" class="pwcp-modal pwcp-hidden">
            <div class="pwcp-modal-overlay"></div>
            <div class="pwcp-modal-container" style="max-width: 580px;">
                <div class="pwcp-modal-header">
                    <h3 class="pwcp-modal-title">
                        <svg style="width: 20px; height: 20px; margin-right: 8px; vertical-align: middle;" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z"/>
                        </svg>
                        <?php esc_html_e('Premium Feature', 'pricewise-calculator-pro'); ?>
                    </h3>
                    <button type="button" class="pwcp-modal-close" id="pwcp-premium-modal-close">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <line x1="18" y1="6" x2="6" y2="18"/>
                            <line x1="6" y1="6" x2="18" y2="18"/>
                        </svg>
                    </button>
                </div>
                
                <div class="pwcp-modal-content" style="text-align: center; padding: 2.5rem 2rem;">
                    <div class="pwcp-premium-icon" style="width: 80px; height: 80px; background: linear-gradient(135deg, #844CA9 0%, #b88cd4 100%); border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 1.5rem; box-shadow: 0 8px 24px rgba(132, 76, 169, 0.3);">
                        <svg style="width: 40px; height: 40px; color: white;" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 3v4M3 5h4M6 17v4m-2-2h4m5-16l2.286 6.857L21 12l-5.714 2.143L13 21l-2.286-6.857L5 12l5.714-2.143L13 3z"/>
                        </svg>
                    </div>
                    
                    <h2 id="pwcp-premium-feature-name" style="font-size: 1.625rem; font-weight: 700; color: #1f2937; margin: 0 0 0.875rem 0;">
                        <?php esc_html_e('Premium Feature', 'pricewise-calculator-pro'); ?>
                    </h2>
                    
                    <p id="pwcp-premium-feature-description" style="font-size: 0.9375rem; color: #6b7280; line-height: 1.6; margin: 0 0 2rem 0;">
                        <?php esc_html_e('This feature is available in the premium version of PriceWise Calculator Pro.', 'pricewise-calculator-pro'); ?>
                    </p>
                    
                    <div class="pwcp-premium-benefits" style="background: #f9fafb; border: 2px solid #e5e7eb; border-radius: 10px; padding: 1.5rem; margin-bottom: 2rem; text-align: left;">
                        <h4 style="font-size: 0.9375rem; font-weight: 700; color: #374151; margin: 0 0 1rem 0;">
                            ✨ <?php esc_html_e('Premium Benefits', 'pricewise-calculator-pro'); ?>
                        </h4>
                        <ul style="margin: 0; padding-left: 1.5rem; font-size: 0.875rem; color: #4b5563; line-height: 2;">
                            <li><strong><?php esc_html_e('Product-Specific Calculators', 'pricewise-calculator-pro'); ?></strong> - <?php esc_html_e('Assign to individual products', 'pricewise-calculator-pro'); ?></li>
                            <li><strong><?php esc_html_e('Category-Based Assignment', 'pricewise-calculator-pro'); ?></strong> - <?php esc_html_e('Automatic category-wide calculators', 'pricewise-calculator-pro'); ?></li>
                            <li><strong><?php esc_html_e('Advanced Field Types', 'pricewise-calculator-pro'); ?></strong> - <?php esc_html_e('Dropdown, radio, checkbox, and more', 'pricewise-calculator-pro'); ?></li>
                            <li><strong><?php esc_html_e('Priority Support', 'pricewise-calculator-pro'); ?></strong> - <?php esc_html_e('Get help when you need it', 'pricewise-calculator-pro'); ?></li>
                            <li><strong><?php esc_html_e('Lifetime Updates', 'pricewise-calculator-pro'); ?></strong> - <?php esc_html_e('Always stay current', 'pricewise-calculator-pro'); ?></li>
                        </ul>
                    </div>
                    
                    <a href="https://devtonicstudios.com" target="_blank" class="pwcp-btn pwcp-btn-primary pwcp-btn-lg" style="display: inline-flex; align-items: center; gap: 0.625rem;">
                        <svg style="width: 20px; height: 20px;" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 10V3L4 14h7v7l9-11h-7z"/>
                        </svg>
                        <?php esc_html_e('Upgrade to Premium', 'pricewise-calculator-pro'); ?>
                    </a>
                    
                    <p style="font-size: 0.8125rem; color: #9ca3af; margin-top: 1.125rem; margin-bottom: 0; font-weight: 500;">
                        <?php esc_html_e('💳 One-time payment • No recurring fees', 'pricewise-calculator-pro'); ?>
                    </p>
                </div>
            </div>
        </div>

        </div>

        <script type="text/html" id="pwcp-field-template">
            <?php $this->pwcp_render_field_template(); ?>
        </script>
        <?php
    }

    /**
     * Render single field editor
     * 
     * @param object $field Field object
     * @since 1.0.0
     */
    private function pwcp_render_field_editor($field) {
        $field_id = isset($field->id) ? absint($field->id) : 0;
        $field_key = isset($field->field_key) ? sanitize_key($field->field_key) : '';
        $field_label = isset($field->field_label) ? sanitize_text_field($field->field_label) : '';
        $field_type = isset($field->field_type) ? sanitize_text_field($field->field_type) : 'text';
        $field_required = isset($field->field_required) ? (bool) $field->field_required : false;
        $field_default = isset($field->field_default) ? sanitize_text_field($field->field_default) : '';
        $field_options = isset($field->field_options) ? $field->field_options : '';
        $field_order = isset($field->field_order) ? absint($field->field_order) : 0;
        
        $validation_rules = array();
        if (isset($field->field_validation) && !empty($field->field_validation)) {
            $decoded = json_decode($field->field_validation, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                $validation_rules = $decoded;
            }
        }
        
        $field_data = array(
            'id' => $field_id,
            'label' => $field_label,
            'key' => $field_key,
            'type' => $field_type,
            'required' => $field_required ? 1 : 0,
            'default_value' => $field_default,
            'order' => $field_order,
            'options' => $field_options,
            'min' => isset($validation_rules['min']) ? $validation_rules['min'] : '',
            'max' => isset($validation_rules['max']) ? $validation_rules['max'] : '',
            'step' => isset($validation_rules['step']) ? $validation_rules['step'] : ''
        );

        ?>
        <div class="pwcp-modern-field" 
             data-field-id="<?php echo esc_attr((string) $field_id); ?>">
            <div class="pwcp-field-header">
                <div class="pwcp-field-info">
                    <div class="pwcp-field-title"><?php echo esc_html($field_label ?: __('New Field', 'pricewise-calculator-pro')); ?></div>
                    <div class="pwcp-field-meta">
                        <span class="pwcp-field-type-badge pwcp-type-<?php echo esc_attr($field_type); ?>">
                            <?php echo esc_html($this->pwcp_type_label($field_type)); ?>
                        </span>
                        <?php if ($field_required): ?>
                            <span class="pwcp-field-required-badge"><?php esc_html_e('Required', 'pricewise-calculator-pro'); ?></span>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div class="pwcp-field-actions">
                    <button type="button" class="pwcp-field-btn pwcp-toggle-field" title="<?php esc_attr_e('Edit Field', 'pricewise-calculator-pro'); ?>">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M11 4H4a2 2 0 00-2 2v14a2 2 0 002 2h14a2 2 0 002-2v-7"/>
                            <path d="M18.5 2.5a2.121 2.121 0 113 3L12 15l-4 1 1-4 9.5-9.5z"/>
                        </svg>
                    </button>
                    <button type="button" class="pwcp-field-btn pwcp-remove-field" title="<?php esc_attr_e('Remove Field', 'pricewise-calculator-pro'); ?>">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <polyline points="3,6 5,6 21,6"/>
                            <path d="M19,6v14a2,2 0 01-2,2H7a2,2 0 01-2-2V6m3,0V4a2,2 0 012-2h4a2,2 0 012,2v2"/>
                            <line x1="10" y1="11" x2="10" y2="17"/>
                            <line x1="14" y1="11" x2="14" y2="17"/>
                        </svg>
                    </button>
                </div>
            </div>
            
            <div class="pwcp-field-content pwcp-field-collapsed">
                <script type="application/json" class="pwcp-field-data-json"><?php echo wp_json_encode($field_data); ?></script>
                
                <input type="hidden" 
                       name="fields[<?php echo esc_attr((string) $field_id); ?>][order]" 
                       value="<?php echo esc_attr((string) $field_order); ?>" 
                       class="pwcp-field-order">
                <input type="hidden" 
                       name="fields[<?php echo esc_attr((string) $field_id); ?>][id]" 
                       value="<?php echo esc_attr((string) $field_id); ?>">
            </div> 
        </div>
        <?php
    }

    /**
     * Render field template for JavaScript
     * 
     * @since 1.0.0
     */
    private function pwcp_render_field_template() {
        ?>
        <div class="pwcp-modern-field" data-field-id="TEMPLATE_INDEX">
            <div class="pwcp-field-header">
                <div class="pwcp-field-info">
                    <div class="pwcp-field-title"><?php esc_html_e('New Field', 'pricewise-calculator-pro'); ?></div>
                    <div class="pwcp-field-meta">
                        <span class="pwcp-field-type-badge pwcp-type-text"><?php esc_html_e('Text', 'pricewise-calculator-pro'); ?></span>
                    </div>
                </div>
                
                <div class="pwcp-field-actions">
                    <button type="button" class="pwcp-field-btn pwcp-toggle-field" title="<?php esc_attr_e('Edit Field', 'pricewise-calculator-pro'); ?>">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M11 4H4a2 2 0 00-2 2v14a2 2 0 002 2h14a2 2 0 002-2v-7"/>
                            <path d="M18.5 2.5a2.121 2.121 0 113 3L12 15l-4 1 1-4 9.5-9.5z"/>
                        </svg>
                    </button>
                    <button type="button" class="pwcp-field-btn pwcp-remove-field" title="<?php esc_attr_e('Remove Field', 'pricewise-calculator-pro'); ?>">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <polyline points="3,6 5,6 21,6"/>
                            <path d="M19,6v14a2,2 0 01-2,2H7a2,2 0 01-2-2V6m3,0V4a2,2 0 012-2h4a2,2 0 012,2v2"/>
                            <line x1="10" y1="11" x2="10" y2="17"/>
                            <line x1="14" y1="11" x2="14" y2="17"/>
                        </svg>
                    </button>
                </div>
            </div>
            
            <div class="pwcp-field-content pwcp-field-collapsed">
                <input type="hidden" name="fields[TEMPLATE_INDEX][order]" value="0" class="pwcp-field-order">
                <input type="hidden" name="fields[TEMPLATE_INDEX][id]" value="0">
                <input type="hidden" name="fields[TEMPLATE_INDEX][temp_id]" value="">
            </div>
        </div>
        <?php
    }

    /**
     * Render add calculator page
     * 
     * @since 1.0.0
     */
    public function pwcp_render_add_page() {
        $this->pwcp_render_calculator_form();
    }

    /**
     * Render edit calculator page
     * 
     * @param int $calculator_id Calculator ID
     * @since 1.0.0
     */
    public function pwcp_render_edit_page($calculator_id) {
        $calculator = null;
        if (class_exists('PWCP_Admin_Calculator_Manager')) {
            $manager = new PWCP_Admin_Calculator_Manager();
            if (method_exists($manager, 'pwcp_get_calculator')) {
                $calculator = $manager->pwcp_get_calculator($calculator_id);
            }
        }
        
        if (!$calculator) {
            global $wpdb;
            $table = $wpdb->prefix . 'pwcp_calculators';
            $calculator = $wpdb->get_row( 
                $wpdb->prepare( 
                    "SELECT id, name, description, formula, status, assignment_type, assignment_ids, 
                     priority, hide_quantity_selector, created_at, updated_at 
                     FROM `{$table}` WHERE id = %d", 
                    $calculator_id 
                ) 
            );
        }
        
        if (!$calculator) {
            wp_die(esc_html__('Calculator not found.', 'pricewise-calculator-pro'));
        }
        
        $this->pwcp_render_calculator_form($calculator);
    }
}