<?php
/**
 * Admin Area Management
 *
 * Handles all admin-side functionality and integrations
 *
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

// Load dependencies
require_once plugin_dir_path(__FILE__) . '../includes/class-pwcp-base-ajax-handler.php';
require_once plugin_dir_path(__FILE__) . '../includes/traits/trait-pwcp-security-validation.php';
require_once plugin_dir_path(__FILE__) . '../includes/class-pwcp-input-sanitizer.php';
require_once plugin_dir_path(__FILE__) . '../includes/class-pwcp-response-factory.php';

/**
 * Admin Class
 *
 * @since 1.0.0
 */
class PWCP_Admin extends PWCP_Base_Ajax_Handler {

    use PWCP_Security_Validation_Trait;

    private $pwcp_loader;

    /**
     * Constructor
     */
    public function __construct($loader = null) {
        if ($loader) {
            $this->pwcp_loader = $loader;
        }
        $this->init_core_admin_hooks();
    }

    /**
     * Initialize core admin hooks
     */
    private function init_core_admin_hooks() {
        if (!is_admin()) {
            return;
        }

        add_action('admin_init', array($this, 'pwcp_admin_init'));
        add_action('admin_enqueue_scripts', array($this, 'pwcp_enqueue_admin_scripts'));
        add_action('admin_notices', array($this, 'pwcp_admin_notices'));

        $this->register_ajax_handlers();
        $this->register_woocommerce_hooks();
    }

    /**
     * Register AJAX handlers
     */
    private function register_ajax_handlers() {
        $ajax_handlers = array(
            'pwcp_save_calculator'      => 'pwcp_ajax_save_calculator',
            'pwcp_delete_calculator'    => 'pwcp_ajax_delete_calculator',
            'pwcp_duplicate_calculator' => 'pwcp_ajax_duplicate_calculator',
        );

        foreach ($ajax_handlers as $action => $method) {
            add_action('wp_ajax_' . $action, array($this, $method));
        }
    }

    /**
     * Register WooCommerce integration hooks
     */
    private function register_woocommerce_hooks() {
        add_action('save_post', array($this, 'pwcp_save_product_meta'));
    }

    /**
     * Admin initialization
     */
    public function pwcp_admin_init() {
        if (class_exists('PWCP_Database')) {
            PWCP_Database::pwcp_maybe_update_database();
        }
        $this->handle_admin_actions();
    }

    /**
     * Handle admin actions
     */
  private function handle_admin_actions() {
    // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce is verified via verify_calculator_nonce() method
    if (!isset($_POST['pwcp_nonce']) || !$this->verify_calculator_nonce()) {
        return;
    }

    $action = $this->get_post_data('action', 'text');
    // phpcs:enable WordPress.Security.NonceVerification.Missing

        $action_handlers = array(
            'save_calculator'      => 'handle_save_calculator',
            'delete_calculator'    => 'handle_delete_calculator',
            'duplicate_calculator' => 'handle_duplicate_calculator',
        );

        if (isset($action_handlers[$action]) && method_exists($this, $action_handlers[$action])) {
            $this->{$action_handlers[$action]}();
        }
    }

    /**
     * Enqueue admin scripts
     */
    public function pwcp_enqueue_admin_scripts($hook) {
        if (strpos($hook, 'pwcp-') === false && $hook !== 'post.php' && $hook !== 'post-new.php') {
            return;
        }

        $this->enqueue_admin_assets();
        $this->localize_admin_scripts();
    }

    /**
     * Enqueue admin assets
     */
    private function enqueue_admin_assets() {
        wp_enqueue_style(
            'pwcp-admin-styles',
            PWCP_ASSETS_URL . 'css/admin.css',
            array(),
            PWCP_PLUGIN_VERSION
        );

        wp_enqueue_script(
            'pwcp-admin-scripts',
            PWCP_ASSETS_URL . 'js/admin.js',
            array('jquery', 'jquery-ui-sortable'),
            PWCP_PLUGIN_VERSION,
            true
        );

        if (function_exists('wp_set_script_translations')) {
            wp_set_script_translations(
                'pwcp-admin-scripts',
                'pricewise-calculator-pro',
                PWCP_PLUGIN_DIR . 'languages'
            );
        }
    }

    /**
     * Localize admin scripts
     */
    private function localize_admin_scripts() {
        wp_localize_script('pwcp-admin-scripts', 'pwcp_admin', array(
            'ajax_url'    => admin_url('admin-ajax.php'),
            'nonce'       => wp_create_nonce('pwcp_admin_nonce'),
            'text_domain' => PWCP_TEXT_DOMAIN,
            'strings'     => array(
                'confirm_delete' => esc_html__('Are you sure you want to delete this calculator?', 'pricewise-calculator-pro'),
                'saving'         => esc_html__('Saving...', 'pricewise-calculator-pro'),
                'saved'          => esc_html__('Saved successfully!', 'pricewise-calculator-pro'),
                'error'          => esc_html__('An error occurred. Please try again.', 'pricewise-calculator-pro'),
            ),
        ));
    }

    /**
     * Display admin notices
     */
  public function pwcp_admin_notices() {
    // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Read-only GET parameter for displaying admin notice
    if (isset($_GET['pwcp_message'])) {
        $message_type = sanitize_text_field(wp_unslash($_GET['pwcp_message']));
        $this->display_admin_notice($message_type);
    }
    // phpcs:enable WordPress.Security.NonceVerification.Recommended

    if (!class_exists('WooCommerce')) {
        $this->display_woocommerce_notice();
    }
}

private function display_admin_notice($message_type) {
    $messages = array(
        'calculator_saved'      => array('type' => 'success', 'text' => esc_html__('Calculator saved successfully!', 'pricewise-calculator-pro')),
        'calculator_deleted'    => array('type' => 'success', 'text' => esc_html__('Calculator deleted successfully!', 'pricewise-calculator-pro')),
        'calculator_duplicated' => array('type' => 'success', 'text' => esc_html__('Calculator duplicated successfully!', 'pricewise-calculator-pro')),
        'settings_saved'        => array('type' => 'success', 'text' => esc_html__('Settings saved successfully!', 'pricewise-calculator-pro')),
        'error'                 => array('type' => 'error',   'text' => esc_html__('An error occurred. Please try again.', 'pricewise-calculator-pro')),
        'permission_error'      => array('type' => 'error',   'text' => esc_html__('You do not have permission to perform this action.', 'pricewise-calculator-pro')),
    );

if (isset($messages[$message_type])) {
    $message = $messages[$message_type];
    printf(
        '<div class="notice notice-%s is-dismissible"><p>%s</p></div>',
        esc_attr($message['type']),
        wp_kses_post($message['text'])  // Escape at point of output
    );
}
}

    /**
     * Display WooCommerce requirement notice
     */
    private function display_woocommerce_notice() {
        $current_screen = get_current_screen();

        if (!$current_screen || strpos($current_screen->id, 'pwcp-') === false) {
            return;
        }

    printf(
    '<div class="notice notice-warning"><p>%s</p></div>',
    wp_kses_post(
        sprintf(
            /* translators: %s: WooCommerce plugin name */
            __('PriceWise Calculator Pro requires %s to be installed and active for full functionality.', 'pricewise-calculator-pro'),
            '<strong>WooCommerce</strong>'
        )
    )
);
    }

    /**
     * Get active calculators
     */
private function get_active_calculators() {
    global $wpdb;

    $table = $wpdb->prefix . 'pwcp_calculators';

    return $wpdb->get_results(
        $wpdb->prepare(
            "SELECT id, name FROM `" . esc_sql($table) . "` WHERE status = %s ORDER BY name ASC",
            'active'
        )
    );
}

    /**
     * Save product meta data
     */
public function pwcp_save_product_meta($post_id) {
    // Product-specific assignment disabled in free version
    // Premium feature available at devtonicstudios.com
    return;
}
    /**
     * Update calculator assignments in database
     */
private function update_calculator_assignments($product_id, $calculator_ids) {
    // Product-specific assignment disabled in free version
    return;
}

    /**
     * Handle save calculator AJAX request
     */
  public function pwcp_ajax_save_calculator() {
    $this->verify_admin_ajax_security('edit_pwcp_calculators', 'pwcp_admin_nonce', 'nonce');

    // phpcs:disable WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput -- Security verified via verify_admin_ajax_security(), data sanitized via PWCP_Input_Sanitizer
    $calculator_data = PWCP_Input_Sanitizer::sanitize_calculator_data(
        isset($_POST['calculator']) ? wp_unslash($_POST['calculator']) : array()
    );
    // phpcs:enable WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput

        if (empty($calculator_data['name'])) {
            PWCP_Response_Factory::validation_error(__('Calculator name is required.', 'pricewise-calculator-pro'));
        }

        $allowed_status = array('active', 'inactive');
        if (!in_array($calculator_data['status'], $allowed_status, true)) {
            $calculator_data['status'] = 'active';
        }

        PWCP_Response_Factory::calculator_success('save', $calculator_data['id'], array(
            'calculator' => $calculator_data,
        ));
    }

    /**
     * Handle delete calculator AJAX request
     */
    public function pwcp_ajax_delete_calculator() {
        $this->verify_admin_ajax_security('delete_pwcp_calculators', 'pwcp_admin_nonce', 'nonce');

        $calculator_id = $this->get_calculator_id();

        PWCP_Response_Factory::calculator_success('delete', $calculator_id);
    }

    /**
     * Handle duplicate calculator AJAX request
     */
    public function pwcp_ajax_duplicate_calculator() {
        $this->verify_admin_ajax_security('edit_pwcp_calculators', 'pwcp_admin_nonce', 'nonce');

        $calculator_id = $this->get_calculator_id();

        PWCP_Response_Factory::calculator_success('duplicate', $calculator_id);
    }

    /**
     * Handle save calculator action
     */
    private function handle_save_calculator() {
        if (class_exists('PWCP_Admin_Calculator_Manager')) {
            $manager = new PWCP_Admin_Calculator_Manager();
        }
    }

    /**
     * Handle delete calculator action
     */
    private function handle_delete_calculator() {
        if (class_exists('PWCP_Admin_Calculator_Manager')) {
            $manager = new PWCP_Admin_Calculator_Manager();
        }
    }

    /**
     * Handle duplicate calculator action
     */
    private function handle_duplicate_calculator() {
        if (class_exists('PWCP_Admin_Calculator_Manager')) {
            $manager = new PWCP_Admin_Calculator_Manager();
        }
    }

    /**
     * Get admin instance
     */
    public static function pwcp_get_instance() {
        static $instance = null;

        if (null === $instance) {
            $instance = new self();
        }

        return $instance;
    }
}