<?php
/**
 * Admin Settings Page
 *
 * Manages plugin configuration options including general settings, design customization,
 * and calculation behavior through a modern AJAX-powered interface.
 * 
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

// Load dependencies
require_once PWCP_PLUGIN_DIR . 'includes/class-pwcp-base-ajax-handler.php';
require_once PWCP_PLUGIN_DIR . 'includes/traits/trait-pwcp-security-validation.php';
require_once PWCP_PLUGIN_DIR . 'includes/class-pwcp-input-sanitizer.php';

/**
 * PWCP Admin Settings Class
 * 
 * @since 1.0.0
 */
class PWCP_Admin_Settings extends PWCP_Base_Ajax_Handler {
    
    use PWCP_Security_Validation_Trait;
    
    /**
     * Enhanced color picker instance
     * 
     * @var PWCP_Enhanced_Color_Picker
     * @since 1.0.0
     */
    private $enhanced_color_picker;
    
    /**
     * Constructor
     * 
     * @since 1.0.0
     */
    public function __construct() {
        add_action('admin_init', array($this, 'pwcp_register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'pwcp_enqueue_settings_assets'));
        
        // AJAX handlers
        add_action('wp_ajax_pwcp_save_settings', array($this, 'ajax_save_settings'));
        add_action('wp_ajax_pwcp_reset_settings', array($this, 'ajax_reset_settings'));
        add_action('wp_ajax_pwcp_validate_color', array($this, 'ajax_validate_color'));
        
        // Initialize enhanced color picker
        if (class_exists('PWCP_Enhanced_Color_Picker')) {
            $this->enhanced_color_picker = new PWCP_Enhanced_Color_Picker();
        }
    }
    
    /**
     * Register settings using WordPress Settings API
     * 
     * @since 1.0.0
     */
    public function pwcp_register_settings() {
        // Register settings groups
        register_setting('pwcp_general_settings', 'pwcp_general_settings', array(
            'sanitize_callback' => array($this, 'pwcp_sanitize_general_settings')
        ));
        
        register_setting('pwcp_design_settings', 'pwcp_design_settings', array(
            'sanitize_callback' => array($this, 'pwcp_sanitize_design_settings')
        ));
        
        register_setting('pwcp_calculation_settings', 'pwcp_calculation_settings', array(
            'sanitize_callback' => array($this, 'pwcp_sanitize_calculation_settings')
        ));
    }
    
    /**
     * Render settings page
     * 
     * @since 1.0.0
     */
    public function pwcp_render() {
        if (!$this->can_manage_calculators()) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'pricewise-calculator-pro'));
        }
        
        $active_tab = isset($_GET['tab']) ? sanitize_key(wp_unslash($_GET['tab'])) : 'general';
        $valid_tabs = array('general', 'design', 'calculation');

        if (!$this->is_action_allowed($active_tab, $valid_tabs)) {
            $active_tab = 'general';
        }
        
        ?>
        <div class="wrap pwcp-modern-admin pwcp-page-settings">
            <!-- Modern Hero Header -->
            <div class="pwcp-hero-header">
                <div class="pwcp-hero-background">
                    <div class="pwcp-hero-particles"></div>
                    <div class="pwcp-hero-gradient"></div>
                </div>
                
                <div class="pwcp-hero-content">
                    <div class="pwcp-hero-left">
                        <div class="pwcp-hero-badge">
                            <span><?php esc_html_e('Configuration', 'pricewise-calculator-pro'); ?></span>
                        </div>
                        <h1 class="pwcp-hero-title">
                            <?php esc_html_e('Settings', 'pricewise-calculator-pro'); ?>
                        </h1>
                        <p class="pwcp-hero-subtitle">
                            <?php esc_html_e('Configure and customize your calculator experience with advanced settings, design options, and behavior controls.', 'pricewise-calculator-pro'); ?>
                        </p>
                    </div>
                </div>
            </div>

            <!-- Modern Container -->
            <div class="pwcp-modern-container">
                <!-- Modern Tab Navigation -->
                <div class="pwcp-tab-navigation">
                    <button type="button" class="pwcp-tab-btn <?php echo $active_tab === 'general' ? 'pwcp-tab-active' : ''; ?>" data-tab="general">
                        <svg class="pwcp-tab-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <circle cx="12" cy="12" r="3"/>
                            <path d="M19.4 15a1.65 1.65 0 00.33 1.82l.06.06a2 2 0 010 2.83 2 2 0 01-2.83 0l-.06-.06a1.65 1.65 0 00-1.82-.33 1.65 1.65 0 00-1 1.51V21a2 2 0 01-2 2 2 2 0 01-2-2v-.09A1.65 1.65 0 009 19.4a1.65 1.65 0 00-1.82.33l-.06.06a2 2 0 01-2.83 0 2 2 0 010-2.83l.06-.06a1.65 1.65 0 00.33-1.82 1.65 1.65 0 00-1.51-1H3a2 2 0 01-2-2 2 2 0 012-2h.09A1.65 1.65 0 004.6 9a1.65 1.65 0 00-.33-1.82l-.06-.06a2 2 0 010-2.83 2 2 0 012.83 0l.06.06a1.65 1.65 0 001.82.33H9a1.65 1.65 0 001-1.51V3a2 2 0 012-2 2 2 0 012 2v.09a1.65 1.65 0 001 1.51 1.65 1.65 0 001.82-.33l.06-.06a2 2 0 012.83 0 2 2 0 010 2.83l-.06.06a1.65 1.65 0 00-.33 1.82V9a1.65 1.65 0 001.51 1H21a2 2 0 012 2 2 2 0 01-2 2h-.09a1.65 1.65 0 00-1.51 1z"/>
                        </svg>
                        <?php esc_html_e('General', 'pricewise-calculator-pro'); ?>
                    </button>
                    
                    <button type="button" class="pwcp-tab-btn <?php echo $active_tab === 'design' ? 'pwcp-tab-active' : ''; ?>" data-tab="design">
                        <svg class="pwcp-tab-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/>
                        </svg>
                        <?php esc_html_e('Design', 'pricewise-calculator-pro'); ?>
                    </button>
                    
                    <button type="button" class="pwcp-tab-btn <?php echo $active_tab === 'calculation' ? 'pwcp-tab-active' : ''; ?>" data-tab="calculation">
                        <svg class="pwcp-tab-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                            <path d="M8 12h8"/>
                            <path d="M12 8v8"/>
                        </svg>
                        <?php esc_html_e('Calculation', 'pricewise-calculator-pro'); ?>
                    </button>
                </div>

                <!-- Tab Content -->
                <div class="pwcp-tab-content">
                    <div class="pwcp-settings-grid">
                        <!-- Main Settings Area -->
                        <div class="pwcp-settings-main">
                            <div class="pwcp-panel-header">
                                <h2 class="pwcp-panel-title" id="pwcp-panel-title">
                                    <?php echo esc_html($this->get_tab_title($active_tab)); ?>
                                </h2>
                                <p class="pwcp-panel-description" id="pwcp-panel-description">
                                    <?php echo esc_html($this->get_tab_description($active_tab)); ?>
                                </p>
                                <div class="pwcp-panel-actions">
                                    <button type="button" class="pwcp-btn pwcp-btn-ghost pwcp-btn-sm" id="pwcp-reset-settings" data-tab="<?php echo esc_attr($active_tab); ?>">
                                        <?php esc_html_e('Reset to Defaults', 'pricewise-calculator-pro'); ?>
                                    </button>
                                </div>
                            </div>
                            
                            <!-- Modern AJAX Form -->
                            <form id="pwcp-settings-form" class="pwcp-settings-form" data-active-tab="<?php echo esc_attr($active_tab); ?>">
                                <?php wp_nonce_field('pwcp_ajax_settings', 'pwcp_settings_nonce'); ?>
                                
                                <!-- General Settings Panel -->
                                <div class="pwcp-tab-panel <?php echo $active_tab === 'general' ? 'pwcp-tab-active' : ''; ?>" data-panel="general">
                                    <?php $this->render_general_settings(); ?>
                                </div>
                                
                                <!-- Design Settings Panel -->
                                <div class="pwcp-tab-panel <?php echo $active_tab === 'design' ? 'pwcp-tab-active' : ''; ?>" data-panel="design">
                                    <?php $this->render_design_settings(); ?>
                                </div>
                                
                                <!-- Calculation Settings Panel -->
                                <div class="pwcp-tab-panel <?php echo $active_tab === 'calculation' ? 'pwcp-tab-active' : ''; ?>" data-panel="calculation">
                                    <?php $this->render_calculation_settings(); ?>
                                </div>
                                
                                <div class="pwcp-settings-footer">
                                    <button type="submit" class="pwcp-btn pwcp-btn-primary pwcp-btn-lg" id="pwcp-save-settings">
                                        <svg class="pwcp-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <path d="M19 21H5a2 2 0 01-2-2V5a2 2 0 012-2h11l5 5v11a2 2 0 01-2 2z"/>
                                            <polyline points="17,21 17,13 7,13 7,21"/>
                                            <polyline points="7,3 7,8 15,8"/>
                                        </svg>
                                        <?php esc_html_e('Save Changes', 'pricewise-calculator-pro'); ?>
                                    </button>
                                </div>
                            </form>
                        </div>
                        
                        <!-- Sidebar -->
                        <div class="pwcp-settings-sidebar">
                            <?php $this->pwcp_render_settings_sidebar($active_tab); ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * AJAX handler for saving settings
     * 
     * @since 1.0.0
     */
    public function ajax_save_settings() {
        $this->verify_ajax_security('manage_options', 'pwcp_ajax_settings', 'pwcp_settings_nonce');
        
        $tab = isset($_POST['tab']) ? sanitize_key(wp_unslash($_POST['tab'])) : 'general';
        $valid_tabs = array('general', 'design', 'calculation');
        
        if (!$this->is_action_allowed($tab, $valid_tabs)) {
            $this->send_error_response(__('Invalid settings tab.', 'pricewise-calculator-pro'));
        }
        
        $response_data = array(
            'tab' => $tab,
            'new_nonce' => wp_create_nonce('pwcp_ajax_settings')
        );
        
        switch ($tab) {
            case 'general':
                $raw_settings = $this->get_post_data('pwcp_general_settings', 'array', array());
                if (!empty($raw_settings)) {
                    $sanitized = $this->pwcp_sanitize_general_settings($raw_settings);
                    update_option('pwcp_general_settings', $sanitized);
                    
                    $this->send_success_response(
                        __('General settings saved successfully!', 'pricewise-calculator-pro'),
                        array_merge($response_data, array('data' => $sanitized))
                    );
                }
                break;
                
            case 'design':
                $raw_settings = $this->get_post_data('pwcp_design_settings', 'array', array());
                if (!empty($raw_settings)) {
                    $sanitized = $this->pwcp_sanitize_design_settings($raw_settings);
                    update_option('pwcp_design_settings', $sanitized);
                    
                    $this->send_success_response(
                        __('Design settings saved successfully!', 'pricewise-calculator-pro'),
                        array_merge($response_data, array('data' => $sanitized))
                    );
                }
                break;
                
            case 'calculation':
                $raw_settings = $this->get_post_data('pwcp_calculation_settings', 'array', array());
                if (!empty($raw_settings)) {
                    $sanitized = $this->pwcp_sanitize_calculation_settings($raw_settings);
                    update_option('pwcp_calculation_settings', $sanitized);
                    
                    $this->send_success_response(
                        __('Calculation settings saved successfully!', 'pricewise-calculator-pro'),
                        array_merge($response_data, array('data' => $sanitized))
                    );
                }
                break;
        }
        
        $this->send_error_response(__('No settings data received.', 'pricewise-calculator-pro'));
    }
    
    /**
     * Get tab title
     *
     * @since 1.0.0
     * @param string $tab Tab identifier
     * @return string
     */
    private function get_tab_title($tab) {
        $titles = array(
            'general' => __('General Settings', 'pricewise-calculator-pro'),
            'design' => __('Design & Styling', 'pricewise-calculator-pro'),
            'calculation' => __('Calculation Settings', 'pricewise-calculator-pro')
        );
        return isset($titles[$tab]) ? $titles[$tab] : $titles['general'];
    }
    
    /**
     * Get tab description
     *
     * @since 1.0.0
     * @param string $tab Tab identifier
     * @return string
     */
    private function get_tab_description($tab) {
        $descriptions = array(
            'general' => __('Configure basic calculator behavior, text labels, and messages shown to users.', 'pricewise-calculator-pro'),
            'design' => __('Customize the visual appearance of your calculators to match your site design', 'pricewise-calculator-pro'),
            'calculation' => __('Configure how calculations are performed and displayed to users.', 'pricewise-calculator-pro')
        );
        return isset($descriptions[$tab]) ? $descriptions[$tab] : $descriptions['general'];
    }
    
    /**
     * Render general settings
     *
     * @since 1.0.0
     * @return void
     */
    private function render_general_settings() {
        $options = get_option('pwcp_general_settings', array());
        ?>
        <table class="form-table">
            <tr>
                <th scope="row"><?php esc_html_e('Calculator Position', 'pricewise-calculator-pro'); ?></th>
                <td>
                    <?php
                    $this->render_select_field(
                        'pwcp_general_settings[calculator_position]',
                        isset($options['calculator_position']) ? $options['calculator_position'] : 'before_add_to_cart',
                        array(
                            'before_add_to_cart' => __('Before Add to Cart', 'pricewise-calculator-pro'),
                            'after_add_to_cart' => __('After Add to Cart', 'pricewise-calculator-pro'),
                            'before_product_summary' => __('Before Product Summary', 'pricewise-calculator-pro'),
                            'after_product_summary' => __('After Product Summary', 'pricewise-calculator-pro'),
                            'in_product_tabs' => __('In Product Tabs', 'pricewise-calculator-pro'),
                        ),
                        __('Where to display the calculator on product pages', 'pricewise-calculator-pro')
                    );
                    ?>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Add to Cart Text', 'pricewise-calculator-pro'); ?></th>
                <td>
                    <?php
                    $this->render_text_field(
                        'pwcp_general_settings[add_to_cart_text]',
                        isset($options['add_to_cart_text']) ? $options['add_to_cart_text'] : 'Add to Cart',
                        __('Text for the add to cart button when calculator is used', 'pricewise-calculator-pro')
                    );
                    ?>
                </td>
            </tr>
            <tr>
    <th scope="row"><?php esc_html_e('Shop Quote Button Text', 'pricewise-calculator-pro'); ?></th>
    <td>
        <?php
        $this->render_text_field(
            'pwcp_general_settings[shop_quote_button_text]',
            isset($options['shop_quote_button_text']) ? $options['shop_quote_button_text'] : 'Get Custom Quote',
            __('Text displayed on shop/archive pages when price is hidden', 'pricewise-calculator-pro')
        );
        ?>
    </td>
</tr>
            <tr>
                <th scope="row"><?php esc_html_e('Add to Cart Button Requirement', 'pricewise-calculator-pro'); ?></th>
                <td>
                    <?php
                    $this->render_select_field(
                        'pwcp_general_settings[add_to_cart_requirement]',
                        isset($options['add_to_cart_requirement']) ? $options['add_to_cart_requirement'] : 'all_required',
                        array(
                            'disabled' => __('Always Enabled (No Requirements)', 'pricewise-calculator-pro'),
                            'all_required' => __('All Required Fields Must Be Filled (Default)', 'pricewise-calculator-pro'),
                            'all_fields' => __('All Fields Must Be Filled (Required + Optional)', 'pricewise-calculator-pro'),
                            'at_least_1' => __('At Least First 1 Field Must Be Filled', 'pricewise-calculator-pro'),
                            'at_least_2' => __('At Least First 2 Fields Must Be Filled', 'pricewise-calculator-pro'),
                            'at_least_3' => __('At Least First 3 Fields Must Be Filled', 'pricewise-calculator-pro'),
                            'at_least_4' => __('At Least First 4 Fields Must Be Filled', 'pricewise-calculator-pro'),
                        ),
                        __('Choose when to enable the Add to Cart button based on field completion', 'pricewise-calculator-pro')
                    );
                    ?>
                    <p class="description pwcp-help-requirements">
                        <strong><?php esc_html_e('Requirement Types Explained:', 'pricewise-calculator-pro'); ?></strong><br>
                        • <strong><?php esc_html_e('Always Enabled:', 'pricewise-calculator-pro'); ?></strong> <?php esc_html_e('Button is always clickable regardless of field completion', 'pricewise-calculator-pro'); ?><br>
                        • <strong><?php esc_html_e('All Required Fields:', 'pricewise-calculator-pro'); ?></strong> <?php esc_html_e('Only fields marked as required must be completed', 'pricewise-calculator-pro'); ?><br>
                        • <strong><?php esc_html_e('All Fields:', 'pricewise-calculator-pro'); ?></strong> <?php esc_html_e('Every field (required and optional) must be filled', 'pricewise-calculator-pro'); ?><br>
                        • <strong><?php esc_html_e('At Least X Fields:', 'pricewise-calculator-pro'); ?></strong> <?php esc_html_e('The first X fields in order must be completed (useful for progressive forms)', 'pricewise-calculator-pro'); ?>
                    </p>
                </td>
            </tr>
            
            <tr>
                <th scope="row"><?php esc_html_e('Required Field Indicator', 'pricewise-calculator-pro'); ?></th>
                <td>
                    <?php
                    $this->render_text_field(
                        'pwcp_general_settings[required_field_text]',
                        isset($options['required_field_text']) ? $options['required_field_text'] : '*',
                        __('Text or symbol to indicate required fields', 'pricewise-calculator-pro')
                    );
                    ?>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Validation Error Message', 'pricewise-calculator-pro'); ?></th>
                <td>
                    <?php
                    $this->render_text_field(
                        'pwcp_general_settings[error_message]',
                        isset($options['error_message']) ? $options['error_message'] : 'Please check your inputs and try again.',
                        __('Generic error message for validation failures', 'pricewise-calculator-pro')
                    );
                    ?>
                </td>
            </tr>
        </table>
        <?php
    }
    
    /**
     * Render design settings
     *
     * @since 1.0.0
     * @return void
     */
    private function render_design_settings() {
        $options = get_option('pwcp_design_settings', array());
        ?>
        
        <div class="pwcp-settings-section">
            <h3 class="pwcp-section-header">
                <svg class="pwcp-section-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <rect x="2" y="3" width="20" height="6" rx="2"/>
                    <path d="M2 9v10a2 2 0 002 2h16a2 2 0 002-2V9"/>
                </svg>
                <?php esc_html_e('Calculator Header Colors', 'pricewise-calculator-pro'); ?>
            </h3>
            <p class="pwcp-section-description"><?php esc_html_e('Customize the colors for the calculator header section including title, subtitle, and header background.', 'pricewise-calculator-pro'); ?></p>
            
            <div class="pwcp-color-picker-grid">
                <?php
                if ($this->enhanced_color_picker) {
                    $this->enhanced_color_picker->pwcp_render_single_color_picker(
                        'header_background_color',
                        __('Header Background', 'pricewise-calculator-pro'),
                        __('Background color for the calculator header section with title and description', 'pricewise-calculator-pro'),
                        '#844CA9',
                        'pwcp_design_settings'
                    );
                    
                    $this->enhanced_color_picker->pwcp_render_single_color_picker(
                        'title_color',
                        __('Title Text Color', 'pricewise-calculator-pro'),
                        __('Color for the main calculator title text', 'pricewise-calculator-pro'),
                        '#ffffff',
                        'pwcp_design_settings'
                    );
                    
                    $this->enhanced_color_picker->pwcp_render_single_color_picker(
                        'subtitle_color',
                        __('Subtitle Text Color', 'pricewise-calculator-pro'),
                        __('Color for the calculator description/subtitle text', 'pricewise-calculator-pro'),
                        '#e5e7eb',
                        'pwcp_design_settings'
                    );
                }
                ?>
            </div>
        </div>
        
        <div class="pwcp-settings-section">
            <h3 class="pwcp-section-header">
                <svg class="pwcp-section-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <rect x="2" y="3" width="20" height="14" rx="2" ry="2"/>
                    <line x1="8" y1="21" x2="16" y2="21"/>
                    <line x1="12" y1="17" x2="12" y2="21"/>
                </svg>
                <?php esc_html_e('Main Content Colors', 'pricewise-calculator-pro'); ?>
            </h3>
            <p class="pwcp-section-description"><?php esc_html_e('Customize colors for the main calculator content area including background, text, buttons, and form elements.', 'pricewise-calculator-pro'); ?></p>
            
            <div class="pwcp-color-picker-grid">
                <?php
                if ($this->enhanced_color_picker) {
                    $this->enhanced_color_picker->pwcp_render_single_color_picker(
                        'background_color',
                        __('Main Background', 'pricewise-calculator-pro'),
                        __('Background color for the main calculator content area', 'pricewise-calculator-pro'),
                        '#ffffff',
                        'pwcp_design_settings'
                    );
                    
                    $this->enhanced_color_picker->pwcp_render_single_color_picker(
                        'text_color',
                        __('Main Text Color', 'pricewise-calculator-pro'),
                        __('Color for main content text, field labels, and descriptions', 'pricewise-calculator-pro'),
                        '#333333',
                        'pwcp_design_settings'
                    );
                    
                    $this->enhanced_color_picker->pwcp_render_single_color_picker(
                        'next_button_color',
                        __('Next Button Color', 'pricewise-calculator-pro'),
                        __('Background color for the Next navigation button', 'pricewise-calculator-pro'),
                        '#844CA9',
                        'pwcp_design_settings'
                    );
                    
                    $this->enhanced_color_picker->pwcp_render_single_color_picker(
                        'previous_button_color',
                        __('Previous Button Color', 'pricewise-calculator-pro'),
                        __('Background color for the Previous navigation button', 'pricewise-calculator-pro'),
                        '#6b7280',
                        'pwcp_design_settings'
                    );
                    
                    $this->enhanced_color_picker->pwcp_render_single_color_picker(
                        'validation_background_color',
                        __('Validation Message Background', 'pricewise-calculator-pro'),
                        __('Background color for validation and warning messages', 'pricewise-calculator-pro'),
                        '#e45949',
                        'pwcp_design_settings'
                    );
                    
                    $this->enhanced_color_picker->pwcp_render_single_color_picker(
                        'validation_text_color',
                        __('Validation Text Color', 'pricewise-calculator-pro'),
                        __('Text color for validation and warning messages', 'pricewise-calculator-pro'),
                        '#ffffff',
                        'pwcp_design_settings'
                    );

$this->enhanced_color_picker->pwcp_render_single_color_picker(
    'shop_quote_button_color',
    __('Shop Quote Button Color', 'pricewise-calculator-pro'),
    __('Background color for the "Get Custom Quote" button on shop and archive pages', 'pricewise-calculator-pro'),
    '#844CA9',
    'pwcp_design_settings'
);
                }
                ?>
            </div>
        </div>
        
        <div class="pwcp-settings-section">
            <h3 class="pwcp-section-header">
                <svg class="pwcp-section-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M9 12l2 2 4-4"/>
                    <circle cx="21" cy="12" r="1"/>
                    <path d="M18 12h-2.5a3.5 3.5 0 0 0-3.5 3.5V18"/>
                    <circle cx="7" cy="7" r="1"/>
                    <path d="M7 10v4h4"/>
                </svg>
                <?php esc_html_e('Option Styling Colors', 'pricewise-calculator-pro'); ?>
            </h3>
            <p class="pwcp-section-description"><?php esc_html_e('Customize colors for radio buttons, checkboxes, price tags, and selected option states.', 'pricewise-calculator-pro'); ?></p>
            
            <div class="pwcp-color-picker-grid">
                <?php
                if ($this->enhanced_color_picker) {
                    $this->enhanced_color_picker->pwcp_render_single_color_picker(
                        'price_badge_background_color',
                        __('Badge/Price Tag Background', 'pricewise-calculator-pro'),
                        __('Background color for price tags like "+20.00Rs" and "+50.00Rs" displayed next to options', 'pricewise-calculator-pro'),
                        '#844CA9',
                        'pwcp_design_settings'
                    );
                    
                    $this->enhanced_color_picker->pwcp_render_single_color_picker(
                        'price_badge_text_color',
                        __('Badge/Price Tag Text Color', 'pricewise-calculator-pro'),
                        __('Text color for the price amounts displayed in price tags next to options', 'pricewise-calculator-pro'),
                        '#ffffff',
                        'pwcp_design_settings'
                    );
                    
                    $this->enhanced_color_picker->pwcp_render_single_color_picker(
                        'selection_indicator_color',
                        __('Selection Indicator Color', 'pricewise-calculator-pro'),
                        __('Color for the checkmark circle (radio) or square (checkbox) graphic indicators', 'pricewise-calculator-pro'),
                        '#6b7280',
                        'pwcp_design_settings'
                    );
                    
                    $this->enhanced_color_picker->pwcp_render_single_color_picker(
                        'selected_option_background_color',
                        __('Selected Option Background', 'pricewise-calculator-pro'),
                        __('Background color when an option is selected - the entire option background changes to this color', 'pricewise-calculator-pro'),
                        '#844CA9',
                        'pwcp_design_settings'
                    );
                }
                ?>
            </div>
        </div>
        
        <div class="pwcp-settings-section">
            <h3 class="pwcp-section-header">
                <svg class="pwcp-section-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <rect x="2" y="15" width="20" height="6" rx="2"/>
                    <path d="M2 15V5a2 2 0 012-2h16a2 2 0 012 2v10"/>
                    <circle cx="8" cy="18" r="1"/>
                    <circle cx="16" cy="18" r="1"/>
                </svg>
                <?php esc_html_e('Calculator Footer Colors', 'pricewise-calculator-pro'); ?>
            </h3>
            <p class="pwcp-section-description"><?php esc_html_e('Customize colors for the calculator footer section including price display, progress indicators, and completion status.', 'pricewise-calculator-pro'); ?></p>
            
            <div class="pwcp-color-picker-grid">
                <?php
                if ($this->enhanced_color_picker) {
                    $this->enhanced_color_picker->pwcp_render_single_color_picker(
                        'footer_background_color',
                        __('Footer Background', 'pricewise-calculator-pro'),
                        __('Background color for the calculator footer section', 'pricewise-calculator-pro'),
                        '#e1ccf0',
                        'pwcp_design_settings'
                    );
                    
                    $this->enhanced_color_picker->pwcp_render_single_color_picker(
                        'footer_price_color',
                        __('Price Display Color', 'pricewise-calculator-pro'),
                        __('Color for the calculated price amount in the footer', 'pricewise-calculator-pro'),
                        '#844CA9',
                        'pwcp_design_settings'
                    );
                    
                    $this->enhanced_color_picker->pwcp_render_single_color_picker(
                        'price_label_color',
                        __('Price Label Color', 'pricewise-calculator-pro'),
                        __('Color for the "Price:" label text in footer section', 'pricewise-calculator-pro'),
                        '#844CA9',
                        'pwcp_design_settings'
                    );
                    
                    $this->enhanced_color_picker->pwcp_render_single_color_picker(
                        'footer_text_color',
                        __('Footer Text Color', 'pricewise-calculator-pro'),
                        __('Color for all footer text including progress indicators, completion status, and helper text', 'pricewise-calculator-pro'),
                        '#844CA9',
                        'pwcp_design_settings'
                    );
                    
                    $this->enhanced_color_picker->pwcp_render_single_color_picker(
                        'progress_bar_color',
                        __('Progress Bar Color', 'pricewise-calculator-pro'),
                        __('Color for the progress bar fill that shows completion status in the footer section', 'pricewise-calculator-pro'),
                        '#844CA9',
                        'pwcp_design_settings'
                    );
                }
                ?>
            </div>
        </div>
        
        <div class="pwcp-settings-section">
            <h3 class="pwcp-section-header">
                <svg class="pwcp-section-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                    <path d="M9 9h6v6H9z"/>
                </svg>
                <?php esc_html_e('Layout & Styling Options', 'pricewise-calculator-pro'); ?>
            </h3>
            <p class="pwcp-section-description"><?php esc_html_e('Configure general layout, sizing, and visual effects for your calculators.', 'pricewise-calculator-pro'); ?></p>
            
            <table class="form-table">
                <tr>
                    <th scope="row"><?php esc_html_e('Border Radius', 'pricewise-calculator-pro'); ?></th>
                    <td>
                        <?php
                        $this->render_number_field(
                            'pwcp_design_settings[border_radius]',
                            isset($options['border_radius']) ? $options['border_radius'] : 12,
                            __('Border radius in pixels for calculator elements', 'pricewise-calculator-pro'),
                            0, 50, 1
                        );
                        ?>
                    </td>
                </tr>
           <tr>
    <th scope="row"><?php esc_html_e('Calculator Width', 'pricewise-calculator-pro'); ?></th>
    <td>
        <?php
        $this->render_select_field(
            'pwcp_design_settings[calculator_width]',
            isset($options['calculator_width']) ? $options['calculator_width'] : 'auto',
            array(
                'auto' => __('Auto', 'pricewise-calculator-pro'),
                '100%' => __('Full Width', 'pricewise-calculator-pro'),
                '400px' => __('Fixed 400px', 'pricewise-calculator-pro'),
                '450px' => __('Fixed 450px', 'pricewise-calculator-pro'),
                '500px' => __('Fixed 500px', 'pricewise-calculator-pro'),
                '550px' => __('Fixed 550px', 'pricewise-calculator-pro'),
                '600px' => __('Fixed 600px', 'pricewise-calculator-pro'),
            ),
            __('Calculator container width', 'pricewise-calculator-pro')
        );
        ?>
    </td>
</tr>
            </table>
        </div>
        <?php
    }
    
    /**
     * Render calculation settings
     *
     * @since 1.0.0
     * @return void
     */
    private function render_calculation_settings() {
        $options = get_option('pwcp_calculation_settings', array());
        ?>
        <table class="form-table">
            <tr>
                <th scope="row"><?php esc_html_e('Live Calculations', 'pricewise-calculator-pro'); ?></th>
                <td>
                    <?php
                    $this->render_checkbox_field(
                        'pwcp_calculation_settings[enable_live_calculation]',
                        isset($options['enable_live_calculation']) ? $options['enable_live_calculation'] : true,
                        __('Update prices instantly as users change values', 'pricewise-calculator-pro')
                    );
                    ?>
                </td>
            </tr>
            
            <tr>
                <th scope="row"><?php esc_html_e('Currency Position', 'pricewise-calculator-pro'); ?></th>
                <td>
                    <?php
                    $this->render_select_field(
                        'pwcp_calculation_settings[currency_position]',
                        isset($options['currency_position']) ? $options['currency_position'] : 'left',
                        array(
                            'left' => __('Left ($100)', 'pricewise-calculator-pro'),
                            'right' => __('Right (100$)', 'pricewise-calculator-pro'),
                            'left_space' => __('Left with space ($ 100)', 'pricewise-calculator-pro'),
                            'right_space' => __('Right with space (100 $)', 'pricewise-calculator-pro')
                        ),
                        __('Position of currency symbol in price display', 'pricewise-calculator-pro')
                    );
                    ?>
                </td>
            </tr>
            
            <tr>
                <th scope="row"><?php esc_html_e('Currency Symbol', 'pricewise-calculator-pro'); ?></th>
                <td>
                    <?php
                    $this->render_text_field(
                        'pwcp_calculation_settings[currency_symbol]',
                        isset($options['currency_symbol']) ? $options['currency_symbol'] : '$',
                        __('Custom currency symbol', 'pricewise-calculator-pro'),
                        5
                    );
                    ?>
                    <div class="pwcp-currency-note-wrapper">
                        <p class="description pwcp-currency-note">
                            <strong><?php esc_html_e('Note:', 'pricewise-calculator-pro'); ?></strong> <?php esc_html_e('To change the currency symbol site-wide (including Add to Cart buttons, product pages, and checkout), change the currency in WooCommerce settings. This setting only affects the calculator display.', 'pricewise-calculator-pro'); ?>
                        </p>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=wc-settings&tab=general')); ?>" 
                           class="button button-secondary pwcp-wc-settings-link" 
                           target="_blank" rel="noopener noreferrer">
                            <?php esc_html_e('Go to WooCommerce Currency Settings', 'pricewise-calculator-pro'); ?>
                        </a>
                    </div>
                </td>
            </tr>
            
            <tr>
                <th scope="row"><?php esc_html_e('Decimal Places', 'pricewise-calculator-pro'); ?></th>
                <td>
                    <?php
                    $this->render_number_field(
                        'pwcp_calculation_settings[decimal_places]',
                        isset($options['decimal_places']) ? $options['decimal_places'] : 2,
                        __('Number of decimal places to show in prices', 'pricewise-calculator-pro'),
                        0, 4, 1
                    );
                    ?>
                </td>
            </tr>
            
            <tr>
                <th scope="row"><?php esc_html_e('Thousands Separator', 'pricewise-calculator-pro'); ?></th>
                <td>
                    <?php
                    $this->render_text_field(
                        'pwcp_calculation_settings[thousand_separator]',
                        isset($options['thousand_separator']) ? $options['thousand_separator'] : ',',
                        __('Character to separate thousands', 'pricewise-calculator-pro'),
                        1
                    );
                    ?>
                </td>
            </tr>
            
            <tr>
                <th scope="row"><?php esc_html_e('Decimal Separator', 'pricewise-calculator-pro'); ?></th>
                <td>
                    <?php
                    $this->render_text_field(
                        'pwcp_calculation_settings[decimal_separator]',
                        isset($options['decimal_separator']) ? $options['decimal_separator'] : '.',
                        __('Character to separate decimal places', 'pricewise-calculator-pro'),
                        1
                    );
                    ?>
                </td>
            </tr>
        </table>
        <?php
    }
    
    /**
     * Render text field
     *
     * @since 1.0.0
     */
    private function render_text_field($name, $value, $description, $maxlength = null) {
        printf(
            '<div class="pwcp-field-group">
                <div class="pwcp-input-wrapper">
                    <input type="text" name="%s" value="%s" class="pwcp-input-modern" %s>
                    <div class="pwcp-input-highlight"></div>
                </div>
                %s
            </div>',
            esc_attr($name),
            esc_attr($value),
            $maxlength ? 'maxlength="' . esc_attr($maxlength) . '"' : '',
            $description ? '<div class="pwcp-field-hint">' . esc_html($description) . '</div>' : ''
        );
    }
    
    /**
     * Render number field
     *
     * @since 1.0.0
     */
    private function render_number_field($name, $value, $description, $min = null, $max = null, $step = 1) {
        printf(
            '<div class="pwcp-field-group">
                <div class="pwcp-input-wrapper">
                    <input type="number" name="%s" value="%s" class="pwcp-input-modern" min="%s" max="%s" step="%s">
                    <div class="pwcp-input-highlight"></div>
                </div>
                %s
            </div>',
            esc_attr($name),
            esc_attr($value),
            esc_attr($min),
            esc_attr($max),
            esc_attr($step),
            $description ? '<div class="pwcp-field-hint">' . esc_html($description) . '</div>' : ''
        );
    }
    
    /**
     * Render checkbox field
     *
     * @since 1.0.0
     */
    private function render_checkbox_field($name, $value, $description) {
        printf(
            '<div class="pwcp-field-group">
                <label class="pwcp-checkbox-label">
                    <input type="checkbox" name="%s" value="1" %s class="pwcp-checkbox-input">
                    <div class="pwcp-checkbox-custom"></div>
                    <span class="pwcp-checkbox-text">%s</span>
                </label>
            </div>',
            esc_attr($name),
            checked($value, true, false),
            esc_html($description)
        );
    }
    
    /**
     * Render select field
     *
     * @since 1.0.0
     */
    private function render_select_field($name, $value, $options, $description) {
        printf('<div class="pwcp-field-group"><div class="pwcp-select-wrapper">');
        printf('<select name="%s" class="pwcp-select-modern">', esc_attr($name));
        
        foreach ($options as $option_value => $option_label) {
            printf(
                '<option value="%s" %s>%s</option>',
                esc_attr($option_value),
                selected($value, $option_value, false),
                esc_html($option_label)
            );
        }
        
        echo '</select><div class="pwcp-select-highlight"></div></div>';
        
        if ($description) {
            printf('<div class="pwcp-field-hint">%s</div>', esc_html($description));
        }
        
        echo '</div>';
    }
    
    /**
     * Render settings sidebar
     *
     * @since 1.0.0
     */
    private function pwcp_render_settings_sidebar($active_tab) {
        ?>
        <div class="pwcp-sidebar-widget">
            <div class="pwcp-widget-header">
                <h3><?php esc_html_e('Quick Actions', 'pricewise-calculator-pro'); ?></h3>
            </div>
            <div class="pwcp-widget-content">
                <div class="pwcp-quick-actions">
                    <a href="<?php echo esc_url(admin_url('admin.php?page=pwcp-add-calculator')); ?>" class="pwcp-quick-action pwcp-action-primary">
                        <div class="pwcp-action-icon">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <line x1="12" y1="5" x2="12" y2="19"/>
                                <line x1="5" y1="12" x2="19" y2="12"/>
                            </svg>
                        </div>
                        <div class="pwcp-action-content">
                            <div class="pwcp-action-title"><?php esc_html_e('New Calculator', 'pricewise-calculator-pro'); ?></div>
                            <div class="pwcp-action-desc"><?php esc_html_e('Create a new pricing calculator', 'pricewise-calculator-pro'); ?></div>
                        </div>
                        <div class="pwcp-action-arrow">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <polyline points="9,18 15,12 9,6"/>
                            </svg>
                        </div>
                    </a>

                    <a href="<?php echo esc_url(admin_url('admin.php?page=pwcp-calculators')); ?>" class="pwcp-quick-action">
                        <div class="pwcp-action-icon">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                                <line x1="9" y1="12" x2="15" y2="12"/>
                                <line x1="12" y1="9" x2="12" y2="15"/>
                            </svg>
                        </div>
                        <div class="pwcp-action-content">
                            <div class="pwcp-action-title"><?php esc_html_e('Manage Calculators', 'pricewise-calculator-pro'); ?></div>
                            <div class="pwcp-action-desc"><?php esc_html_e('Edit existing calculators', 'pricewise-calculator-pro'); ?></div>
                        </div>
                        <div class="pwcp-action-arrow">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <polyline points="9,18 15,12 9,6"/>
                            </svg>
                        </div>
                    </a>

                    <a href="<?php echo esc_url(admin_url('admin.php?page=pwcp-dashboard')); ?>" class="pwcp-quick-action">
                        <div class="pwcp-action-icon">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <circle cx="12" cy="12" r="3"/>
                                <path d="M19.4 15a1.65 1.65 0 00.33 1.82l.06.06a2 2 0 010 2.83 2 2 0 01-2.83 0l-.06-.06a1.65 1.65 0 00-1.82-.33 1.65 1.65 0 00-1 1.51V21a2 2 0 01-2 2 2 2 0 01-2-2v-.09A1.65 1.65 0 009 19.4a1.65 1.65 0 00-1.82.33l-.06.06a2 2 0 01-2.83 0 2 2 0 010-2.83l.06-.06a1.65 1.65 0 00.33-1.82 1.65 1.65 0 00-1.51-1H3a2 2 0 01-2-2 2 2 0 012-2h.09A1.65 1.65 0 004.6 9a1.65 1.65 0 00-.33-1.82l-.06-.06a2 2 0 010-2.83 2 2 0 012.83 0l.06.06a1.65 1.65 0 001.82.33H9a1.65 1.65 0 001-1.51V3a2 2 0 012-2 2 2 0 012 2v.09a1.65 1.65 0 001 1.51 1.65 1.65 0 001.82-.33l.06-.06a2 2 0 012.83 0 2 2 0 010 2.83l-.06.06a1.65 1.65 0 00-.33 1.82V9a1.65 1.65 0 001.51 1H21a2 2 0 012 2 2 2 0 01-2 2h-.09a1.65 1.65 0 00-1.51 1z"/>
                            </svg>
                        </div>
                        <div class="pwcp-action-content">
                            <div class="pwcp-action-title"><?php esc_html_e('Dashboard', 'pricewise-calculator-pro'); ?></div>
                            <div class="pwcp-action-desc"><?php esc_html_e('View overview and analytics', 'pricewise-calculator-pro'); ?></div>
                        </div>
                        <div class="pwcp-action-arrow">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <polyline points="9,18 15,12 9,6"/>
                            </svg>
                        </div>
                    </a>

                    <?php if (class_exists('WooCommerce')) : ?>
                    <a href="<?php echo esc_url(admin_url('edit.php?post_type=product')); ?>" class="pwcp-quick-action">
                        <div class="pwcp-action-icon">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M20 7l-8-4-8 4m16 0l-8 4m8-4v10l-8 4m0-10L4 7m8 4v10M4 7v10l8 4"/>
                            </svg>
                        </div>
                        <div class="pwcp-action-content">
                            <div class="pwcp-action-title"><?php esc_html_e('Products', 'pricewise-calculator-pro'); ?></div>
                            <div class="pwcp-action-desc"><?php esc_html_e('Manage WooCommerce products', 'pricewise-calculator-pro'); ?></div>
                        </div>
                        <div class="pwcp-action-arrow">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <polyline points="9,18 15,12 9,6"/>
                            </svg>
                        </div>
                    </a>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <div class="pwcp-sidebar-widget">
            <div class="pwcp-widget-header">
                <h3><?php esc_html_e('Quick Tips', 'pricewise-calculator-pro'); ?></h3>
            </div>
            <div class="pwcp-widget-content">
                <div class="pwcp-tips-list">
                    <div class="pwcp-tip-item">
                        <div class="pwcp-tip-number">1</div>
                        <div class="pwcp-tip-content">
                            <h4><?php esc_html_e('Live Updates', 'pricewise-calculator-pro'); ?></h4>
                            <p><?php esc_html_e('Enable live calculation to show prices instantly as customers select options - creates a more engaging experience.', 'pricewise-calculator-pro'); ?></p>
                        </div>
                    </div>

                    <div class="pwcp-tip-item">
                        <div class="pwcp-tip-number">2</div>
                        <div class="pwcp-tip-content">
                            <h4><?php esc_html_e('Reset Per Tab', 'pricewise-calculator-pro'); ?></h4>
                            <p><?php esc_html_e('Each tab (General, Design, Calculation) has its own reset button. Resetting one tab won\'t affect settings in other tabs.', 'pricewise-calculator-pro'); ?></p>
                        </div>
                    </div>

                    <div class="pwcp-tip-item">
                        <div class="pwcp-tip-number">3</div>
                        <div class="pwcp-tip-content">
                            <h4><?php esc_html_e('Currency Formats', 'pricewise-calculator-pro'); ?></h4>
                            <p><?php esc_html_e('US: 1,234.56 | EU: 1.234,56 | Set separators to match your region.', 'pricewise-calculator-pro'); ?></p>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="pwcp-sidebar-widget">
            <div class="pwcp-widget-header">
                <h3><?php esc_html_e('System Diagnostics', 'pricewise-calculator-pro'); ?></h3>
            </div>
            <div class="pwcp-widget-content">
                <div class="pwcp-plugin-info">
                    <div class="pwcp-plugin-stats-list">
                        <div class="pwcp-info-item">
                            <div class="pwcp-info-left">
                                <div class="pwcp-info-indicator"></div>
                                <span class="pwcp-info-label"><?php esc_html_e('Configuration Status:', 'pricewise-calculator-pro'); ?></span>
                            </div>
                            <span class="pwcp-info-value"><?php esc_html_e('Active', 'pricewise-calculator-pro'); ?></span>
                        </div>
                        <div class="pwcp-info-item">
                            <div class="pwcp-info-left">
                                <div class="pwcp-info-indicator"></div>
                                <span class="pwcp-info-label"><?php esc_html_e('Version:', 'pricewise-calculator-pro'); ?></span>
                            </div>
                            <span class="pwcp-info-value"><?php echo esc_html(PWCP_PLUGIN_VERSION); ?></span>
                        </div>
                        <div class="pwcp-info-item">
                            <div class="pwcp-info-left">
                                <div class="pwcp-info-indicator"></div>
                                <span class="pwcp-info-label"><?php esc_html_e('Last Updated:', 'pricewise-calculator-pro'); ?></span>
                            </div>
                            <span class="pwcp-info-value"><?php echo esc_html(date_i18n(get_option('date_format'), current_time('timestamp'))); ?></span>
                        </div>
                    </div>
                </div>
                
                <div class="pwcp-plugin-actions">
                    <a href="<?php echo esc_url(admin_url('admin.php?page=pwcp-help')); ?>" class="pwcp-btn pwcp-btn-outline pwcp-btn-sm">
                        <?php esc_html_e('Documentation', 'pricewise-calculator-pro'); ?>
                    </a>
                    <a href="mailto:support@pricewise-calculator.com" class="pwcp-btn pwcp-btn-outline pwcp-btn-sm">
                        <?php esc_html_e('Support', 'pricewise-calculator-pro'); ?>
                    </a>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Sanitize general settings
     *
     * @since 1.0.0
     * @param array $input Raw input data
     * @return array Sanitized settings
     */
    public function pwcp_sanitize_general_settings($input) {
        if (!is_array($input)) {
            return $this->get_default_values('general');
        }
        
        $sanitized = array();
        
        $sanitized['calculator_position'] = isset($input['calculator_position']) 
            ? sanitize_text_field($input['calculator_position']) 
            : 'before_add_to_cart';
            
        $sanitized['add_to_cart_text'] = isset($input['add_to_cart_text']) 
            ? sanitize_text_field($input['add_to_cart_text']) 
            : 'Add to Cart';
            
        $sanitized['add_to_cart_requirement'] = isset($input['add_to_cart_requirement']) 
            ? sanitize_text_field($input['add_to_cart_requirement']) 
            : 'all_required';
            
        $sanitized['loading_message'] = isset($input['loading_message']) 
            ? sanitize_text_field($input['loading_message']) 
            : 'Calculating...';
            
        $sanitized['required_field_text'] = isset($input['required_field_text']) 
            ? sanitize_text_field($input['required_field_text']) 
            : '*';
            
        $sanitized['error_message'] = isset($input['error_message']) 
            ? sanitize_text_field($input['error_message']) 
            : 'Please check your inputs and try again.';
        $sanitized['shop_quote_button_text'] = isset($input['shop_quote_button_text']) 
    ? sanitize_text_field($input['shop_quote_button_text']) 
    : 'Get Custom Quote';
        return $sanitized;
    }
    
    /**
     * Sanitize design settings
     *
     * @since 1.0.0
     * @param array $input Raw input data
     * @return array Sanitized settings
     */
    public function pwcp_sanitize_design_settings($input) {
        if (!is_array($input)) {
            return $this->get_default_values('design');
        }
        
        $sanitized = array();
        
        $sanitized['header_background_color'] = sanitize_hex_color($input['header_background_color'] ?? '#844CA9');
        $sanitized['title_color'] = sanitize_hex_color($input['title_color'] ?? '#ffffff');
        $sanitized['subtitle_color'] = sanitize_hex_color($input['subtitle_color'] ?? '#e5e7eb');
        
        $sanitized['background_color'] = sanitize_hex_color($input['background_color'] ?? '#ffffff');
        $sanitized['text_color'] = sanitize_hex_color($input['text_color'] ?? '#333333');
        $sanitized['next_button_color'] = sanitize_hex_color($input['next_button_color'] ?? '#844CA9');
        $sanitized['previous_button_color'] = sanitize_hex_color($input['previous_button_color'] ?? '#6b7280');
        $sanitized['validation_background_color'] = sanitize_hex_color($input['validation_background_color'] ?? '#e45949');
        $sanitized['validation_text_color'] = sanitize_hex_color($input['validation_text_color'] ?? '#ffffff');
        $sanitized['shop_quote_button_color'] = sanitize_hex_color($input['shop_quote_button_color'] ?? '#844CA9');

        
        $sanitized['price_badge_background_color'] = sanitize_hex_color($input['price_badge_background_color'] ?? '#844CA9');
        $sanitized['price_badge_text_color'] = sanitize_hex_color($input['price_badge_text_color'] ?? '#ffffff');
        $sanitized['selection_indicator_color'] = sanitize_hex_color($input['selection_indicator_color'] ?? '#6b7280');
        $sanitized['selected_option_background_color'] = sanitize_hex_color($input['selected_option_background_color'] ?? '#844CA9');
        
        $sanitized['footer_background_color'] = sanitize_hex_color($input['footer_background_color'] ?? '#e1ccf0');
        $sanitized['footer_price_color'] = sanitize_hex_color($input['footer_price_color'] ?? '#844CA9');
        $sanitized['price_label_color'] = sanitize_hex_color($input['price_label_color'] ?? '#844CA9');
        $sanitized['footer_text_color'] = sanitize_hex_color($input['footer_text_color'] ?? '#844CA9');
        $sanitized['progress_bar_color'] = sanitize_hex_color($input['progress_bar_color'] ?? '#844CA9');
        
        $sanitized['border_radius'] = absint($input['border_radius'] ?? 12);
        $sanitized['box_shadow'] = !empty($input['box_shadow']);
        $sanitized['animation_effects'] = !empty($input['animation_effects']);
        $sanitized['calculator_width'] = sanitize_text_field($input['calculator_width'] ?? 'auto');
        
        return $sanitized;
    }
    
    /**
     * Sanitize calculation settings
     *
     * @since 1.0.0
     * @param array $input Raw input data
     * @return array Sanitized settings
     */
    public function pwcp_sanitize_calculation_settings($input) {
        if (!is_array($input)) {
            return $this->get_default_values('calculation');
        }
        
        $sanitized = array();
        
        $sanitized['enable_live_calculation'] = !empty($input['enable_live_calculation']);
        $sanitized['currency_position'] = sanitize_text_field($input['currency_position'] ?? 'left');
        $sanitized['decimal_places'] = absint($input['decimal_places'] ?? 2);
        $sanitized['thousand_separator'] = sanitize_text_field($input['thousand_separator'] ?? ',');
        $sanitized['decimal_separator'] = sanitize_text_field($input['decimal_separator'] ?? '.');
        $sanitized['currency_symbol'] = sanitize_text_field($input['currency_symbol'] ?? '$');
        
        return $sanitized;
    }
    
    /**
     * AJAX handler for resetting settings
     * 
     * @since 1.0.0
     */
    public function ajax_reset_settings() {
        $this->verify_ajax_security('manage_options', 'pwcp_ajax_settings', 'pwcp_settings_nonce');
        
        $tab = isset($_POST['tab']) ? sanitize_key(wp_unslash($_POST['tab'])) : 'general';
        $valid_tabs = array('general', 'design', 'calculation');
        
        if (!$this->is_action_allowed($tab, $valid_tabs)) {
            $this->send_error_response(__('Invalid settings tab.', 'pricewise-calculator-pro'));
        }
        
        $defaults = $this->get_default_values($tab);
        
        switch ($tab) {
            case 'general':
                update_option('pwcp_general_settings', $defaults);
                break;
            case 'design':
                update_option('pwcp_design_settings', $defaults);
                break;
            case 'calculation':
                update_option('pwcp_calculation_settings', $defaults);
                break;
        }
        
        $this->send_success_response(
            __('Settings reset to defaults successfully!', 'pricewise-calculator-pro'),
            array(
                'tab' => $tab,
                'defaults' => $defaults
            )
        );
    }
    
    /**
     * Get default values for settings tab
     *
     * @since 1.0.0
     * @param string $tab Tab identifier
     * @return array Default values
     */
    private function get_default_values($tab) {
        $defaults = array(
            'general' => array(
                'calculator_position' => 'before_add_to_cart',
                'add_to_cart_text' => 'Add to Cart',
                'add_to_cart_requirement' => 'all_required',
                'loading_message' => 'Calculating...',
                'required_field_text' => '*',
                'error_message' => 'Please check your inputs and try again.',
                    'shop_quote_button_text' => 'Get Custom Quote'  

            ),
            'design' => array(
                'header_background_color' => '#844CA9',
                'title_color' => '#FFFFFF',
                'subtitle_color' => '#E5E7EB',
                
                'background_color' => '#ffffff',
                'text_color' => '#333333',
                'next_button_color' => '#844CA9',
                'previous_button_color' => '#6b7280',
                'validation_background_color' => '#e45949',
                'shop_quote_button_color' => '#844CA9',
                'validation_text_color' => '#FFFFFF',
                
                'price_badge_background_color' => '#844CA9',
                'price_badge_text_color' => '#FFFFFF',
                'selection_indicator_color' => '#6b7280',
                'selected_option_background_color' => '#844CA9',
                
                'footer_background_color' => '#e1ccf0',
                'footer_price_color' => '#844CA9',
                'price_label_color' => '#844CA9',
                'footer_text_color' => '#844CA9',
                'progress_bar_color' => '#844CA9',
                
                'border_radius' => 12,
                'box_shadow' => true,
                'animation_effects' => true,
                'calculator_width' => 'auto'
            ),
            'calculation' => array(
                'enable_live_calculation' => true,
                'currency_symbol' => '$',
                'currency_position' => 'left',
                'decimal_places' => 2,
                'thousand_separator' => ',',
                'decimal_separator' => '.'
            )
        );
        
        return $defaults[$tab] ?? array();
    }
    
    /**
     * AJAX handler for validating colors
     * 
     * @since 1.0.0
     */
    public function ajax_validate_color() {
        $this->verify_ajax_security('manage_options', 'pwcp_ajax_settings', 'pwcp_settings_nonce');
        
        $color = $this->get_post_data('color', 'text', '');
        
        if (!$this->enhanced_color_picker) {
            $this->send_error_response(__('Color picker not available.', 'pricewise-calculator-pro'));
        }
        
        $validation_result = $this->enhanced_color_picker->pwcp_validate_hex_color($color);
        
        $this->send_success_response(__('Color validated', 'pricewise-calculator-pro'), $validation_result);
    }
    
    /**
     * Enqueue settings assets
     * 
     * @since 1.0.0
     */
    public function pwcp_enqueue_settings_assets($hook) {
        $screen = function_exists('get_current_screen') ? get_current_screen() : null;
        if (!$screen || $screen->id !== 'calculator-pro_page_pwcp-settings') {
            return;
        }
        
        wp_enqueue_style(
            'pwcp-admin-settings',
            PWCP_ASSETS_URL . 'css/all-admin-css/admin-pages.css',
            array(),
            PWCP_PLUGIN_VERSION
        );
        
        wp_enqueue_script(
            'pwcp-admin-settings',
            PWCP_ASSETS_URL . 'js/all-admin-js/admin-settings.js',
            array('jquery'),
            PWCP_PLUGIN_VERSION,
            true
        );
        
        wp_localize_script('pwcp-admin-settings', 'pwcpSettings', array(
    'ajax_url' => admin_url('admin-ajax.php'),
    'nonce'    => wp_create_nonce('pwcp_ajax_settings'),
    'strings'  => array(
        'saving'              => __('Saving...', 'pricewise-calculator-pro'),
        'save_changes'        => __('Save Changes', 'pricewise-calculator-pro'),
        'settings_saved'      => __('Settings saved successfully!', 'pricewise-calculator-pro'),
        'save_failed'         => __('Failed to save settings.', 'pricewise-calculator-pro'),
        'network_error'       => __('Network error occurred.', 'pricewise-calculator-pro'),
        'reset_confirm'       => __('Are you sure you want to reset these settings to defaults?', 'pricewise-calculator-pro'),
        'resetting'           => __('Resetting...', 'pricewise-calculator-pro'),
        'reset_failed'        => __('Reset failed', 'pricewise-calculator-pro'),
        'invalid_hex'         => __('Invalid hex color format', 'pricewise-calculator-pro'),
        'valid_hex'           => __('Valid hex color', 'pricewise-calculator-pro'),
        'settings_data_unavailable' => __('Settings data not available', 'pricewise-calculator-pro'),
        'panel_titles' => array(
            'general'     => __('General Settings', 'pricewise-calculator-pro'),
            'design'      => __('Design & Styling', 'pricewise-calculator-pro'),
            'calculation' => __('Calculation Settings', 'pricewise-calculator-pro'),
        ),
        'panel_descriptions' => array(
            'general'     => __('Configure basic calculator behavior, text labels, and messages shown to users.', 'pricewise-calculator-pro'),
            'design'      => __('Customize the visual appearance of your calculators to match your site design', 'pricewise-calculator-pro'),
            'calculation' => __('Configure how calculations are performed and displayed to users.', 'pricewise-calculator-pro'),
        ),
    ),
));
        
        if (function_exists('wp_set_script_translations')) {
            wp_set_script_translations(
                'pwcp-admin-settings',
                'pricewise-calculator-pro',
                plugin_dir_path(PWCP_PLUGIN_FILE) . 'languages'
            );
        }
    }
}