<?php
/**
 * Admin Menus Manager
 *
 * Handles all admin menu registration and navigation
 *
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * PWCP Admin Menus Class
 *
 * @since 1.0.0
 */
class PWCP_Admin_Menus {

    /**
     * Constructor
     */
    public function __construct() {
        $this->pwcp_init_menu_hooks();
    }

    /**
     * Initialize menu hooks
     */
    private function pwcp_init_menu_hooks() {
        add_action('admin_menu', array($this, 'pwcp_register_admin_menus'));
        add_action('admin_init', array($this, 'pwcp_handle_menu_redirects'));
        add_filter('submenu_file', array($this, 'pwcp_highlight_menu_items'), 10, 2);
        add_filter('parent_file', array($this, 'pwcp_fix_parent_menu_highlight'));
        add_action('admin_head', array($this, 'pwcp_add_menu_styles'));
    }

    /**
     * Register admin menus
     */
    public function pwcp_register_admin_menus() {
        if (!current_user_can('manage_pwcp_calculators')) {
            return;
        }

        $main_menu = add_menu_page(
            esc_html__('Calculator Pro', 'pricewise-calculator-pro'),
            esc_html__('Calculator Pro', 'pricewise-calculator-pro'),
            'manage_pwcp_calculators',
            'pwcp-dashboard',
            array($this, 'pwcp_render_dashboard_page'),
            $this->pwcp_get_menu_icon(),
            30
        );

        add_submenu_page(
            'pwcp-dashboard',
            esc_html__('Dashboard', 'pricewise-calculator-pro'),
            esc_html__('Dashboard', 'pricewise-calculator-pro'),
            'manage_pwcp_calculators',
            'pwcp-dashboard',
            array($this, 'pwcp_render_dashboard_page')
        );

        add_submenu_page(
            'pwcp-dashboard',
            esc_html__('All Calculators', 'pricewise-calculator-pro'),
            esc_html__('All Calculators', 'pricewise-calculator-pro'),
            'manage_pwcp_calculators',
            'pwcp-calculators',
            array($this, 'pwcp_render_calculators_page')
        );

        add_submenu_page(
            'pwcp-dashboard',
            esc_html__('Add New Calculator', 'pricewise-calculator-pro'),
            esc_html__('Add New', 'pricewise-calculator-pro'),
            'edit_pwcp_calculators',
            'pwcp-add-calculator',
            array($this, 'pwcp_render_add_calculator_page')
        );

        add_submenu_page(
            'pwcp-dashboard',
            esc_html__('Calculator Settings', 'pricewise-calculator-pro'),
            esc_html__('Settings', 'pricewise-calculator-pro'),
            'manage_pwcp_calculators',
            'pwcp-settings',
            array($this, 'pwcp_render_settings_page')
        );

        add_submenu_page(
            'pwcp-dashboard',
            esc_html__('Import/Export Calculators', 'pricewise-calculator-pro'),
            esc_html__('Import/Export', 'pricewise-calculator-pro'),
            'manage_pwcp_calculators',
            'pwcp-import-export',
            array($this, 'pwcp_render_import_export_page')
        );

        add_action('admin_print_styles-' . $main_menu, array($this, 'pwcp_enqueue_admin_styles'));
        add_action('admin_bar_menu', array($this, 'pwcp_add_admin_bar_menu'), 100);
    }

    /**
     * Get menu icon
     */
    private function pwcp_get_menu_icon() {
        return 'dashicons-calculator';
    }

 /**
 * Handle menu redirects
 */
public function pwcp_handle_menu_redirects() {
    if (defined('DOING_AJAX') && DOING_AJAX) {
        return;
    }

    if (!is_admin()) {
        return;
    }

    // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Read-only GET parameter check for page redirect
    if (isset($_GET['page']) && sanitize_text_field(wp_unslash($_GET['page'])) === 'pwcp-edit-calculator') {
        $calculator_id = isset($_GET['calculator_id']) ? absint(wp_unslash($_GET['calculator_id'])) : 0;

        if ($calculator_id) {
            wp_safe_redirect(esc_url_raw(admin_url(add_query_arg(array(
                'page'          => 'pwcp-add-calculator',
                'action'        => 'edit',
                'calculator_id' => $calculator_id,
            ), 'admin.php'))));
            exit;
        }
    }
    // phpcs:enable WordPress.Security.NonceVerification.Recommended
}

/**
 * Highlight correct menu items
 */
public function pwcp_highlight_menu_items($submenu_file, $parent_file) {
    global $pagenow;

    // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Read-only GET parameter check for menu highlighting
    if ($pagenow === 'admin.php' && isset($_GET['page'])) {
        $current_page = sanitize_text_field(wp_unslash($_GET['page']));

        if ($current_page === 'pwcp-add-calculator') {
            $action = isset($_GET['action']) ? sanitize_text_field(wp_unslash($_GET['action'])) : 'add';

            if ($action === 'edit' && isset($_GET['calculator_id'])) {
                $submenu_file = 'pwcp-calculators';
            } else {
                $submenu_file = 'pwcp-add-calculator';
            }
        }
    }
    // phpcs:enable WordPress.Security.NonceVerification.Recommended

    return $submenu_file;
}

/**
 * Fix parent menu highlighting
 */
public function pwcp_fix_parent_menu_highlight($parent_file) {
    global $pagenow;

    // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Read-only GET parameter check for menu highlighting
    if ($pagenow === 'admin.php' && isset($_GET['page'])) {
        $current_page = sanitize_text_field(wp_unslash($_GET['page']));

        if (strpos($current_page, 'pwcp-') === 0) {
            $parent_file = 'pwcp-dashboard';
        }
    }
    // phpcs:enable WordPress.Security.NonceVerification.Recommended

    return $parent_file;
}

    /**
     * Add admin bar menu
     */
    public function pwcp_add_admin_bar_menu($wp_admin_bar) {
        if (!current_user_can('manage_pwcp_calculators')) {
            return;
        }

        $wp_admin_bar->add_node(array(
            'id'    => 'pwcp-menu',
            'title' => esc_html__('Calculator Pro', 'pricewise-calculator-pro'),
            'href'  => admin_url('admin.php?page=pwcp-dashboard'),
            'meta'  => array(
                'title' => esc_html__('Calculator Pro Dashboard', 'pricewise-calculator-pro'),
            ),
        ));

        $wp_admin_bar->add_node(array(
            'parent' => 'pwcp-menu',
            'id'     => 'pwcp-dashboard',
            'title'  => esc_html__('Dashboard', 'pricewise-calculator-pro'),
            'href'   => admin_url('admin.php?page=pwcp-dashboard'),
        ));

        $wp_admin_bar->add_node(array(
            'parent' => 'pwcp-menu',
            'id'     => 'pwcp-add-new',
            'title'  => esc_html__('Add New Calculator', 'pricewise-calculator-pro'),
            'href'   => admin_url('admin.php?page=pwcp-add-calculator'),
        ));

        $wp_admin_bar->add_node(array(
            'parent' => 'pwcp-menu',
            'id'     => 'pwcp-all-calculators',
            'title'  => esc_html__('All Calculators', 'pricewise-calculator-pro'),
            'href'   => admin_url('admin.php?page=pwcp-calculators'),
        ));

        $wp_admin_bar->add_node(array(
            'parent' => 'pwcp-menu',
            'id'     => 'pwcp-settings',
            'title'  => esc_html__('Settings', 'pricewise-calculator-pro'),
            'href'   => admin_url('admin.php?page=pwcp-settings'),
        ));
    }

    /**
     * Add menu styles
     */
public function pwcp_add_menu_styles() {
    $screen = get_current_screen();
    if (!$screen || strpos($screen->id, 'pwcp-') === false) {
        return;
    }
    
    wp_register_style('pwcp-menu-inline', false);
    wp_enqueue_style('pwcp-menu-inline');
    
    $custom_css = '.pwcp-admin-menu { /* Menu customization */ }';
    wp_add_inline_style('pwcp-menu-inline', $custom_css);
}

    /**
     * Enqueue admin styles
     */
    public function pwcp_enqueue_admin_styles() {
        wp_enqueue_style(
            'pwcp-admin-menu-styles',
            PWCP_ASSETS_URL . 'css/all-admin-css/admin-menu.css',
            array(),
            PWCP_PLUGIN_VERSION
        );
    }

    /**
     * Render dashboard page
     */
    public function pwcp_render_dashboard_page() {
        if ($this->pwcp_load_admin_class('PWCP_Admin_Dashboard', 'class-pwcp-admin-dashboard.php')) {
            $dashboard = new PWCP_Admin_Dashboard();
            $dashboard->pwcp_render();
        } else {
            $this->pwcp_render_missing_class_page('Dashboard');
        }
    }

    /**
     * Render calculators list page
     */
    public function pwcp_render_calculators_page() {
        if ($this->pwcp_load_admin_class('PWCP_Admin_Calculator_List', 'class-pwcp-admin-calculator-list.php')) {
            $calculator_list = new PWCP_Admin_Calculator_List();
            $calculator_list->pwcp_render_list_page();
        } else {
            if ($this->pwcp_load_admin_class('PWCP_Admin_Calculator_Manager', 'class-pwcp-admin-calculator-manager.php')) {
                $this->pwcp_render_simple_calculator_list();
            } else {
                $this->pwcp_render_missing_class_page('Calculator List Manager');
            }
        }
    }

    /**
     * Render simple calculator list
     */
    private function pwcp_render_simple_calculator_list() {
        $manager     = new PWCP_Admin_Calculator_Manager();
        $calculators = $manager->pwcp_get_calculators_list();
        $total_count = $manager->pwcp_get_calculators_count();
        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline"><?php esc_html_e('All Calculators', 'pricewise-calculator-pro'); ?></h1>
            <a href="<?php echo esc_url(admin_url('admin.php?page=pwcp-add-calculator')); ?>" class="page-title-action">
                <?php esc_html_e('Add New', 'pricewise-calculator-pro'); ?>
            </a>
            <hr class="wp-header-end">

            <?php 
            if (isset($_GET['message'])): ?>
                <?php $this->pwcp_display_list_notices(); ?>
            <?php endif; ?>

            <div class="pwcp-calculator-stats">
    <p><?php 
    /* translators: %d: Number of calculators */
    printf(esc_html__('Total: %d calculators', 'pricewise-calculator-pro'), absint($total_count)); 
    ?></p>
</div>

            <?php if (!empty($calculators)): ?>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th scope="col"><?php esc_html_e('Name', 'pricewise-calculator-pro'); ?></th>
                            <th scope="col"><?php esc_html_e('Type', 'pricewise-calculator-pro'); ?></th>
                            <th scope="col"><?php esc_html_e('Status', 'pricewise-calculator-pro'); ?></th>
                            <th scope="col"><?php esc_html_e('Created', 'pricewise-calculator-pro'); ?></th>
                            <th scope="col"><?php esc_html_e('Actions', 'pricewise-calculator-pro'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($calculators as $calculator): ?>
                            <tr>
                                <td>
                                    <strong>
                                        <a href="<?php echo esc_url(admin_url('admin.php?page=pwcp-add-calculator&action=edit&calculator_id=' . absint($calculator->id))); ?>">
                                            <?php echo esc_html($calculator->name); ?>
                                        </a>
                                    </strong>
                                    <?php if (!empty($calculator->description)): ?>
                                        <p class="description"><?php echo esc_html(wp_trim_words($calculator->description, 10)); ?></p>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo esc_html(ucfirst($calculator->assignment_type)); ?></td>
                                <td>
                                    <span class="status-<?php echo esc_attr($calculator->status); ?>">
                                        <?php echo esc_html(ucfirst($calculator->status)); ?>
                                    </span>
                                </td>
                                <td><?php echo esc_html(mysql2date(get_option('date_format'), $calculator->created_at)); ?></td>
                                <td>
                                    <a href="<?php echo esc_url(admin_url('admin.php?page=pwcp-add-calculator&action=edit&calculator_id=' . absint($calculator->id))); ?>" class="button button-small">
                                        <?php esc_html_e('Edit', 'pricewise-calculator-pro'); ?>
                                    </a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php else: ?>
                <div class="notice notice-info">
                    <p>
                        <?php esc_html_e('No calculators found.', 'pricewise-calculator-pro'); ?>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=pwcp-add-calculator')); ?>">
                            <?php esc_html_e('Create your first calculator', 'pricewise-calculator-pro'); ?>
                        </a>
                    </p>
                </div>
            <?php endif; ?>
     </div>
<?php
// Add inline styles for calculator list
wp_register_style('pwcp-calculator-list-inline', false);
wp_enqueue_style('pwcp-calculator-list-inline');

$list_css = '
.status-active { color: #46b450; font-weight: bold; }
.status-inactive { color: #dc3232; }
.pwcp-calculator-stats { margin: 10px 0; }
';
wp_add_inline_style('pwcp-calculator-list-inline', $list_css);
?>
<?php
    }

    /**
     * Display list page notices
     */
private function pwcp_display_list_notices() {
    // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Read-only GET parameter for displaying notice
    if (!isset($_GET['message'])) {
        return;
    }

    $message = sanitize_text_field(wp_unslash($_GET['message']));
    // phpcs:enable WordPress.Security.NonceVerification.Recommended
    
    $notices = array(
        'deleted'         => array('type' => 'success', 'text' => esc_html__('Calculator deleted successfully!', 'pricewise-calculator-pro')),
        'duplicated'      => array('type' => 'success', 'text' => esc_html__('Calculator duplicated successfully!', 'pricewise-calculator-pro')),
        'bulk_delete'     => array('type' => 'success', 'text' => esc_html__('Selected calculators deleted successfully!', 'pricewise-calculator-pro')),
        'bulk_activate'   => array('type' => 'success', 'text' => esc_html__('Selected calculators activated successfully!', 'pricewise-calculator-pro')),
        'bulk_deactivate' => array('type' => 'success', 'text' => esc_html__('Selected calculators deactivated successfully!', 'pricewise-calculator-pro')),
        'error'           => array('type' => 'error',   'text' => esc_html__('An error occurred. Please try again.', 'pricewise-calculator-pro')),
    );

    if (isset($notices[$message])) {
        $notice = $notices[$message];
        printf(
            '<div class="notice notice-%1$s is-dismissible"><p>%2$s</p></div>',
            esc_attr($notice['type']),
            wp_kses_post($notice['text'])
        );
    }
}

    /**
     * Render add/edit calculator page
     */
public function pwcp_render_add_calculator_page() {
    // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Read-only GET parameters for page display
    $action        = isset($_GET['action']) ? sanitize_text_field(wp_unslash($_GET['action'])) : 'add';
    $calculator_id = isset($_GET['calculator_id']) ? absint(wp_unslash($_GET['calculator_id'])) : 0;
    // phpcs:enable WordPress.Security.NonceVerification.Recommended

        $valid_actions = array('add', 'edit');
        if (!in_array($action, $valid_actions, true)) {
            $action = 'add';
        }

        if ($calculator_id && $action === 'add') {
            $action = 'edit';
        }

        if ($action === 'edit' && !$calculator_id) {
            wp_safe_redirect(esc_url_raw(admin_url('admin.php?page=pwcp-add-calculator')));
            exit;
        }

        if (!$this->pwcp_validate_calculator_access($calculator_id, $action)) {
            if ($action === 'edit' && $calculator_id) {
                wp_die(esc_html__('Calculator not found or you do not have permission to edit it.', 'pricewise-calculator-pro'));
            } else {
                wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'pricewise-calculator-pro'));
            }
        }

        if ($this->pwcp_load_admin_class('PWCP_Admin_Calculator_Form_Editor', 'class-pwcp-admin-calculator-form-editor.php')) {
            $form_editor = new PWCP_Admin_Calculator_Form_Editor();

            if ($action === 'edit' && $calculator_id) {
                global $wpdb;
                $calculator = $wpdb->get_row(
                    $wpdb->prepare(
                        "SELECT id, name, status, created_at, updated_at, description, assignment_type, formula, priority
                         FROM `{$wpdb->prefix}pwcp_calculators`
                         WHERE id = %d",
                        $calculator_id
                    )
                );

                if (!$calculator) {
                    wp_die(esc_html__('Calculator not found.', 'pricewise-calculator-pro'));
                }

                $form_editor->pwcp_render_edit_page($calculator_id);
            } else {
                $form_editor->pwcp_render_add_page();
            }
        } else {
            $this->pwcp_render_missing_class_page('Calculator Form Editor');
        }
    }

    /**
     * Validate calculator access
     */
    private function pwcp_validate_calculator_access($calculator_id, $action = 'edit') {
        if (!current_user_can('edit_pwcp_calculators')) {
            return false;
        }

        if ($action === 'add') {
            return true;
        }

        if ($action === 'edit' && $calculator_id) {
            global $wpdb;
            $calculator = $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT id FROM `{$wpdb->prefix}pwcp_calculators` WHERE id = %d",
                    $calculator_id
                )
            );

            return !empty($calculator);
        }

        return false;
    }

    /**
     * Render settings page
     */
    public function pwcp_render_settings_page() {
        if ($this->pwcp_load_admin_class('PWCP_Admin_Settings', 'class-pwcp-admin-settings.php')) {
            $settings = new PWCP_Admin_Settings();
            $settings->pwcp_render();
        } else {
            $this->pwcp_render_missing_class_page('Settings');
        }
    }

    /**
     * Render import/export page
     */
    public function pwcp_render_import_export_page() {
        if ($this->pwcp_load_admin_class('PWCP_Admin_Import_Export', 'class-pwcp-admin-import-export.php')) {
            $import_export = new PWCP_Admin_Import_Export();
            $import_export->pwcp_render();
        } else {
            $this->pwcp_render_missing_class_page('Import/Export');
        }
    }

    /**
     * Load admin class safely
     */
    private function pwcp_load_admin_class($class_name, $filename) {
        if (class_exists($class_name)) {
            return true;
        }

        $filename = sanitize_file_name($filename);
        if (empty($filename) || strpos($filename, '..') !== false) {
            return false;
        }

        $file_path = PWCP_ADMIN_DIR . $filename;

        if (file_exists($file_path) && is_readable($file_path)) {
            require_once $file_path;
            return class_exists($class_name);
        }

        return false;
    }

    /**
     * Render missing class page
     */
    private function pwcp_render_missing_class_page($feature_name) {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html($feature_name); ?></h1>

            <div class="notice notice-error">
                <h3><?php esc_html_e('Component Not Available', 'pricewise-calculator-pro'); ?></h3>
          <p>
    <?php
    printf(
        /* translators: %s: Component name */
        esc_html__('The %s component could not be loaded. This may be due to a file permission issue or incomplete installation.', 'pricewise-calculator-pro'),
        '<strong>' . esc_html($feature_name) . '</strong>'
    );
    ?>
</p>
            </div>

            <div class="card">
                <h2><?php esc_html_e('Troubleshooting Steps', 'pricewise-calculator-pro'); ?></h2>
                <ol>
                    <li>
                        <strong><?php esc_html_e('Check File Permissions', 'pricewise-calculator-pro'); ?></strong>
                        <p><?php esc_html_e('Ensure the plugin directory has proper read permissions.', 'pricewise-calculator-pro'); ?></p>
                    </li>
                    <li>
                        <strong><?php esc_html_e('Reinstall Plugin', 'pricewise-calculator-pro'); ?></strong>
                        <p><?php esc_html_e('Deactivate and reinstall the plugin to restore missing files.', 'pricewise-calculator-pro'); ?></p>
                    </li>
                    <li>
                        <strong><?php esc_html_e('Contact Support', 'pricewise-calculator-pro'); ?></strong>
                        <p><?php esc_html_e('If the issue persists, please contact our support team.', 'pricewise-calculator-pro'); ?></p>
                    </li>
                </ol>
            </div>
        </div>
        <?php
    }

    /**
     * Get current admin page
     */
    public function pwcp_get_current_page() {
        return isset($_GET['page']) ? sanitize_text_field(wp_unslash($_GET['page'])) : '';
    }

    /**
     * Get admin page URL
     */
    public function pwcp_get_admin_url($page, $args = array()) {
        $base_url = admin_url('admin.php?page=' . $page);

        if (!empty($args)) {
            $base_url = add_query_arg($args, $base_url);
        }

        return esc_url($base_url);
    }
}