<?php
/**
 * Admin Import Export Manager
 * Handles calculator import/export functionality
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * PWCP Admin Import Export Class
 * 
 * @since 1.0.0
 */
class PWCP_Admin_Import_Export {
    
    /**
     * Maximum upload file size (5MB)
     * 
     * @var int
     */
    private $max_upload_size = 5242880;
    
    /**
     * Constructor
     * 
     * @since 1.0.0
     */
    public function __construct() {
        add_action('admin_init', array($this, 'pwcp_handle_import_export'));
        add_action('admin_enqueue_scripts', array($this, 'pwcp_enqueue_assets'));
    }
    
    /**
     * Render import/export page
     * 
     * @since 1.0.0
     */
    public function pwcp_render() {
        // Check user capabilities
        if (!current_user_can('manage_pwcp_calculators')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'pricewise-calculator-pro'));
        }
        
        ?>
        <div class="wrap pwcp-modern-admin pwcp-page-settings">
            <!-- Modern Hero Header -->
            <div class="pwcp-hero-header">
                <div class="pwcp-hero-background">
                    <div class="pwcp-hero-particles"></div>
                    <div class="pwcp-hero-gradient"></div>
                </div>
                
                <div class="pwcp-hero-content">
                    <div class="pwcp-hero-left">
                        <div class="pwcp-hero-badge">
                            <span><?php esc_html_e('Data Management', 'pricewise-calculator-pro'); ?></span>
                        </div>
                        <h1 class="pwcp-hero-title">
                            <?php esc_html_e('Import & Export', 'pricewise-calculator-pro'); ?>
                        </h1>
                        <p class="pwcp-hero-subtitle">
                            <?php esc_html_e('Backup your calculators, migrate between sites, or restore from previous exports with powerful import and export tools.', 'pricewise-calculator-pro'); ?>
                        </p>
                    </div>
                </div>
            </div>

            <!-- Modern Container -->
            <div class="pwcp-modern-container">
                <?php $this->pwcp_display_notices(); ?>
                
                <div class="pwcp-settings-grid">
                    <!-- Main Content Area -->
                    <div class="pwcp-settings-main">
                        
                        <!-- Export Section -->
                        <div class="pwcp-settings-section pwcp-export-section">
                            <h3 class="pwcp-section-header">
                                <svg class="pwcp-section-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <path d="M21 15v4a2 2 0 01-2 2H5a2 2 0 01-2-2v-4"/>
                                    <polyline points="7,10 12,15 17,10"/>
                                    <line x1="12" y1="15" x2="12" y2="3"/>
                                </svg>
                                <?php esc_html_e('Export Data', 'pricewise-calculator-pro'); ?>
                            </h3>
                            <p class="pwcp-section-description">
                                <?php esc_html_e('Choose what to export: calculators, settings, or both.', 'pricewise-calculator-pro'); ?>
                            </p>
                            
                            <form method="post" class="pwcp-export-form">
                                <?php wp_nonce_field('pwcp_export_data', 'pwcp_export_nonce'); ?>
                                <input type="hidden" name="pwcp_action" value="export">
                                
                                <table class="form-table pwcp-form-table-reset">
                                    <tr>
                                        <th scope="row" class="pwcp-table-label">
                                            <?php esc_html_e('Export Type', 'pricewise-calculator-pro'); ?>
                                        </th>
                                        <td>
                                            <div class="pwcp-field-group">
                                                <label class="pwcp-checkbox-label pwcp-radio-label-with-desc">
                                                    <input type="radio" name="export_type" value="calculators" checked class="pwcp-checkbox-input">
                                                    <div class="pwcp-checkbox-custom pwcp-radio-custom"></div>
                                                    <span class="pwcp-checkbox-text">
                                                        <strong class="pwcp-radio-title"><?php esc_html_e('Calculators Only', 'pricewise-calculator-pro'); ?></strong>
                                                        <span class="pwcp-radio-description">
                                                            <?php esc_html_e('Export all calculators with their fields and formulas', 'pricewise-calculator-pro'); ?>
                                                        </span>
                                                    </span>
                                                </label>
                                            </div>
                                            <div class="pwcp-field-group pwcp-field-group-spaced">
                                                <label class="pwcp-checkbox-label pwcp-radio-label-with-desc">
                                                    <input type="radio" name="export_type" value="settings" class="pwcp-checkbox-input">
                                                    <div class="pwcp-checkbox-custom pwcp-radio-custom"></div>
                                                    <span class="pwcp-checkbox-text">
                                                        <strong class="pwcp-radio-title"><?php esc_html_e('Settings Only', 'pricewise-calculator-pro'); ?></strong>
                                                        <span class="pwcp-radio-description">
                                                            <?php esc_html_e('Export plugin configuration, design, and calculation settings', 'pricewise-calculator-pro'); ?>
                                                        </span>
                                                    </span>
                                                </label>
                                            </div>
                                            <div class="pwcp-field-group pwcp-field-group-spaced">
                                                <label class="pwcp-checkbox-label pwcp-radio-label-with-desc">
                                                    <input type="radio" name="export_type" value="both" class="pwcp-checkbox-input">
                                                    <div class="pwcp-checkbox-custom pwcp-radio-custom"></div>
                                                    <span class="pwcp-checkbox-text">
                                                        <strong class="pwcp-radio-title"><?php esc_html_e('Everything', 'pricewise-calculator-pro'); ?></strong>
                                                        <span class="pwcp-radio-description">
                                                            <?php esc_html_e('Export both calculators and settings in one file', 'pricewise-calculator-pro'); ?>
                                                        </span>
                                                    </span>
                                                </label>
                                            </div>
                                        </td>
                                    </tr>
                                </table>
                                
                                <div class="pwcp-button-container">
                                    <button type="submit" name="export" class="pwcp-btn pwcp-btn-primary">
                                        <svg class="pwcp-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <path d="M21 15v4a2 2 0 01-2 2H5a2 2 0 01-2-2v-4"/>
                                            <polyline points="7,10 12,15 17,10"/>
                                            <line x1="12" y1="15" x2="12" y2="3"/>
                                        </svg>
                                        <?php esc_html_e('Export Data', 'pricewise-calculator-pro'); ?>
                                    </button>
                                </div>
                            </form>
                        </div>
                        
                        <!-- Import Section -->
                        <div class="pwcp-settings-section">
                            <h3 class="pwcp-section-header">
                                <svg class="pwcp-section-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <path d="M21 15v4a2 2 0 01-2 2H5a2 2 0 01-2-2v-4"/>
                                    <polyline points="17,8 12,3 7,8"/>
                                    <line x1="12" y1="3" x2="12" y2="15"/>
                                </svg>
                                <?php esc_html_e('Import Data', 'pricewise-calculator-pro'); ?>
                            </h3>
                            <p class="pwcp-section-description">
                                <?php esc_html_e('Import calculators or settings from a previously exported JSON file.', 'pricewise-calculator-pro'); ?>
                            </p>
                            
                            <form method="post" enctype="multipart/form-data" class="pwcp-import-form">
                                <?php wp_nonce_field('pwcp_import_data', 'pwcp_import_nonce'); ?>
                                <input type="hidden" name="pwcp_action" value="import">
                                
                                <table class="form-table pwcp-form-table-reset">
                                    <tr>
                                        <th scope="row" class="pwcp-table-label">
                                            <?php esc_html_e('Import File', 'pricewise-calculator-pro'); ?>
                                        </th>
                                        <td>
                                            <div class="pwcp-field-group">
                                                <input type="file" name="import_file" accept=".json" required class="pwcp-file-input">
                                                <div class="pwcp-field-hint pwcp-field-hint-spaced">
                                                    <?php esc_html_e('Select a JSON file exported from PriceWise Calculator Pro.', 'pricewise-calculator-pro'); ?>
                                                </div>
                                            </div>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row" class="pwcp-table-label">
                                            <?php esc_html_e('Import Type', 'pricewise-calculator-pro'); ?>
                                        </th>
                                        <td>
                                            <div class="pwcp-field-group">
                                                <label class="pwcp-checkbox-label pwcp-radio-label-with-desc">
                                                    <input type="radio" name="import_type" value="auto" checked class="pwcp-checkbox-input">
                                                    <div class="pwcp-checkbox-custom pwcp-radio-custom"></div>
                                                    <span class="pwcp-checkbox-text">
                                                        <strong class="pwcp-radio-title"><?php esc_html_e('Auto-Detect', 'pricewise-calculator-pro'); ?></strong>
                                                        <span class="pwcp-radio-description">
                                                            <?php esc_html_e('Automatically import whatever the file contains (recommended)', 'pricewise-calculator-pro'); ?>
                                                        </span>
                                                    </span>
                                                </label>
                                            </div>
                                            <div class="pwcp-field-group pwcp-field-group-spaced">
                                                <label class="pwcp-checkbox-label pwcp-radio-label-with-desc">
                                                    <input type="radio" name="import_type" value="calculators_only" class="pwcp-checkbox-input">
                                                    <div class="pwcp-checkbox-custom pwcp-radio-custom"></div>
                                                    <span class="pwcp-checkbox-text">
                                                        <strong class="pwcp-radio-title"><?php esc_html_e('Calculators Only', 'pricewise-calculator-pro'); ?></strong>
                                                        <span class="pwcp-radio-description">
                                                            <?php esc_html_e('Import only calculators, ignore settings even if present', 'pricewise-calculator-pro'); ?>
                                                        </span>
                                                    </span>
                                                </label>
                                            </div>
                                            <div class="pwcp-field-group pwcp-field-group-spaced">
                                                <label class="pwcp-checkbox-label pwcp-radio-label-with-desc">
                                                    <input type="radio" name="import_type" value="settings_only" class="pwcp-checkbox-input">
                                                    <div class="pwcp-checkbox-custom pwcp-radio-custom"></div>
                                                    <span class="pwcp-checkbox-text">
                                                        <strong class="pwcp-radio-title"><?php esc_html_e('Settings Only', 'pricewise-calculator-pro'); ?></strong>
                                                        <span class="pwcp-radio-description">
                                                            <?php esc_html_e('Import only settings, ignore calculators even if present', 'pricewise-calculator-pro'); ?>
                                                        </span>
                                                    </span>
                                                </label>
                                            </div>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row" class="pwcp-table-label">
                                            <?php esc_html_e('Import Options', 'pricewise-calculator-pro'); ?>
                                        </th>
                                        <td>
                                            <div class="pwcp-field-group">
                                                <label class="pwcp-checkbox-label">
                                                    <input type="checkbox" name="overwrite_existing" value="1" class="pwcp-checkbox-input">
                                                    <div class="pwcp-checkbox-custom"></div>
                                                    <span class="pwcp-checkbox-text">
                                                        <?php esc_html_e('Overwrite existing calculators with same name', 'pricewise-calculator-pro'); ?>
                                                    </span>
                                                </label>
                                            </div>
                                        </td>
                                    </tr>
                                </table>
                                
                                <div class="pwcp-button-container">
                                    <button type="submit" name="import" class="pwcp-btn pwcp-btn-primary">
                                        <svg class="pwcp-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <path d="M21 15v4a2 2 0 01-2 2H5a2 2 0 01-2-2v-4"/>
                                            <polyline points="17,8 12,3 7,8"/>
                                            <line x1="12" y1="3" x2="12" y2="15"/>
                                        </svg>
                                        <?php esc_html_e('Import Data', 'pricewise-calculator-pro'); ?>
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                    
                    <!-- Sidebar -->
                    <div class="pwcp-settings-sidebar">
                        <?php $this->pwcp_render_sidebar(); ?>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render modern sidebar
     */
    private function pwcp_render_sidebar() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'pwcp_calculators';
// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name from $wpdb->prefix
$calculator_count = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM `" . esc_sql( $table_name ) . "` WHERE 1=%d", 1 ) );
        
        ?>
        <!-- Quick Actions Widget -->
        <div class="pwcp-sidebar-widget">
            <div class="pwcp-widget-header">
                <h3><?php esc_html_e('Quick Actions', 'pricewise-calculator-pro'); ?></h3>
            </div>
            <div class="pwcp-widget-content">
                <div class="pwcp-quick-actions">
                    <a href="<?php echo esc_url(admin_url('admin.php?page=pwcp-calculators')); ?>" class="pwcp-quick-action pwcp-action-primary">
                        <div class="pwcp-action-icon">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                                <line x1="9" y1="12" x2="15" y2="12"/>
                                <line x1="12" y1="9" x2="12" y2="15"/>
                            </svg>
                        </div>
                        <div class="pwcp-action-content">
                            <div class="pwcp-action-title"><?php esc_html_e('Manage Calculators', 'pricewise-calculator-pro'); ?></div>
                            <div class="pwcp-action-desc"><?php esc_html_e('View and edit all calculators', 'pricewise-calculator-pro'); ?></div>
                        </div>
                        <div class="pwcp-action-arrow">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <polyline points="9,18 15,12 9,6"/>
                            </svg>
                        </div>
                    </a>

                    <a href="<?php echo esc_url(admin_url('admin.php?page=pwcp-settings')); ?>" class="pwcp-quick-action">
                        <div class="pwcp-action-icon">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <circle cx="12" cy="12" r="3"/>
                                <path d="M19.4 15a1.65 1.65 0 00.33 1.82l.06.06a2 2 0 010 2.83 2 2 0 01-2.83 0l-.06-.06a1.65 1.65 0 00-1.82-.33 1.65 1.65 0 00-1 1.51V21a2 2 0 01-2 2 2 2 0 01-2-2v-.09A1.65 1.65 0 009 19.4a1.65 1.65 0 00-1.82.33l-.06.06a2 2 0 01-2.83 0 2 2 0 010-2.83l.06-.06a1.65 1.65 0 00.33-1.82 1.65 1.65 0 00-1.51-1H3a2 2 0 01-2-2 2 2 0 012-2h.09A1.65 1.65 0 004.6 9a1.65 1.65 0 00-.33-1.82l-.06-.06a2 2 0 010-2.83 2 2 0 012.83 0l.06.06a1.65 1.65 0 001.82.33H9a1.65 1.65 0 001-1.51V3a2 2 0 012-2 2 2 0 012 2v.09a1.65 1.65 0 001 1.51 1.65 1.65 0 001.82-.33l.06-.06a2 2 0 012.83 0 2 2 0 010 2.83l-.06.06a1.65 1.65 0 00-.33 1.82V9a1.65 1.65 0 001.51 1H21a2 2 0 012 2 2 2 0 01-2 2h-.09a1.65 1.65 0 00-1.51 1z"/>
                            </svg>
                        </div>
                        <div class="pwcp-action-content">
                            <div class="pwcp-action-title"><?php esc_html_e('Settings', 'pricewise-calculator-pro'); ?></div>
                            <div class="pwcp-action-desc"><?php esc_html_e('Configure plugin settings', 'pricewise-calculator-pro'); ?></div>
                        </div>
                        <div class="pwcp-action-arrow">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <polyline points="9,18 15,12 9,6"/>
                            </svg>
                        </div>
                    </a>
                </div>
            </div>
        </div>

        <!-- Import/Export Guide Widget -->
        <div class="pwcp-sidebar-widget">
            <div class="pwcp-widget-header">
                <h3><?php esc_html_e('Import/Export Guide', 'pricewise-calculator-pro'); ?></h3>
            </div>
            <div class="pwcp-widget-content">
                <div class="pwcp-tips-list">
                    <div class="pwcp-tip-item">
                        <div class="pwcp-tip-number">1</div>
                        <div class="pwcp-tip-content">
                            <h4><?php esc_html_e('Calculators Export', 'pricewise-calculator-pro'); ?></h4>
                            <p><?php esc_html_e('Export only calculator data with fields and formulas. Perfect for migrating calculators between sites.', 'pricewise-calculator-pro'); ?></p>
                        </div>
                    </div>

                    <div class="pwcp-tip-item">
                        <div class="pwcp-tip-number">2</div>
                        <div class="pwcp-tip-content">
                            <h4><?php esc_html_e('Settings Export', 'pricewise-calculator-pro'); ?></h4>
                            <p><?php esc_html_e('Export only plugin configuration, design, and calculation settings without any calculators.', 'pricewise-calculator-pro'); ?></p>
                        </div>
                    </div>

                    <div class="pwcp-tip-item">
                        <div class="pwcp-tip-number">3</div>
                        <div class="pwcp-tip-content">
                            <h4><?php esc_html_e('Complete Backup', 'pricewise-calculator-pro'); ?></h4>
                            <p><?php esc_html_e('Export everything in one file for a complete backup. Import with auto-detect for easy restoration.', 'pricewise-calculator-pro'); ?></p>
                        </div>
                    </div>

                    <div class="pwcp-tip-item">
                        <div class="pwcp-tip-number">4</div>
                        <div class="pwcp-tip-content">
                            <h4><?php esc_html_e('Selective Import', 'pricewise-calculator-pro'); ?></h4>
                            <p><?php esc_html_e('Choose to import only calculators or only settings, even if the file contains both.', 'pricewise-calculator-pro'); ?></p>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- System Info Widget -->
        <div class="pwcp-sidebar-widget">
            <div class="pwcp-widget-header">
                <h3><?php esc_html_e('System Information', 'pricewise-calculator-pro'); ?></h3>
            </div>
            <div class="pwcp-widget-content">
                <div class="pwcp-plugin-info">
                    <div class="pwcp-plugin-stats-list">
                        <div class="pwcp-info-item">
                            <div class="pwcp-info-left">
                                <div class="pwcp-info-indicator"></div>
                                <span class="pwcp-info-label"><?php esc_html_e('Total Calculators:', 'pricewise-calculator-pro'); ?></span>
                            </div>
                            <span class="pwcp-info-value"><?php echo esc_html($calculator_count); ?></span>
                        </div>
                        <div class="pwcp-info-item">
                            <div class="pwcp-info-left">
                                <div class="pwcp-info-indicator"></div>
                                <span class="pwcp-info-label"><?php esc_html_e('Plugin Version:', 'pricewise-calculator-pro'); ?></span>
                            </div>
                            <span class="pwcp-info-value"><?php echo esc_html(PWCP_PLUGIN_VERSION); ?></span>
                        </div>
                        <div class="pwcp-info-item">
                            <div class="pwcp-info-left">
                                <div class="pwcp-info-indicator"></div>
                                <span class="pwcp-info-label"><?php esc_html_e('Export Format:', 'pricewise-calculator-pro'); ?></span>
                            </div>
                            <span class="pwcp-info-value">JSON</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Handle import/export actions
     * 
     * @since 1.0.0
     */
  public function pwcp_handle_import_export() {
    // Check if either nonce exists first
    $has_export_nonce = isset($_POST['pwcp_export_nonce']);
    $has_import_nonce = isset($_POST['pwcp_import_nonce']);
    
    if (!$has_export_nonce && !$has_import_nonce) {
        return;
    }
    
    if (!isset($_POST['pwcp_action'])) {
        return;
    }
    
    $action = sanitize_text_field(wp_unslash($_POST['pwcp_action']));
    
    if ($action === 'export') {
        $this->pwcp_handle_export();
    } elseif ($action === 'import') {
        $this->pwcp_handle_import();
    }
}
    
    /**
     * Handle export action
     * 
     * @since 1.0.0
     */
    private function pwcp_handle_export() {
        // Verify nonce
if (!isset($_POST['pwcp_export_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['pwcp_export_nonce'])), 'pwcp_export_data')) {
            wp_die(esc_html__('Security verification failed.', 'pricewise-calculator-pro'));
        }
        
        // Check permissions
        if (!current_user_can('manage_pwcp_calculators')) {
            wp_die(esc_html__('Insufficient permissions.', 'pricewise-calculator-pro'));
        }
        
        // Get export type
$export_type = isset($_POST['export_type']) ? sanitize_text_field(wp_unslash($_POST['export_type'])) : 'calculators';
        
        // Create base export data
        $export_data = array(
            'version' => PWCP_PLUGIN_VERSION,
            'timestamp' => current_time('timestamp'),
            'export_type' => $export_type
        );
        
        // Add data based on export type
        switch ($export_type) {
            case 'calculators':
                $export_data['calculators'] = $this->pwcp_get_export_calculators();
                $filename_prefix = 'calculators';
                break;
                
            case 'settings':
                $export_data['settings'] = $this->pwcp_get_export_settings();
                $filename_prefix = 'settings';
                update_option('pwcp_last_settings_export', current_time('mysql'), false);
                break;
                
            case 'both':
                $export_data['calculators'] = $this->pwcp_get_export_calculators();
                $export_data['settings'] = $this->pwcp_get_export_settings();
                $filename_prefix = 'complete';
                update_option('pwcp_last_settings_export', current_time('mysql'), false);
                break;
                
            default:
                $export_data['calculators'] = $this->pwcp_get_export_calculators();
                $filename_prefix = 'calculators';
                break;
        }
        
        // Generate filename
$filename = 'pwcp-' . $filename_prefix . '-export-' . gmdate('Y-m-d-H-i-s') . '.json';
        
        // Set headers for download
        header('Content-Type: application/json');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Cache-Control: no-cache, must-revalidate');
        header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');
        
        // Output JSON data
        echo wp_json_encode($export_data, JSON_PRETTY_PRINT);
        exit;
    }
    
    /**
     * Handle import action with validation
     * 
     * @since 1.0.0
     */
    private function pwcp_handle_import() {
        // Verify nonce
if (!isset($_POST['pwcp_import_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['pwcp_import_nonce'])), 'pwcp_import_data')) {
            wp_die(esc_html__('Security verification failed.', 'pricewise-calculator-pro'));
        }
        
        // Check permissions
        if (!current_user_can('manage_pwcp_calculators')) {
            wp_die(esc_html__('Insufficient permissions.', 'pricewise-calculator-pro'));
        }
        
        // Validate file upload
if (!isset($_FILES['import_file']['error']) || $_FILES['import_file']['error'] !== UPLOAD_ERR_OK) {
            $this->pwcp_add_notice('error', esc_html__('Please select a valid file to import.', 'pricewise-calculator-pro'));
            return;
        }
        
        // Validate file size
if (isset($_FILES['import_file']['size']) && $_FILES['import_file']['size'] > $this->max_upload_size) {
            $this->pwcp_add_notice('error', esc_html__('File is too large. Maximum size is 5MB.', 'pricewise-calculator-pro'));
            return;
        }
        
        // Validate file extension
$file_ext = isset($_FILES['import_file']['name']) ? strtolower(pathinfo(sanitize_file_name(wp_unslash($_FILES['import_file']['name'])), PATHINFO_EXTENSION)) : '';
        if ($file_ext !== 'json') {
            $this->pwcp_add_notice('error', esc_html__('Invalid file type. Only JSON files are allowed.', 'pricewise-calculator-pro'));
            return;
        }
        
        // Validate MIME type with fallback
        if (function_exists('finfo_open')) {
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
$mime_type = isset($_FILES['import_file']['tmp_name']) ? finfo_file($finfo, sanitize_text_field(wp_unslash($_FILES['import_file']['tmp_name']))) : '';
            finfo_close($finfo);
            
            $allowed_mimes = array('application/json', 'text/plain');
            if (!in_array($mime_type, $allowed_mimes, true)) {
                $this->pwcp_add_notice('error', esc_html__('Invalid file MIME type. Only JSON files are allowed.', 'pricewise-calculator-pro'));
                return;
            }
        }
        
        // Read and validate file
$file_content = isset($_FILES['import_file']['tmp_name']) ? file_get_contents(sanitize_text_field(wp_unslash($_FILES['import_file']['tmp_name']))) : false;
        
        if ($file_content === false) {
            $this->pwcp_add_notice('error', esc_html__('Failed to read the uploaded file.', 'pricewise-calculator-pro'));
            return;
        }
        
        $import_data = json_decode($file_content, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->pwcp_add_notice('error', esc_html__('Invalid JSON file.', 'pricewise-calculator-pro'));
            return;
        }
        
        // Validate import data structure
        if (!$this->pwcp_validate_import_data($import_data)) {
            $this->pwcp_add_notice('error', esc_html__('Invalid import file format.', 'pricewise-calculator-pro'));
            return;
        }
        
        // Get import type
$import_type = isset($_POST['import_type']) ? sanitize_text_field(wp_unslash($_POST['import_type'])) : 'auto';
        
        // Determine what to import
        $import_calculators = false;
        $import_settings = false;
        
        switch ($import_type) {
            case 'calculators_only':
                $import_calculators = isset($import_data['calculators']) && !empty($import_data['calculators']);
                break;
                
            case 'settings_only':
                $import_settings = isset($import_data['settings']) && !empty($import_data['settings']);
                break;
                
            case 'auto':
            default:
                $import_calculators = isset($import_data['calculators']) && !empty($import_data['calculators']);
                $import_settings = isset($import_data['settings']) && !empty($import_data['settings']);
                break;
        }
        
        // Track import results
        $results = array(
            'calculators_imported' => 0,
            'settings_imported' => false
        );
        
        // Import calculators if applicable
        if ($import_calculators) {
            $calculator_result = $this->pwcp_import_calculators($import_data);
            if ($calculator_result['success']) {
                $results['calculators_imported'] = $calculator_result['imported_count'];
            }
        }
        
        // Import settings if applicable
        if ($import_settings) {
            $this->pwcp_import_settings($import_data['settings']);
            $results['settings_imported'] = true;
        }
        
        // Generate success message
        $messages = array();
        
if ($results['calculators_imported'] > 0) {
    $messages[] = sprintf(
        // translators: %d: Number of calculators imported
        esc_html__('Successfully imported %d calculator(s).', 'pricewise-calculator-pro'),
        $results['calculators_imported']
    );
}
        
        if ($results['settings_imported']) {
            $messages[] = esc_html__('Successfully imported plugin settings.', 'pricewise-calculator-pro');
        }
        
        if (empty($messages)) {
            $this->pwcp_add_notice('error', esc_html__('No data was imported. Please check your import file and type selection.', 'pricewise-calculator-pro'));
        } else {
            $this->pwcp_add_notice('success', implode(' ', $messages));
        }
    }
    
    /**
     * Get calculators for export
     * 
     * @return array Export calculator data
     * @since 1.0.0
     */
    private function pwcp_get_export_calculators() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'pwcp_calculators';
        
// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name from $wpdb->prefix
$calculators = $wpdb->get_results(
    $wpdb->prepare(
        "SELECT id, name, description, formula, settings, status, assignment_type, assignment_ids, priority, created_at, updated_at 
         FROM `" . esc_sql( $table_name ) . "` 
         WHERE 1=%d
         ORDER BY id ASC",
        1
    ),
    ARRAY_A
);
        
        if (!$calculators) {
            return array();
        }
        
        $fields_table = $wpdb->prefix . 'pwcp_calculator_fields';
        
        foreach ($calculators as &$calculator) {
    $fields = $wpdb->get_results(
    $wpdb->prepare(
        "SELECT id, calculator_id, field_key, field_label, field_type, field_options, 
                field_validation, field_default, field_required, field_order, 
                created_at, updated_at 
         FROM `" . esc_sql($fields_table) . "` 
         WHERE calculator_id = %d 
         ORDER BY field_order ASC",
        $calculator['id']
    ),
    ARRAY_A
);
            
            $calculator['fields'] = $fields ? $fields : array();
        }
        
        return $calculators;
    }
    
    /**
     * Get settings for export
     * 
     * @return array Export settings data
     * @since 1.0.0
     */
    private function pwcp_get_export_settings() {
        return array(
            'pwcp_general_settings' => get_option('pwcp_general_settings', array()),
            'pwcp_design_settings' => get_option('pwcp_design_settings', array()),
            'pwcp_calculation_settings' => get_option('pwcp_calculation_settings', array()),
            'pwcp_text_settings' => get_option('pwcp_text_settings', array()),
            'pwcp_advanced_settings' => get_option('pwcp_advanced_settings', array()),
        );
    }
    
    /**
     * Validate import data structure
     * 
     * @param array $data Import data
     * @return bool True if valid
     * @since 1.0.0
     */
    private function pwcp_validate_import_data($data) {
        // Must have version
        if (!isset($data['version'])) {
            return false;
        }
        
        // Must have either calculators or settings
        $has_calculators = isset($data['calculators']) && is_array($data['calculators']);
        $has_settings = isset($data['settings']) && is_array($data['settings']);
        
        if (!$has_calculators && !$has_settings) {
            return false;
        }
        
        // Validate calculators if present
        if ($has_calculators) {
            foreach ($data['calculators'] as $calculator) {
                if (!isset($calculator['name']) || !isset($calculator['formula'])) {
                    return false;
                }
                
                // Validate fields if they exist
                if (isset($calculator['fields']) && is_array($calculator['fields'])) {
                    foreach ($calculator['fields'] as $field) {
                        if (empty($field['field_key']) && empty($field['field_id'])) {
                            return false;
                        }
                    }
                }
            }
        }
        
        // Settings validation
        if ($has_settings && !is_array($data['settings'])) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Import calculators with proper sanitization
     * 
     * @param array $data Import data
     * @return array Import result
     * @since 1.0.0
     */
    private function pwcp_import_calculators($data) {
        global $wpdb;
        
        $imported_count = 0;
$overwrite = isset($_POST['overwrite_existing']) && !empty($_POST['overwrite_existing']);
        $table_name = $wpdb->prefix . 'pwcp_calculators';
        
        foreach ($data['calculators'] as $calculator_data) {
            $existing_id = null;
            if ($overwrite) {
              $existing_id = $wpdb->get_var(
    $wpdb->prepare(
        "SELECT id FROM `" . esc_sql($table_name) . "` WHERE name = %s",
        $calculator_data['name']
    )
);
            }
            
            // Sanitize calculator settings
            $settings_raw = $calculator_data['settings'] ?? '';
            
            if (is_array($settings_raw)) {
                $settings_sanitized = wp_json_encode($this->pwcp_sanitize_settings_array($settings_raw));
            } elseif (is_string($settings_raw)) {
                $decoded = json_decode($settings_raw, true);
                if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                    $settings_sanitized = wp_json_encode($this->pwcp_sanitize_settings_array($decoded));
                } else {
                    $settings_sanitized = sanitize_text_field($settings_raw);
                }
            } else {
                $settings_sanitized = '';
            }
            
            $calculator_insert_data = array(
                'name' => sanitize_text_field($calculator_data['name']),
                'description' => sanitize_textarea_field($calculator_data['description'] ?? ''),
                'formula' => sanitize_textarea_field($calculator_data['formula']),
                'settings' => $settings_sanitized,
                'status' => sanitize_text_field($calculator_data['status'] ?? 'active'),
                'assignment_type' => sanitize_text_field($calculator_data['assignment_type'] ?? 'global'),
                'assignment_ids' => sanitize_text_field($calculator_data['assignment_ids'] ?? ''),
                'priority' => absint($calculator_data['priority'] ?? 1),
                'updated_at' => current_time('mysql')
            );
            
            if ($existing_id) {
                $result = $wpdb->update(
                    $table_name,
                    $calculator_insert_data,
                    array('id' => $existing_id),
                    array('%s', '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%s'),
                    array('%d')
                );
                
                $calculator_id = $existing_id;
            } else {
                $calculator_insert_data['created_at'] = current_time('mysql');
                
                $result = $wpdb->insert(
                    $table_name,
                    $calculator_insert_data,
                    array('%s', '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%s', '%s')
                );
                
                $calculator_id = $wpdb->insert_id;
            }
            
            if ($result !== false && $calculator_id) {
                if (isset($calculator_data['fields']) && is_array($calculator_data['fields'])) {
                    $this->pwcp_import_calculator_fields($calculator_id, $calculator_data['fields']);
                }
                
                $imported_count++;
            }
        }
        
        return array(
            'success' => true,
            'imported_count' => $imported_count
        );
    }
    
    /**
     * Import calculator fields with proper sanitization
     * 
     * @param int $calculator_id Calculator ID
     * @param array $fields Field data
     * @since 1.0.0
     */
    private function pwcp_import_calculator_fields($calculator_id, $fields) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'pwcp_calculator_fields';
        
        $wpdb->delete(
            $table_name,
            array('calculator_id' => $calculator_id),
            array('%d')
        );
        
        foreach ($fields as $field) {
            // Handle both old and new field key formats
            $field_key = '';
            if (!empty($field['field_key'])) {
                $field_key = $field['field_key'];
            } elseif (!empty($field['field_id'])) {
                $field_key = $field['field_id'];
            }
            
            if (empty($field_key)) {
                continue;
            }
            
            $field_label = '';
            if (!empty($field['field_label'])) {
                $field_label = $field['field_label'];
            }
            
            $field_type = '';
            if (!empty($field['field_type'])) {
                $field_type = $field['field_type'];
            }
            
            if (empty($field_label) || empty($field_type)) {
                continue;
            }
            
            // Handle field default value
            $field_default = '';
            if (isset($field['field_default'])) {
                $field_default = $field['field_default'];
            } elseif (isset($field['default_value'])) {
                $field_default = $field['default_value'];
            }
            
            // Handle field required flag
            $field_required = 0;
            if (isset($field['field_required'])) {
                $field_required = absint($field['field_required']);
            } elseif (isset($field['is_required'])) {
                $field_required = absint($field['is_required']);
            }
            
          // Sanitize field options (preserve newlines for multi-line options)
$field_options_raw = isset($field['field_options']) ? $field['field_options'] : '';
if (is_array($field_options_raw)) {
    $field_options_safe = wp_json_encode($this->pwcp_sanitize_settings_array($field_options_raw));
} elseif (is_string($field_options_raw)) {
    $fo_dec = json_decode($field_options_raw, true);
    if (json_last_error() === JSON_ERROR_NONE && is_array($fo_dec)) {
        $field_options_safe = wp_json_encode($this->pwcp_sanitize_settings_array($fo_dec));
    } else {
        // Use sanitize_textarea_field to preserve newlines
        $field_options_safe = sanitize_textarea_field($field_options_raw); 
    }
} else {
    $field_options_safe = '';
}
            
            // Sanitize field validation
            $field_validation_raw = isset($field['field_validation']) ? $field['field_validation'] : '';
            if (is_array($field_validation_raw)) {
                $field_validation_safe = wp_json_encode($this->pwcp_sanitize_settings_array($field_validation_raw));
            } elseif (is_string($field_validation_raw)) {
                $fv_dec = json_decode($field_validation_raw, true);
                if (json_last_error() === JSON_ERROR_NONE && is_array($fv_dec)) {
                    $field_validation_safe = wp_json_encode($this->pwcp_sanitize_settings_array($fv_dec));
                } else {
                    $field_validation_safe = sanitize_text_field($field_validation_raw);
                }
            } else {
                $field_validation_safe = '';
            }
            
            $wpdb->insert(
                $table_name,
                array(
                    'calculator_id' => $calculator_id,
                    'field_key' => sanitize_key($field_key),
                    'field_type' => sanitize_text_field($field_type),
                    'field_label' => sanitize_text_field($field_label),
                    'field_options' => $field_options_safe,
                    'field_validation' => $field_validation_safe,
                    'field_order' => isset($field['field_order']) ? absint($field['field_order']) : 0,
                    'field_required' => $field_required,
                    'field_default' => sanitize_text_field($field_default),
                    'created_at' => current_time('mysql'),
                    'updated_at' => current_time('mysql')
                ),
                array('%d', '%s', '%s', '%s', '%s', '%s', '%d', '%d', '%s', '%s', '%s')
            );
        }
    }
    
    /**
     * Import settings
     * 
     * @param array $settings Settings data
     * @since 1.0.0
     */
    private function pwcp_import_settings($settings) {
        $allowed_options = array(
            'pwcp_general_settings',
            'pwcp_design_settings',
            'pwcp_calculation_settings',
            'pwcp_text_settings',
            'pwcp_advanced_settings'
        );
        
        foreach ($allowed_options as $option_name) {
            if (isset($settings[$option_name]) && is_array($settings[$option_name])) {
                $sanitized_settings = $this->pwcp_sanitize_settings_array($settings[$option_name]);
                update_option($option_name, $sanitized_settings);
            }
        }
    }
    
    /**
     * Recursively sanitize settings array
     * 
     * @param array $settings Settings array to sanitize
     * @return array Sanitized settings
     * @since 1.0.0
     */
    private function pwcp_sanitize_settings_array($settings) {
        $sanitized = array();
        
        foreach ($settings as $key => $value) {
            $sanitized_key = sanitize_key($key);
            
            if (is_array($value)) {
                $sanitized[$sanitized_key] = $this->pwcp_sanitize_settings_array($value);
            } elseif (is_bool($value)) {
                $sanitized[$sanitized_key] = (bool) $value;
            } elseif (is_numeric($value)) {
                $sanitized[$sanitized_key] = is_float($value) ? floatval($value) : absint($value);
            } elseif (filter_var($value, FILTER_VALIDATE_URL)) {
                $sanitized[$sanitized_key] = esc_url_raw($value);
            } elseif (strpos($value, '<') !== false || strpos($value, '>') !== false) {
                $sanitized[$sanitized_key] = wp_kses_post($value);
            } else {
                $sanitized[$sanitized_key] = sanitize_text_field($value);
            }
        }
        
        return $sanitized;
    }
    
    /**
     * Add admin notice
     * 
     * @param string $type Notice type
     * @param string $message Notice message
     * @since 1.0.0
     */
    private function pwcp_add_notice($type, $message) {
        set_transient('pwcp_import_export_notice', array(
            'type' => sanitize_key($type),
            'message' => sanitize_text_field($message)
        ), 30);
    }
    
    /**
     * Display admin notices
     * 
     * @since 1.0.0
     */
    private function pwcp_display_notices() {
        $notice = get_transient('pwcp_import_export_notice');
        if ($notice) {
            $icon = $notice['type'] === 'success' ? '✓' : '!';
            $notice_class = 'pwcp-notification pwcp-notification-' . esc_attr($notice['type']) . ' pwcp-notification-show pwcp-notification-import-export';
            
            printf(
                '<div class="%s"><span class="pwcp-notification-icon">%s</span><p>%s</p></div>',
                esc_attr($notice_class),
                esc_html($icon),
                esc_html($notice['message'])
            );
            delete_transient('pwcp_import_export_notice');
        }
    }
    
    /**
     * Enqueue assets
     * 
     * @param string $hook Current page hook
     * @since 1.0.0
     */
    public function pwcp_enqueue_assets($hook) {
        if ($hook !== 'calculator-pro_page_pwcp-import-export') {
            return;
        }
        
        wp_enqueue_style(
            'pwcp-admin-settings',
            PWCP_ASSETS_URL . 'css/all-admin-css/admin-pages.css',
            array(),
            PWCP_PLUGIN_VERSION
        );
        
        wp_enqueue_media();
        wp_enqueue_script('jquery');
    }
}