<?php
/**
 * Enhanced Admin Dashboard
 *
 * Modern dashboard interface for managing calculator statistics and quick actions.
 *
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * PWCP Admin Dashboard Class
 *
 * Handles dashboard display and statistics management
 *
 * @since 1.0.0
 */
class PWCP_Admin_Dashboard {

    /**
     * Constructor
     *
     * @since 1.0.0
     */
    public function __construct() {
        add_action( 'admin_enqueue_scripts', array( $this, 'pwcp_enqueue_admin_assets' ) );
    }

    /**
     * Enqueue dashboard assets
     *
     * @param string $hook Current admin page hook
     * @since 1.0.0
     * @return void
     */
    public function pwcp_enqueue_admin_assets( $hook ) {
        if ( ! current_user_can( 'manage_pwcp_calculators' ) ) {
            return;
        }

        $screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
        if ( ! $screen ) {
            return;
        }

        $allowed_screen_ids = array(
            'calculator-pro_page_pwcp-dashboard',
            'toplevel_page_pwcp-dashboard', 
            'admin_page_pwcp-dashboard',
            'pricewise-calculator-pro_page_pwcp-dashboard'
        );

        if ( ! in_array( $screen->id, $allowed_screen_ids, true ) ) {
            return;
        }

        wp_enqueue_style(
            'pwcp-admin-dashboard',
            PWCP_ASSETS_URL . 'css/all-admin-css/admin-pages.css',
            array(),
            PWCP_PLUGIN_VERSION,
            'all'
        );

        wp_enqueue_script(
            'pwcp-admin-dashboard',
            PWCP_ASSETS_URL . 'js/all-admin-js/admin-dashboard.js',
            array( 'jquery', 'wp-i18n' ),
            PWCP_PLUGIN_VERSION,
            true
        );

        wp_set_script_translations(
            'pwcp-admin-dashboard',
            'pricewise-calculator-pro',
            PWCP_PLUGIN_DIR . 'languages'
        );

        wp_localize_script(
            'pwcp-admin-dashboard',
            'pwcpDashboard',
            array(
                'ajaxurl' => admin_url( 'admin-ajax.php' ),
                'nonce'   => wp_create_nonce( 'pwcp_dashboard_nonce' ),
                'strings' => array(
                    'success' => esc_html__( 'Operation completed successfully!', 'pricewise-calculator-pro' ),
                    'error'   => esc_html__( 'An error occurred', 'pricewise-calculator-pro' ),
                ),
            )
        );
    }

    /**
     * Render the dashboard page
     *
     * @since 1.0.0
     * @return void
     */
    public function pwcp_render() {
        if ( ! current_user_can( 'manage_pwcp_calculators' ) ) {
            wp_die( 
                esc_html__( 'You do not have sufficient permissions to access this page.', 'pricewise-calculator-pro' ),
                esc_html__( 'Access Denied', 'pricewise-calculator-pro' ),
                array( 'response' => 403 )
            );
        }

        $stats = $this->pwcp_get_dashboard_stats();
        $recent_calculators = $this->pwcp_get_recent_calculators( 5 );

        // Allowed SVG tags for wp_kses
        $allowed_svg = array(
            'line'     => array( 'x1' => true, 'y1' => true, 'x2' => true, 'y2' => true ),
            'rect'     => array( 'x' => true, 'y' => true, 'width' => true, 'height' => true, 'rx' => true, 'ry' => true ),
            'circle'   => array( 'cx' => true, 'cy' => true, 'r' => true ),
            'path'     => array( 'd' => true ),
            'polyline' => array( 'points' => true ),
        );

        ?>
        <div class="wrap pwcp-modern-admin pwcp-page-dashboard pwcp-dashboard">
            <!-- Modern Hero Header -->
            <div class="pwcp-hero-header">
                <div class="pwcp-hero-background">
                    <div class="pwcp-hero-particles"></div>
                    <div class="pwcp-hero-gradient"></div>
                </div>
                
                <div class="pwcp-hero-content">
                    <div class="pwcp-hero-left">
                        <div class="pwcp-hero-badge">
                            <span><?php esc_html_e( 'Overview', 'pricewise-calculator-pro' ); ?></span>
                        </div>
                        <h1 class="pwcp-hero-title">
                            <?php esc_html_e( 'Dashboard', 'pricewise-calculator-pro' ); ?>
                        </h1>
                        <p class="pwcp-hero-subtitle">
                            <?php esc_html_e( 'Advanced pricing calculators for WooCommerce products with custom fields, formulas, and automatic cart updates.', 'pricewise-calculator-pro' ); ?>
                        </p>
                    </div>
                </div>
            </div>

            <!-- Modern Container -->
            <div class="pwcp-modern-container">
                <!-- Welcome Panel - Only show if no calculators -->
                <?php if ( empty( $recent_calculators ) ): ?>
                <div class="pwcp-welcome-panel">
                    <div class="pwcp-welcome-content">
                        <div class="pwcp-welcome-header">
                            <h2><?php esc_html_e( 'Welcome to PriceWise Calculator Pro!', 'pricewise-calculator-pro' ); ?></h2>
                            <p><?php esc_html_e( 'Create powerful, interactive calculators for your WooCommerce products with live pricing, modern design, and seamless cart integration.', 'pricewise-calculator-pro' ); ?></p>
                        </div>
                    </div>
                </div>
                <?php endif; ?>

                <!-- Main Content Grid -->
                <div class="pwcp-dashboard-grid">
                    <!-- Main Content Area -->
                    <div class="pwcp-dashboard-main">
                        <!-- Performance Metrics Section -->
                        <div class="pwcp-dashboard-section">
                            <div class="pwcp-section-header">
                                <div class="pwcp-section-title">
                                    <h2><?php esc_html_e( 'Performance Overview', 'pricewise-calculator-pro' ); ?></h2>
                                    <p><?php esc_html_e( 'Key metrics and system performance indicators', 'pricewise-calculator-pro' ); ?></p>
                                </div>
                            </div>
                            
                            <div class="pwcp-section-content">
                                <div class="pwcp-metrics-grid">
                                    <div class="pwcp-metric-card">
                                        <div class="pwcp-metric-icon pwcp-metric-primary">
                                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <path d="M12 2v20m0-20l-7 7m7-7 7 7"/>
                                            </svg>
                                        </div>
                                        <div class="pwcp-metric-content">
                                            <div class="pwcp-metric-value"><?php echo esc_html( number_format_i18n( (int) $stats['total_calculators'] ) ); ?></div>
                                            <div class="pwcp-metric-label"><?php esc_html_e( 'Total Calculators', 'pricewise-calculator-pro' ); ?></div>
                                            <div class="pwcp-metric-change pwcp-metric-positive">
                                                <span class="pwcp-change-indicator">+</span>
                                                <span><?php esc_html_e( 'Growing', 'pricewise-calculator-pro' ); ?></span>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="pwcp-metric-card">
                                        <div class="pwcp-metric-icon pwcp-metric-success">
                                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <polyline points="20,6 9,17 4,12"/>
                                            </svg>
                                        </div>
                                        <div class="pwcp-metric-content">
                                            <div class="pwcp-metric-value"><?php echo esc_html( number_format_i18n( (int) $stats['active_calculators'] ) ); ?></div>
                                            <div class="pwcp-metric-label"><?php esc_html_e( 'Active Calculators', 'pricewise-calculator-pro' ); ?></div>
                                            <div class="pwcp-metric-change pwcp-metric-stable">
                                                <span><?php esc_html_e( 'Operational', 'pricewise-calculator-pro' ); ?></span>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="pwcp-metric-card">
                                        <div class="pwcp-metric-icon pwcp-metric-info">
                                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <path d="M20 7l-8-4-8 4m16 0l-8 4m8-4v10l-8 4m0-10L4 7m8 4v10M4 7v10l8 4"/>
                                            </svg>
                                        </div>
                                        <div class="pwcp-metric-content">
                                            <div class="pwcp-metric-value"><?php echo esc_html( number_format_i18n( (int) $stats['products_with_calculators'] ) ); ?></div>
                                            <div class="pwcp-metric-label"><?php esc_html_e( 'Connected Products', 'pricewise-calculator-pro' ); ?></div>
                                            <div class="pwcp-metric-change pwcp-metric-positive">
                                                <span class="pwcp-change-indicator">+</span>
                                                <span><?php esc_html_e( 'Expanding', 'pricewise-calculator-pro' ); ?></span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Recent Calculators Section -->
                        <div class="pwcp-dashboard-section">
                            <div class="pwcp-section-header">
                                <div class="pwcp-section-title">
                                    <h2><?php esc_html_e( 'Recent Calculators', 'pricewise-calculator-pro' ); ?></h2>
                                    <p><?php esc_html_e( 'Your latest calculator configurations and their current status', 'pricewise-calculator-pro' ); ?></p>
                                </div>
                            </div>

                            <div class="pwcp-section-content">
                                <?php if ( ! empty( $recent_calculators ) ) : ?>
                                    <div class="pwcp-calculator-list">
                                        <?php foreach ( $recent_calculators as $calculator ) : 
                                            $edit_url = add_query_arg(
                                                array(
                                                    'page'          => 'pwcp-add-calculator',
                                                    'action'        => 'edit',
                                                    'calculator_id' => absint( $calculator['id'] ),
                                                ),
                                                admin_url( 'admin.php' )
                                            );
                                        ?>
                                            <div class="pwcp-calculator-item">
                                                <div class="pwcp-calculator-avatar">
                                                    <?php echo esc_html( $this->pwcp_get_calculator_initials( $calculator['name'] ) ); ?>
                                                </div>
                                                <div class="pwcp-calculator-info">
                                                    <h3 class="pwcp-calculator-name"><?php echo esc_html( $calculator['name'] ); ?></h3>
                                                    <div class="pwcp-calculator-meta">
                                                        <span class="pwcp-calculator-status pwcp-status-<?php echo esc_attr( $calculator['status'] ); ?>">
                                                            <?php echo ( 'active' === $calculator['status'] ) ? esc_html__( 'Active', 'pricewise-calculator-pro' ) : esc_html__( 'Inactive', 'pricewise-calculator-pro' ); ?>
                                                        </span>
                                                        <span class="pwcp-calculator-assignment"><?php echo esc_html( $this->pwcp_format_assignment_type( $calculator['assignment_type'] ) ); ?></span>
                                                        <span class="pwcp-calculator-stats-inline">
                                                            <?php 
                                                            echo esc_html( number_format_i18n( (int) $calculator['field_count'] ) ) . ' ' .
                                                                 esc_html__( 'fields', 'pricewise-calculator-pro' ) . ' • ' .
                                                                 esc_html( human_time_diff( strtotime( $calculator['created_at'] ), current_time( 'timestamp' ) ) ) . ' ' .
                                                                 esc_html__( 'ago', 'pricewise-calculator-pro' );
                                                            ?>
                                                        </span>
                                                    </div>
                                                    <?php if ( ! empty( $calculator['description'] ) ) : ?>
                                                        <p class="pwcp-calculator-description"><?php echo esc_html( wp_trim_words( $calculator['description'], 15 ) ); ?></p>
                                                    <?php endif; ?>
                                                </div>
                                                <div class="pwcp-calculator-actions">
                                                    <a href="<?php echo esc_url( $edit_url ); ?>" 
                                                       data-url="<?php echo esc_url( $edit_url ); ?>"
                                                       class="pwcp-btn pwcp-btn-outline pwcp-btn-sm">
                                                        <svg class="pwcp-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                            <path d="M11 4H4a2 2 0 00-2 2v14a2 2 0 002 2h14a2 2 0 002-2v-7"/>
                                                            <path d="M18.5 2.5a2.121 2.121 0 013 3L12 15l-4 1 1-4 9.5-9.5z"/>
                                                        </svg>
                                                        <?php esc_html_e( 'Edit', 'pricewise-calculator-pro' ); ?>
                                                    </a>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>

                                    <div class="pwcp-section-footer">
                                        <a href="<?php echo esc_url( admin_url( 'admin.php?page=pwcp-calculators' ) ); ?>" class="pwcp-btn pwcp-btn-primary">
                                            <?php esc_html_e( 'View All Calculators', 'pricewise-calculator-pro' ); ?>
                                        </a>
                                    </div>
                                <?php else : ?>
                                    <div class="pwcp-empty-state">
                                        <div class="pwcp-empty-illustration">
                                            <svg class="pwcp-empty-icon" viewBox="0 0 200 150">
                                                <defs>
                                                    <linearGradient id="calc-gradient" x1="0%" y1="0%" x2="100%" y2="100%">
                                                        <stop offset="0%" style="stop-color:#6366f1;stop-opacity:0.1"/>
                                                        <stop offset="100%" style="stop-color:#a855f7;stop-opacity:0.1"/>
                                                    </linearGradient>
                                                </defs>
                                                <rect x="30" y="30" width="140" height="90" rx="8" fill="url(#calc-gradient)" stroke="#6366f1" stroke-width="2"/>
                                                <circle cx="50" cy="50" r="4" fill="#6366f1"/>
                                                <circle cx="70" cy="50" r="4" fill="#8b5cf6"/>
                                                <circle cx="90" cy="50" r="4" fill="#a855f7"/>
                                                <rect x="40" y="70" width="120" height="4" rx="2" fill="#6366f1" opacity="0.3"/>
                                                <rect x="40" y="85" width="80" height="4" rx="2" fill="#8b5cf6" opacity="0.3"/>
                                                <rect x="40" y="100" width="100" height="4" rx="2" fill="#a855f7" opacity="0.3"/>
                                            </svg>
                                        </div>
                                        <h3><?php esc_html_e( 'No Calculators Yet', 'pricewise-calculator-pro' ); ?></h3>
                                        <p><?php esc_html_e( 'Create your first calculator to start collecting pricing data and enhancing your product pages.', 'pricewise-calculator-pro' ); ?></p>
                                        <a href="<?php echo esc_url( admin_url( 'admin.php?page=pwcp-add-calculator' ) ); ?>" class="pwcp-btn pwcp-btn-primary pwcp-btn-icon">
                                            <svg class="pwcp-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <line x1="12" y1="5" x2="12" y2="19"/>
                                                <line x1="5" y1="12" x2="19" y2="12"/>
                                            </svg>
                                            <?php esc_html_e( 'Create Your First Calculator', 'pricewise-calculator-pro' ); ?>
                                        </a>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                    <!-- Sidebar -->
                    <div class="pwcp-dashboard-sidebar">
                        <!-- Quick Actions Widget -->
                        <div class="pwcp-sidebar-widget">
                            <div class="pwcp-widget-header">
                                <h3><?php esc_html_e( 'Quick Actions', 'pricewise-calculator-pro' ); ?></h3>
                            </div>
                            <div class="pwcp-widget-content">
                                <div class="pwcp-quick-actions">
                                    <?php
                                    $quick_actions = array(
                                        array(
                                            'url'   => admin_url( 'admin.php?page=pwcp-add-calculator' ),
                                            'title' => __( 'New Calculator', 'pricewise-calculator-pro' ),
                                            'desc'  => __( 'Create a new pricing calculator', 'pricewise-calculator-pro' ),
                                            'icon'  => '<line x1="12" y1="5" x2="12" y2="19"/><line x1="5" y1="12" x2="19" y2="12"/>',
                                            'class' => 'pwcp-action-primary',
                                        ),
                                        array(
                                            'url'   => admin_url( 'admin.php?page=pwcp-calculators' ),
                                            'title' => __( 'Manage Calculators', 'pricewise-calculator-pro' ),
                                            'desc'  => __( 'Edit existing calculators', 'pricewise-calculator-pro' ),
                                            'icon'  => '<rect x="3" y="3" width="18" height="18" rx="2" ry="2"/><line x1="9" y1="12" x2="15" y2="12"/><line x1="12" y1="9" x2="12" y2="15"/>',
                                            'class' => '',
                                        ),
                                        array(
                                            'url'   => admin_url( 'admin.php?page=pwcp-settings' ),
                                            'title' => __( 'Settings', 'pricewise-calculator-pro' ),
                                            'desc'  => __( 'Configure global options', 'pricewise-calculator-pro' ),
                                            'icon'  => '<circle cx="12" cy="12" r="3"/><path d="M19.4 15a1.65 1.65 0 00.33 1.82l.06.06a2 2 0 010 2.83 2 2 0 01-2.83 0l-.06-.06a1.65 1.65 0 00-1.82-.33 1.65 1.65 0 00-1 1.51V21a2 2 0 01-2 2 2 2 0 01-2-2v-.09A1.65 1.65 0 009 19.4a1.65 1.65 0 00-1.82.33l-.06.06a2 2 0 01-2.83 0 2 2 0 010-2.83l.06-.06a1.65 1.65 0 00.33-1.82 1.65 1.65 0 00-1.51-1H3a2 2 0 01-2-2 2 2 0 012-2h.09A1.65 1.65 0 004.6 9a1.65 1.65 0 00-.33-1.82l-.06-.06a2 2 0 010-2.83 2 2 0 012.83 0l.06.06a1.65 1.65 0 001.82.33H9a1.65 1.65 0 001-1.51V3a2 2 0 012-2 2 2 0 012 2v.09a1.65 1.65 0 001 1.51 1.65 1.65 0 001.82-.33l.06-.06a2 2 0 012.83 0 2 2 0 010 2.83l-.06.06a1.65 1.65 0 00-.33 1.82V9a1.65 1.65 0 001.51 1H21a2 2 0 012 2 2 2 0 01-2 2h-.09a1.65 1.65 0 00-1.51 1z"/>',
                                            'class' => '',
                                        ),
                                    );

                                    if ( class_exists( 'WooCommerce' ) ) {
                                        $quick_actions[] = array(
                                            'url'   => admin_url( 'edit.php?post_type=product' ),
                                            'title' => __( 'Products', 'pricewise-calculator-pro' ),
                                            'desc'  => __( 'Manage WooCommerce products', 'pricewise-calculator-pro' ),
                                            'icon'  => '<path d="M20 7l-8-4-8 4m16 0l-8 4m8-4v10l-8 4m0-10L4 7m8 4v10M4 7v10l8 4"/>',
                                            'class' => '',
                                        );
                                    }

                                    foreach ( $quick_actions as $action ) :
                                    ?>
                                    <a href="<?php echo esc_url( $action['url'] ); ?>" 
                                       data-url="<?php echo esc_url( $action['url'] ); ?>"
                                       class="pwcp-quick-action <?php echo esc_attr( $action['class'] ); ?>">
                                        <div class="pwcp-action-icon">
                                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <?php echo wp_kses( $action['icon'], $allowed_svg ); ?>
                                            </svg>
                                        </div>
                                        <div class="pwcp-action-content">
                                            <div class="pwcp-action-title"><?php echo esc_html( $action['title'] ); ?></div>
                                            <div class="pwcp-action-desc"><?php echo esc_html( $action['desc'] ); ?></div>
                                        </div>
                                        <div class="pwcp-action-arrow">
                                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <polyline points="9,18 15,12 9,6"/>
                                            </svg>
                                        </div>
                                    </a>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>

                        <!-- Tips & Help Widget -->
                        <div class="pwcp-sidebar-widget">
                            <div class="pwcp-widget-header">
                                <h3><?php esc_html_e( 'Getting Started', 'pricewise-calculator-pro' ); ?></h3>
                            </div>
                            <div class="pwcp-widget-content">
                                <div class="pwcp-tips-list">
                                    <div class="pwcp-tip-item">
                                        <div class="pwcp-tip-number">1</div>
                                        <div class="pwcp-tip-content">
                                            <h4><?php esc_html_e( 'Custom Product Pricing', 'pricewise-calculator-pro' ); ?></h4>
                                            <p><?php esc_html_e( 'Let customers configure products with size, quantity, material options - prices update automatically based on your formulas.', 'pricewise-calculator-pro' ); ?></p>
                                        </div>
                                    </div>

                                    <div class="pwcp-tip-item">
                                        <div class="pwcp-tip-number">2</div>
                                        <div class="pwcp-tip-content">
                                            <h4><?php esc_html_e( 'Works With WooCommerce', 'pricewise-calculator-pro' ); ?></h4>
                                            <p><?php esc_html_e( 'Calculated prices sync directly to cart - no manual price entry needed. Customers see final costs before checkout.', 'pricewise-calculator-pro' ); ?></p>
                                        </div>
                                    </div>

                                    <div class="pwcp-tip-item">
                                        <div class="pwcp-tip-number">3</div>
                                        <div class="pwcp-tip-content">
                                            <h4><?php esc_html_e( 'Start Simple', 'pricewise-calculator-pro' ); ?></h4>
                                            <p><?php esc_html_e( 'Create your first calculator with 2-3 fields. Once comfortable, add complex formulas and conditional logic.', 'pricewise-calculator-pro' ); ?></p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Plugin Info Widget -->
                        <div class="pwcp-sidebar-widget">
                            <div class="pwcp-widget-header">
                                <h3><?php esc_html_e( 'Plugin Diagnostics', 'pricewise-calculator-pro' ); ?></h3>
                            </div>
                            <div class="pwcp-widget-content">
                                <div class="pwcp-plugin-info">
                                    <div class="pwcp-plugin-stats-list">
                                        <div class="pwcp-info-item">
                                            <div class="pwcp-info-left">
                                                <div class="pwcp-info-indicator"></div>
                                                <span class="pwcp-info-label"><?php esc_html_e( 'Plugin Status:', 'pricewise-calculator-pro' ); ?></span>
                                            </div>
                                            <span class="pwcp-info-value"><?php esc_html_e( 'Active', 'pricewise-calculator-pro' ); ?></span>
                                        </div>
                                        <div class="pwcp-info-item">
                                            <div class="pwcp-info-left">
                                                <div class="pwcp-info-indicator"></div>
                                                <span class="pwcp-info-label"><?php esc_html_e( 'Version:', 'pricewise-calculator-pro' ); ?></span>
                                            </div>
                                            <span class="pwcp-info-value"><?php echo esc_html( PWCP_PLUGIN_VERSION ); ?></span>
                                        </div>
                                        <div class="pwcp-info-item">
                                            <div class="pwcp-info-left">
                                                <div class="pwcp-info-indicator"></div>
                                                <span class="pwcp-info-label"><?php esc_html_e( 'Author:', 'pricewise-calculator-pro' ); ?></span>
                                            </div>
                                            <span class="pwcp-info-value"><?php esc_html_e( 'Devtonic Studios Team', 'pricewise-calculator-pro' ); ?></span>
                                        </div>
                                    </div>
                                </div>
                                
                             <div class="pwcp-plugin-actions">
                              <a href="https://devtonicstudios.com/documentation/" class="pwcp-btn pwcp-btn-outline pwcp-btn-sm" target="_blank" rel="noopener noreferrer">
                                    <?php esc_html_e( 'Documentation', 'pricewise-calculator-pro' ); ?>
                                   </a>
    
                                        <a href="https://devtonicstudios.com/contact/" class="pwcp-btn pwcp-btn-outline pwcp-btn-sm" target="_blank" rel="noopener noreferrer">
                                          <?php esc_html_e( 'Support', 'pricewise-calculator-pro' ); ?>
                                                        </a>
                                                   </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Get dashboard statistics with proper error handling
     *
     * @since 1.0.0
     * @return array
     */
    private function pwcp_get_dashboard_stats() {
        global $wpdb;

        $stats = array(
            'total_calculators'         => 0,
            'active_calculators'        => 0,
            'products_with_calculators' => 0,
        );

        $calculators_table = $wpdb->prefix . 'pwcp_calculators';
        
        // Verify table exists
        if ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $wpdb->esc_like( $calculators_table ) ) ) !== $calculators_table ) {
            return $stats;
        }

        // Total calculators with error handling
// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name from $wpdb->prefix
        $total_count = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM `" . esc_sql( $calculators_table ) . "` WHERE 1=%d", 1 ) );
        if ( $wpdb->last_error ) {
            return $stats;
        }
        $stats['total_calculators'] = (int) $total_count;

        // Active calculators with error handling
$active_count = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM `" . esc_sql($calculators_table) . "` WHERE status = %s", 'active' ) );
        if ( $wpdb->last_error ) {
        } else {
            $stats['active_calculators'] = (int) $active_count;
        }

        // Get products with calculators
        if ( class_exists( 'WooCommerce' ) ) {
            $products_with_calculators = 0;
            
            // Check if there are any active global calculators
            $has_global_calculator = $wpdb->get_var(
                $wpdb->prepare(
"SELECT COUNT(*) FROM `" . esc_sql($calculators_table) . "` WHERE status = %s AND assignment_type = %s",
                    'active',
                    'global'
                )
            );
            
            if ( $has_global_calculator > 0 ) {
                // If global calculator exists, count all published WooCommerce products
                $products_with_calculators = (int) $wpdb->get_var(
                    $wpdb->prepare(
                        "SELECT COUNT(ID) FROM {$wpdb->posts} WHERE post_type = %s AND post_status = %s",
                        'product',
                        'publish'
                    )
                );
            } else {
                // Compute union of product-specific and category-based assignments
                $product_ids = array();
                
                // Get product-specific assignments
                $product_specific_ids = $wpdb->get_col(
                    $wpdb->prepare(
                        "SELECT DISTINCT pm.post_id
                         FROM {$wpdb->postmeta} pm
INNER JOIN `" . esc_sql($calculators_table) . "` c ON FIND_IN_SET(c.id, pm.meta_value) > 0
                         WHERE pm.meta_key = %s 
                         AND pm.meta_value IS NOT NULL 
                         AND pm.meta_value <> ''
                         AND c.status = %s
                         AND c.assignment_type = %s",
                        '_pwcp_assigned_calculators',
                        'active',
                        'product'
                    )
                );
                
                if ( is_array( $product_specific_ids ) ) {
                    $product_ids = array_merge( $product_ids, array_map( 'intval', $product_specific_ids ) );
                }
                
                // Get category-based assignments
                $category_calculators = $wpdb->get_col(
                    $wpdb->prepare(
"SELECT id FROM `" . esc_sql($calculators_table) . "` WHERE status = %s AND assignment_type = %s",
                        'active',
                        'category'
                    )
                );
                
                if ( ! empty( $category_calculators ) ) {
                    foreach ( $category_calculators as $calc_id ) {
                        $assigned_categories = get_option( 'pwcp_category_assignments_' . absint( $calc_id ), array() );
                        if ( is_array( $assigned_categories ) && ! empty( $assigned_categories ) ) {
                            $category_ids = array_map( 'absint', array_column( $assigned_categories, 'id' ) );
                            if ( ! empty( $category_ids ) ) {
                                // Get products in categories
                                $products_in_cats = get_posts(
                                    array(
                                        'post_type'               => 'product',
                                        'post_status'             => 'publish',
                                        'posts_per_page'          => -1,
                                        'fields'                  => 'ids',
                                        'no_found_rows'           => true,
                                        'update_post_meta_cache'  => false,
                                        'update_post_term_cache'  => false,
                                        'tax_query'               => array(
                                            array(
                                                'taxonomy' => 'product_cat',
                                                'field'    => 'term_id',
                                                'terms'    => $category_ids,
                                            ),
                                        ),
                                    )
                                );
                                $product_ids = array_merge( $product_ids, $products_in_cats );
                            }
                        }
                    }
                }
                
                // Count unique products
                $products_with_calculators = count( array_unique( array_map( 'intval', $product_ids ) ) );
            }
            
            $stats['products_with_calculators'] = $products_with_calculators;
        }

        return $stats;
    }

    /**
     * Get recent calculators with proper sanitization
     *
     * @param int $limit Number of calculators to retrieve
     * @since 1.0.0
     * @return array
     */
    private function pwcp_get_recent_calculators( $limit = 5 ) {
        global $wpdb;

        // Sanitize limit with bounds checking
        $limit = absint( $limit );
        if ( $limit <= 0 || $limit > 100 ) {
            $limit = 5;
        }

        $calculators_table = $wpdb->prefix . 'pwcp_calculators';
        $fields_table      = $wpdb->prefix . 'pwcp_calculator_fields';

        // Check both tables exist
        $has_calcs  = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $wpdb->esc_like( $calculators_table ) ) ) === $calculators_table;
        $has_fields = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $wpdb->esc_like( $fields_table ) ) ) === $fields_table;

        if ( ! $has_calcs ) {
            return array();
        }

        // Build field_count SQL conditionally
        $safe_fields_table = esc_sql( $fields_table );
        $safe_calculators_table = esc_sql( $calculators_table );
        
        $field_count_sql = $has_fields
            ? "(SELECT COUNT(*) FROM `{$safe_fields_table}` f WHERE f.calculator_id = c.id)"
            : "0";

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table names from $wpdb->prefix, field_count_sql is hardcoded subquery
        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT 
                    c.id,
                    c.name,
                    c.description,
                    c.status,
                    c.assignment_type,
                    c.created_at,
                    {$field_count_sql} as field_count
                FROM `{$safe_calculators_table}` c
                ORDER BY c.created_at DESC
                LIMIT %d",
                $limit
            ),
            ARRAY_A
        );
        
        // Error handling
        if ( $wpdb->last_error ) {
            return array();
        }

        return $results ?: array();
    }

    /**
     * Format assignment type for display
     *
     * @param string $assignment_type Assignment type
     * @since 1.0.0
     * @return string
     */
    private function pwcp_format_assignment_type( $assignment_type ) {
        $assignment_type = sanitize_text_field( $assignment_type );
        
        switch ( $assignment_type ) {
            case 'global':
                return esc_html__( 'Global', 'pricewise-calculator-pro' );
            case 'category':
                return esc_html__( 'Category', 'pricewise-calculator-pro' );
            case 'product':
                return esc_html__( 'Product', 'pricewise-calculator-pro' );
            default:
                return esc_html__( 'Unknown', 'pricewise-calculator-pro' );
        }
    }

    /**
     * Get calculator initials for avatar display
     *
     * @param string $name Calculator name
     * @since 1.0.0
     * @return string
     */
    private function pwcp_get_calculator_initials( $name ) {
        $name = sanitize_text_field( $name );
        
        if ( empty( $name ) ) {
            return 'PC';
        }

        // Split name into words and get first letter of each
        $words = explode( ' ', trim( $name ) );
        $initials = '';
        
        foreach ( array_slice( $words, 0, 2 ) as $word ) {
            if ( ! empty( $word ) ) {
                $initials .= strtoupper( substr( $word, 0, 1 ) );
            }
        }
        
        return ! empty( $initials ) ? $initials : 'PC';
    }
}