<?php
/**
 * Calculator Manager
 * 
 * Handles calculator operations including CRUD, AJAX handlers, and bulk actions
 *
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

// Load dependencies
require_once PWCP_PLUGIN_DIR . 'includes/class-pwcp-base-ajax-handler.php';
require_once PWCP_PLUGIN_DIR . 'includes/traits/trait-pwcp-security-validation.php';
require_once PWCP_PLUGIN_DIR . 'includes/class-pwcp-input-sanitizer.php';
require_once PWCP_PLUGIN_DIR . 'includes/class-pwcp-response-factory.php';

/**
 * Calculator Manager Class
 * 
 * Manages calculator operations throughout the admin interface
 * 
 * @since 1.0.0
 */
class PWCP_Admin_Calculator_Manager extends PWCP_Base_Ajax_Handler {
    
    use PWCP_Security_Validation_Trait;
    
    private $field_manager;
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_init', array($this, 'pwcp_handle_form_submissions'));
        
        if (class_exists('PWCP_Field_Manager')) {
            $this->field_manager = new PWCP_Field_Manager();
        }
        
        $this->register_ajax_handlers();
    }
    
    /**
     * Register AJAX handlers
     */
    private function register_ajax_handlers() {
        $ajax_handlers = array(
            'pwcp_delete_calculator' => 'pwcp_ajax_delete_calculator',
            'pwcp_duplicate_calculator' => 'pwcp_ajax_duplicate_calculator',
            'pwcp_bulk_action_calculators' => 'pwcp_ajax_bulk_action_calculators'
        );
        
        foreach ($ajax_handlers as $action => $method) {
            if (!has_action('wp_ajax_' . $action)) {
                add_action('wp_ajax_' . $action, array($this, $method));
            }
        }
    }
    
    /**
     * Handle form submissions
     */
public function pwcp_handle_form_submissions() {
    // Check nonce FIRST before checking any $_POST data
    if (!isset($_POST['pwcp_calculator_nonce']) || 
        !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['pwcp_calculator_nonce'])), 'pwcp_calculator_action')) {
        return; // Silently return if nonce is missing/invalid
    }
    
    if (!isset($_POST['pwcp_calculator_action'])) {
        return;
    }
    
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have sufficient permissions to perform this action.', 'pricewise-calculator-pro'));
    }
    
    $action = sanitize_text_field(wp_unslash($_POST['pwcp_calculator_action']));
        
        switch ($action) {
            case 'delete':
                $this->handle_delete_calculator_form();
                break;
            case 'duplicate':
                $this->handle_duplicate_calculator_form();
                break;
        }
    }
    
    /**
     * Get calculators list
     */
    public function pwcp_get_calculators_list() {
        global $wpdb;
        $table = $wpdb->prefix . 'pwcp_calculators';
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name from $wpdb->prefix
        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT id, name, description, formula, settings, status, 
                        assignment_type, assignment_ids, priority, hide_quantity_selector,
                        created_at, updated_at
                 FROM `" . esc_sql( $table ) . "` 
                 WHERE 1=%d
                 ORDER BY created_at DESC",
                1
            )
        );
        
        return $results ?: array();
    }

    /**
     * Get calculators count
     */
    public function pwcp_get_calculators_count() {
        global $wpdb;
        $table = $wpdb->prefix . 'pwcp_calculators';
        
// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name from $wpdb->prefix
$count = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(id) FROM `" . esc_sql( $table ) . "` WHERE 1=%d", 1 ) );
        return (int) $count;
    }

    /**
     * Get calculator by ID
     */
    public function pwcp_get_calculator($calculator_id) {
        global $wpdb;
        $table = $wpdb->prefix . 'pwcp_calculators';
        
        return $wpdb->get_row(
            $wpdb->prepare(
                "SELECT id, name, description, formula, settings, status, 
                        assignment_type, assignment_ids, priority, hide_quantity_selector,
                        created_at, updated_at
                 FROM `" . esc_sql($table) . "` 
                 WHERE id = %d",
                $calculator_id
            )
        );
    }

    /**
     * Get calculator fields
     */
    public function pwcp_get_calculator_fields($calculator_id) {
        if ($this->field_manager) {
            return $this->field_manager->get_calculator_fields($calculator_id);
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'pwcp_calculator_fields';
        
        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT id, calculator_id, field_key, field_label, field_type, field_options, field_validation, field_default, field_required, field_order, created_at, updated_at 
                 FROM `{$table}` 
                 WHERE calculator_id = %d 
                 ORDER BY field_order ASC",
                $calculator_id
            )
        );
        return $results ?: array();
    }
    
    /**
     * Delete calculator by ID
     */
    public function pwcp_delete_calculator($calculator_id) {
        global $wpdb;
        
        $calculator_id = absint($calculator_id);
        if (!$calculator_id) {
            return false;
        }
        
        $wpdb->delete(
            $wpdb->prefix . 'pwcp_calculator_fields',
            array('calculator_id' => $calculator_id),
            array('%d')
        );
        
        $result = $wpdb->delete(
            $wpdb->prefix . 'pwcp_calculators',
            array('id' => $calculator_id),
            array('%d')
        );
        
        return $result !== false;
    }
    
    /**
     * Duplicate calculator
     */
    public function pwcp_duplicate_calculator($calculator_id) {
        global $wpdb;
        
        $calculator_id = absint($calculator_id);
        if (!$calculator_id) {
            return false;
        }
        
        $original = $this->pwcp_get_calculator($calculator_id);
        if (!$original) {
            return false;
        }
        
        $copy_name = $this->generate_unique_calculator_name($original->name);
        
        $duplicate_data = array(
            'name' => $copy_name,
            'description' => $original->description,
            'formula' => $original->formula,
            'settings' => $original->settings,
            'status' => 'inactive',
            'assignment_type' => $original->assignment_type,
            'assignment_ids' => $original->assignment_ids,
            'priority' => $original->priority,
            'created_at' => current_time('mysql'),
            'updated_at' => current_time('mysql'),
        );
        
        $result = $wpdb->insert(
            $wpdb->prefix . 'pwcp_calculators',
            $duplicate_data,
            array('%s', '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%s', '%s')
        );
        
        if ($result === false) {
            return false;
        }
        
        $new_calculator_id = $wpdb->insert_id;
        $this->duplicate_calculator_fields($calculator_id, $new_calculator_id);
        
        return $new_calculator_id;
    }
    
    /**
     * Bulk delete calculators
     */
    public function pwcp_bulk_delete_calculators($calculator_ids) {
        if (empty($calculator_ids)) {
            return false;
        }
        
        $success_count = 0;
        foreach ($calculator_ids as $calculator_id) {
            if ($this->pwcp_delete_calculator($calculator_id)) {
                $success_count++;
            }
        }
        
        return $success_count > 0;
    }
    
    /**
     * Bulk update calculator status
     */
    public function pwcp_bulk_update_status($calculator_ids, $status) {
        if (empty($calculator_ids)) {
            return false;
        }
        
        $status = in_array($status, array('active', 'inactive'), true) ? $status : 'inactive';
        
        global $wpdb;
        $table = $wpdb->prefix . 'pwcp_calculators';
        
        $placeholders = implode(',', array_fill(0, count($calculator_ids), '%d'));
        
  // Prepare placeholders for each ID
$placeholders = implode(',', array_fill(0, count($calculator_ids), '%d'));

// Combine all parameters: status, timestamp, then all IDs
$params = array_merge(
    array($status, current_time('mysql')),
    $calculator_ids
);

// Use esc_sql for table name since it's from $wpdb->prefix (safe)
$result = $wpdb->query(
    $wpdb->prepare(
        "UPDATE `" . esc_sql($table) . "` SET status = %s, updated_at = %s WHERE id IN ($placeholders)",
        $params
    )
);
        return $result !== false;
    }
    
    /**
     * Save calculator
     */
    public function pwcp_save_calculator($calculator_data, $calculator_id = 0) {
        global $wpdb;
        $table = $wpdb->prefix . 'pwcp_calculators';
        
        $allowed_fields = array(
            'name', 'description', 'formula', 'settings', 'status',
            'assignment_type', 'assignment_ids', 'priority', 
            'hide_quantity_selector'
        );
        
        $clean_data = array();
        foreach ($calculator_data as $key => $value) {
            if (!in_array($key, $allowed_fields, true)) {
                continue;
            }
            
          if (in_array($key, array('priority', 'hide_quantity_selector'), true)) {
    $clean_data[$key] = absint($value);

} elseif ($key === 'status') {
    $clean_data[$key] = in_array($value, array('active', 'inactive'), true) ? $value : 'inactive';

} elseif ($key === 'assignment_type') {
    $clean_data[$key] = in_array($value, array('global', 'product', 'category'), true) ? $value : 'global';

} elseif ($key === 'settings') {
    // Store settings as JSON
    if (is_array($value)) {
        $clean_data[$key] = wp_json_encode($value);
    } else {
        $decoded = json_decode(wp_unslash($value), true);
        $clean_data[$key] = is_array($decoded) ? wp_json_encode($decoded) : '';
    }

} elseif ($key === 'assignment_ids') {
    // Store assignment IDs as JSON array of ints
    $ids = is_string($value) ? json_decode(wp_unslash($value), true) : $value;
    $ids = is_array($ids) ? array_map('absint', $ids) : array();
    $clean_data[$key] = wp_json_encode($ids);

} else {
    $clean_data[$key] = sanitize_textarea_field(wp_unslash($value));
}
        }
        
        if (!isset($clean_data['hide_quantity_selector'])) {
            $clean_data['hide_quantity_selector'] = 0;
        }
        if (!isset($clean_data['settings'])) {
            $clean_data['settings'] = '';
        }
        
        if ($calculator_id > 0) {
            $clean_data['updated_at'] = current_time('mysql');
            
            $format = array();
            foreach ($clean_data as $key => $value) {
                if (in_array($key, array('priority', 'hide_quantity_selector'), true)) {
                    $format[] = '%d';
                } else {
                    $format[] = '%s';
                }
            }
            
            $result = $wpdb->update(
                $table,
                $clean_data,
                array('id' => $calculator_id),
                $format,
                array('%d')
            );
            
            if ($result !== false) {
                $this->bust_counts_cache();
                return true;
            }
            return false;
        } else {
            $clean_data['created_at'] = current_time('mysql');
            $clean_data['updated_at'] = current_time('mysql');
            
            $format = array();
            foreach ($clean_data as $key => $value) {
                if (in_array($key, array('priority', 'hide_quantity_selector'), true)) {
                    $format[] = '%d';
                } else {
                    $format[] = '%s';
                }
            }
            
            $result = $wpdb->insert($table, $clean_data, $format);
            
            if ($result) {
                $this->bust_counts_cache();
                return $wpdb->insert_id;
            }
            return false;
        }
    }
    
    /**
     * Save calculator fields
     */
    public function pwcp_save_calculator_fields_extended($calculator_id, $fields) {
        if ($this->field_manager) {
            return $this->field_manager->bulk_save_fields($calculator_id, $fields);
        }
        
        return array();
    }
    
    /**
     * AJAX delete calculator handler
     */
    public function pwcp_ajax_delete_calculator() {
        if (!defined('DOING_AJAX') || !DOING_AJAX) {
            wp_die(esc_html__('Invalid request', 'pricewise-calculator-pro'));
        }
        
        $this->verify_ajax_security('manage_options');
        
        $calculator_id = $this->get_calculator_id();
        $result = $this->pwcp_delete_calculator($calculator_id);
        
        if ($result) {
            $this->bust_counts_cache();
            PWCP_Response_Factory::calculator_success('delete', $calculator_id);
        } else {
            PWCP_Response_Factory::database_error('delete_calculator');
        }
    }
    
    /**
     * AJAX duplicate calculator handler
     */
    public function pwcp_ajax_duplicate_calculator() {
        if (!defined('DOING_AJAX') || !DOING_AJAX) {
            wp_die(esc_html__('Invalid request', 'pricewise-calculator-pro'));
        }
        
        $this->verify_ajax_security('manage_options');
        
        $calculator_id = $this->get_calculator_id();
        $new_calculator_id = $this->pwcp_duplicate_calculator($calculator_id);
        
        if ($new_calculator_id) {
            $this->bust_counts_cache();
            
            $new_calculator = $this->pwcp_get_calculator($new_calculator_id);
            $new_card_html = $new_calculator ? $this->generate_calculator_card_html($new_calculator) : '';
            
            PWCP_Response_Factory::calculator_success('duplicate', $new_calculator_id, array(
                'new_card_html' => $new_card_html,
                'original_calculator_id' => $calculator_id
            ));
        } else {
            PWCP_Response_Factory::database_error('duplicate_calculator');
        }
    }
    
    /**
     * AJAX bulk actions handler
     */
    public function pwcp_ajax_bulk_action_calculators() {
        if (!defined('DOING_AJAX') || !DOING_AJAX) {
            wp_die(esc_html__('Invalid request', 'pricewise-calculator-pro'));
        }
        
        $this->verify_ajax_security('manage_options', 'pwcp_bulk_calculators', 'nonce');
        
        $action_type = sanitize_key($this->get_post_data('action_type', 'text'));
        $calculator_ids = $this->get_post_data('calculator_ids', 'array');
        
        $valid_actions = array('delete', 'duplicate', 'activate', 'deactivate');
        if (!in_array($action_type, $valid_actions, true)) {
            PWCP_Response_Factory::validation_error(
                esc_html__('Invalid bulk action specified.', 'pricewise-calculator-pro')
            );
        }
        
        if (empty($calculator_ids)) {
            PWCP_Response_Factory::validation_error(
                esc_html__('No calculators selected.', 'pricewise-calculator-pro')
            );
        }
        
        $calculator_ids = array_map('absint', $calculator_ids);
        $result = $this->process_bulk_action($action_type, $calculator_ids);
        
        if ($result['success']) {
            $this->bust_counts_cache();
            $this->send_success_response($result['message'], $result['data']);
        } else {
            PWCP_Response_Factory::generic_error($result['message']);
        }
    }
    
    /**
     * Process bulk action
     */
    private function process_bulk_action($action_type, $calculator_ids) {
        switch ($action_type) {
            case 'delete':
                if (!$this->can_manage_calculators()) {
                    return array(
                        'success' => false,
                        'message' => esc_html__('Insufficient permissions to delete calculators.', 'pricewise-calculator-pro')
                    );
                }
                
                $success_count = 0;
                foreach ($calculator_ids as $calculator_id) {
                    if ($this->pwcp_delete_calculator($calculator_id)) {
                        $success_count++;
                    }
                }
                
                return array(
                    'success' => $success_count > 0,
                    'message' => sprintf(
                        /* translators: %d: number of calculators deleted */
                        esc_html__('Deleted %d calculators successfully!', 'pricewise-calculator-pro'),
                        $success_count
                    ),
                    'data' => array(
                        'action_type' => 'delete',
                        'success_count' => $success_count,
                        'calculator_ids' => $calculator_ids
                    )
                );
                
            case 'duplicate':
                if (!$this->can_manage_calculators()) {
                    return array(
                        'success' => false,
                        'message' => esc_html__('Insufficient permissions to duplicate calculators.', 'pricewise-calculator-pro')
                    );
                }
                
                $success_count = 0;
                $new_cards_html = array();
                
                foreach ($calculator_ids as $calculator_id) {
                    $new_calculator_id = $this->pwcp_duplicate_calculator($calculator_id);
                    if ($new_calculator_id) {
                        $success_count++;
                        $new_calculator = $this->pwcp_get_calculator($new_calculator_id);
                        if ($new_calculator) {
                            $new_cards_html[] = array(
                                'id' => $new_calculator_id,
                                'html' => $this->generate_calculator_card_html($new_calculator)
                            );
                        }
                    }
                }
                
                return array(
                    'success' => $success_count > 0,
                    'message' => sprintf(
                        /* translators: %d: number of calculators duplicated */
                        esc_html__('Duplicated %d calculators successfully!', 'pricewise-calculator-pro'),
                        $success_count
                    ),
                    'data' => array(
                        'action_type' => 'duplicate',
                        'success_count' => $success_count,
                        'new_cards_html' => $new_cards_html
                    )
                );
                
            case 'activate':
            case 'deactivate':
                if (!$this->can_manage_calculators()) {
                    return array(
                        'success' => false,
                        'message' => esc_html__('Insufficient permissions to change calculator status.', 'pricewise-calculator-pro')
                    );
                }
                
                $status = $action_type === 'activate' ? 'active' : 'inactive';
                $result = $this->pwcp_bulk_update_status($calculator_ids, $status);
                
                $action_label = $action_type === 'activate' 
                    ? esc_html__('Activated', 'pricewise-calculator-pro')
                    : esc_html__('Deactivated', 'pricewise-calculator-pro');
                
                return array(
                    'success' => $result,
                    'message' => sprintf(
                        /* translators: 1: action label, 2: number of calculators */
                        esc_html__('%1$s %2$d calculators successfully!', 'pricewise-calculator-pro'),
                        $action_label,
                        count($calculator_ids)
                    ),
                    'data' => array(
                        'action_type' => $action_type,
                        'calculator_ids' => $calculator_ids
                    )
                );
                
            default:
                return array(
                    'success' => false,
                    'message' => esc_html__('Invalid bulk action.', 'pricewise-calculator-pro')
                );
        }
    }
    
    /**
     * Generate unique calculator name
     */
    private function generate_unique_calculator_name($base_name) {
        $copy_name = $base_name . ' ' . __('(Copy)', 'pricewise-calculator-pro');
        $copy_count = 1;
        
        while ($this->calculator_name_exists($copy_name)) {
            $copy_count++;
            $copy_name = $base_name . ' ' . sprintf(
                /* translators: %d: copy number */
                __('(Copy %d)', 'pricewise-calculator-pro'),
                $copy_count
            );
        }
        
        return $copy_name;
    }
    
    /**
     * Check if calculator name exists
     */
    private function calculator_name_exists($name) {
        global $wpdb;
        $table = $wpdb->prefix . 'pwcp_calculators';
        
        $count = $wpdb->get_var(
            $wpdb->prepare("SELECT COUNT(*) FROM `{$table}` WHERE name = %s", $name)
        );
        
        return (int) $count > 0;
    }
    
    /**
     * Duplicate calculator fields
     */
    private function duplicate_calculator_fields($original_id, $new_id) {
        $fields = $this->pwcp_get_calculator_fields($original_id);
        
        if (empty($fields)) {
            return;
        }
        
        global $wpdb;
        $fields_table = $wpdb->prefix . 'pwcp_calculator_fields';
        
        foreach ($fields as $field) {
            $field_data = array(
                'calculator_id' => $new_id,
                'field_key' => $field->field_key,
                'field_label' => $field->field_label,
                'field_type' => $field->field_type,
                'field_options' => $field->field_options,
                'field_validation' => $field->field_validation,
                'field_default' => $field->field_default,
                'field_required' => (int) $field->field_required,
                'field_order' => (int) $field->field_order,
                'created_at' => current_time('mysql'),
                'updated_at' => current_time('mysql'),
            );
            
            $wpdb->insert(
                $fields_table,
                $field_data,
                array('%d', '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%d', '%s', '%s')
            );
        }
    }
    
    /**
     * Clear cache
     */
    private function bust_counts_cache() {
        if (function_exists('wp_cache_delete')) {
            wp_cache_delete('pwcp_calculator_counts', 'pwcp_calculators');
        }
    }
    
    /**
     * Handle delete calculator form submission
     */
    private function handle_delete_calculator_form() {
        $calculator_id = $this->get_post_data('calculator_id', 'int');
        
        if (!$calculator_id) {
            wp_die(esc_html__('Invalid calculator ID.', 'pricewise-calculator-pro'));
        }
        
        $result = $this->pwcp_delete_calculator($calculator_id);
        
        if ($result) {
            $this->bust_counts_cache();
        }
        
        $message = $result ? 'deleted' : 'error';
        $redirect_url = add_query_arg(
            array(
                'page' => 'pwcp-calculators',
                'message' => $message,
            ),
            admin_url('admin.php')
        );
        
        wp_safe_redirect($redirect_url);
        exit;
    }
    
    /**
     * Handle duplicate calculator form submission
     */
    private function handle_duplicate_calculator_form() {
        $calculator_id = $this->get_post_data('calculator_id', 'int');
        
        if (!$calculator_id) {
            wp_die(esc_html__('Invalid calculator ID.', 'pricewise-calculator-pro'));
        }
        
        $new_calculator_id = $this->pwcp_duplicate_calculator($calculator_id);
        
        if ($new_calculator_id) {
            $this->bust_counts_cache();
        }
        
        $message = $new_calculator_id ? 'duplicated' : 'error';
        $redirect_url = add_query_arg(
            array(
                'page' => 'pwcp-calculators',
                'message' => $message,
            ),
            admin_url('admin.php')
        );
        
        wp_safe_redirect($redirect_url);
        exit;
    }
    
    /**
     * Generate calculator card HTML
     */
    public function generate_calculator_card_html($calculator) {
        if (!$calculator) {
            return '';
        }
        
        require_once PWCP_PLUGIN_DIR . 'admin/class-pwcp-admin-calculator-list.php';
        
        if (!class_exists('PWCP_Admin_Calculator_List')) {
            return '';
        }
        
        $list = new PWCP_Admin_Calculator_List();
        
        if (method_exists($list, 'render_calculator_card')) {
            ob_start();
            $list->render_calculator_card($calculator);
            return ob_get_clean();
        }
        
        if (method_exists($list, 'pwcp_render_calculator_card')) {
            try {
                $method = new ReflectionMethod($list, 'pwcp_render_calculator_card');
                $method->setAccessible(true);
                
                ob_start();
                $method->invoke($list, $calculator);
                return ob_get_clean();
            } catch (ReflectionException $e) {
                return '';
            }
        }
        
        return '';
    }
    
    /**
     * Get assignment type label
     */
    private function get_assignment_label($type) {
        $labels = array(
            'global' => __('Global', 'pricewise-calculator-pro'),
            'product' => __('Product', 'pricewise-calculator-pro'),
            'category' => __('Category', 'pricewise-calculator-pro'),
        );
        
        return isset($labels[$type]) ? $labels[$type] : __('Unknown', 'pricewise-calculator-pro');
    }
}

if (is_admin() && class_exists('PWCP_Admin_Calculator_Manager')) {
    new PWCP_Admin_Calculator_Manager();
}