<?php
/**
 * Calculator List Page
 *
 * Manages the display and operations for the calculator list interface.
 * Provides grid/list view options, search, filtering, and bulk actions.
 *
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

// Load dependencies
require_once PWCP_PLUGIN_DIR . 'includes/class-pwcp-base-ajax-handler.php';
require_once PWCP_PLUGIN_DIR . 'includes/traits/trait-pwcp-security-validation.php';
require_once PWCP_PLUGIN_DIR . 'includes/class-pwcp-input-sanitizer.php';

/**
 * PWCP Admin Calculator List Class
 *
 * Handles calculator list display and management operations.
 *
 * @since 1.0.0
 */
class PWCP_Admin_Calculator_List extends PWCP_Base_Ajax_Handler {

    use PWCP_Security_Validation_Trait;

    /**
     * Calculator manager instance
     *
     * @var PWCP_Admin_Calculator_Manager
     */
    private $manager;

    /**
     * Constructor - Initialize hooks
     *
     * @since 1.0.0
     */
    public function __construct() {
        $this->manager = new PWCP_Admin_Calculator_Manager();
        
        add_action('admin_enqueue_scripts', array($this, 'pwcp_enqueue_list_assets'));
        add_action('admin_init', array($this, 'pwcp_handle_bulk_actions'));
        
        // Register AJAX handlers
        add_action('wp_ajax_pwcp_save_view_preference', array($this, 'pwcp_save_view_preference'));
        add_action('wp_ajax_pwcp_delete_calculator', array($this, 'pwcp_ajax_delete_calculator'));
        add_action('wp_ajax_pwcp_duplicate_calculator', array($this, 'pwcp_ajax_duplicate_calculator'));
        add_action('wp_ajax_pwcp_get_calculator_counts', array($this, 'pwcp_ajax_get_calculator_counts'));
    }

    /**
     * Centralized cache busting helper
     *
     * @since 1.0.0
     * @return void
     */
    private function pwcp_bust_counts_cache() {
        wp_cache_delete('pwcp_calculator_counts', 'pwcp_calculators');
    }

    /**
     * AJAX handler to get current calculator counts
     *
     * @since 1.0.0
     * @return void
     */
    public function pwcp_ajax_get_calculator_counts() {
        $this->verify_ajax_security('manage_options', 'pwcp_calculator_action', 'nonce');

        // Get accurate counts from database
        $counts = $this->pwcp_get_accurate_counts();
        
        $this->send_success_response(
            __('Counts updated successfully', 'pricewise-calculator-pro'),
            array('counts' => $counts)
        );
    }

    /**
     * AJAX handler for deleting calculator
     *
     * @since 1.0.0
     * @return void
     */
    public function pwcp_ajax_delete_calculator() {
        $this->verify_ajax_security('manage_options', 'pwcp_calculator_action', 'nonce');

        $calculator_id = $this->get_post_data('calculator_id', 'int', 0);
        
        if (empty($calculator_id)) {
            $this->send_error_response(__('Invalid calculator ID', 'pricewise-calculator-pro'));
        }

        // Delete calculator using manager method
        $result = $this->manager->pwcp_delete_calculator($calculator_id);
        
        if ($result) {
            // Bust cache after delete
            $this->pwcp_bust_counts_cache();
            
            $this->send_success_response(
                __('Calculator deleted successfully!', 'pricewise-calculator-pro'),
                array('calculator_id' => $calculator_id)
            );
        } else {
            $this->send_error_response(__('Failed to delete calculator. Please try again.', 'pricewise-calculator-pro'));
        }
    }

    /**
     * AJAX handler for duplicating calculator
     *
     * @since 1.0.0
     * @return void
     */
    public function pwcp_ajax_duplicate_calculator() {
        $this->verify_ajax_security('manage_options', 'pwcp_calculator_action', 'nonce');

        $calculator_id = $this->get_post_data('calculator_id', 'int', 0);
        
        if (empty($calculator_id)) {
            $this->send_error_response(__('Invalid calculator ID', 'pricewise-calculator-pro'));
        }

        // Duplicate calculator using manager method
        $new_calculator_id = $this->manager->pwcp_duplicate_calculator($calculator_id);
        
        if ($new_calculator_id) {
            // Bust cache after duplicate
            $this->pwcp_bust_counts_cache();
            
            // Get the new calculator data for frontend rendering
            $new_calculator = $this->manager->pwcp_get_calculator($new_calculator_id);
            
            if ($new_calculator) {
                // Generate the HTML for the new calculator card
                ob_start();
                $this->pwcp_render_calculator_card($new_calculator);
                $new_card_html = ob_get_clean();
                
                $this->send_success_response(
                    __('Calculator duplicated successfully!', 'pricewise-calculator-pro'),
                    array(
                        'new_calculator_id' => $new_calculator_id,
                        'new_card_html' => $new_card_html,
                        'original_calculator_id' => $calculator_id
                    )
                );
            } else {
                $this->send_error_response(__('Calculator duplicated but failed to retrieve data. Please refresh the page.', 'pricewise-calculator-pro'));
            }
        } else {
            $this->send_error_response(__('Failed to duplicate calculator. Please try again.', 'pricewise-calculator-pro'));
        }
    }

    /**
     * Handle view preference saving
     *
     * @since 1.0.0
     * @return void
     */
    public function pwcp_save_view_preference() {
        $this->verify_ajax_security('manage_options', 'pwcp_calculator_action', 'nonce');

        $view = $this->get_post_data('view', 'text', '');
        
        if (!$this->is_action_allowed($view, array('grid', 'list'))) {
            $this->send_error_response(__('Invalid view type.', 'pricewise-calculator-pro'));
        }

        $user_id = get_current_user_id();
        update_user_meta($user_id, 'pwcp_calculator_view', $view);
        
        $this->send_success_response(
            __('View preference saved successfully!', 'pricewise-calculator-pro'),
            array('view' => $view)
        );
    }

    /**
     * Render calculators list page
     *
     * @since 1.0.0
     * @return void
     */
    public function pwcp_render_list_page() {
        if (!$this->can_manage_calculators()) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'pricewise-calculator-pro'));
        }

        $calculators = $this->manager->pwcp_get_calculators_list();
        
        // Get accurate counts from server
        $counts = $this->pwcp_get_accurate_counts();
        
        // Detect saved view preference for server-side rendering
        $user_id = get_current_user_id();
        $saved_view = get_user_meta($user_id, 'pwcp_calculator_view', true);
        if (empty($saved_view)) {
            $saved_view = 'grid'; // Default view
        }

        ?>
        <div class="wrap pwcp-modern-admin pwcp-page-calculator-list">
            <!-- Modern Hero Header -->
            <div class="pwcp-hero-header">
                <div class="pwcp-hero-background">
                    <div class="pwcp-hero-particles"></div>
                    <div class="pwcp-hero-gradient"></div>
                </div>
                
                <div class="pwcp-hero-content">
                    <div class="pwcp-hero-left">
                        <div class="pwcp-hero-badge">
                            <?php esc_html_e('Management Hub', 'pricewise-calculator-pro'); ?>
                        </div>
                        <h1 class="pwcp-hero-title">
                            <?php esc_html_e('All Calculators', 'pricewise-calculator-pro'); ?>
                        </h1>
                        <p class="pwcp-hero-subtitle">
                            <?php esc_html_e('Manage all your calculators from one powerful interface with modern tools and smart features', 'pricewise-calculator-pro'); ?>
                        </p>
                        
                        <!-- Stats Overview with accurate server-side counts -->
                        <div class="pwcp-stats-overview">
                            <div class="pwcp-stat-item">
                                <div class="pwcp-stat-number" id="pwcp-total-count"><?php echo esc_html(number_format_i18n($counts['total'])); ?></div>
                                <div class="pwcp-stat-label"><?php esc_html_e('Total', 'pricewise-calculator-pro'); ?></div>
                            </div>
                            <div class="pwcp-stat-item">
                                <div class="pwcp-stat-number" id="pwcp-active-count"><?php echo esc_html(number_format_i18n($counts['active'])); ?></div>
                                <div class="pwcp-stat-label"><?php esc_html_e('Active', 'pricewise-calculator-pro'); ?></div>
                            </div>
                            <div class="pwcp-stat-item">
                                <div class="pwcp-stat-number" id="pwcp-recent-count"><?php echo esc_html(number_format_i18n($counts['recent'])); ?></div>
                                <div class="pwcp-stat-label"><?php esc_html_e('Recent', 'pricewise-calculator-pro'); ?></div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="pwcp-hero-right">
                        <div class="pwcp-hero-actions">
                            <a href="<?php echo esc_url(admin_url('admin.php?page=pwcp-add-calculator')); ?>" 
                               class="pwcp-btn pwcp-btn-primary">
                                <svg class="pwcp-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <line x1="12" y1="5" x2="12" y2="19"/>
                                    <line x1="5" y1="12" x2="19" y2="12"/>
                                </svg>
                                <?php esc_html_e('Add New Calculator', 'pricewise-calculator-pro'); ?>
                            </a>
                            
                            <div class="pwcp-view-toggle" data-mobile-behavior="hide">
                                <button type="button" 
                                        class="pwcp-view-btn pwcp-view-grid <?php echo 'grid' === $saved_view ? 'pwcp-view-active' : ''; ?>" 
                                        data-view="grid" 
                                        title="<?php esc_attr_e('Grid View', 'pricewise-calculator-pro'); ?>">
                                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                        <rect x="3" y="3" width="7" height="7"/>
                                        <rect x="14" y="3" width="7" height="7"/>
                                        <rect x="14" y="14" width="7" height="7"/>
                                        <rect x="3" y="14" width="7" height="7"/>
                                    </svg>
                                </button>
                                <button type="button" 
                                        class="pwcp-view-btn pwcp-view-list <?php echo 'list' === $saved_view ? 'pwcp-view-active' : ''; ?>" 
                                        data-view="list" 
                                        title="<?php esc_attr_e('List View', 'pricewise-calculator-pro'); ?>">
                                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                        <line x1="8" y1="6" x2="21" y2="6"/>
                                        <line x1="8" y1="12" x2="21" y2="12"/>
                                        <line x1="8" y1="18" x2="21" y2="18"/>
                                        <line x1="3" y1="6" x2="3.01" y2="6"/>
                                        <line x1="3" y1="12" x2="3.01" y2="12"/>
                                        <line x1="3" y1="18" x2="3.01" y2="18"/>
                                    </svg>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <?php $this->pwcp_display_admin_notices(); ?>

            <div class="pwcp-modern-container">
                <!-- Control Bar -->
                <div class="pwcp-control-bar">
                    <div class="pwcp-control-left">
                        <!-- Search -->
                        <div class="pwcp-search-wrapper">
                            <input type="text" 
                                   id="pwcp-calculator-search" 
                                   class="pwcp-search-input" 
                                   placeholder="<?php esc_attr_e('Search calculators...', 'pricewise-calculator-pro'); ?>">
                            <svg class="pwcp-search-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <circle cx="11" cy="11" r="8"/>
                                <path d="m21 21-4.35-4.35"/>
                            </svg>
                        </div>
                        
                        <!-- Filters -->
                        <div class="pwcp-filter-group">
                            <select id="pwcp-status-filter" class="pwcp-filter-select">
                                <option value=""><?php esc_html_e('All Status', 'pricewise-calculator-pro'); ?></option>
                                <option value="active"><?php esc_html_e('Active', 'pricewise-calculator-pro'); ?></option>
                                <option value="inactive"><?php esc_html_e('Inactive', 'pricewise-calculator-pro'); ?></option>
                            </select>
                            
                            <select id="pwcp-type-filter" class="pwcp-filter-select">
                                <option value=""><?php esc_html_e('All Types', 'pricewise-calculator-pro'); ?></option>
                                <option value="global"><?php esc_html_e('Global', 'pricewise-calculator-pro'); ?></option>
                                <option value="product"><?php esc_html_e('Product', 'pricewise-calculator-pro'); ?></option>
                                <option value="category"><?php esc_html_e('Category', 'pricewise-calculator-pro'); ?></option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="pwcp-control-right">
                        <!-- Select All -->
                        <div class="pwcp-select-all-wrap">
                            <label>
                                <input type="checkbox" id="pwcp-select-all-calculators" />
                                <?php esc_html_e('Select all', 'pricewise-calculator-pro'); ?>
                            </label>
                        </div>
                        
                        <!-- Bulk Actions -->
                        <div class="pwcp-bulk-actions">
                            <select id="pwcp-bulk-action" class="pwcp-bulk-select">
                                <option value=""><?php esc_html_e('Bulk Actions', 'pricewise-calculator-pro'); ?></option>
                                <?php if (current_user_can('manage_options')) : ?>
                                    <option value="activate"><?php esc_html_e('Activate', 'pricewise-calculator-pro'); ?></option>
                                    <option value="deactivate"><?php esc_html_e('Deactivate', 'pricewise-calculator-pro'); ?></option>
                                    <option value="duplicate"><?php esc_html_e('Duplicate', 'pricewise-calculator-pro'); ?></option>
                                    <option value="delete"><?php esc_html_e('Delete', 'pricewise-calculator-pro'); ?></option>
                                <?php endif; ?>
                            </select>
                            <button type="button" id="pwcp-apply-bulk" class="pwcp-btn pwcp-btn-ghost pwcp-btn-sm pwcp-btn-apply-bulk">
                                <?php esc_html_e('Apply', 'pricewise-calculator-pro'); ?>
                            </button>
                        </div>
                    </div>
                </div>

                <!-- Calculator Grid/List -->
                <div class="pwcp-calculator-container">
                    <div id="pwcp-calculator-grid" 
                         class="pwcp-calculator-grid pwcp-view-<?php echo esc_attr($saved_view); ?>" 
                         data-initial-view="<?php echo esc_attr($saved_view); ?>"
                         data-mobile-view="grid">
                        <?php if (!empty($calculators)) : ?>
                            <?php foreach ($calculators as $calculator) : ?>
                                <?php $this->pwcp_render_calculator_card($calculator); ?>
                            <?php endforeach; ?>
                        <?php else : ?>
                            <?php $this->pwcp_render_empty_state(); ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

        <!-- Loading Overlay -->
        <div id="pwcp-loading-overlay" class="pwcp-loading-overlay pwcp-hidden">
            <div class="pwcp-loading-spinner">
                <div class="pwcp-spinner"></div>
                <p><?php esc_html_e('Processing...', 'pricewise-calculator-pro'); ?></p>
            </div>
        </div>

        <!-- Accessibility - ARIA live region for screen reader announcements -->
        <div aria-live="polite" aria-atomic="true" class="pwcp-aria-live pwcp-sr-only"></div>
        <?php
    }

    /**
     * Get accurate counts from database with caching
     *
     * @since 1.0.0
     * @return array
     */
    private function pwcp_get_accurate_counts() {
        $cache_key = 'pwcp_calculator_counts';
        $cache_group = 'pwcp_calculators';
        
        // Try to get from cache first
        $counts = wp_cache_get($cache_key, $cache_group);
        
        if (false === $counts) {
            global $wpdb;
            $table = $wpdb->prefix . 'pwcp_calculators';
            
            // Get total count
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name from $wpdb->prefix
            $total = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(id) FROM `" . esc_sql( $table ) . "` WHERE 1=%d", 1 ) );
            
            // Get active count
            $active = (int) $wpdb->get_var(
                $wpdb->prepare("SELECT COUNT(id) FROM `{$table}` WHERE status = %s", 'active')
            );
            
            // Get recent count (last 24 hours)
            $twenty_four_hours_ago = gmdate('Y-m-d H:i:s', strtotime('-24 hours'));
            $recent = (int) $wpdb->get_var(
                $wpdb->prepare("SELECT COUNT(id) FROM `{$table}` WHERE created_at >= %s", $twenty_four_hours_ago)
            );
            
            $counts = array(
                'total'    => $total,
                'active'   => $active,
                'inactive' => max(0, $total - $active),
                'recent'   => $recent
            );
            
            // Cache for 5 minutes
            wp_cache_set($cache_key, $counts, $cache_group, 300);
        }
        
        return $counts;
    }

    /**
     * Render calculator card with proper data attributes
     *
     * @param object $calculator Calculator object
     * @since 1.0.0
     * @return void
     */
    private function pwcp_render_calculator_card($calculator) {
        $edit_url = add_query_arg(
            array(
                'page'          => 'pwcp-add-calculator',
                'action'        => 'edit',
                'calculator_id' => absint($calculator->id),
            ),
            admin_url('admin.php')
        );
        
        $fields = $this->manager->pwcp_get_calculator_fields($calculator->id);
        $fields_count = is_array($fields) ? count($fields) : 0;
        $assignment_label = $this->pwcp_get_assignment_label(isset($calculator->assignment_type) ? $calculator->assignment_type : 'global');
        $status = isset($calculator->status) ? $calculator->status : 'inactive';
        $status_class = 'active' === $status ? 'pwcp-status-active' : 'pwcp-status-inactive';
        $status_text = 'active' === $status ? esc_html__('Active', 'pricewise-calculator-pro') : esc_html__('Inactive', 'pricewise-calculator-pro');
        $created_at = isset($calculator->created_at) ? $calculator->created_at : current_time('mysql');
        $created_date = mysql2date(get_option('date_format'), (string) $created_at);
        
        // Check if calculator is recent (within 24 hours)
        $created_timestamp = strtotime((string) $created_at);
        $is_recent = $created_timestamp >= strtotime('-24 hours');
        
        // Multibyte safe name conversion for data attribute
        $data_name = function_exists('mb_strtolower') ? mb_strtolower((string) $calculator->name, 'UTF-8') : strtolower((string) $calculator->name);
        
        ?>
        <div class="pwcp-calculator-card" 
             data-id="<?php echo esc_attr((string) $calculator->id); ?>"
             data-status="<?php echo esc_attr((string) $calculator->status); ?>"
             data-type="<?php echo esc_attr((string) $calculator->assignment_type); ?>"
             data-name="<?php echo esc_attr($data_name); ?>"
             data-created="<?php echo esc_attr(gmdate('c', $created_timestamp)); ?>"
             data-is-recent="<?php echo $is_recent ? 'true' : 'false'; ?>">
             
            <!-- Card Header -->
            <div class="pwcp-card-header">
                <div class="pwcp-card-checkbox">
                    <input type="checkbox" 
                           class="pwcp-calculator-checkbox" 
                           value="<?php echo esc_attr((string) $calculator->id); ?>"
                           aria-label="<?php esc_attr_e('Select calculator', 'pricewise-calculator-pro'); ?>">
                </div>
            </div>
            
            <!-- Card Content -->
            <div class="pwcp-card-content">
                <div class="pwcp-card-icon" role="img" aria-label="<?php esc_attr_e('Calculator icon', 'pricewise-calculator-pro'); ?>">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.1">
                        <rect x="5" y="3" width="14" height="18" rx="2"/>
                        <path d="M7 7h10v2H7z"/>
                        <circle cx="8" cy="11" r="0.8"/>
                        <circle cx="12" cy="11" r="0.8"/>
                        <circle cx="16" cy="11" r="0.8"/>
                        <circle cx="8" cy="14" r="0.8"/>
                        <circle cx="12" cy="14" r="0.8"/>
                        <circle cx="16" cy="14" r="0.8"/>
                        <circle cx="8" cy="17" r="0.8"/>
                        <circle cx="12" cy="17" r="0.8"/>
                        <circle cx="16" cy="17" r="0.8"/>
                    </svg>
                </div>
                
                <div class="pwcp-card-title-section">
                    <h3 class="pwcp-card-title">
                        <a href="<?php echo esc_url($edit_url); ?>"><?php echo esc_html($calculator->name); ?></a>
                    </h3>
                    
                    <p class="pwcp-card-description">
                        <?php 
                        if (!empty($calculator->description)) {
                            echo esc_html(wp_trim_words((string) $calculator->description, 15));
                        } else {
                            esc_html_e('No description available', 'pricewise-calculator-pro');
                        }
                        ?>
                    </p>
                    
                    <div class="pwcp-card-meta-inline">
                        <div class="pwcp-meta-inline">
                            <span class="pwcp-meta-inline-label"><?php esc_html_e('Type:', 'pricewise-calculator-pro'); ?></span>
                            <span class="pwcp-meta-inline-value"><?php echo esc_html($assignment_label); ?></span>
                        </div>
                        <div class="pwcp-meta-inline">
                            <span class="pwcp-meta-inline-label"><?php esc_html_e('Fields:', 'pricewise-calculator-pro'); ?></span>
                            <span class="pwcp-meta-inline-value"><?php echo esc_html((string) $fields_count); ?></span>
                        </div>
                        <div class="pwcp-meta-inline">
                            <span class="pwcp-meta-inline-label"><?php esc_html_e('Priority:', 'pricewise-calculator-pro'); ?></span>
                            <span class="pwcp-meta-inline-value"><?php echo esc_html((string) absint(isset($calculator->priority) ? $calculator->priority : 1)); ?></span>
                        </div>
                        <div class="pwcp-meta-inline">
                            <span class="pwcp-meta-inline-label"></span>
                            <span class="pwcp-meta-inline-value">
                                <?php 
                                printf( 
                                  /* translators: %s: date when the calculator was created */
                                    esc_html__('Created %s', 'pricewise-calculator-pro'), 
                                    esc_html($created_date) 
                                ); 
                                ?>
                            </span>
                        </div>
                    </div>
                </div>
                
                <div class="pwcp-card-status">
                    <span class="pwcp-status-indicator <?php echo esc_attr($status_class); ?>">
                        <?php echo esc_html($status_text); ?>
                    </span>
                </div>
                
                <div class="pwcp-card-actions">
                    <div class="pwcp-action-buttons">
                        <a href="<?php echo esc_url($edit_url); ?>" 
                           class="pwcp-action-btn pwcp-action-edit" 
                           title="<?php esc_attr_e('Edit Calculator', 'pricewise-calculator-pro'); ?>">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M11 4H4a2 2 0 00-2 2v14a2 2 0 002 2h14a2 2 0 002-2v-7"/>
                                <path d="M18.5 2.5a2.121 2.121 0 013 3L12 15l-4 1 1-4 9.5-9.5z"/>
                            </svg>
                        </a>
                        
                        <button type="button" 
                                class="pwcp-action-btn pwcp-action-duplicate pwcp-duplicate-calculator" 
                                data-calculator-id="<?php echo esc_attr((string) $calculator->id); ?>"
                                title="<?php esc_attr_e('Duplicate Calculator', 'pricewise-calculator-pro'); ?>">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <rect x="9" y="9" width="13" height="13" rx="2" ry="2"/>
                                <path d="M5 15H4a2 2 0 01-2-2V4a2 2 0 012-2h9a2 2 0 012 2v1"/>
                            </svg>
                        </button>
                        
                        <button type="button" 
                                class="pwcp-action-btn pwcp-action-delete pwcp-delete-calculator" 
                                data-calculator-id="<?php echo esc_attr((string) $calculator->id); ?>"
                                title="<?php esc_attr_e('Delete Calculator', 'pricewise-calculator-pro'); ?>">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <polyline points="3,6 5,6 21,6"/>
                                <path d="M19,6v14a2,2 0 01-2,2H7a2,2 0 01-2-2V6m3,0V4a2,2 0 012-2h4a2,2 0 012,2v2"/>
                                <line x1="10" y1="11" x2="10" y2="17"/>
                                <line x1="14" y1="11" x2="14" y2="17"/>
                            </svg>
                        </button>
                    </div>
                </div>
            </div>
            
            <div class="pwcp-card-meta">
                <div class="pwcp-meta-item">
                    <span class="pwcp-meta-label"><?php esc_html_e('Type', 'pricewise-calculator-pro'); ?></span>
                    <span class="pwcp-meta-value"><?php echo esc_html($assignment_label); ?></span>
                </div>
                <div class="pwcp-meta-item">
                    <span class="pwcp-meta-label"><?php esc_html_e('Fields', 'pricewise-calculator-pro'); ?></span>
                    <span class="pwcp-meta-value"><?php echo esc_html((string) $fields_count); ?></span>
                </div>
                <div class="pwcp-meta-item">
                    <span class="pwcp-meta-label"><?php esc_html_e('Priority', 'pricewise-calculator-pro'); ?></span>
                    <span class="pwcp-meta-value"><?php echo esc_html((string) absint($calculator->priority)); ?></span>
                </div>
            </div>
            
            <div class="pwcp-card-footer">
                <div class="pwcp-footer-meta">
                    <span class="pwcp-created-date">
                        <?php 
                        printf( 
                               /* translators: %s: date when the calculator was created */
                            esc_html__('Created %s', 'pricewise-calculator-pro'), 
                            esc_html($created_date) 
                        ); 
                        ?>
                    </span>
                </div>
                
                <div class="pwcp-footer-actions">
                    <div class="pwcp-card-status pwcp-list-only">
                        <span class="pwcp-status-indicator <?php echo esc_attr($status_class); ?>">
                            <?php echo esc_html($status_text); ?>
                        </span>
                    </div>
                    
                    <div class="pwcp-action-buttons">
                        <a href="<?php echo esc_url($edit_url); ?>" 
                           class="pwcp-action-btn pwcp-action-edit" 
                           title="<?php esc_attr_e('Edit Calculator', 'pricewise-calculator-pro'); ?>">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M11 4H4a2 2 0 00-2 2v14a2 2 0 002 2h14a2 2 0 002-2v-7"/>
                                <path d="M18.5 2.5a2.121 2.121 0 113 3L12 15l-4 1 1-4 9.5-9.5z"/>
                            </svg>
                        </a>
                        
                        <button type="button" 
                                class="pwcp-action-btn pwcp-action-duplicate pwcp-duplicate-calculator" 
                                data-calculator-id="<?php echo esc_attr((string) $calculator->id); ?>"
                                title="<?php esc_attr_e('Duplicate Calculator', 'pricewise-calculator-pro'); ?>">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <rect x="9" y="9" width="13" height="13" rx="2" ry="2"/>
                                <path d="M5 15H4a2 2 0 01-2-2V4a2 2 0 012-2h9a2 2 0 012 2v1"/>
                            </svg>
                        </button>
                        
                        <button type="button" 
                                class="pwcp-action-btn pwcp-action-delete pwcp-delete-calculator" 
                                data-calculator-id="<?php echo esc_attr((string) $calculator->id); ?>"
                                title="<?php esc_attr_e('Delete Calculator', 'pricewise-calculator-pro'); ?>">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <polyline points="3,6 5,6 21,6"/>
                                <path d="M19,6v14a2,2 0 01-2,2H7a2,2 0 01-2-2V6m3,0V4a2,2 0 012-2h4a2,2 0 012,2v2"/>
                                <line x1="10" y1="11" x2="10" y2="17"/>
                                <line x1="14" y1="11" x2="14" y2="17"/>
                            </svg>
                        </button>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render empty state
     *
     * @since 1.0.0
     * @return void
     */
    private function pwcp_render_empty_state() {
        ?>
        <div class="pwcp-empty-state pwcp-empty-calculators pwcp-empty-wrap">
            <div class="pwcp-empty-illustration">
                <svg class="pwcp-empty-icon" viewBox="0 0 200 150">
                    <defs>
                        <linearGradient id="calc-gradient" x1="0%" y1="0%" x2="100%" y2="100%">
                            <stop offset="0%" style="stop-color:#6366f1;stop-opacity:0.1"/>
                            <stop offset="100%" style="stop-color:#a855f7;stop-opacity:0.1"/>
                        </linearGradient>
                    </defs>
                    <rect x="40" y="30" width="120" height="90" rx="8" fill="url(#calc-gradient)" stroke="#6366f1" stroke-width="2"/>
                    <rect x="50" y="45" width="25" height="15" rx="2" fill="#6366f1" opacity="0.3"/>
                    <rect x="80" y="45" width="25" height="15" rx="2" fill="#8b5cf6" opacity="0.3"/>
                    <rect x="110" y="45" width="25" height="15" rx="2" fill="#a855f7" opacity="0.3"/>
                    <rect x="50" y="65" width="25" height="15" rx="2" fill="#6366f1" opacity="0.3"/>
                    <rect x="80" y="65" width="25" height="15" rx="2" fill="#8b5cf6" opacity="0.3"/>
                    <rect x="110" y="65" width="25" height="15" rx="2" fill="#a855f7" opacity="0.3"/>
                    <rect x="50" y="85" width="85" height="20" rx="3" fill="#6366f1" opacity="0.2"/>
                    <circle cx="180" cy="40" r="8" fill="#6366f1" opacity="0.2"/>
                    <circle cx="20" cy="80" r="6" fill="#8b5cf6" opacity="0.2"/>
                    <circle cx="25" cy="45" r="4" fill="#a855f7" opacity="0.2"/>
                </svg>
            </div>
            <h3><?php esc_html_e('No Calculators Found', 'pricewise-calculator-pro'); ?></h3>
            <p><?php esc_html_e('Start building powerful calculators for your users with our easy-to-use form builder.', 'pricewise-calculator-pro'); ?></p>
            <a href="<?php echo esc_url(admin_url('admin.php?page=pwcp-add-calculator')); ?>" 
               class="pwcp-btn pwcp-btn-primary pwcp-btn-icon">
                <svg class="pwcp-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <line x1="12" y1="5" x2="12" y2="19"/>
                    <line x1="5" y1="12" x2="19" y2="12"/>
                </svg>
                <?php esc_html_e('Create Your First Calculator', 'pricewise-calculator-pro'); ?>
            </a>
        </div>
        <?php
    }

    /**
     * Get assignment type label
     *
     * @param string $type Assignment type
     * @since 1.0.0
     * @return string
     */
    private function pwcp_get_assignment_label($type) {
        $labels = array(
            'global'   => __('Global', 'pricewise-calculator-pro'),
            'product'  => __('Product', 'pricewise-calculator-pro'),
            'category' => __('Category', 'pricewise-calculator-pro'),
        );
        
        return isset($labels[$type]) ? $labels[$type] : __('Unknown', 'pricewise-calculator-pro');
    }

    /**
     * Handle bulk actions (kept for fallback)
     *
     * @since 1.0.0
     * @return void
     */
    public function pwcp_handle_bulk_actions() {
        if (!isset($_POST['action']) || '-1' === $_POST['action']) {
            return;
        }

        if (!isset($_POST['calculator']) || !is_array($_POST['calculator'])) {
            return;
        }

        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'bulk-calculators')) {
            wp_die(esc_html__('Security verification failed.', 'pricewise-calculator-pro'));
        }

        if (!$this->can_manage_calculators()) {
            wp_die(esc_html__('Insufficient permissions.', 'pricewise-calculator-pro'));
        }

        $action = sanitize_text_field(wp_unslash($_POST['action']));
        $calculator_ids = array_map('absint', wp_unslash($_POST['calculator']));

        switch ($action) {
            case 'delete':
                $this->manager->pwcp_bulk_delete_calculators($calculator_ids);
                $this->pwcp_bust_counts_cache();
                $this->pwcp_redirect_with_message('bulk_delete');
                break;
            case 'deactivate':
                $this->manager->pwcp_bulk_update_status($calculator_ids, 'inactive');
                $this->pwcp_bust_counts_cache();
                $this->pwcp_redirect_with_message('bulk_deactivate');
                break;
        }
    }

    /**
     * Redirect with message
     *
     * @since 1.0.0
     * @param string $message Message key
     * @return void
     */
    private function pwcp_redirect_with_message($message) {
        $redirect_url = add_query_arg(
            array(
                'page'    => 'pwcp-calculators',
                'message' => $message,
            ),
            admin_url('admin.php')
        );

        wp_safe_redirect($redirect_url);
        exit;
    }

    /**
     * Display admin notices
     *
     * @since 1.0.0
     * @return void
     */
    private function pwcp_display_admin_notices() {
        if (!isset($_GET['message'])) {
            return;
        }

        $message = sanitize_text_field(wp_unslash($_GET['message']));
        $notices = array(
            'deleted'         => array('type' => 'success', 'text' => esc_html__('Calculator deleted successfully!', 'pricewise-calculator-pro')),
            'duplicated'      => array('type' => 'success', 'text' => esc_html__('Calculator duplicated successfully!', 'pricewise-calculator-pro')),
            'bulk_delete'     => array('type' => 'success', 'text' => esc_html__('Selected calculators deleted successfully!', 'pricewise-calculator-pro')),
            'bulk_activate'   => array('type' => 'success', 'text' => esc_html__('Selected calculators activated successfully!', 'pricewise-calculator-pro')),
            'bulk_deactivate' => array('type' => 'success', 'text' => esc_html__('Selected calculators deactivated successfully!', 'pricewise-calculator-pro')),
            'error'           => array('type' => 'error', 'text' => esc_html__('An error occurred. Please try again.', 'pricewise-calculator-pro')),
        );

        if (isset($notices[$message])) {
            $notice = $notices[$message];
            printf(
                '<div class="notice notice-%1$s is-dismissible"><p>%2$s</p></div>',
                esc_attr($notice['type']),
                esc_html($notice['text'])
            );
        }
    }

    /**
     * Enqueue list page assets
     *
     * @since 1.0.0
     * @param string $hook Current admin page hook
     * @return void
     */
    public function pwcp_enqueue_list_assets($hook) {
        $screen = function_exists('get_current_screen') ? get_current_screen() : null;
        if (!$screen) {
            return;
        }

        if ('calculator-pro_page_pwcp-calculators' !== $screen->id) {
            return;
        }

        wp_enqueue_style(
            'pwcp-calculator-list-styles',
            PWCP_ASSETS_URL . 'css/all-admin-css/admin-calculator-list.css',
            array(),
            PWCP_PLUGIN_VERSION
        );

        wp_enqueue_script(
            'pwcp-calculator-list-script',
            PWCP_ASSETS_URL . 'js/all-admin-js/admin-calculator-list.js',
            array('jquery'),
            PWCP_PLUGIN_VERSION,
            true
        );

        $user_id = get_current_user_id();
        $saved_view = get_user_meta($user_id, 'pwcp_calculator_view', true);
        if (empty($saved_view)) {
            $saved_view = 'grid';
        }

        wp_localize_script(
            'pwcp-calculator-list-script',
            'pwcpCalculatorList',
            array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce'    => wp_create_nonce('pwcp_calculator_action'),
                'bulk_nonce' => wp_create_nonce('pwcp_bulk_calculators'),
                'initial_view' => $saved_view,
                'confirmations' => true,
                'debug' => defined('WP_DEBUG') && WP_DEBUG,
                'strings'  => array(
                    'confirm_delete'         => esc_html__('Are you sure you want to delete this calculator? This action cannot be undone.', 'pricewise-calculator-pro'),
                    'confirm_bulk_delete'    => esc_html__('Are you sure you want to delete the selected calculators? This action cannot be undone.', 'pricewise-calculator-pro'),
                    'confirm_duplicate'      => esc_html__('Create a duplicate of this calculator?', 'pricewise-calculator-pro'),
                    'confirm_bulk_duplicate' => esc_html__('Create duplicates of the selected calculators?', 'pricewise-calculator-pro'),
                    'deleting'               => esc_html__('Deleting calculator...', 'pricewise-calculator-pro'),
                    'duplicating'            => esc_html__('Creating duplicate...', 'pricewise-calculator-pro'),
                    'processing'             => esc_html__('Processing request...', 'pricewise-calculator-pro'),
                    'error'                  => esc_html__('An error occurred. Please try again.', 'pricewise-calculator-pro'),
                    'no_selection'           => esc_html__('Please select one or more calculators first.', 'pricewise-calculator-pro'),
                    'no_action'              => esc_html__('Please select a bulk action to perform.', 'pricewise-calculator-pro'),
                    'search_placeholder'     => esc_html__('Type to search calculators...', 'pricewise-calculator-pro'),
                    'view_switched'          => esc_html__('View mode changed successfully.', 'pricewise-calculator-pro'),
                    'ajax_config_missing'    => esc_html__('AJAX configuration not found.', 'pricewise-calculator-pro'),
                    'request_failed'         => esc_html__('Request failed. Please try again.', 'pricewise-calculator-pro'),
                    'counts_invalid'         => esc_html__('Invalid count response from server.', 'pricewise-calculator-pro'),
                    'no_results_title'       => esc_html__('No Results Found', 'pricewise-calculator-pro'),
                    'no_results_body'        => esc_html__('Try adjusting your search or filter criteria to find what you\'re looking for.', 'pricewise-calculator-pro'),
                    'clear_filters'          => esc_html__('Clear Filters', 'pricewise-calculator-pro'),
                    'status_active'          => esc_html__('Active', 'pricewise-calculator-pro'),
                    'status_inactive'        => esc_html__('Inactive', 'pricewise-calculator-pro'),
                    'empty_state_title'       => esc_html__('No Calculators Found', 'pricewise-calculator-pro'),
                    'empty_state_description' => esc_html__('Start building powerful calculators for your users with our easy-to-use form builder.', 'pricewise-calculator-pro'),
                    'create_first_calculator' => esc_html__('Create Your First Calculator', 'pricewise-calculator-pro'),
                ),
            )
        );
    }
}

if (is_admin() && class_exists('PWCP_Admin_Calculator_List')) {
    new PWCP_Admin_Calculator_List();
}