<?php
/**
 * Calculator Form Editor
 *
 * Handles the admin interface for creating and editing calculator forms.
 * FREE VERSION - Global assignment and number fields only
 *
 * @package PriceWise_Calculator_Pro
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

// Load dependencies
require_once PWCP_PLUGIN_DIR . 'includes/class-pwcp-base-ajax-handler.php';
require_once PWCP_PLUGIN_DIR . 'includes/traits/trait-pwcp-security-validation.php';
require_once PWCP_PLUGIN_DIR . 'includes/class-pwcp-input-sanitizer.php';

/**
 * PWCP Admin Calculator Form Editor Class
 *
 * @since 1.0.0
 */
class PWCP_Admin_Calculator_Form_Editor extends PWCP_Base_Ajax_Handler {

    use PWCP_Security_Validation_Trait;

    private $manager;
    private $field_manager;
    public $category_assignment;
    private $templates;
    
    /**
     * Static guard to prevent duplicate hook registration
     */
    private static $hooks_registered = false;

    public function __construct() {
        if (!$this->pwcp_init_dependencies()) {
            return;
        }
        
        // Register hooks only once to avoid duplication
        if (!self::$hooks_registered) {
            add_action('admin_enqueue_scripts', array($this, 'pwcp_enqueue_admin_assets'));
            self::$hooks_registered = true;
        }
        
        // AJAX actions handled by router via PWCP_Ajax_Handler::handle_form_editor_request()
    }

    private function pwcp_init_dependencies() {
        if (!class_exists('PWCP_Admin_Calculator_Manager')) {
            add_action('admin_notices', array($this, 'pwcp_show_manager_error'));
            return false;
        }
        
        $this->manager = new PWCP_Admin_Calculator_Manager();
        
        if (!class_exists('PWCP_Field_Manager')) {
            add_action('admin_notices', array($this, 'pwcp_show_field_manager_error'));
            return false;
        }
        
        $this->field_manager = new PWCP_Field_Manager();
        
        $template_file = plugin_dir_path(__FILE__) . 'class-pwcp-calculator-form-templates.php';
        if (file_exists($template_file)) {
            require_once $template_file;
            if (class_exists('PWCP_Calculator_Form_Templates')) {
                $this->templates = new PWCP_Calculator_Form_Templates();
            }
        }
    
        if (class_exists('PWCP_Category_Assignment')) {
            $this->category_assignment = new PWCP_Category_Assignment();
        }
        
        return true;
    }

    public function pwcp_show_manager_error() {
        printf(
            '<div class="notice notice-error"><p>%s</p></div>',
            esc_html__('Calculator Form Editor requires Calculator Manager class. Please check your plugin files.', 'pricewise-calculator-pro')
        );
    }

    public function pwcp_show_field_manager_error() {
        printf(
            '<div class="notice notice-error"><p>%s</p></div>',
            esc_html__('Calculator Form Editor requires Field Manager class. Please check your plugin files.', 'pricewise-calculator-pro')
        );
    }

    /**
     * AJAX: Save calculator
     * 
     * Handles saving calculator data including fields and formula.
     * FREE VERSION: Only global assignment allowed
     *
     * @since 1.0.0
     */
    public function ajax_save_calculator() {
        ob_start();
        
        // Verify security
        $this->verify_ajax_security('manage_options', 'pwcp_calculator_action');

        $calculator_id = $this->get_post_data('calculator_id', 'int', 0);
        $name = $this->get_post_data('calculator_name', 'text', '');
        $description = $this->get_post_data('calculator_description', 'textarea', '');
        $formula = $this->get_post_data('calculator_formula', 'textarea', '');
        $status = $this->get_post_data('calculator_status', 'text', 'active');
        $assignment_type = $this->get_post_data('assignment_type', 'text', 'global');
        $priority = $this->get_post_data('calculator_priority', 'int', 1);
        $hide_quantity_selector = $this->get_post_data('hide_quantity_selector', 'int', 0);
        $is_auto_save = (bool) $this->get_post_data('auto_save', 'int', 0);

        // Validate required fields
        if ('' === trim($name)) {
            ob_end_clean();
            $this->send_error_response(__('Calculator name is required.', 'pricewise-calculator-pro'));
        }

        // FREE VERSION: Force global assignment only
        if ($assignment_type !== 'global') {
            ob_end_clean();
            $this->send_error_response(__('Only Global assignment is available in free version. Visit devtonicstudios.com for premium features.', 'pricewise-calculator-pro'));
        }

        $calculator_data = array(
            'name' => $name,
            'description' => $description,
            'formula' => $formula,
            'status' => $status,
            'assignment_type' => 'global', // FORCED TO GLOBAL
            'assignment_ids' => wp_json_encode(array()),
            'priority' => $priority,
            'hide_quantity_selector' => $hide_quantity_selector,
        );

        $result = $this->manager->pwcp_save_calculator($calculator_data, $calculator_id);

        if ($calculator_id > 0) {
            if ($is_auto_save) {
                $message = esc_html__('Calculator auto-saved successfully!', 'pricewise-calculator-pro');
            } else {
                $message = esc_html__('Calculator updated successfully!', 'pricewise-calculator-pro');
            }
            $action = 'updated';
        } else {
            $calculator_id = $result;
            $message = esc_html__('Calculator created successfully!', 'pricewise-calculator-pro');
            $action = 'created';
        }
        
        // Process category assignments - DISABLED IN FREE VERSION
        // Category assignment is a premium feature

        if (false === $result || !$calculator_id) {
            ob_end_clean();
            $this->send_error_response(__('Database error occurred.', 'pricewise-calculator-pro'));
        }

        $field_id_mappings = array();
        if ($calculator_id && isset($_POST['fields']) && is_array($_POST['fields'])) {
            $fields_raw = (array) wp_unslash($_POST['fields']);
            
            // Sanitize and validate field data - numeric fields preserve their type
            $fields_raw = array_map(static function($f) {
                if (!is_array($f)) {
                    return array();
                }
                return array(
                    'id'       => isset($f['id']) ? (int) $f['id'] : 0,
                    'label'    => isset($f['label']) ? sanitize_text_field($f['label']) : '',
                    'key'      => isset($f['key']) ? sanitize_key($f['key']) : '',
                    'type'     => 'number', // FORCED TO NUMBER
                    'required' => !empty($f['required']) ? 1 : 0,
                    'default'  => isset($f['default']) ? sanitize_text_field($f['default']) : '',
                    'order'    => isset($f['order']) ? (int) $f['order'] : 0,
                    // Preserve numeric values for min/max/step
                    'min'  => ( isset( $f['min'] )  && $f['min']  !== '' && is_numeric( $f['min'] )  ) ? 0 + $f['min']  : '',
                    'max'  => ( isset( $f['max'] )  && $f['max']  !== '' && is_numeric( $f['max'] )  ) ? 0 + $f['max']  : '',
                    'step' => ( isset( $f['step'] ) && $f['step'] !== '' && is_numeric( $f['step'] ) ) ? 0 + $f['step'] : '',
                    'options'  => '', // No options for number field
                );
            }, $fields_raw);
            
            if (!empty($fields_raw)) {
                $field_id_mappings = $this->field_manager->bulk_save_fields($calculator_id, $fields_raw);
            }
        }

        ob_end_clean();

        $edit_url = admin_url('admin.php?page=pwcp-add-calculator&action=edit&calculator_id=' . $calculator_id);

        if ($is_auto_save) {
            $this->send_success_response(
                esc_html__('Auto-saved', 'pricewise-calculator-pro'),
                array(
                    'calculator_id' => $calculator_id,
                    'action' => $action,
                    'field_id_mappings' => $field_id_mappings,
                    'is_auto_save' => true
                )
            );
        }

        $this->send_success_response(
            $message,
            array(
                'calculator_id' => $calculator_id,
                'action' => $action,
                'field_id_mappings' => $field_id_mappings,
                'edit_url' => $edit_url,
                'is_new' => ($action === 'created'),
                'stay_on_page' => true,
            )
        );
    }

    /**
     * AJAX: Validate formula
     * 
     * Validates calculator formula syntax and returns any errors.
     *
     * @since 1.0.0
     */
    public function ajax_validate_formula() {
        $this->verify_ajax_security('manage_options', 'pwcp_calculator_action');

        $formula = $this->get_post_data('formula', 'textarea', '');

        if (empty($formula)) {
            $this->send_success_response(
                __('Formula is empty (optional)', 'pricewise-calculator-pro'),
                array('valid' => true)
            );
        }

        $errors = array();
        
        $open_parens = substr_count($formula, '(');
        $close_parens = substr_count($formula, ')');
        if ($open_parens !== $close_parens) {
            $errors[] = esc_html__('Unbalanced parentheses', 'pricewise-calculator-pro');
        }

        if (!preg_match('/^[a-zA-Z0-9_+\-*\/().\s]+$/', $formula)) {
            $errors[] = esc_html__('Invalid characters in formula', 'pricewise-calculator-pro');
        }

        if (preg_match('/[+\-*\/]{2,}/', $formula)) {
            $errors[] = esc_html__('Consecutive operators found', 'pricewise-calculator-pro');
        }

        preg_match_all('/[a-zA-Z_][a-zA-Z0-9_]*/', $formula, $matches);
        $variables = isset($matches[0]) ? $matches[0] : array();
        
        if (empty($variables) && !preg_match('/\d/', $formula)) {
            $errors[] = esc_html__('Formula must contain variables or numbers', 'pricewise-calculator-pro');
        }

        if (!empty($errors)) {
            $this->send_error_response(
                implode(', ', $errors),
                array(
                    'valid' => false,
                    'errors' => $errors
                )
            );
        }

        $this->send_success_response(
            __('Formula is valid', 'pricewise-calculator-pro'),
            array(
                'valid' => true,
                'variables' => array_unique($variables)
            )
        );
    }

    /**
     * AJAX: Auto-save calculator
     * 
     * Handles automatic saving of calculator data during editing.
     *
     * @since 1.0.0
     */
    public function ajax_auto_save_calculator() {
        $this->verify_ajax_security('manage_options', 'pwcp_calculator_action');

        $calculator_id = $this->get_post_data('calculator_id', 'int', 0);
        
        if (!$calculator_id) {
            $this->send_error_response(__('Calculator ID is required for auto-save.', 'pricewise-calculator-pro'));
        }

        $data = isset($_POST['data']) ? (array) wp_unslash($_POST['data']) : array();

        $auto_save_data = array();
        if (isset($data['name'])) {
            $auto_save_data['name'] = sanitize_text_field($data['name']);
        }
        if (isset($data['description'])) {
            $auto_save_data['description'] = sanitize_textarea_field($data['description']);
        }
        if (isset($data['formula'])) {
            $auto_save_data['formula'] = sanitize_textarea_field($data['formula']);
        }

        if (empty($auto_save_data)) {
            $this->send_error_response(__('No data to save.', 'pricewise-calculator-pro'));
        }

        $result = $this->manager->pwcp_save_calculator($auto_save_data, $calculator_id);

        if (false === $result) {
            $this->send_error_response(__('Auto-save failed.', 'pricewise-calculator-pro'));
        }

        $this->send_success_response(
            __('Auto-saved successfully.', 'pricewise-calculator-pro'),
            array(
                'timestamp' => current_time('timestamp'),
                'calculator_id' => $calculator_id
            )
        );
    }

    /**
     * Enqueue admin assets
     * 
     * Loads CSS and JavaScript files needed for the calculator form editor.
     * Includes localization for JavaScript strings and nonce values.
     *
     * @since 1.0.0
     */
    public function pwcp_enqueue_admin_assets($hook) {
        $screen = function_exists('get_current_screen') ? get_current_screen() : null;
        if (!$screen) {
            return;
        }

        $allowed_screens = array(
            'calculator-pro_page_pwcp-add-calculator',
            'calculator-pro_page_pwcp-edit-calculator',
        );

        if (!in_array($screen->id, $allowed_screens, true)) {
            return;
        }

        wp_enqueue_style(
            'pwcp-modern-calculator-form',
            PWCP_ASSETS_URL . 'css/all-admin-css/admin-calculator-form.css',
            array(),
            PWCP_PLUGIN_VERSION
        );

        wp_enqueue_script('jquery');
        wp_enqueue_script('jquery-ui-sortable');

        // Core script with i18n support
        wp_enqueue_script(
            'pwcp-calculator-form-core',
            PWCP_ASSETS_URL . 'js/all-admin-js/admin-calculator-form-core.js',
            array('jquery', 'wp-i18n'),
            PWCP_PLUGIN_VERSION,
            true
        );

        // Set up translations for core script
        if (function_exists('wp_set_script_translations')) {
            wp_set_script_translations(
                'pwcp-calculator-form-core',
                'pricewise-calculator-pro',
                plugin_dir_path(dirname(__FILE__)) . 'languages'
            );
        }

        wp_enqueue_script(
            'pwcp-calculator-field-formula',
            PWCP_ASSETS_URL . 'js/all-admin-js/admin-calculator-field-formula.js',
            array('jquery', 'jquery-ui-sortable', 'pwcp-calculator-form-core', 'wp-i18n'), 
            PWCP_PLUGIN_VERSION,
            true
        );

 
// Premium lock CSS - enqueue BEFORE JavaScript
wp_enqueue_style(
    'pwcp-premium-lock-styles',
    PWCP_ASSETS_URL . 'css/all-admin-css/admin-premium-lock.css',
    array('pwcp-modern-calculator-form'), // Depends on main admin CSS
    PWCP_PLUGIN_VERSION
);

// Premium lock JavaScript handler
wp_enqueue_script(
    'pwcp-premium-lock-handler',
    PWCP_ASSETS_URL . 'js/all-admin-js/admin-premium-lock.js',
    array('jquery', 'wp-i18n', 'pwcp-calculator-form-core'), // EXPLICIT dependencies
    PWCP_PLUGIN_VERSION,
    true // Load in footer
);

// Set up translations for premium lock script
if (function_exists('wp_set_script_translations')) {
    wp_set_script_translations(
        'pwcp-premium-lock-handler',
        'pricewise-calculator-pro', // Text domain
        plugin_dir_path(dirname(__FILE__)) . 'languages'
    );
}

        $calculator_id = 0;
        if (isset($_GET['calculator_id'])) {
            $calculator_id = absint($_GET['calculator_id']);
        } elseif (isset($_POST['calculator_id'])) {
            $calculator_id = absint($_POST['calculator_id']);
        }

        wp_localize_script(
            'pwcp-calculator-field-formula',
            'pwcpCalculatorForm',
            array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce'    => wp_create_nonce('pwcp_calculator_action'),
                'calculator_id' => $calculator_id,
                'is_edit_mode' => $calculator_id > 0,
                'strings'  => array(
                    'confirm_delete'    => esc_html__('Are you sure you want to delete this field?', 'pricewise-calculator-pro'),
                    'saving'            => esc_html__('Saving...', 'pricewise-calculator-pro'),
                    'saved'             => esc_html__('Saved!', 'pricewise-calculator-pro'),
                    'auto_saved'        => esc_html__('Auto-saved', 'pricewise-calculator-pro'),
                    'error'             => esc_html__('Error occurred', 'pricewise-calculator-pro'),
                    'formula_required'  => esc_html__('Formula is required', 'pricewise-calculator-pro'),
                    'name_required'     => esc_html__('Calculator name is required', 'pricewise-calculator-pro'),
                    'save_first'        => esc_html__('Please save the calculator first', 'pricewise-calculator-pro'),
                    'network_error'     => esc_html__('Network error. Please check your connection.', 'pricewise-calculator-pro'),
                    'timeout_error'     => esc_html__('Request timed out. Please try again.', 'pricewise-calculator-pro'),
                    'success'           => esc_html__('Operation completed successfully', 'pricewise-calculator-pro'),
                    'field_generated'   => esc_html__('Field generated automatically', 'pricewise-calculator-pro'),
                    'formula_valid'     => esc_html__('Formula is valid', 'pricewise-calculator-pro'),
                    'formula_invalid'   => esc_html__('Formula has errors', 'pricewise-calculator-pro'),
                ),
            )
        );
    }

    /**
     * Render add calculator page
     */
    public function pwcp_render_add_page() {
        if ($this->templates && method_exists($this->templates, 'pwcp_render_calculator_form')) {
            $this->templates->pwcp_render_calculator_form();
        }
    }

    /**
     * Render edit calculator page
     */
    public function pwcp_render_edit_page($calculator_id) {
        $calculator = $this->manager->pwcp_get_calculator($calculator_id);
        
        if (!$calculator) {
            wp_die(esc_html__('Calculator not found.', 'pricewise-calculator-pro'));
        }
        
        if ($this->templates && method_exists($this->templates, 'pwcp_render_calculator_form')) {
            $this->templates->pwcp_render_calculator_form($calculator);
        }
    }
}

// Initialize in admin context
if (is_admin() && class_exists('PWCP_Admin_Calculator_Form_Editor')) {
    global $pwcp_calculator_form_editor;
    if (!isset($pwcp_calculator_form_editor)) {
        $pwcp_calculator_form_editor = new PWCP_Admin_Calculator_Form_Editor();
    }
}