<?php

/**
 * Plugin Name: Price Schema VAT Fixer
 * Description: Fixes WooCommerce JSON-LD when prices are entered ex-VAT but displayed inc-VAT.
 * Version: 1.0
 * Author: ArrayGator
 * Author URI: https://arraygator.com
 * Requires Plugins: woocommerce
 * Requires at least: 6.5
 * Requires PHP: 7.4
 * License: GPLv2 or later
 * Text Domain: price-schema-vat-fixer
*/

namespace ArrayGator\PriceSchemaVatFix;

defined( 'ABSPATH' ) || exit;


function check_wc_on_activation() {
    if ( ! class_exists( '\\WooCommerce' ) ) {
        deactivate_plugins( plugin_basename( __FILE__ ) );
        wp_die(
            esc_html__( 'Price Schema VAT Fixer requires WooCommerce to be installed and active.', 'price-schema-vat-fixer' ),
            esc_html__( 'Plugin dependency check', 'price-schema-vat-fixer' ), [ 'back_link' => true ] );
    }
}
register_activation_hook( __FILE__, __NAMESPACE__ . '\\check_wc_on_activation' );


function runtime_wc_check() {
    if ( class_exists( '\\WooCommerce' ) ) {
        init_plugin();
        return;
    }
    add_action( 'admin_notices', function () {
        echo '<div class="notice notice-error"><p>';
        echo esc_html__(
            'Price Schema VAT Fixer is inactive because WooCommerce is not active.',
            'price-schema-vat-fixer'
        );
        echo '</p></div>';
    } );
}
add_action( 'plugins_loaded', __NAMESPACE__ . '\\runtime_wc_check', 11 );


function init_plugin() {
    add_action( 'admin_menu', __NAMESPACE__ . '\\register_menu_page' );
    add_action( 'admin_init', __NAMESPACE__ . '\\register_settings' );
    add_filter( 'woocommerce_structured_data_product', __NAMESPACE__ . '\\fix_structured_price', 100, 2 );
    add_action( 'admin_enqueue_scripts', __NAMESPACE__ . '\\enqueue_admin_assets' );
}


function register_menu_page () {
    add_menu_page(
        esc_html__( 'Schema VAT', 'price-schema-vat-fixer' ),
        esc_html__( 'Schema VAT', 'price-schema-vat-fixer' ),
        'manage_options',
        'psvf-settings',
        __NAMESPACE__ . '\\render_page',
        'dashicons-editor-code',
        60
    );
}


function register_settings () {
    register_setting(
        'psvf_settings_group',
        'psvf_use_custom_address',
        [
            'type'              => 'boolean',
            'sanitize_callback' => '\\rest_sanitize_boolean',
            'default'           => false,
        ]
    );
    register_setting(
        'psvf_settings_group',
        'psvf_default_address',
        [
            'type'              => 'array',
            'sanitize_callback' => __NAMESPACE__ . '\\sanitize_address',
            'default'           => [],
        ]
    );
    add_settings_section(
        'psvf_address_section',
        '',
        '\\__return_false',
        'psvf-settings'
    );
    add_settings_field(
        'psvf_use_custom_address',
        esc_html__( 'Override store base address', 'price-schema-vat-fixer' ),
        function () {
            $enabled = (bool) get_option( 'psvf_use_custom_address', false );
            printf(
                '<label><input type="checkbox" id="psvf_use_custom_address" name="psvf_use_custom_address" value="1" %s /> %s</label>',
                checked( $enabled, true, false ),
                esc_html__( 'Use custom address (set below) instead of the WooCommerce base location', 'price-schema-vat-fixer' )
            );
        },
        'psvf-settings',
        'psvf_address_section'
    );
    $fields = array(
        'city'     => esc_html__( 'City', 'price-schema-vat-fixer' ),
        'state'    => esc_html__( 'State / Province', 'price-schema-vat-fixer' ),
        'postcode' => esc_html__( 'Postal Code', 'price-schema-vat-fixer' ),
        'country'  => esc_html__( 'Country (2-letter ISO)', 'price-schema-vat-fixer' ),
    );
    foreach ( $fields as $key => $label ) {
        add_settings_field(
            "psvf_field_$key",
            $label,
            function () use ( $key ) {
                $opts = get_option( 'psvf_default_address', [] );
                $value = isset( $opts[ $key ] ) ? $opts[ $key ] : '';
                $attr = [];
                if ( ! get_option( 'psvf_use_custom_address', false ) ) {
                    $attr[] = 'readonly="readonly"';
                }
                if ( 'country' === $key ) {
                    $attr[] = 'maxlength="2"';
                    $attr[] = 'pattern="[A-Za-z]{2}"';
                    $attr[] = 'inputmode="latin"';
                    $attr[] = 'style="text-transform:uppercase"';
                }
                $attr_str = implode( ' ', $attr );
                printf(
                    '<input type="text" name="psvf_default_address[%1$s]" id="%1$s" value="%2$s" class="regular-text psvf-row" %3$s />',
                    esc_attr( $key ),
                    esc_attr( $value ),
                    wp_kses( $attr_str, [] )
                );
            },
            'psvf-settings',
            'psvf_address_section'
        );
    }
}


function sanitize_address( $input ) {
    $output = array();
    foreach ( (array) $input as $key => $value ) {
        $output[ $key ] = sanitize_text_field( $value );
    }
    if ( ! empty( $output['country'] ) ) {
        $output['country'] = strtoupper( $output['country'] );
    }
    return $output;
}


function get_structured_data_address() {
    if ( get_option( 'psvf_use_custom_address', false ) ) {
        $a = get_option( 'psvf_default_address', [] );
        return [
            'country'  => $a['country']  ?? '',
            'state'    => $a['state']    ?? '',
            'postcode' => $a['postcode'] ?? '',
            'city'     => $a['city']     ?? '',
        ];
    }
    return [
        'country'  => \WC()->countries->get_base_country(),
        'state'    => \WC()->countries->get_base_state(),
        'postcode' => \WC()->countries->get_base_postcode(),
        'city'     => \WC()->countries->get_base_city(),
    ];
}


function render_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die();
    }
    if ( wc_prices_include_tax() ) {
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Default Tax Address', 'price-schema-vat-fixer' ); ?></h1>
            <div class="notice notice-warning" style="margin-top:1em;">
                <p>
                    <strong><?php esc_html_e( 'Nothing to configure:', 'price-schema-vat-fixer' ); ?></strong>
                    <?php
                    /* translators: 1,3: <em>; 2,4: </em> */
                    $notice = __( 'WooCommerce is set to enter prices %1$sincluding%2$s tax. This plugin only helps when prices are entered %3$sexcluding%4$s tax, so its settings are disabled.', 'price-schema-vat-fixer' );
                    echo wp_kses( sprintf($notice, '<em>', '</em>', '<em>', '</em>'), [ 'em' => [] ] );
                    ?>
                </p>
            </div>
        </div>
        <?php
        return;
    }

    $address    = get_structured_data_address();
    $tax_rates  = \WC_Tax::find_rates( $address );
    $first_rate = reset( $tax_rates );
    $tax_rate   = $first_rate && isset( $first_rate['rate'] )
        ? wc_format_localized_decimal( $first_rate['rate'], 2 ) . '%'
        : 'N/A';
    $address_lines = array_filter( $address );

    ?>
    <div class="wrap">
        <h1><?php esc_html_e( 'Default Tax Address', 'price-schema-vat-fixer' ); ?></h1>
        <div class="psvf-summary notice notice-alt">
            <h2 class="psvf-heading">
                <span class="dashicons dashicons-location-alt" aria-hidden="true"></span>
                <?php esc_html_e( 'Structured-data tax location', 'price-schema-vat-fixer' ); ?>
            </h2>
            <div class="psvf-address-block">
                <p class="psvf-address-label">
                    <?php esc_html_e( 'Currently used address:', 'price-schema-vat-fixer' ); ?>
                </p>
                <address class="psvf-address">
                    <?php echo implode( '<br>', array_map( 'esc_html', $address_lines ) ); ?>
                </address>
            </div>
            <p class="psvf-vat">
                <?php esc_html_e( 'Effective VAT rate for this location:', 'price-schema-vat-fixer' ); ?>
                <strong class="psvf-vat-rate"><?php echo esc_html( $tax_rate ); ?></strong>
            </p>
        </div>
        <form action="options.php" method="post">
            <?php
            settings_fields( 'psvf_settings_group' );
            do_settings_sections( 'psvf-settings' );
            submit_button();
            ?>
        </form>
    </div>
    <?php
}


function fix_structured_price ( $markup, $product ) {
    if ( wc_prices_include_tax() || empty( $markup['offers'] ) || ! is_array( $markup['offers'] ) ) {
        return $markup;
    }
    $address = get_structured_data_address();
    $tax_class = [ 'tax_class' => $product->get_tax_class() ] + $address;
    $rates = \WC_Tax::find_rates( $tax_class );

    $add_vat = static function ( $net ) use ( $rates ) {
        return wc_round_tax_total( $net + array_sum(
            \WC_Tax::calc_tax( $net, $rates, false )
        ) );
    };

    $offers = &$markup['offers'];

    foreach ( $offers as &$offer ) {
        if ( ! is_array( $offer ) || ! isset( $offer['priceSpecification'] ) || ! is_array( $offer['priceSpecification'] ) ) {
            continue;
        }
        foreach ( $offer['priceSpecification'] as &$spec ) {
            if ( isset( $spec['valueAddedTaxIncluded'] ) && $spec['valueAddedTaxIncluded'] === false ) {
                $spec['price'] = $add_vat( (float) $spec['price'] );
                $spec['valueAddedTaxIncluded'] = true;
            }
        }
        unset( $offer['price'], $offer['priceCurrency'], $offer['valueAddedTaxIncluded'] );
    }
    return $markup;
}


function enqueue_admin_assets ( $hook ) {
    if ( $hook !== 'toplevel_page_psvf-settings' ) {
        return;
    }
    wp_enqueue_script('psvf-admin', plugin_dir_url( __FILE__ ) . 'assets/js/admin-ui.js', [], '1.0.0', true);
    wp_script_add_data( 'psvf-admin', 'type', 'module' );
    wp_enqueue_style('psvf-admin', plugin_dir_url( __FILE__ ) . 'assets/css/admin-ui.css', [], '1.0.0');
}


function uninstall() {
    $option_keys = array(
        'psvf_use_custom_address',
        'psvf_default_address',
    );
    foreach ( $option_keys as $key ) {
        delete_option( $key );
    }
}
register_uninstall_hook( __FILE__, __NAMESPACE__ . '\\uninstall' );
