<?php
/**
 * Integration functions for the prevent user name leakage feataure.
 *
 * @package calmpress\integration\prevent_username_leakage.
 * @version 1.0.0
 * @license GPLv2
 */

namespace calmpress\integration\prevent_username_leakage;

if ( ! defined( 'ABSPATH' ) ) {
	die( '403' );
}

const VERSION = '1.0.0';

/**
 * Filters the canonical redirect URL.
 *
 * Stops redirection of author=id type of urls to their canonical
 * forms on pretty permalink settings to avoid user name discloser.
 *
 * @since 1.0.0
 *
 * @param string $redirect_url  The redirect URL.
 * @param string $requested_url The requested URL.
 *
 * @return mixed false if the author page is being redirected to canonical form
 *               otherwise returns the value of $redirect_url.
 */
function redirect_canonical( string $redirect_url, string $requested_url ) {
	global $wp_rewrite;

	if ( is_object( $wp_rewrite ) && $wp_rewrite->using_permalinks() ) {
		if ( is_author() && ! empty( $_GET['author'] ) && preg_match( '|^[0-9]+$|', $_GET['author'] ) ) {
			return false;
		}
	}

	return $redirect_url;
}

/**
 * Filters user data returned from the REST API. Removes any field
 * that can be used for user name discloser when the request is not
 * done by an admin.
 *
 * @since 1.0.0
 *
 * @param \WP_REST_Response $response The response object.
 * @param \WP_User          $user     User object used to create response.
 * @param \WP_REST_Request  $request  Request object.
 *
 * @return \WP_REST_Response The response object without the author page link field
 *                          and without the user login name.
 */
function rest_prepare_user( \WP_REST_Response $response, \WP_User $user, \WP_REST_Request $request ) {

	// Remove data that is generated by code which can leak user information
	// when the requesting user can not get access to that data in other ways
	// like not having access to the profile page.
	if ( ! current_user_can( 'edit_users', $user->ID ) ) {
		$data = $response->data;
		$params = $request->get_params();

		// The link to the posts page might leak the user name when pretty permalinks
		// are on, but lets do the code smaller and not add checks for something
		// that is unlikely to be used in non pretty permalinks setting.
		if ( isset( $data['link'] ) ) {
			unset( $data['link'] );
		}

		// slug is usually the login name with no UI to change it.
		if ( isset( $data['slug'] ) ) {
			unset( $data['slug'] );
		}

		// The display name by default is the same as login name, so lets remove
		// it if it was not changed as security takes precedence.
		if ( isset( $data['name'] ) && ( $data['name'] === $user->user_login ) ) {
			unset( $data['name'] );
		}

		$response->data = $data;
	}

	return $response;
}

/**
 * Filter authentication result and remove user or password specific
 * authentication errors in favor of a generic one.
 *
 * @since 1.0.0
 *
 * @param null|WP_User|WP_Error $user     WP_User if the user is authenticated.
 *                                        WP_Error or null otherwise.
 * @param string                $username Username or email address.
 * @param string                $password User password.
 *
 * @return null|WP_User|WP_Error returns a user if $user is a user, null When
 *                               $user is a user or password specific error (to trigger
 *                               a more generic error), otherwise the value of $user.
 */
function authenticate( $user, string $username, string $password ) {

	if ( is_wp_error( $user ) ) {
		if (
			( 'invalid_username' === $user->get_error_code() ) ||
			( 'incorrect_password' === $user->get_error_code() )
			) {
			return null;
		}
	}
	return $user;
}

/**
 * Action triggered on the lost password form. Ensure redirection to the
 * "check your mail" page even if the email/user name are wrong to prevent
 * the possibility to finding out which ones are legit on the site.
 *
 * @sinc 1.0.0
 */
function lost_password() {
	$http_post = ( 'POST' === $_SERVER['REQUEST_METHOD'] );

	// If this is not a post request we are more likely after password reset.
	// If the user/email info was correct a redirect would have already happened
	// so just force a redirect in the exact same way.
	if ( $http_post ) {
		$redirect_to = ! empty( $_REQUEST['redirect_to'] ) ? $_REQUEST['redirect_to'] : 'wp-login.php?checkemail=confirm';
		wp_safe_redirect( $redirect_to );
		exit();
	}
}
