/*!
 * ScrollMagic v2.0.7 (2019-05-07)
 * The javascript library for magical scroll interactions.
 * (c) 2019 Jan Paepke (@janpaepke)
 * Project Website: http://scrollmagic.io
 *
 * @version 2.0.7
 * @license Dual licensed under MIT license and GPL.
 * @author Jan Paepke - e-mail@janpaepke.de
 *
 * @file ScrollMagic main library.
 */
/**
 * @namespace ScrollMagic
 */
(function (root, factory) {
    if (typeof define === 'function' && define.amd) {
        // AMD. Register as an anonymous module.
        define(factory);
    } else if (typeof exports === 'object') {
        // CommonJS
        module.exports = factory();
    } else {
        // Browser global
        root.ScrollMagic = factory();
    }
}(this, function () {
    "use strict";

    var ScrollMagic = function () {
        _util.log(2, '(COMPATIBILITY NOTICE) -> As of ScrollMagic 2.0.0 you need to use \'new ScrollMagic.Controller()\' to create a new controller instance. Use \'new ScrollMagic.Scene()\' to instance a scene.');
    };

    ScrollMagic.version = "2.0.7";

    // TODO: temporary workaround for chrome's scroll jitter bug
    window.addEventListener("mousewheel", function () { });

    // global const
    var PIN_SPACER_ATTRIBUTE = "data-scrollmagic-pin-spacer";

    /**
     * The main class that is needed once per scroll container.
     *
     * @class
     *
     * @example
     * // basic initialization
     * var controller = new ScrollMagic.Controller();
     *
     * // passing options
     * var controller = new ScrollMagic.Controller({container: "#myContainer", loglevel: 3});
     *
     * @param {object} [options] - An object containing one or more options for the controller.
     * @param {(string|object)} [options.container=window] - A selector, DOM object that references the main container for scrolling.
     * @param {boolean} [options.vertical=true] - Sets the scroll mode to vertical (`true`) or horizontal (`false`) scrolling.
     * @param {object} [options.globalSceneOptions={}] - These options will be passed to every Scene that is added to the controller using the addScene method. For more information on Scene options see {@link ScrollMagic.Scene}.
     * @param {number} [options.loglevel=2] Loglevel for debugging. Note that logging is disabled in the minified version of ScrollMagic.
                                             ** `0` => silent
                                             ** `1` => errors
                                             ** `2` => errors, warnings
                                             ** `3` => errors, warnings, debuginfo
     * @param {boolean} [options.refreshInterval=100] - Some changes don't call events by default, like changing the container size or moving a scene trigger element.
                                                                                                                 This interval polls these parameters to fire the necessary events.
                                                                                                                 If you don't use custom containers, trigger elements or have static layouts, where the positions of the trigger elements don't change, you can set this to 0 disable interval checking and improve performance.
     *
     */
    ScrollMagic.Controller = function (options) {
        /*
         * ----------------------------------------------------------------
         * settings
         * ----------------------------------------------------------------
         */
        var
            NAMESPACE = 'ScrollMagic.Controller',
            SCROLL_DIRECTION_FORWARD = 'FORWARD',
            SCROLL_DIRECTION_REVERSE = 'REVERSE',
            SCROLL_DIRECTION_PAUSED = 'PAUSED',
            DEFAULT_OPTIONS = CONTROLLER_OPTIONS.defaults;

        /*
         * ----------------------------------------------------------------
         * private vars
         * ----------------------------------------------------------------
         */
        var
            Controller = this,
            _options = _util.extend({}, DEFAULT_OPTIONS, options),
            _sceneObjects = [],
            _updateScenesOnNextCycle = false, // can be boolean (true => all scenes) or an array of scenes to be updated
            _scrollPos = 0,
            _scrollDirection = SCROLL_DIRECTION_PAUSED,
            _isDocument = true,
            _viewPortSize = 0,
            _enabled = true,
            _updateTimeout,
            _refreshTimeout;

        /*
         * ----------------------------------------------------------------
         * private functions
         * ----------------------------------------------------------------
         */

        /**
         * Internal constructor function of the ScrollMagic Controller
         * @private
         */
        var construct = function () {
            for (var key in _options) {
                if (!DEFAULT_OPTIONS.hasOwnProperty(key)) {
                    log(2, "WARNING: Unknown option \"" + key + "\"");
                    delete _options[key];
                }
            }
            _options.container = _util.get.elements(_options.container)[0];
            // check ScrollContainer
            if (!_options.container) {
                log(1, "ERROR creating object " + NAMESPACE + ": No valid scroll container supplied");
                throw NAMESPACE + " init failed."; // cancel
            }
            _isDocument = _options.container === window || _options.container === document.body || !document.body.contains(_options.container);
            // normalize to window
            if (_isDocument) {
                _options.container = window;
            }
            // update container size immediately
            _viewPortSize = getViewportSize();
            // set event handlers
            _options.container.addEventListener("resize", onChange);
            _options.container.addEventListener("scroll", onChange);

            var ri = parseInt(_options.refreshInterval, 10);
            _options.refreshInterval = _util.type.Number(ri) ? ri : DEFAULT_OPTIONS.refreshInterval;
            scheduleRefresh();

            log(3, "added new " + NAMESPACE + " controller (v" + ScrollMagic.version + ")");
        };

        /**
         * Schedule the next execution of the refresh function
         * @private
         */
        var scheduleRefresh = function () {
            if (_options.refreshInterval > 0) {
                _refreshTimeout = window.setTimeout(refresh, _options.refreshInterval);
            }
        };

        /**
         * Default function to get scroll pos - overwriteable using `Controller.scrollPos(newFunction)`
         * @private
         */
        var getScrollPos = function () {
            return _options.vertical ? _util.get.scrollTop(_options.container) : _util.get.scrollLeft(_options.container);
        };

        /**
         * Returns the current viewport Size (width vor horizontal, height for vertical)
         * @private
         */
        var getViewportSize = function () {
            return _options.vertical ? _util.get.height(_options.container) : _util.get.width(_options.container);
        };

        /**
         * Default function to set scroll pos - overwriteable using `Controller.scrollTo(newFunction)`
         * Make available publicly for pinned mousewheel workaround.
         * @private
         */
        var setScrollPos = this._setScrollPos = function (pos) {
            if (_options.vertical) {
                if (_isDocument) {
                    window.scrollTo(_util.get.scrollLeft(), pos);
                } else {
                    _options.container.scrollTop = pos;
                }
            } else {
                if (_isDocument) {
                    window.scrollTo(pos, _util.get.scrollTop());
                } else {
                    _options.container.scrollLeft = pos;
                }
            }
        };

        /**
         * Handle updates in cycles instead of on scroll (performance)
         * @private
         */
        var updateScenes = function () {
            if (_enabled && _updateScenesOnNextCycle) {
                // determine scenes to update
                var scenesToUpdate = _util.type.Array(_updateScenesOnNextCycle) ? _updateScenesOnNextCycle : _sceneObjects.slice(0);
                // reset scenes
                _updateScenesOnNextCycle = false;
                var oldScrollPos = _scrollPos;
                // update scroll pos now instead of onChange, as it might have changed since scheduling (i.e. in-browser smooth scroll)
                _scrollPos = Controller.scrollPos();
                var deltaScroll = _scrollPos - oldScrollPos;
                if (deltaScroll !== 0) { // scroll position changed?
                    _scrollDirection = (deltaScroll > 0) ? SCROLL_DIRECTION_FORWARD : SCROLL_DIRECTION_REVERSE;
                }
                // reverse order of scenes if scrolling reverse
                if (_scrollDirection === SCROLL_DIRECTION_REVERSE) {
                    scenesToUpdate.reverse();
                }
                // update scenes
                scenesToUpdate.forEach(function (scene, index) {
                    log(3, "updating Scene " + (index + 1) + "/" + scenesToUpdate.length + " (" + _sceneObjects.length + " total)");
                    scene.update(true);
                });
                if (scenesToUpdate.length === 0 && _options.loglevel >= 3) {
                    log(3, "updating 0 Scenes (nothing added to controller)");
                }
            }
        };

        /**
         * Initializes rAF callback
         * @private
         */
        var debounceUpdate = function () {
            _updateTimeout = _util.rAF(updateScenes);
        };

        /**
         * Handles Container changes
         * @private
         */
        var onChange = function (e) {
            log(3, "event fired causing an update:", e.type);
            if (e.type == "resize") {
                // resize
                _viewPortSize = getViewportSize();
                _scrollDirection = SCROLL_DIRECTION_PAUSED;
            }
            // schedule update
            if (_updateScenesOnNextCycle !== true) {
                _updateScenesOnNextCycle = true;
                debounceUpdate();
            }
        };

        var refresh = function () {
            if (!_isDocument) {
                // simulate resize event. Only works for viewport relevant param (performance)
                if (_viewPortSize != getViewportSize()) {
                    var resizeEvent;
                    try {
                        resizeEvent = new Event('resize', {
                            bubbles: false,
                            cancelable: false
                        });
                    } catch (e) { // stupid IE
                        resizeEvent = document.createEvent("Event");
                        resizeEvent.initEvent("resize", false, false);
                    }
                    _options.container.dispatchEvent(resizeEvent);
                }
            }
            _sceneObjects.forEach(function (scene, index) { // refresh all scenes
                scene.refresh();
            });
            scheduleRefresh();
        };

        /**
         * Send a debug message to the console.
         * provided publicly with _log for plugins
         * @private
         *
         * @param {number} loglevel - The loglevel required to initiate output for the message.
         * @param {...mixed} output - One or more variables that should be passed to the console.
         */
        var log = this._log = function (loglevel, output) {
            if (_options.loglevel >= loglevel) {
                Array.prototype.splice.call(arguments, 1, 0, "(" + NAMESPACE + ") ->");
                _util.log.apply(window, arguments);
            }
        };
        // for scenes we have getters for each option, but for the controller we don't, so we need to make it available externally for plugins
        this._options = _options;

        /**
         * Sort scenes in ascending order of their start offset.
         * @private
         *
         * @param {array} ScenesArray - an array of ScrollMagic Scenes that should be sorted
         * @return {array} The sorted array of Scenes.
         */
        var sortScenes = function (ScenesArray) {
            if (ScenesArray.length <= 1) {
                return ScenesArray;
            } else {
                var scenes = ScenesArray.slice(0);
                scenes.sort(function (a, b) {
                    return a.scrollOffset() > b.scrollOffset() ? 1 : -1;
                });
                return scenes;
            }
        };

        /**
         * ----------------------------------------------------------------
         * public functions
         * ----------------------------------------------------------------
         */

        /**
         * Add one ore more scene(s) to the controller.
         * This is the equivalent to `Scene.addTo(controller)`.
         * @public
         * @example
         * // with a previously defined scene
         * controller.addScene(scene);
         *
         * // with a newly created scene.
         * controller.addScene(new ScrollMagic.Scene({duration : 0}));
         *
         * // adding multiple scenes
         * controller.addScene([scene, scene2, new ScrollMagic.Scene({duration : 0})]);
         *
         * @param {(ScrollMagic.Scene|array)} newScene - ScrollMagic Scene or Array of Scenes to be added to the controller.
         * @return {Controller} Parent object for chaining.
         */
        this.addScene = function (newScene) {
            if (_util.type.Array(newScene)) {
                newScene.forEach(function (scene, index) {
                    Controller.addScene(scene);
                });
            } else if (newScene instanceof ScrollMagic.Scene) {
                if (newScene.controller() !== Controller) {
                    newScene.addTo(Controller);
                } else if (_sceneObjects.indexOf(newScene) < 0) {
                    // new scene
                    _sceneObjects.push(newScene); // add to array
                    _sceneObjects = sortScenes(_sceneObjects); // sort
                    newScene.on("shift.controller_sort", function () { // resort whenever scene moves
                        _sceneObjects = sortScenes(_sceneObjects);
                    });
                    // insert Global defaults.
                    for (var key in _options.globalSceneOptions) {
                        if (newScene[key]) {
                            newScene[key].call(newScene, _options.globalSceneOptions[key]);
                        }
                    }
                    log(3, "adding Scene (now " + _sceneObjects.length + " total)");
                }
            } else {
                log(1, "ERROR: invalid argument supplied for '.addScene()'");
            }
            return Controller;
        };

        /**
         * Remove one ore more scene(s) from the controller.
         * This is the equivalent to `Scene.remove()`.
         * @public
         * @example
         * // remove a scene from the controller
         * controller.removeScene(scene);
         *
         * // remove multiple scenes from the controller
         * controller.removeScene([scene, scene2, scene3]);
         *
         * @param {(ScrollMagic.Scene|array)} Scene - ScrollMagic Scene or Array of Scenes to be removed from the controller.
         * @returns {Controller} Parent object for chaining.
         */
        this.removeScene = function (Scene) {
            if (_util.type.Array(Scene)) {
                Scene.forEach(function (scene, index) {
                    Controller.removeScene(scene);
                });
            } else {
                var index = _sceneObjects.indexOf(Scene);
                if (index > -1) {
                    Scene.off("shift.controller_sort");
                    _sceneObjects.splice(index, 1);
                    log(3, "removing Scene (now " + _sceneObjects.length + " left)");
                    Scene.remove();
                }
            }
            return Controller;
        };

        /**
     * Update one ore more scene(s) according to the scroll position of the container.
     * This is the equivalent to `Scene.update()`.
     * The update method calculates the scene's start and end position (based on the trigger element, trigger hook, duration and offset) and checks it against the current scroll position of the container.
     * It then updates the current scene state accordingly (or does nothing, if the state is already correct) â€“ Pins will be set to their correct position and tweens will be updated to their correct progress.
     * _**Note:** This method gets called constantly whenever Controller detects a change. The only application for you is if you change something outside of the realm of ScrollMagic, like moving the trigger or changing tween parameters._
     * @public
     * @example
     * // update a specific scene on next cycle
       * controller.updateScene(scene);
       *
     * // update a specific scene immediately
     * controller.updateScene(scene, true);
       *
     * // update multiple scenes scene on next cycle
     * controller.updateScene([scene1, scene2, scene3]);
     *
     * @param {ScrollMagic.Scene} Scene - ScrollMagic Scene or Array of Scenes that is/are supposed to be updated.
     * @param {boolean} [immediately=false] - If `true` the update will be instant, if `false` it will wait until next update cycle.
                                                  This is useful when changing multiple properties of the scene - this way it will only be updated once all new properties are set (updateScenes).
     * @return {Controller} Parent object for chaining.
     */
        this.updateScene = function (Scene, immediately) {
            if (_util.type.Array(Scene)) {
                Scene.forEach(function (scene, index) {
                    Controller.updateScene(scene, immediately);
                });
            } else {
                if (immediately) {
                    Scene.update(true);
                } else if (_updateScenesOnNextCycle !== true && Scene instanceof ScrollMagic.Scene) { // if _updateScenesOnNextCycle is true, all connected scenes are already scheduled for update
                    // prep array for next update cycle
                    _updateScenesOnNextCycle = _updateScenesOnNextCycle || [];
                    if (_updateScenesOnNextCycle.indexOf(Scene) == -1) {
                        _updateScenesOnNextCycle.push(Scene);
                    }
                    _updateScenesOnNextCycle = sortScenes(_updateScenesOnNextCycle); // sort
                    debounceUpdate();
                }
            }
            return Controller;
        };

        /**
         * Updates the controller params and calls updateScene on every scene, that is attached to the controller.
         * See `Controller.updateScene()` for more information about what this means.
         * In most cases you will not need this function, as it is called constantly, whenever ScrollMagic detects a state change event, like resize or scroll.
         * The only application for this method is when ScrollMagic fails to detect these events.
         * One application is with some external scroll libraries (like iScroll) that move an internal container to a negative offset instead of actually scrolling. In this case the update on the controller needs to be called whenever the child container's position changes.
         * For this case there will also be the need to provide a custom function to calculate the correct scroll position. See `Controller.scrollPos()` for details.
         * @public
         * @example
         * // update the controller on next cycle (saves performance due to elimination of redundant updates)
         * controller.update();
         *
         * // update the controller immediately
         * controller.update(true);
         *
         * @param {boolean} [immediately=false] - If `true` the update will be instant, if `false` it will wait until next update cycle (better performance)
         * @return {Controller} Parent object for chaining.
         */
        this.update = function (immediately) {
            onChange({
                type: "resize"
            }); // will update size and set _updateScenesOnNextCycle to true
            if (immediately) {
                updateScenes();
            }
            return Controller;
        };

        /**
         * Scroll to a numeric scroll offset, a DOM element, the start of a scene or provide an alternate method for scrolling.
         * For vertical controllers it will change the top scroll offset and for horizontal applications it will change the left offset.
         * @public
         *
         * @since 1.1.0
         * @example
         * // scroll to an offset of 100
         * controller.scrollTo(100);
         *
         * // scroll to a DOM element
         * controller.scrollTo("#anchor");
         *
         * // scroll to the beginning of a scene
         * var scene = new ScrollMagic.Scene({offset: 200});
         * controller.scrollTo(scene);
         *
         * // define a new scroll position modification function (jQuery animate instead of jump)
         * controller.scrollTo(function (newScrollPos) {
         *	$("html, body").animate({scrollTop: newScrollPos});
         * });
         * controller.scrollTo(100); // call as usual, but the new function will be used instead
         *
         * // define a new scroll function with an additional parameter
         * controller.scrollTo(function (newScrollPos, message) {
         *  console.log(message);
         *	$(this).animate({scrollTop: newScrollPos});
         * });
         * // call as usual, but supply an extra parameter to the defined custom function
         * controller.scrollTo(100, "my message");
         *
         * // define a new scroll function with an additional parameter containing multiple variables
         * controller.scrollTo(function (newScrollPos, options) {
         *  someGlobalVar = options.a + options.b;
         *	$(this).animate({scrollTop: newScrollPos});
         * });
         * // call as usual, but supply an extra parameter containing multiple options
         * controller.scrollTo(100, {a: 1, b: 2});
         *
         * // define a new scroll function with a callback supplied as an additional parameter
         * controller.scrollTo(function (newScrollPos, callback) {
         *	$(this).animate({scrollTop: newScrollPos}, 400, "swing", callback);
         * });
         * // call as usual, but supply an extra parameter, which is used as a callback in the previously defined custom scroll function
         * controller.scrollTo(100, function() {
         *	console.log("scroll has finished.");
         * });
         *
         * @param {mixed} scrollTarget - The supplied argument can be one of these types:
         * 1. `number` -> The container will scroll to this new scroll offset.
         * 2. `string` or `object` -> Can be a selector or a DOM object.
         *  The container will scroll to the position of this element.
         * 3. `ScrollMagic Scene` -> The container will scroll to the start of this scene.
         * 4. `function` -> This function will be used for future scroll position modifications.
         *  This provides a way for you to change the behaviour of scrolling and adding new behaviour like animation. The function receives the new scroll position as a parameter and a reference to the container element using `this`.
         *  It may also optionally receive an optional additional parameter (see below)
         *  _**NOTE:**
         *  All other options will still work as expected, using the new function to scroll._
         * @param {mixed} [additionalParameter] - If a custom scroll function was defined (see above 4.), you may want to supply additional parameters to it, when calling it. You can do this using this parameter â€“ see examples for details. Please note, that this parameter will have no effect, if you use the default scrolling function.
         * @returns {Controller} Parent object for chaining.
         */
        this.scrollTo = function (scrollTarget, additionalParameter) {
            if (_util.type.Number(scrollTarget)) { // execute
                setScrollPos.call(_options.container, scrollTarget, additionalParameter);
            } else if (scrollTarget instanceof ScrollMagic.Scene) { // scroll to scene
                if (scrollTarget.controller() === Controller) { // check if the controller is associated with this scene
                    Controller.scrollTo(scrollTarget.scrollOffset(), additionalParameter);
                } else {
                    log(2, "scrollTo(): The supplied scene does not belong to this controller. Scroll cancelled.", scrollTarget);
                }
            } else if (_util.type.Function(scrollTarget)) { // assign new scroll function
                setScrollPos = scrollTarget;
            } else { // scroll to element
                var elem = _util.get.elements(scrollTarget)[0];
                if (elem) {
                    // if parent is pin spacer, use spacer position instead so correct start position is returned for pinned elements.
                    while (elem.parentNode.hasAttribute(PIN_SPACER_ATTRIBUTE)) {
                        elem = elem.parentNode;
                    }

                    var
                        param = _options.vertical ? "top" : "left", // which param is of interest ?
                        containerOffset = _util.get.offset(_options.container), // container position is needed because element offset is returned in relation to document, not in relation to container.
                        elementOffset = _util.get.offset(elem);

                    if (!_isDocument) { // container is not the document root, so substract scroll Position to get correct trigger element position relative to scrollcontent
                        containerOffset[param] -= Controller.scrollPos();
                    }

                    Controller.scrollTo(elementOffset[param] - containerOffset[param], additionalParameter);
                } else {
                    log(2, "scrollTo(): The supplied argument is invalid. Scroll cancelled.", scrollTarget);
                }
            }
            return Controller;
        };

        /**
         * **Get** the current scrollPosition or **Set** a new method to calculate it.
         * -> **GET**:
         * When used as a getter this function will return the current scroll position.
         * To get a cached value use Controller.info("scrollPos"), which will be updated in the update cycle.
         * For vertical controllers it will return the top scroll offset and for horizontal applications it will return the left offset.
         *
         * -> **SET**:
         * When used as a setter this method prodes a way to permanently overwrite the controller's scroll position calculation.
         * A typical usecase is when the scroll position is not reflected by the containers scrollTop or scrollLeft values, but for example by the inner offset of a child container.
         * Moving a child container inside a parent is a commonly used method for several scrolling frameworks, including iScroll.
         * By providing an alternate calculation function you can make sure ScrollMagic receives the correct scroll position.
         * Please also bear in mind that your function should return y values for vertical scrolls an x for horizontals.
         *
         * To change the current scroll position please use `Controller.scrollTo()`.
         * @public
         *
         * @example
         * // get the current scroll Position
         * var scrollPos = controller.scrollPos();
         *
         * // set a new scroll position calculation method
         * controller.scrollPos(function () {
         *	return this.info("vertical") ? -mychildcontainer.y : -mychildcontainer.x
         * });
         *
         * @param {function} [scrollPosMethod] - The function to be used for the scroll position calculation of the container.
         * @returns {(number|Controller)} Current scroll position or parent object for chaining.
         */
        this.scrollPos = function (scrollPosMethod) {
            if (!arguments.length) { // get
                return getScrollPos.call(Controller);
            } else { // set
                if (_util.type.Function(scrollPosMethod)) {
                    getScrollPos = scrollPosMethod;
                } else {
                    log(2, "Provided value for method 'scrollPos' is not a function. To change the current scroll position use 'scrollTo()'.");
                }
            }
            return Controller;
        };

        /**
         * **Get** all infos or one in particular about the controller.
         * @public
         * @example
         * // returns the current scroll position (number)
         * var scrollPos = controller.info("scrollPos");
         *
         * // returns all infos as an object
         * var infos = controller.info();
         *
         * @param {string} [about] - If passed only this info will be returned instead of an object containing all.
                                         Valid options are:
                                         ** `"size"` => the current viewport size of the container
                                         ** `"vertical"` => true if vertical scrolling, otherwise false
                                         ** `"scrollPos"` => the current scroll position
                                         ** `"scrollDirection"` => the last known direction of the scroll
                                         ** `"container"` => the container element
                                         ** `"isDocument"` => true if container element is the document.
         * @returns {(mixed|object)} The requested info(s).
         */
        this.info = function (about) {
            var values = {
                size: _viewPortSize, // contains height or width (in regard to orientation);
                vertical: _options.vertical,
                scrollPos: _scrollPos,
                scrollDirection: _scrollDirection,
                container: _options.container,
                isDocument: _isDocument
            };
            if (!arguments.length) { // get all as an object
                return values;
            } else if (values[about] !== undefined) {
                return values[about];
            } else {
                log(1, "ERROR: option \"" + about + "\" is not available");
                return;
            }
        };

        /**
         * **Get** or **Set** the current loglevel option value.
         * @public
         *
         * @example
         * // get the current value
         * var loglevel = controller.loglevel();
         *
         * // set a new value
         * controller.loglevel(3);
         *
         * @param {number} [newLoglevel] - The new loglevel setting of the Controller. `[0-3]`
         * @returns {(number|Controller)} Current loglevel or parent object for chaining.
         */
        this.loglevel = function (newLoglevel) {
            if (!arguments.length) { // get
                return _options.loglevel;
            } else if (_options.loglevel != newLoglevel) { // set
                _options.loglevel = newLoglevel;
            }
            return Controller;
        };

        /**
         * **Get** or **Set** the current enabled state of the controller.
         * This can be used to disable all Scenes connected to the controller without destroying or removing them.
         * @public
         *
         * @example
         * // get the current value
         * var enabled = controller.enabled();
         *
         * // disable the controller
         * controller.enabled(false);
         *
         * @param {boolean} [newState] - The new enabled state of the controller `true` or `false`.
         * @returns {(boolean|Controller)} Current enabled state or parent object for chaining.
         */
        this.enabled = function (newState) {
            if (!arguments.length) { // get
                return _enabled;
            } else if (_enabled != newState) { // set
                _enabled = !!newState;
                Controller.updateScene(_sceneObjects, true);
            }
            return Controller;
        };

        /**
         * Destroy the Controller, all Scenes and everything.
         * @public
         *
         * @example
         * // without resetting the scenes
         * controller = controller.destroy();
         *
         * // with scene reset
         * controller = controller.destroy(true);
         *
         * @param {boolean} [resetScenes=false] - If `true` the pins and tweens (if existent) of all scenes will be reset.
         * @returns {null} Null to unset handler variables.
         */
        this.destroy = function (resetScenes) {
            window.clearTimeout(_refreshTimeout);
            var i = _sceneObjects.length;
            while (i--) {
                _sceneObjects[i].destroy(resetScenes);
            }
            _options.container.removeEventListener("resize", onChange);
            _options.container.removeEventListener("scroll", onChange);
            _util.cAF(_updateTimeout);
            log(3, "destroyed " + NAMESPACE + " (reset: " + (resetScenes ? "true" : "false") + ")");
            return null;
        };

        // INIT
        construct();
        return Controller;
    };

    // store pagewide controller options
    var CONTROLLER_OPTIONS = {
        defaults: {
            container: window,
            vertical: true,
            globalSceneOptions: {},
            loglevel: 2,
            refreshInterval: 100
        }
    };
    /*
     * method used to add an option to ScrollMagic Scenes.
     */
    ScrollMagic.Controller.addOption = function (name, defaultValue) {
        CONTROLLER_OPTIONS.defaults[name] = defaultValue;
    };
    // instance extension function for plugins
    ScrollMagic.Controller.extend = function (extension) {
        var oldClass = this;
        ScrollMagic.Controller = function () {
            oldClass.apply(this, arguments);
            this.$super = _util.extend({}, this); // copy parent state
            return extension.apply(this, arguments) || this;
        };
        _util.extend(ScrollMagic.Controller, oldClass); // copy properties
        ScrollMagic.Controller.prototype = oldClass.prototype; // copy prototype
        ScrollMagic.Controller.prototype.constructor = ScrollMagic.Controller; // restore constructor
    };


    /**
     * A Scene defines where the controller should react and how.
     *
     * @class
     *
     * @example
     * // create a standard scene and add it to a controller
     * new ScrollMagic.Scene()
     *		.addTo(controller);
     *
     * // create a scene with custom options and assign a handler to it.
     * var scene = new ScrollMagic.Scene({
     * 		duration: 100,
     *		offset: 200,
     *		triggerHook: "onEnter",
     *		reverse: false
     * });
     *
     * @param {object} [options] - Options for the Scene. The options can be updated at any time.
                                       Instead of setting the options for each scene individually you can also set them globally in the controller as the controllers `globalSceneOptions` option. The object accepts the same properties as the ones below.
                                       When a scene is added to the controller the options defined using the Scene constructor will be overwritten by those set in `globalSceneOptions`.
     * @param {(number|string|function)} [options.duration=0] - The duration of the scene.
                            Please see `Scene.duration()` for details.
     * @param {number} [options.offset=0] - Offset Value for the Trigger Position. If no triggerElement is defined this will be the scroll distance from the start of the page, after which the scene will start.
     * @param {(string|object)} [options.triggerElement=null] - Selector or DOM object that defines the start of the scene. If undefined the scene will start right at the start of the page (unless an offset is set).
     * @param {(number|string)} [options.triggerHook="onCenter"] - Can be a number between 0 and 1 defining the position of the trigger Hook in relation to the viewport.
                                                                      Can also be defined using a string:
                                                                      ** `"onEnter"` => `1`
                                                                      ** `"onCenter"` => `0.5`
                                                                      ** `"onLeave"` => `0`
     * @param {boolean} [options.reverse=true] - Should the scene reverse, when scrolling up?
     * @param {number} [options.loglevel=2] - Loglevel for debugging. Note that logging is disabled in the minified version of ScrollMagic.
                                                  ** `0` => silent
                                                  ** `1` => errors
                                                  ** `2` => errors, warnings
                                                  ** `3` => errors, warnings, debuginfo
     *
     */
    ScrollMagic.Scene = function (options) {

        /*
         * ----------------------------------------------------------------
         * settings
         * ----------------------------------------------------------------
         */

        var
            NAMESPACE = 'ScrollMagic.Scene',
            SCENE_STATE_BEFORE = 'BEFORE',
            SCENE_STATE_DURING = 'DURING',
            SCENE_STATE_AFTER = 'AFTER',
            DEFAULT_OPTIONS = SCENE_OPTIONS.defaults;

        /*
         * ----------------------------------------------------------------
         * private vars
         * ----------------------------------------------------------------
         */

        var
            Scene = this,
            _options = _util.extend({}, DEFAULT_OPTIONS, options),
            _state = SCENE_STATE_BEFORE,
            _progress = 0,
            _scrollOffset = {
                start: 0,
                end: 0
            }, // reflects the controllers's scroll position for the start and end of the scene respectively
            _triggerPos = 0,
            _enabled = true,
            _durationUpdateMethod,
            _controller;

        /**
         * Internal constructor function of the ScrollMagic Scene
         * @private
         */
        var construct = function () {
            for (var key in _options) { // check supplied options
                if (!DEFAULT_OPTIONS.hasOwnProperty(key)) {
                    log(2, "WARNING: Unknown option \"" + key + "\"");
                    delete _options[key];
                }
            }
            // add getters/setters for all possible options
            for (var optionName in DEFAULT_OPTIONS) {
                addSceneOption(optionName);
            }
            // validate all options
            validateOption();
        };

        /*
         * ----------------------------------------------------------------
         * Event Management
         * ----------------------------------------------------------------
         */

        var _listeners = {};
        /**
         * Scene start event.
         * Fires whenever the scroll position its the starting point of the scene.
         * It will also fire when scrolling back up going over the start position of the scene. If you want something to happen only when scrolling down/right, use the scrollDirection parameter passed to the callback.
         *
         * For details on this event and the order in which it is fired, please review the {@link Scene.progress} method.
         *
         * @event ScrollMagic.Scene#start
         *
         * @example
         * scene.on("start", function (event) {
         * 	console.log("Hit start point of scene.");
         * });
         *
         * @property {object} event - The event Object passed to each callback
         * @property {string} event.type - The name of the event
         * @property {Scene} event.target - The Scene object that triggered this event
         * @property {number} event.progress - Reflects the current progress of the scene
         * @property {string} event.state - The current state of the scene `"BEFORE"` or `"DURING"`
         * @property {string} event.scrollDirection - Indicates which way we are scrolling `"PAUSED"`, `"FORWARD"` or `"REVERSE"`
         */
        /**
         * Scene end event.
         * Fires whenever the scroll position its the ending point of the scene.
         * It will also fire when scrolling back up from after the scene and going over its end position. If you want something to happen only when scrolling down/right, use the scrollDirection parameter passed to the callback.
         *
         * For details on this event and the order in which it is fired, please review the {@link Scene.progress} method.
         *
         * @event ScrollMagic.Scene#end
         *
         * @example
         * scene.on("end", function (event) {
         * 	console.log("Hit end point of scene.");
         * });
         *
         * @property {object} event - The event Object passed to each callback
         * @property {string} event.type - The name of the event
         * @property {Scene} event.target - The Scene object that triggered this event
         * @property {number} event.progress - Reflects the current progress of the scene
         * @property {string} event.state - The current state of the scene `"DURING"` or `"AFTER"`
         * @property {string} event.scrollDirection - Indicates which way we are scrolling `"PAUSED"`, `"FORWARD"` or `"REVERSE"`
         */
        /**
         * Scene enter event.
         * Fires whenever the scene enters the "DURING" state.
         * Keep in mind that it doesn't matter if the scene plays forward or backward: This event always fires when the scene enters its active scroll timeframe, regardless of the scroll-direction.
         *
         * For details on this event and the order in which it is fired, please review the {@link Scene.progress} method.
         *
         * @event ScrollMagic.Scene#enter
         *
         * @example
         * scene.on("enter", function (event) {
         * 	console.log("Scene entered.");
         * });
         *
         * @property {object} event - The event Object passed to each callback
         * @property {string} event.type - The name of the event
         * @property {Scene} event.target - The Scene object that triggered this event
         * @property {number} event.progress - Reflects the current progress of the scene
         * @property {string} event.state - The current state of the scene - always `"DURING"`
         * @property {string} event.scrollDirection - Indicates which way we are scrolling `"PAUSED"`, `"FORWARD"` or `"REVERSE"`
         */
        /**
         * Scene leave event.
         * Fires whenever the scene's state goes from "DURING" to either "BEFORE" or "AFTER".
         * Keep in mind that it doesn't matter if the scene plays forward or backward: This event always fires when the scene leaves its active scroll timeframe, regardless of the scroll-direction.
         *
         * For details on this event and the order in which it is fired, please review the {@link Scene.progress} method.
         *
         * @event ScrollMagic.Scene#leave
         *
         * @example
         * scene.on("leave", function (event) {
         * 	console.log("Scene left.");
         * });
         *
         * @property {object} event - The event Object passed to each callback
         * @property {string} event.type - The name of the event
         * @property {Scene} event.target - The Scene object that triggered this event
         * @property {number} event.progress - Reflects the current progress of the scene
         * @property {string} event.state - The current state of the scene `"BEFORE"` or `"AFTER"`
         * @property {string} event.scrollDirection - Indicates which way we are scrolling `"PAUSED"`, `"FORWARD"` or `"REVERSE"`
         */
        /**
         * Scene update event.
         * Fires whenever the scene is updated (but not necessarily changes the progress).
         *
         * @event ScrollMagic.Scene#update
         *
         * @example
         * scene.on("update", function (event) {
         * 	console.log("Scene updated.");
         * });
         *
         * @property {object} event - The event Object passed to each callback
         * @property {string} event.type - The name of the event
         * @property {Scene} event.target - The Scene object that triggered this event
         * @property {number} event.startPos - The starting position of the scene (in relation to the conainer)
         * @property {number} event.endPos - The ending position of the scene (in relation to the conainer)
         * @property {number} event.scrollPos - The current scroll position of the container
         */
        /**
         * Scene progress event.
         * Fires whenever the progress of the scene changes.
         *
         * For details on this event and the order in which it is fired, please review the {@link Scene.progress} method.
         *
         * @event ScrollMagic.Scene#progress
         *
         * @example
         * scene.on("progress", function (event) {
         * 	console.log("Scene progress changed to " + event.progress);
         * });
         *
         * @property {object} event - The event Object passed to each callback
         * @property {string} event.type - The name of the event
         * @property {Scene} event.target - The Scene object that triggered this event
         * @property {number} event.progress - Reflects the current progress of the scene
         * @property {string} event.state - The current state of the scene `"BEFORE"`, `"DURING"` or `"AFTER"`
         * @property {string} event.scrollDirection - Indicates which way we are scrolling `"PAUSED"`, `"FORWARD"` or `"REVERSE"`
         */
        /**
         * Scene change event.
         * Fires whenvever a property of the scene is changed.
         *
         * @event ScrollMagic.Scene#change
         *
         * @example
         * scene.on("change", function (event) {
         * 	console.log("Scene Property \"" + event.what + "\" changed to " + event.newval);
         * });
         *
         * @property {object} event - The event Object passed to each callback
         * @property {string} event.type - The name of the event
         * @property {Scene} event.target - The Scene object that triggered this event
         * @property {string} event.what - Indicates what value has been changed
         * @property {mixed} event.newval - The new value of the changed property
         */
        /**
         * Scene shift event.
         * Fires whenvever the start or end **scroll offset** of the scene change.
         * This happens explicitely, when one of these values change: `offset`, `duration` or `triggerHook`.
         * It will fire implicitly when the `triggerElement` changes, if the new element has a different position (most cases).
         * It will also fire implicitly when the size of the container changes and the triggerHook is anything other than `onLeave`.
         *
         * @event ScrollMagic.Scene#shift
         * @since 1.1.0
         *
         * @example
         * scene.on("shift", function (event) {
         * 	console.log("Scene moved, because the " + event.reason + " has changed.)");
         * });
         *
         * @property {object} event - The event Object passed to each callback
         * @property {string} event.type - The name of the event
         * @property {Scene} event.target - The Scene object that triggered this event
         * @property {string} event.reason - Indicates why the scene has shifted
         */
        /**
         * Scene destroy event.
         * Fires whenvever the scene is destroyed.
         * This can be used to tidy up custom behaviour used in events.
         *
         * @event ScrollMagic.Scene#destroy
         * @since 1.1.0
         *
         * @example
         * scene.on("enter", function (event) {
         *        // add custom action
         *        $("#my-elem").left("200");
         *      })
         *      .on("destroy", function (event) {
         *        // reset my element to start position
         *        if (event.reset) {
         *          $("#my-elem").left("0");
         *        }
         *      });
         *
         * @property {object} event - The event Object passed to each callback
         * @property {string} event.type - The name of the event
         * @property {Scene} event.target - The Scene object that triggered this event
         * @property {boolean} event.reset - Indicates if the destroy method was called with reset `true` or `false`.
         */
        /**
         * Scene add event.
         * Fires when the scene is added to a controller.
         * This is mostly used by plugins to know that change might be due.
         *
         * @event ScrollMagic.Scene#add
         * @since 2.0.0
         *
         * @example
         * scene.on("add", function (event) {
         * 	console.log('Scene was added to a new controller.');
         * });
         *
         * @property {object} event - The event Object passed to each callback
         * @property {string} event.type - The name of the event
         * @property {Scene} event.target - The Scene object that triggered this event
         * @property {boolean} event.controller - The controller object the scene was added to.
         */
        /**
         * Scene remove event.
         * Fires when the scene is removed from a controller.
         * This is mostly used by plugins to know that change might be due.
         *
         * @event ScrollMagic.Scene#remove
         * @since 2.0.0
         *
         * @example
         * scene.on("remove", function (event) {
         * 	console.log('Scene was removed from its controller.');
         * });
         *
         * @property {object} event - The event Object passed to each callback
         * @property {string} event.type - The name of the event
         * @property {Scene} event.target - The Scene object that triggered this event
         */

        /**
         * Add one ore more event listener.
         * The callback function will be fired at the respective event, and an object containing relevant data will be passed to the callback.
         * @method ScrollMagic.Scene#on
         *
         * @example
         * function callback (event) {
         * 		console.log("Event fired! (" + event.type + ")");
         * }
         * // add listeners
         * scene.on("change update progress start end enter leave", callback);
         *
         * @param {string} names - The name or names of the event the callback should be attached to.
         * @param {function} callback - A function that should be executed, when the event is dispatched. An event object will be passed to the callback.
         * @returns {Scene} Parent object for chaining.
         */
        this.on = function (names, callback) {
            if (_util.type.Function(callback)) {
                names = names.trim().split(' ');
                names.forEach(function (fullname) {
                    var
                        nameparts = fullname.split('.'),
                        eventname = nameparts[0],
                        namespace = nameparts[1];
                    if (eventname != "*") { // disallow wildcards
                        if (!_listeners[eventname]) {
                            _listeners[eventname] = [];
                        }
                        _listeners[eventname].push({
                            namespace: namespace || '',
                            callback: callback
                        });
                    }
                });
            } else {
                log(1, "ERROR when calling '.on()': Supplied callback for '" + names + "' is not a valid function!");
            }
            return Scene;
        };

        /**
         * Remove one or more event listener.
         * @method ScrollMagic.Scene#off
         *
         * @example
         * function callback (event) {
         * 		console.log("Event fired! (" + event.type + ")");
         * }
         * // add listeners
         * scene.on("change update", callback);
         * // remove listeners
         * scene.off("change update", callback);
         *
         * @param {string} names - The name or names of the event that should be removed.
         * @param {function} [callback] - A specific callback function that should be removed. If none is passed all callbacks to the event listener will be removed.
         * @returns {Scene} Parent object for chaining.
         */
        this.off = function (names, callback) {
            if (!names) {
                log(1, "ERROR: Invalid event name supplied.");
                return Scene;
            }
            names = names.trim().split(' ');
            names.forEach(function (fullname, key) {
                var
                    nameparts = fullname.split('.'),
                    eventname = nameparts[0],
                    namespace = nameparts[1] || '',
                    removeList = eventname === '*' ? Object.keys(_listeners) : [eventname];
                removeList.forEach(function (remove) {
                    var
                        list = _listeners[remove] || [],
                        i = list.length;
                    while (i--) {
                        var listener = list[i];
                        if (listener && (namespace === listener.namespace || namespace === '*') && (!callback || callback == listener.callback)) {
                            list.splice(i, 1);
                        }
                    }
                    if (!list.length) {
                        delete _listeners[remove];
                    }
                });
            });
            return Scene;
        };

        /**
         * Trigger an event.
         * @method ScrollMagic.Scene#trigger
         *
         * @example
         * this.trigger("change");
         *
         * @param {string} name - The name of the event that should be triggered.
         * @param {object} [vars] - An object containing info that should be passed to the callback.
         * @returns {Scene} Parent object for chaining.
         */
        this.trigger = function (name, vars) {
            if (name) {
                var
                    nameparts = name.trim().split('.'),
                    eventname = nameparts[0],
                    namespace = nameparts[1],
                    listeners = _listeners[eventname];
                log(3, 'event fired:', eventname, vars ? "->" : '', vars || '');
                if (listeners) {
                    listeners.forEach(function (listener, key) {
                        if (!namespace || namespace === listener.namespace) {
                            listener.callback.call(Scene, new ScrollMagic.Event(eventname, listener.namespace, Scene, vars));
                        }
                    });
                }
            } else {
                log(1, "ERROR: Invalid event name supplied.");
            }
            return Scene;
        };

        // set event listeners
        Scene
            .on("change.internal", function (e) {
                if (e.what !== "loglevel" && e.what !== "tweenChanges") { // no need for a scene update scene with these options...
                    if (e.what === "triggerElement") {
                        updateTriggerElementPosition();
                    } else if (e.what === "reverse") { // the only property left that may have an impact on the current scene state. Everything else is handled by the shift event.
                        Scene.update();
                    }
                }
            })
            .on("shift.internal", function (e) {
                updateScrollOffset();
                Scene.update(); // update scene to reflect new position
            });

        /**
         * Send a debug message to the console.
         * @private
         * but provided publicly with _log for plugins
         *
         * @param {number} loglevel - The loglevel required to initiate output for the message.
         * @param {...mixed} output - One or more variables that should be passed to the console.
         */
        var log = this._log = function (loglevel, output) {
            if (_options.loglevel >= loglevel) {
                Array.prototype.splice.call(arguments, 1, 0, "(" + NAMESPACE + ") ->");
                _util.log.apply(window, arguments);
            }
        };

        /**
         * Add the scene to a controller.
         * This is the equivalent to `Controller.addScene(scene)`.
         * @method ScrollMagic.Scene#addTo
         *
         * @example
         * // add a scene to a ScrollMagic Controller
         * scene.addTo(controller);
         *
         * @param {ScrollMagic.Controller} controller - The controller to which the scene should be added.
         * @returns {Scene} Parent object for chaining.
         */
        this.addTo = function (controller) {
            if (!(controller instanceof ScrollMagic.Controller)) {
                log(1, "ERROR: supplied argument of 'addTo()' is not a valid ScrollMagic Controller");
            } else if (_controller != controller) {
                // new controller
                if (_controller) { // was associated to a different controller before, so remove it...
                    _controller.removeScene(Scene);
                }
                _controller = controller;
                validateOption();
                updateDuration(true);
                updateTriggerElementPosition(true);
                updateScrollOffset();
                _controller.info("container").addEventListener('resize', onContainerResize);
                controller.addScene(Scene);
                Scene.trigger("add", {
                    controller: _controller
                });
                log(3, "added " + NAMESPACE + " to controller");
                Scene.update();
            }
            return Scene;
        };

        /**
         * **Get** or **Set** the current enabled state of the scene.
         * This can be used to disable this scene without removing or destroying it.
         * @method ScrollMagic.Scene#enabled
         *
         * @example
         * // get the current value
         * var enabled = scene.enabled();
         *
         * // disable the scene
         * scene.enabled(false);
         *
         * @param {boolean} [newState] - The new enabled state of the scene `true` or `false`.
         * @returns {(boolean|Scene)} Current enabled state or parent object for chaining.
         */
        this.enabled = function (newState) {
            if (!arguments.length) { // get
                return _enabled;
            } else if (_enabled != newState) { // set
                _enabled = !!newState;
                Scene.update(true);
            }
            return Scene;
        };

        /**
         * Remove the scene from the controller.
         * This is the equivalent to `Controller.removeScene(scene)`.
         * The scene will not be updated anymore until you readd it to a controller.
         * To remove the pin or the tween you need to call removeTween() or removePin() respectively.
         * @method ScrollMagic.Scene#remove
         * @example
         * // remove the scene from its controller
         * scene.remove();
         *
         * @returns {Scene} Parent object for chaining.
         */
        this.remove = function () {
            if (_controller) {
                _controller.info("container").removeEventListener('resize', onContainerResize);
                var tmpParent = _controller;
                _controller = undefined;
                tmpParent.removeScene(Scene);
                Scene.trigger("remove");
                log(3, "removed " + NAMESPACE + " from controller");
            }
            return Scene;
        };

        /**
         * Destroy the scene and everything.
         * @method ScrollMagic.Scene#destroy
         * @example
         * // destroy the scene without resetting the pin and tween to their initial positions
         * scene = scene.destroy();
         *
         * // destroy the scene and reset the pin and tween
         * scene = scene.destroy(true);
         *
         * @param {boolean} [reset=false] - If `true` the pin and tween (if existent) will be reset.
         * @returns {null} Null to unset handler variables.
         */
        this.destroy = function (reset) {
            Scene.trigger("destroy", {
                reset: reset
            });
            Scene.remove();
            Scene.off("*.*");
            log(3, "destroyed " + NAMESPACE + " (reset: " + (reset ? "true" : "false") + ")");
            return null;
        };


        /**
         * Updates the Scene to reflect the current state.
         * This is the equivalent to `Controller.updateScene(scene, immediately)`.
         * The update method calculates the scene's start and end position (based on the trigger element, trigger hook, duration and offset) and checks it against the current scroll position of the container.
         * It then updates the current scene state accordingly (or does nothing, if the state is already correct) â€“ Pins will be set to their correct position and tweens will be updated to their correct progress.
         * This means an update doesn't necessarily result in a progress change. The `progress` event will be fired if the progress has indeed changed between this update and the last.
         * _**NOTE:** This method gets called constantly whenever ScrollMagic detects a change. The only application for you is if you change something outside of the realm of ScrollMagic, like moving the trigger or changing tween parameters._
         * @method ScrollMagic.Scene#update
         * @example
         * // update the scene on next tick
         * scene.update();
         *
         * // update the scene immediately
         * scene.update(true);
         *
         * @fires Scene.update
         *
         * @param {boolean} [immediately=false] - If `true` the update will be instant, if `false` it will wait until next update cycle (better performance).
         * @returns {Scene} Parent object for chaining.
         */
        this.update = function (immediately) {
            if (_controller) {
                if (immediately) {
                    if (_controller.enabled() && _enabled) {
                        var
                            scrollPos = _controller.info("scrollPos"),
                            newProgress;

                        if (_options.duration > 0) {
                            newProgress = (scrollPos - _scrollOffset.start) / (_scrollOffset.end - _scrollOffset.start);
                        } else {
                            newProgress = scrollPos >= _scrollOffset.start ? 1 : 0;
                        }

                        Scene.trigger("update", {
                            startPos: _scrollOffset.start,
                            endPos: _scrollOffset.end,
                            scrollPos: scrollPos
                        });

                        Scene.progress(newProgress);
                    } else if (_pin && _state === SCENE_STATE_DURING) {
                        updatePinState(true); // unpin in position
                    }
                } else {
                    _controller.updateScene(Scene, false);
                }
            }
            return Scene;
        };

        /**
         * Updates dynamic scene variables like the trigger element position or the duration.
         * This method is automatically called in regular intervals from the controller. See {@link ScrollMagic.Controller} option `refreshInterval`.
         *
         * You can call it to minimize lag, for example when you intentionally change the position of the triggerElement.
         * If you don't it will simply be updated in the next refresh interval of the container, which is usually sufficient.
         *
         * @method ScrollMagic.Scene#refresh
         * @since 1.1.0
         * @example
         * scene = new ScrollMagic.Scene({triggerElement: "#trigger"});
         *
         * // change the position of the trigger
         * $("#trigger").css("top", 500);
         * // immediately let the scene know of this change
         * scene.refresh();
         *
         * @fires {@link Scene.shift}, if the trigger element position or the duration changed
         * @fires {@link Scene.change}, if the duration changed
         *
         * @returns {Scene} Parent object for chaining.
         */
        this.refresh = function () {
            updateDuration();
            updateTriggerElementPosition();
            // update trigger element position
            return Scene;
        };

        /**
         * **Get** or **Set** the scene's progress.
         * Usually it shouldn't be necessary to use this as a setter, as it is set automatically by scene.update().
         * The order in which the events are fired depends on the duration of the scene:
         *  1. Scenes with `duration == 0`:
         *  Scenes that have no duration by definition have no ending. Thus the `end` event will never be fired.
         *  When the trigger position of the scene is passed the events are always fired in this order:
         *  `enter`, `start`, `progress` when scrolling forward
         *  and
         *  `progress`, `start`, `leave` when scrolling in reverse
         *  2. Scenes with `duration > 0`:
         *  Scenes with a set duration have a defined start and end point.
         *  When scrolling past the start position of the scene it will fire these events in this order:
         *  `enter`, `start`, `progress`
         *  When continuing to scroll and passing the end point it will fire these events:
         *  `progress`, `end`, `leave`
         *  When reversing through the end point these events are fired:
         *  `enter`, `end`, `progress`
         *  And when continuing to scroll past the start position in reverse it will fire:
         *  `progress`, `start`, `leave`
         *  In between start and end the `progress` event will be called constantly, whenever the progress changes.
         *
         * In short:
         * `enter` events will always trigger **before** the progress update and `leave` envents will trigger **after** the progress update.
         * `start` and `end` will always trigger at their respective position.
         *
         * Please review the event descriptions for details on the events and the event object that is passed to the callback.
         *
         * @method ScrollMagic.Scene#progress
         * @example
         * // get the current scene progress
         * var progress = scene.progress();
         *
         * // set new scene progress
         * scene.progress(0.3);
         *
         * @fires {@link Scene.enter}, when used as setter
         * @fires {@link Scene.start}, when used as setter
         * @fires {@link Scene.progress}, when used as setter
         * @fires {@link Scene.end}, when used as setter
         * @fires {@link Scene.leave}, when used as setter
         *
         * @param {number} [progress] - The new progress value of the scene `[0-1]`.
         * @returns {number} `get` -  Current scene progress.
         * @returns {Scene} `set` -  Parent object for chaining.
         */
        this.progress = function (progress) {
            if (!arguments.length) { // get
                return _progress;
            } else { // set
                var
                    doUpdate = false,
                    oldState = _state,
                    scrollDirection = _controller ? _controller.info("scrollDirection") : 'PAUSED',
                    reverseOrForward = _options.reverse || progress >= _progress;
                if (_options.duration === 0) {
                    // zero duration scenes
                    doUpdate = _progress != progress;
                    _progress = progress < 1 && reverseOrForward ? 0 : 1;
                    _state = _progress === 0 ? SCENE_STATE_BEFORE : SCENE_STATE_DURING;
                } else {
                    // scenes with start and end
                    if (progress < 0 && _state !== SCENE_STATE_BEFORE && reverseOrForward) {
                        // go back to initial state
                        _progress = 0;
                        _state = SCENE_STATE_BEFORE;
                        doUpdate = true;
                    } else if (progress >= 0 && progress < 1 && reverseOrForward) {
                        _progress = progress;
                        _state = SCENE_STATE_DURING;
                        doUpdate = true;
                    } else if (progress >= 1 && _state !== SCENE_STATE_AFTER) {
                        _progress = 1;
                        _state = SCENE_STATE_AFTER;
                        doUpdate = true;
                    } else if (_state === SCENE_STATE_DURING && !reverseOrForward) {
                        updatePinState(); // in case we scrolled backwards mid-scene and reverse is disabled => update the pin position, so it doesn't move back as well.
                    }
                }
                if (doUpdate) {
                    // fire events
                    var
                        eventVars = {
                            progress: _progress,
                            state: _state,
                            scrollDirection: scrollDirection
                        },
                        stateChanged = _state != oldState;

                    var trigger = function (eventName) { // tmp helper to simplify code
                        Scene.trigger(eventName, eventVars);
                    };

                    if (stateChanged) { // enter events
                        if (oldState !== SCENE_STATE_DURING) {
                            trigger("enter");
                            trigger(oldState === SCENE_STATE_BEFORE ? "start" : "end");
                        }
                    }
                    trigger("progress");
                    if (stateChanged) { // leave events
                        if (_state !== SCENE_STATE_DURING) {
                            trigger(_state === SCENE_STATE_BEFORE ? "start" : "end");
                            trigger("leave");
                        }
                    }
                }

                return Scene;
            }
        };


        /**
         * Update the start and end scrollOffset of the container.
         * The positions reflect what the controller's scroll position will be at the start and end respectively.
         * Is called, when:
         *   - Scene event "change" is called with: offset, triggerHook, duration
         *   - scroll container event "resize" is called
         *   - the position of the triggerElement changes
         *   - the controller changes -> addTo()
         * @private
         */
        var updateScrollOffset = function () {
            _scrollOffset = {
                start: _triggerPos + _options.offset
            };
            if (_controller && _options.triggerElement) {
                // take away triggerHook portion to get relative to top
                _scrollOffset.start -= _controller.info("size") * _options.triggerHook;
            }
            _scrollOffset.end = _scrollOffset.start + _options.duration;
        };

        /**
         * Updates the duration if set to a dynamic function.
         * This method is called when the scene is added to a controller and in regular intervals from the controller through scene.refresh().
         *
         * @fires {@link Scene.change}, if the duration changed
         * @fires {@link Scene.shift}, if the duration changed
         *
         * @param {boolean} [suppressEvents=false] - If true the shift event will be suppressed.
         * @private
         */
        var updateDuration = function (suppressEvents) {
            // update duration
            if (_durationUpdateMethod) {
                var varname = "duration";
                if (changeOption(varname, _durationUpdateMethod.call(Scene)) && !suppressEvents) { // set
                    Scene.trigger("change", {
                        what: varname,
                        newval: _options[varname]
                    });
                    Scene.trigger("shift", {
                        reason: varname
                    });
                }
            }
        };

        /**
         * Updates the position of the triggerElement, if present.
         * This method is called ...
         *  - ... when the triggerElement is changed
         *  - ... when the scene is added to a (new) controller
         *  - ... in regular intervals from the controller through scene.refresh().
         *
         * @fires {@link Scene.shift}, if the position changed
         *
         * @param {boolean} [suppressEvents=false] - If true the shift event will be suppressed.
         * @private
         */
        var updateTriggerElementPosition = function (suppressEvents) {
            var
                elementPos = 0,
                telem = _options.triggerElement;
            if (_controller && (telem || _triggerPos > 0)) { // either an element exists or was removed and the triggerPos is still > 0
                if (telem) { // there currently a triggerElement set
                    if (telem.parentNode) { // check if element is still attached to DOM
                        var
                            controllerInfo = _controller.info(),
                            containerOffset = _util.get.offset(controllerInfo.container), // container position is needed because element offset is returned in relation to document, not in relation to container.
                            param = controllerInfo.vertical ? "top" : "left"; // which param is of interest ?

                        // if parent is spacer, use spacer position instead so correct start position is returned for pinned elements.
                        while (telem.parentNode.hasAttribute(PIN_SPACER_ATTRIBUTE)) {
                            telem = telem.parentNode;
                        }

                        var elementOffset = _util.get.offset(telem);

                        if (!controllerInfo.isDocument) { // container is not the document root, so substract scroll Position to get correct trigger element position relative to scrollcontent
                            containerOffset[param] -= _controller.scrollPos();
                        }

                        elementPos = elementOffset[param] - containerOffset[param];

                    } else { // there was an element, but it was removed from DOM
                        log(2, "WARNING: triggerElement was removed from DOM and will be reset to", undefined);
                        Scene.triggerElement(undefined); // unset, so a change event is triggered
                    }
                }

                var changed = elementPos != _triggerPos;
                _triggerPos = elementPos;
                if (changed && !suppressEvents) {
                    Scene.trigger("shift", {
                        reason: "triggerElementPosition"
                    });
                }
            }
        };

        /**
         * Trigger a shift event, when the container is resized and the triggerHook is > 1.
         * @private
         */
        var onContainerResize = function (e) {
            if (_options.triggerHook > 0) {
                Scene.trigger("shift", {
                    reason: "containerResize"
                });
            }
        };


        var _validate = _util.extend(SCENE_OPTIONS.validate, {
            // validation for duration handled internally for reference to private var _durationMethod
            duration: function (val) {
                if (_util.type.String(val) && val.match(/^(\.|\d)*\d+%$/)) {
                    // percentage value
                    var perc = parseFloat(val) / 100;
                    val = function () {
                        return _controller ? _controller.info("size") * perc : 0;
                    };
                }
                if (_util.type.Function(val)) {
                    // function
                    _durationUpdateMethod = val;
                    try {
                        val = parseFloat(_durationUpdateMethod.call(Scene));
                    } catch (e) {
                        val = -1; // will cause error below
                    }
                }
                // val has to be float
                val = parseFloat(val);
                if (!_util.type.Number(val) || val < 0) {
                    if (_durationUpdateMethod) {
                        _durationUpdateMethod = undefined;
                        throw ["Invalid return value of supplied function for option \"duration\":", val];
                    } else {
                        throw ["Invalid value for option \"duration\":", val];
                    }
                }
                return val;
            }
        });

        /**
         * Checks the validity of a specific or all options and reset to default if neccessary.
         * @private
         */
        var validateOption = function (check) {
            check = arguments.length ? [check] : Object.keys(_validate);
            check.forEach(function (optionName, key) {
                var value;
                if (_validate[optionName]) { // there is a validation method for this option
                    try { // validate value
                        value = _validate[optionName](_options[optionName]);
                    } catch (e) { // validation failed -> reset to default
                        value = DEFAULT_OPTIONS[optionName];
                        var logMSG = _util.type.String(e) ? [e] : e;
                        if (_util.type.Array(logMSG)) {
                            logMSG[0] = "ERROR: " + logMSG[0];
                            logMSG.unshift(1); // loglevel 1 for error msg
                            log.apply(this, logMSG);
                        } else {
                            log(1, "ERROR: Problem executing validation callback for option '" + optionName + "':", e.message);
                        }
                    } finally {
                        _options[optionName] = value;
                    }
                }
            });
        };

        /**
         * Helper used by the setter/getters for scene options
         * @private
         */
        var changeOption = function (varname, newval) {
            var
                changed = false,
                oldval = _options[varname];
            if (_options[varname] != newval) {
                _options[varname] = newval;
                validateOption(varname); // resets to default if necessary
                changed = oldval != _options[varname];
            }
            return changed;
        };

        // generate getters/setters for all options
        var addSceneOption = function (optionName) {
            if (!Scene[optionName]) {
                Scene[optionName] = function (newVal) {
                    if (!arguments.length) { // get
                        return _options[optionName];
                    } else {
                        if (optionName === "duration") { // new duration is set, so any previously set function must be unset
                            _durationUpdateMethod = undefined;
                        }
                        if (changeOption(optionName, newVal)) { // set
                            Scene.trigger("change", {
                                what: optionName,
                                newval: _options[optionName]
                            });
                            if (SCENE_OPTIONS.shifts.indexOf(optionName) > -1) {
                                Scene.trigger("shift", {
                                    reason: optionName
                                });
                            }
                        }
                    }
                    return Scene;
                };
            }
        };

        /**
         * **Get** or **Set** the duration option value.
         *
         * As a **setter** it accepts three types of parameters:
         * 1. `number`: Sets the duration of the scene to exactly this amount of pixels.
         *   This means the scene will last for exactly this amount of pixels scrolled. Sub-Pixels are also valid.
         *   A value of `0` means that the scene is 'open end' and no end will be triggered. Pins will never unpin and animations will play independently of scroll progress.
         * 2. `string`: Always updates the duration relative to parent scroll container.
         *   For example `"100%"` will keep the duration always exactly at the inner height of the scroll container.
         *   When scrolling vertically the width is used for reference respectively.
         * 3. `function`: The supplied function will be called to return the scene duration.
         *   This is useful in setups where the duration depends on other elements who might change size. By supplying a function you can return a value instead of updating potentially multiple scene durations.
         *   The scene can be referenced inside the callback using `this`.
         *   _**WARNING:** This is an easy way to kill performance, as the callback will be executed every time `Scene.refresh()` is called, which happens a lot. The interval is defined by the controller (see ScrollMagic.Controller option `refreshInterval`).
         *   It's recomended to avoid calculations within the function and use cached variables as return values.
         *   This counts double if you use the same function for multiple scenes._
         *
         * @method ScrollMagic.Scene#duration
         * @example
         * // get the current duration value
         * var duration = scene.duration();
         *
         * // set a new duration
         * scene.duration(300);
         *
         * // set duration responsively to container size
         * scene.duration("100%");
         *
         * // use a function to randomize the duration for some reason.
         * var durationValueCache;
         * function durationCallback () {
         *   return durationValueCache;
         * }
         * function updateDuration () {
         *   durationValueCache = Math.random() * 100;
         * }
         * updateDuration(); // set to initial value
         * scene.duration(durationCallback); // set duration callback
         *
         * @fires {@link Scene.change}, when used as setter
         * @fires {@link Scene.shift}, when used as setter
         * @param {(number|string|function)} [newDuration] - The new duration setting for the scene.
         * @returns {number} `get` -  Current scene duration.
         * @returns {Scene} `set` -  Parent object for chaining.
         */

        /**
         * **Get** or **Set** the offset option value.
         * @method ScrollMagic.Scene#offset
         * @example
         * // get the current offset
         * var offset = scene.offset();
         *
         * // set a new offset
         * scene.offset(100);
         *
         * @fires {@link Scene.change}, when used as setter
         * @fires {@link Scene.shift}, when used as setter
         * @param {number} [newOffset] - The new offset of the scene.
         * @returns {number} `get` -  Current scene offset.
         * @returns {Scene} `set` -  Parent object for chaining.
         */

        /**
         * **Get** or **Set** the triggerElement option value.
         * Does **not** fire `Scene.shift`, because changing the trigger Element doesn't necessarily mean the start position changes. This will be determined in `Scene.refresh()`, which is automatically triggered.
         * @method ScrollMagic.Scene#triggerElement
         * @example
         * // get the current triggerElement
         * var triggerElement = scene.triggerElement();
         *
         * // set a new triggerElement using a selector
         * scene.triggerElement("#trigger");
         * // set a new triggerElement using a DOM object
         * scene.triggerElement(document.getElementById("trigger"));
         *
         * @fires {@link Scene.change}, when used as setter
         * @param {(string|object)} [newTriggerElement] - The new trigger element for the scene.
         * @returns {(string|object)} `get` -  Current triggerElement.
         * @returns {Scene} `set` -  Parent object for chaining.
         */

        /**
         * **Get** or **Set** the triggerHook option value.
         * @method ScrollMagic.Scene#triggerHook
         * @example
         * // get the current triggerHook value
         * var triggerHook = scene.triggerHook();
         *
         * // set a new triggerHook using a string
         * scene.triggerHook("onLeave");
         * // set a new triggerHook using a number
         * scene.triggerHook(0.7);
         *
         * @fires {@link Scene.change}, when used as setter
         * @fires {@link Scene.shift}, when used as setter
         * @param {(number|string)} [newTriggerHook] - The new triggerHook of the scene. See {@link Scene} parameter description for value options.
         * @returns {number} `get` -  Current triggerHook (ALWAYS numerical).
         * @returns {Scene} `set` -  Parent object for chaining.
         */

        /**
         * **Get** or **Set** the reverse option value.
         * @method ScrollMagic.Scene#reverse
         * @example
         * // get the current reverse option
         * var reverse = scene.reverse();
         *
         * // set new reverse option
         * scene.reverse(false);
         *
         * @fires {@link Scene.change}, when used as setter
         * @param {boolean} [newReverse] - The new reverse setting of the scene.
         * @returns {boolean} `get` -  Current reverse option value.
         * @returns {Scene} `set` -  Parent object for chaining.
         */

        /**
         * **Get** or **Set** the loglevel option value.
         * @method ScrollMagic.Scene#loglevel
         * @example
         * // get the current loglevel
         * var loglevel = scene.loglevel();
         *
         * // set new loglevel
         * scene.loglevel(3);
         *
         * @fires {@link Scene.change}, when used as setter
         * @param {number} [newLoglevel] - The new loglevel setting of the scene. `[0-3]`
         * @returns {number} `get` -  Current loglevel.
         * @returns {Scene} `set` -  Parent object for chaining.
         */

        /**
         * **Get** the associated controller.
         * @method ScrollMagic.Scene#controller
         * @example
         * // get the controller of a scene
         * var controller = scene.controller();
         *
         * @returns {ScrollMagic.Controller} Parent controller or `undefined`
         */
        this.controller = function () {
            return _controller;
        };

        /**
         * **Get** the current state.
         * @method ScrollMagic.Scene#state
         * @example
         * // get the current state
         * var state = scene.state();
         *
         * @returns {string} `"BEFORE"`, `"DURING"` or `"AFTER"`
         */
        this.state = function () {
            return _state;
        };

        /**
         * **Get** the current scroll offset for the start of the scene.
         * Mind, that the scrollOffset is related to the size of the container, if `triggerHook` is bigger than `0` (or `"onLeave"`).
         * This means, that resizing the container or changing the `triggerHook` will influence the scene's start offset.
         * @method ScrollMagic.Scene#scrollOffset
         * @example
         * // get the current scroll offset for the start and end of the scene.
         * var start = scene.scrollOffset();
         * var end = scene.scrollOffset() + scene.duration();
         * console.log("the scene starts at", start, "and ends at", end);
         *
         * @returns {number} The scroll offset (of the container) at which the scene will trigger. Y value for vertical and X value for horizontal scrolls.
         */
        this.scrollOffset = function () {
            return _scrollOffset.start;
        };

        /**
         * **Get** the trigger position of the scene (including the value of the `offset` option).
         * @method ScrollMagic.Scene#triggerPosition
         * @example
         * // get the scene's trigger position
         * var triggerPosition = scene.triggerPosition();
         *
         * @returns {number} Start position of the scene. Top position value for vertical and left position value for horizontal scrolls.
         */
        this.triggerPosition = function () {
            var pos = _options.offset; // the offset is the basis
            if (_controller) {
                // get the trigger position
                if (_options.triggerElement) {
                    // Element as trigger
                    pos += _triggerPos;
                } else {
                    // return the height of the triggerHook to start at the beginning
                    pos += _controller.info("size") * Scene.triggerHook();
                }
            }
            return pos;
        };


        var
            _pin,
            _pinOptions;

        Scene
            .on("shift.internal", function (e) {
                var durationChanged = e.reason === "duration";
                if ((_state === SCENE_STATE_AFTER && durationChanged) || (_state === SCENE_STATE_DURING && _options.duration === 0)) {
                    // if [duration changed after a scene (inside scene progress updates pin position)] or [duration is 0, we are in pin phase and some other value changed].
                    updatePinState();
                }
                if (durationChanged) {
                    updatePinDimensions();
                }
            })
            .on("progress.internal", function (e) {
                updatePinState();
            })
            .on("add.internal", function (e) {
                updatePinDimensions();
            })
            .on("destroy.internal", function (e) {
                Scene.removePin(e.reset);
            });
        /**
         * Update the pin state.
         * @private
         */
        var updatePinState = function (forceUnpin) {

            if (_pin && _controller) {
                var
                    containerInfo = _controller.info(),
                    pinTarget = _pinOptions.spacer.firstChild; // may be pin element or another spacer, if cascading pins

                if (!forceUnpin && _state === SCENE_STATE_DURING) { // during scene or if duration is 0 and we are past the trigger
                    // pinned state
                    if (_util.css(pinTarget, "position") != "fixed") {
                        // change state before updating pin spacer (position changes due to fixed collapsing might occur.)
                        _util.css(pinTarget, {
                            "position": "fixed"
                        });
                        // update pin spacer
                        updatePinDimensions();
                    }

                    var
                        fixedPos = _util.get.offset(_pinOptions.spacer, true), // get viewport position of spacer
                        scrollDistance = _options.reverse || _options.duration === 0 ?
                            containerInfo.scrollPos - _scrollOffset.start // quicker
                            :
                            Math.round(_progress * _options.duration * 10) / 10; // if no reverse and during pin the position needs to be recalculated using the progress

                    // add scrollDistance
                    fixedPos[containerInfo.vertical ? "top" : "left"] += scrollDistance;

                    // set new values
                    _util.css(_pinOptions.spacer.firstChild, {
                        top: fixedPos.top,
                        left: fixedPos.left
                    });
                } else {
                    // unpinned state
                    var
                        newCSS = {
                            position: _pinOptions.inFlow ? "relative" : "absolute",
                            top: 0,
                            left: 0
                        },
                        change = _util.css(pinTarget, "position") != newCSS.position;

                    if (!_pinOptions.pushFollowers) {
                        newCSS[containerInfo.vertical ? "top" : "left"] = _options.duration * _progress;
                    } else if (_options.duration > 0) { // only concerns scenes with duration
                        if (_state === SCENE_STATE_AFTER && parseFloat(_util.css(_pinOptions.spacer, "padding-top")) === 0) {
                            change = true; // if in after state but havent updated spacer yet (jumped past pin)
                        } else if (_state === SCENE_STATE_BEFORE && parseFloat(_util.css(_pinOptions.spacer, "padding-bottom")) === 0) { // before
                            change = true; // jumped past fixed state upward direction
                        }
                    }
                    // set new values
                    _util.css(pinTarget, newCSS);
                    if (change) {
                        // update pin spacer if state changed
                        updatePinDimensions();
                    }
                }
            }
        };

        /**
         * Update the pin spacer and/or element size.
         * The size of the spacer needs to be updated whenever the duration of the scene changes, if it is to push down following elements.
         * @private
         */
        var updatePinDimensions = function () {
            if (_pin && _controller && _pinOptions.inFlow) { // no spacerresize, if original position is absolute
                var
                    after = (_state === SCENE_STATE_AFTER),
                    before = (_state === SCENE_STATE_BEFORE),
                    during = (_state === SCENE_STATE_DURING),
                    vertical = _controller.info("vertical"),
                    pinTarget = _pinOptions.spacer.firstChild, // usually the pined element but can also be another spacer (cascaded pins)
                    marginCollapse = _util.isMarginCollapseType(_util.css(_pinOptions.spacer, "display")),
                    css = {};

                // set new size
                // if relsize: spacer -> pin | else: pin -> spacer
                if (_pinOptions.relSize.width || _pinOptions.relSize.autoFullWidth) {
                    if (during) {
                        _util.css(_pin, {
                            "width": _util.get.width(_pinOptions.spacer)
                        });
                    } else {
                        _util.css(_pin, {
                            "width": "100%"
                        });
                    }
                } else {
                    // minwidth is needed for cascaded pins.
                    css["min-width"] = _util.get.width(vertical ? _pin : pinTarget, true, true);
                    css.width = during ? css["min-width"] : "auto";
                }
                if (_pinOptions.relSize.height) {
                    if (during) {
                        // the only padding the spacer should ever include is the duration (if pushFollowers = true), so we need to substract that.
                        _util.css(_pin, {
                            "height": _util.get.height(_pinOptions.spacer) - (_pinOptions.pushFollowers ? _options.duration : 0)
                        });
                    } else {
                        _util.css(_pin, {
                            "height": "100%"
                        });
                    }
                } else {
                    // margin is only included if it's a cascaded pin to resolve an IE9 bug
                    css["min-height"] = _util.get.height(vertical ? pinTarget : _pin, true, !marginCollapse); // needed for cascading pins

                    css.height = during ? css["min-height"] : "auto";
                }

                // add space for duration if pushFollowers is true
                if (_pinOptions.pushFollowers) {
                    css["padding" + (vertical ? "Top" : "Left")] = _options.duration * _progress;
                    css["padding" + (vertical ? "Bottom" : "Right")] = _options.duration * (1 - _progress);
                }
                _util.css(_pinOptions.spacer, css);
            }
        };

        /**
         * Updates the Pin state (in certain scenarios)
         * If the controller container is not the document and we are mid-pin-phase scrolling or resizing the main document can result to wrong pin positions.
         * So this function is called on resize and scroll of the document.
         * @private
         */
        var updatePinInContainer = function () {
            if (_controller && _pin && _state === SCENE_STATE_DURING && !_controller.info("isDocument")) {
                updatePinState();
            }
        };

        /**
         * Updates the Pin spacer size state (in certain scenarios)
         * If container is resized during pin and relatively sized the size of the pin might need to be updated...
         * So this function is called on resize of the container.
         * @private
         */
        var updateRelativePinSpacer = function () {
            if (_controller && _pin && // well, duh
                _state === SCENE_STATE_DURING && // element in pinned state?
                ( // is width or height relatively sized, but not in relation to body? then we need to recalc.
                    ((_pinOptions.relSize.width || _pinOptions.relSize.autoFullWidth) && _util.get.width(window) != _util.get.width(_pinOptions.spacer.parentNode)) ||
                    (_pinOptions.relSize.height && _util.get.height(window) != _util.get.height(_pinOptions.spacer.parentNode))
                )
            ) {
                updatePinDimensions();
            }
        };

        /**
         * Is called, when the mousewhel is used while over a pinned element inside a div container.
         * If the scene is in fixed state scroll events would be counted towards the body. This forwards the event to the scroll container.
         * @private
         */
        var onMousewheelOverPin = function (e) {
            if (_controller && _pin && _state === SCENE_STATE_DURING && !_controller.info("isDocument")) { // in pin state
                e.preventDefault();
                _controller._setScrollPos(_controller.info("scrollPos") - ((e.wheelDelta || e[_controller.info("vertical") ? "wheelDeltaY" : "wheelDeltaX"]) / 3 || -e.detail * 30));
            }
        };

        /**
         * Pin an element for the duration of the scene.
         * If the scene duration is 0 the element will only be unpinned, if the user scrolls back past the start position.
         * Make sure only one pin is applied to an element at the same time.
         * An element can be pinned multiple times, but only successively.
         * _**NOTE:** The option `pushFollowers` has no effect, when the scene duration is 0._
         * @method ScrollMagic.Scene#setPin
         * @example
         * // pin element and push all following elements down by the amount of the pin duration.
         * scene.setPin("#pin");
         *
         * // pin element and keeping all following elements in their place. The pinned element will move past them.
         * scene.setPin("#pin", {pushFollowers: false});
         *
         * @param {(string|object)} element - A Selector targeting an element or a DOM object that is supposed to be pinned.
         * @param {object} [settings] - settings for the pin
         * @param {boolean} [settings.pushFollowers=true] - If `true` following elements will be "pushed" down for the duration of the pin, if `false` the pinned element will just scroll past them.
                                                               Ignored, when duration is `0`.
         * @param {string} [settings.spacerClass="scrollmagic-pin-spacer"] - Classname of the pin spacer element, which is used to replace the element.
         *
         * @returns {Scene} Parent object for chaining.
         */
        this.setPin = function (element, settings) {
            var
                defaultSettings = {
                    pushFollowers: true,
                    spacerClass: "scrollmagic-pin-spacer"
                };
            var pushFollowersActivelySet = settings && settings.hasOwnProperty('pushFollowers');
            settings = _util.extend({}, defaultSettings, settings);

            // validate Element
            element = _util.get.elements(element)[0];

            if (!element) {
                log(1, "ERROR calling method 'setPin()': Invalid pin element supplied.");
                return Scene; // cancel
            } else if (_util.css(element, "position") === "fixed") {

                log(1, "ERROR calling method 'setPin()': Pin does not work with elements that are positioned 'fixed'.");
                return Scene; // cancel
            }

            if (_pin) { // preexisting pin?
                if (_pin === element) {
                    // same pin we already have -> do nothing
                    return Scene; // cancel
                } else {
                    // kill old pin
                    Scene.removePin();
                }

            }
            _pin = element;

            var
                parentDisplay = _pin.parentNode.style.display,
                boundsParams = ["top", "left", "bottom", "right", "margin", "marginLeft", "marginRight", "marginTop", "marginBottom"];

            _pin.parentNode.style.display = 'none'; // hack start to force css to return stylesheet values instead of calculated px values.
            var
                inFlow = _util.css(_pin, "position") != "absolute",
                pinCSS = _util.css(_pin, boundsParams.concat(["display"])),
                sizeCSS = _util.css(_pin, ["width", "height"]);
            _pin.parentNode.style.display = parentDisplay; // hack end.

            if (!inFlow && settings.pushFollowers) {
                log(2, "WARNING: If the pinned element is positioned absolutely pushFollowers will be disabled.");
                settings.pushFollowers = false;
            }
            window.setTimeout(function () { // wait until all finished, because with responsive duration it will only be set after scene is added to controller
                if (_pin && _options.duration === 0 && pushFollowersActivelySet && settings.pushFollowers) {
                    log(2, "WARNING: pushFollowers =", true, "has no effect, when scene duration is 0.");
                }
            }, 0);

            // create spacer and insert
            var
                spacer = _pin.parentNode.insertBefore(document.createElement('div'), _pin),
                spacerCSS = _util.extend(pinCSS, {
                    position: inFlow ? "relative" : "absolute",
                    boxSizing: "content-box",
                    mozBoxSizing: "content-box",
                    webkitBoxSizing: "content-box"
                });

            if (!inFlow) { // copy size if positioned absolutely, to work for bottom/right positioned elements.
                _util.extend(spacerCSS, _util.css(_pin, ["width", "height"]));
            }

            _util.css(spacer, spacerCSS);
            spacer.setAttribute(PIN_SPACER_ATTRIBUTE, "");
            _util.addClass(spacer, settings.spacerClass);

            // set the pin Options
            _pinOptions = {
                spacer: spacer,
                relSize: { // save if size is defined using % values. if so, handle spacer resize differently...
                    width: sizeCSS.width.slice(-1) === "%",
                    height: sizeCSS.height.slice(-1) === "%",
                    autoFullWidth: sizeCSS.width === "auto" && inFlow && _util.isMarginCollapseType(pinCSS.display)
                },
                pushFollowers: settings.pushFollowers,
                inFlow: inFlow, // stores if the element takes up space in the document flow
            };

            if (!_pin.___origStyle) {
                _pin.___origStyle = {};
                var
                    pinInlineCSS = _pin.style,
                    copyStyles = boundsParams.concat(["width", "height", "position", "boxSizing", "mozBoxSizing", "webkitBoxSizing"]);
                copyStyles.forEach(function (val) {
                    _pin.___origStyle[val] = pinInlineCSS[val] || "";
                });
            }

            // if relative size, transfer it to spacer and make pin calculate it...
            if (_pinOptions.relSize.width) {
                _util.css(spacer, {
                    width: sizeCSS.width
                });
            }
            if (_pinOptions.relSize.height) {
                _util.css(spacer, {
                    height: sizeCSS.height
                });
            }

            // now place the pin element inside the spacer
            spacer.appendChild(_pin);
            // and set new css
            _util.css(_pin, {
                position: inFlow ? "relative" : "absolute",
                margin: "auto",
                top: "auto",
                left: "auto",
                bottom: "auto",
                right: "auto"
            });

            if (_pinOptions.relSize.width || _pinOptions.relSize.autoFullWidth) {
                _util.css(_pin, {
                    boxSizing: "border-box",
                    mozBoxSizing: "border-box",
                    webkitBoxSizing: "border-box"
                });
            }

            // add listener to document to update pin position in case controller is not the document.
            window.addEventListener('scroll', updatePinInContainer);
            window.addEventListener('resize', updatePinInContainer);
            window.addEventListener('resize', updateRelativePinSpacer);
            // add mousewheel listener to catch scrolls over fixed elements
            _pin.addEventListener("mousewheel", onMousewheelOverPin);
            _pin.addEventListener("DOMMouseScroll", onMousewheelOverPin);

            log(3, "added pin");

            // finally update the pin to init
            updatePinState();

            return Scene;
        };

        /**
         * Remove the pin from the scene.
         * @method ScrollMagic.Scene#removePin
         * @example
         * // remove the pin from the scene without resetting it (the spacer is not removed)
         * scene.removePin();
         *
         * // remove the pin from the scene and reset the pin element to its initial position (spacer is removed)
         * scene.removePin(true);
         *
         * @param {boolean} [reset=false] - If `false` the spacer will not be removed and the element's position will not be reset.
         * @returns {Scene} Parent object for chaining.
         */
        this.removePin = function (reset) {
            if (_pin) {
                if (_state === SCENE_STATE_DURING) {
                    updatePinState(true); // force unpin at position
                }
                if (reset || !_controller) { // if there's no controller no progress was made anyway...
                    var pinTarget = _pinOptions.spacer.firstChild; // usually the pin element, but may be another spacer (cascaded pins)...
                    if (pinTarget.hasAttribute(PIN_SPACER_ATTRIBUTE)) { // copy margins to child spacer
                        var
                            style = _pinOptions.spacer.style,
                            values = ["margin", "marginLeft", "marginRight", "marginTop", "marginBottom"],
                            margins = {};
                        values.forEach(function (val) {
                            margins[val] = style[val] || "";
                        });
                        _util.css(pinTarget, margins);
                    }
                    _pinOptions.spacer.parentNode.insertBefore(pinTarget, _pinOptions.spacer);
                    _pinOptions.spacer.parentNode.removeChild(_pinOptions.spacer);
                    if (!_pin.parentNode.hasAttribute(PIN_SPACER_ATTRIBUTE)) { // if it's the last pin for this element -> restore inline styles
                        // TODO: only correctly set for first pin (when cascading) - how to fix?
                        _util.css(_pin, _pin.___origStyle);
                        delete _pin.___origStyle;
                    }
                }
                window.removeEventListener('scroll', updatePinInContainer);
                window.removeEventListener('resize', updatePinInContainer);
                window.removeEventListener('resize', updateRelativePinSpacer);
                _pin.removeEventListener("mousewheel", onMousewheelOverPin);
                _pin.removeEventListener("DOMMouseScroll", onMousewheelOverPin);
                _pin = undefined;
                log(3, "removed pin (reset: " + (reset ? "true" : "false") + ")");
            }
            return Scene;
        };


        var
            _cssClasses,
            _cssClassElems = [];

        Scene
            .on("destroy.internal", function (e) {
                Scene.removeClassToggle(e.reset);
            });
        /**
         * Define a css class modification while the scene is active.
         * When the scene triggers the classes will be added to the supplied element and removed, when the scene is over.
         * If the scene duration is 0 the classes will only be removed if the user scrolls back past the start position.
         * @method ScrollMagic.Scene#setClassToggle
         * @example
         * // add the class 'myclass' to the element with the id 'my-elem' for the duration of the scene
         * scene.setClassToggle("#my-elem", "myclass");
         *
         * // add multiple classes to multiple elements defined by the selector '.classChange'
         * scene.setClassToggle(".classChange", "class1 class2 class3");
         *
         * @param {(string|object)} element - A Selector targeting one or more elements or a DOM object that is supposed to be modified.
         * @param {string} classes - One or more Classnames (separated by space) that should be added to the element during the scene.
         *
         * @returns {Scene} Parent object for chaining.
         */
        this.setClassToggle = function (element, classes) {
            var elems = _util.get.elements(element);
            if (elems.length === 0 || !_util.type.String(classes)) {
                log(1, "ERROR calling method 'setClassToggle()': Invalid " + (elems.length === 0 ? "element" : "classes") + " supplied.");
                return Scene;
            }
            if (_cssClassElems.length > 0) {
                // remove old ones
                Scene.removeClassToggle();
            }
            _cssClasses = classes;
            _cssClassElems = elems;
            Scene.on("enter.internal_class leave.internal_class", function (e) {
                var toggle = e.type === "enter" ? _util.addClass : _util.removeClass;
                _cssClassElems.forEach(function (elem, key) {
                    toggle(elem, _cssClasses);
                });
            });
            return Scene;
        };

        /**
         * Remove the class binding from the scene.
         * @method ScrollMagic.Scene#removeClassToggle
         * @example
         * // remove class binding from the scene without reset
         * scene.removeClassToggle();
         *
         * // remove class binding and remove the changes it caused
         * scene.removeClassToggle(true);
         *
         * @param {boolean} [reset=false] - If `false` and the classes are currently active, they will remain on the element. If `true` they will be removed.
         * @returns {Scene} Parent object for chaining.
         */
        this.removeClassToggle = function (reset) {
            if (reset) {
                _cssClassElems.forEach(function (elem, key) {
                    _util.removeClass(elem, _cssClasses);
                });
            }
            Scene.off("start.internal_class end.internal_class");
            _cssClasses = undefined;
            _cssClassElems = [];
            return Scene;
        };

        // INIT
        construct();
        return Scene;
    };

    // store pagewide scene options
    var SCENE_OPTIONS = {
        defaults: {
            duration: 0,
            offset: 0,
            triggerElement: undefined,
            triggerHook: 0.5,
            reverse: true,
            loglevel: 2
        },
        validate: {
            offset: function (val) {
                val = parseFloat(val);
                if (!_util.type.Number(val)) {
                    throw ["Invalid value for option \"offset\":", val];
                }
                return val;
            },
            triggerElement: function (val) {
                val = val || undefined;

                if (val) {
                    var elem = _util.get.elements(val)[0];
                    if (elem && elem.parentNode) {
                        val = elem;
                    } else {
                        throw ["Element defined in option \"triggerElement\" was not found:", val];
                    }
                }
                return val;
            },
            triggerHook: function (val) {
                var translate = {
                    "onCenter": 0.5,
                    "onEnter": 1,
                    "onLeave": 0
                };
                if (_util.type.Number(val)) {
                    val = Math.max(0, Math.min(parseFloat(val), 1)); //  make sure its betweeen 0 and 1
                } else if (val in translate) {
                    val = translate[val];
                } else {
                    throw ["Invalid value for option \"triggerHook\": ", val];
                }
                return val;
            },
            reverse: function (val) {
                return !!val; // force boolean
            },
            loglevel: function (val) {
                val = parseInt(val);
                if (!_util.type.Number(val) || val < 0 || val > 3) {
                    throw ["Invalid value for option \"loglevel\":", val];
                }
                return val;
            }
        }, // holder for  validation methods. duration validation is handled in 'getters-setters.js'
        shifts: ["duration", "offset", "triggerHook"], // list of options that trigger a `shift` event
    };
    /*
     * method used to add an option to ScrollMagic Scenes.
     * TODO: DOC (private for dev)
     */
    ScrollMagic.Scene.addOption = function (name, defaultValue, validationCallback, shifts) {
        if (!(name in SCENE_OPTIONS.defaults)) {
            SCENE_OPTIONS.defaults[name] = defaultValue;
            SCENE_OPTIONS.validate[name] = validationCallback;
            if (shifts) {
                SCENE_OPTIONS.shifts.push(name);
            }
        } else {
            ScrollMagic._util.log(1, "[static] ScrollMagic.Scene -> Cannot add Scene option '" + name + "', because it already exists.");
        }
    };
    // instance extension function for plugins
    // TODO: DOC (private for dev)
    ScrollMagic.Scene.extend = function (extension) {
        var oldClass = this;
        ScrollMagic.Scene = function () {
            oldClass.apply(this, arguments);
            this.$super = _util.extend({}, this); // copy parent state
            return extension.apply(this, arguments) || this;
        };
        _util.extend(ScrollMagic.Scene, oldClass); // copy properties
        ScrollMagic.Scene.prototype = oldClass.prototype; // copy prototype
        ScrollMagic.Scene.prototype.constructor = ScrollMagic.Scene; // restore constructor
    };



    /**
     * TODO: DOCS (private for dev)
     * @class
     * @private
     */

    ScrollMagic.Event = function (type, namespace, target, vars) {
        vars = vars || {};
        for (var key in vars) {
            this[key] = vars[key];
        }
        this.type = type;
        this.target = this.currentTarget = target;
        this.namespace = namespace || '';
        this.timeStamp = this.timestamp = Date.now();
        return this;
    };

    /*
     * TODO: DOCS (private for dev)
     */

    var _util = ScrollMagic._util = (function (window) {
        var U = {},
            i;

        /**
         * ------------------------------
         * internal helpers
         * ------------------------------
         */

        // parse float and fall back to 0.
        var floatval = function (number) {
            return parseFloat(number) || 0;
        };
        // get current style IE safe (otherwise IE would return calculated values for 'auto')
        var _getComputedStyle = function (elem) {
            return elem.currentStyle ? elem.currentStyle : window.getComputedStyle(elem);
        };

        // get element dimension (width or height)
        var _dimension = function (which, elem, outer, includeMargin) {

            elem = (elem === document) ? window : elem;
            if (elem === window) {
                includeMargin = false;
            } else if (!_type.DomElement(elem)) {
                return 0;
            }
            which = which.charAt(0).toUpperCase() + which.substr(1).toLowerCase();
            var dimension = (outer ? elem['offset' + which] || elem['outer' + which] : elem['client' + which] || elem['inner' + which]) || 0;
            if (outer && includeMargin) {
                var style = _getComputedStyle(elem);

                dimension += which === 'Height' ? floatval(style.marginTop) + floatval(style.marginBottom) : floatval(style.marginLeft) + floatval(style.marginRight);
            }

            return dimension;
        };
        // converts 'margin-top' into 'marginTop'
        var _camelCase = function (str) {
            return str.replace(/^[^a-z]+([a-z])/g, '$1').replace(/-([a-z])/g, function (g) {
                return g[1].toUpperCase();
            });
        };

        /**
         * ------------------------------
         * external helpers
         * ------------------------------
         */

        // extend obj â€“ same as jQuery.extend({}, objA, objB)
        U.extend = function (obj) {
            obj = obj || {};
            for (i = 1; i < arguments.length; i++) {
                if (!arguments[i]) {
                    continue;
                }
                for (var key in arguments[i]) {
                    if (arguments[i].hasOwnProperty(key)) {
                        obj[key] = arguments[i][key];
                    }
                }
            }
            return obj;
        };

        // check if a css display type results in margin-collapse or not
        U.isMarginCollapseType = function (str) {
            return ["block", "flex", "list-item", "table", "-webkit-box"].indexOf(str) > -1;
        };

        // implementation of requestAnimationFrame
        // based on https://gist.github.com/paulirish/1579671
        var
            lastTime = 0,
            vendors = ['ms', 'moz', 'webkit', 'o'];
        var _requestAnimationFrame = window.requestAnimationFrame;
        var _cancelAnimationFrame = window.cancelAnimationFrame;
        // try vendor prefixes if the above doesn't work
        for (i = 0; !_requestAnimationFrame && i < vendors.length; ++i) {
            _requestAnimationFrame = window[vendors[i] + 'RequestAnimationFrame'];
            _cancelAnimationFrame = window[vendors[i] + 'CancelAnimationFrame'] || window[vendors[i] + 'CancelRequestAnimationFrame'];
        }

        // fallbacks
        if (!_requestAnimationFrame) {
            _requestAnimationFrame = function (callback) {
                var
                    currTime = new Date().getTime(),
                    timeToCall = Math.max(0, 16 - (currTime - lastTime)),
                    id = window.setTimeout(function () {
                        callback(currTime + timeToCall);
                    }, timeToCall);
                lastTime = currTime + timeToCall;
                return id;
            };
        }
        if (!_cancelAnimationFrame) {
            _cancelAnimationFrame = function (id) {
                window.clearTimeout(id);
            };
        }
        U.rAF = _requestAnimationFrame.bind(window);
        U.cAF = _cancelAnimationFrame.bind(window);

        var
            loglevels = ["error", "warn", "log"],
            console = window.console || {};

        console.log = console.log || function () { }; // no console log, well - do nothing then...
        // make sure methods for all levels exist.
        for (i = 0; i < loglevels.length; i++) {
            var method = loglevels[i];
            if (!console[method]) {
                console[method] = console.log; // prefer .log over nothing
            }
        }
        U.log = function (loglevel) {
            if (loglevel > loglevels.length || loglevel <= 0) loglevel = loglevels.length;
            var now = new Date(),
                time = ("0" + now.getHours()).slice(-2) + ":" + ("0" + now.getMinutes()).slice(-2) + ":" + ("0" + now.getSeconds()).slice(-2) + ":" + ("00" + now.getMilliseconds()).slice(-3),
                method = loglevels[loglevel - 1],
                args = Array.prototype.splice.call(arguments, 1),
                func = Function.prototype.bind.call(console[method], console);
            args.unshift(time);
            func.apply(console, args);
        };

        /**
         * ------------------------------
         * type testing
         * ------------------------------
         */

        var _type = U.type = function (v) {
            return Object.prototype.toString.call(v).replace(/^\[object (.+)\]$/, "$1").toLowerCase();
        };
        _type.String = function (v) {
            return _type(v) === 'string';
        };
        _type.Function = function (v) {
            return _type(v) === 'function';
        };
        _type.Array = function (v) {
            return Array.isArray(v);
        };
        _type.Number = function (v) {
            return !_type.Array(v) && (v - parseFloat(v) + 1) >= 0;
        };
        _type.DomElement = function (o) {

            if ("DIV" === o.nodeName || "SECTION" === o.nodeName) {
                return true;
            }

            return (
                typeof HTMLElement === "object" || typeof HTMLElement === "function" ? o instanceof HTMLElement || o instanceof SVGElement : //DOM2
                    o && typeof o === "object" && o !== null && o.nodeType === 1 && typeof o.nodeName === "string"
            );
        };

        /**
         * ------------------------------
         * DOM Element info
         * ------------------------------
         */
        // always returns a list of matching DOM elements, from a selector, a DOM element or an list of elements or even an array of selectors
        var _get = U.get = {};
        _get.elements = function (selector) {

            var arr = [];
            if (_type.String(selector)) {
                try {
                    selector = document.querySelectorAll(selector);
                } catch (e) { // invalid selector
                    return arr;
                }
            }

            if (_type(selector) === 'nodelist' || _type.Array(selector) || selector instanceof NodeList) {

                for (var i = 0, ref = arr.length = selector.length; i < ref; i++) { // list of elements
                    var elem = selector[i];
                    // arr[i] = elem;
                    // if not an element, try to resolve recursively
                    arr[i] = _type.DomElement(elem) ? elem : _get.elements(elem);
                }
            } else if (_type.DomElement(selector) || selector === document || selector === window) {
                arr = [selector]; // only the element
            }

            return arr;
        };
        // get scroll top value
        _get.scrollTop = function (elem) {
            return (elem && typeof elem.scrollTop === 'number') ? elem.scrollTop : window.pageYOffset || 0;
        };
        // get scroll left value
        _get.scrollLeft = function (elem) {
            return (elem && typeof elem.scrollLeft === 'number') ? elem.scrollLeft : window.pageXOffset || 0;
        };
        // get element height
        _get.width = function (elem, outer, includeMargin) {
            return _dimension('width', elem, outer, includeMargin);
        };
        // get element width
        _get.height = function (elem, outer, includeMargin) {
            return _dimension('height', elem, outer, includeMargin);
        };

        // get element position (optionally relative to viewport)
        _get.offset = function (elem, relativeToViewport) {
            var offset = {
                top: 0,
                left: 0
            };
            if (elem && elem.getBoundingClientRect) { // check if available
                var rect = elem.getBoundingClientRect();
                offset.top = rect.top;
                offset.left = rect.left;
                if (!relativeToViewport) { // clientRect is by default relative to viewport...
                    offset.top += _get.scrollTop();
                    offset.left += _get.scrollLeft();
                }
            }
            return offset;
        };

        /**
         * ------------------------------
         * DOM Element manipulation
         * ------------------------------
         */

        U.addClass = function (elem, classname) {
            if (classname) {
                if (elem.classList)
                    elem.classList.add(classname);
                else
                    elem.className += ' ' + classname;
            }
        };
        U.removeClass = function (elem, classname) {
            if (classname) {
                if (elem.classList)
                    elem.classList.remove(classname);
                else
                    elem.className = elem.className.replace(new RegExp('(^|\\b)' + classname.split(' ').join('|') + '(\\b|$)', 'gi'), ' ');
            }
        };
        // if options is string -> returns css value
        // if options is array -> returns object with css value pairs
        // if options is object -> set new css values
        U.css = function (elem, options) {
            if (_type.String(options)) {
                return _getComputedStyle(elem)[_camelCase(options)];
            } else if (_type.Array(options)) {
                var
                    obj = {},
                    style = _getComputedStyle(elem);
                options.forEach(function (option, key) {
                    obj[option] = style[_camelCase(option)];
                });
                return obj;
            } else {
                for (var option in options) {
                    var val = options[option];
                    if (val == parseFloat(val)) { // assume pixel for seemingly numerical values
                        val += 'px';
                    }
                    elem.style[_camelCase(option)] = val;
                }
            }
        };

        return U;
    }(window || {}));


    ScrollMagic.Scene.prototype.addIndicators = function () {
        ScrollMagic._util.log(1, '(ScrollMagic.Scene) -> ERROR calling addIndicators() due to missing Plugin \'debug.addIndicators\'. Please make sure to include plugins/debug.addIndicators.js');
        return this;
    }
    ScrollMagic.Scene.prototype.removeIndicators = function () {
        ScrollMagic._util.log(1, '(ScrollMagic.Scene) -> ERROR calling removeIndicators() due to missing Plugin \'debug.addIndicators\'. Please make sure to include plugins/debug.addIndicators.js');
        return this;
    }
    ScrollMagic.Scene.prototype.setTween = function () {
        ScrollMagic._util.log(1, '(ScrollMagic.Scene) -> ERROR calling setTween() due to missing Plugin \'animation.gsap\'. Please make sure to include plugins/animation.gsap.js');
        return this;
    }
    ScrollMagic.Scene.prototype.removeTween = function () {
        ScrollMagic._util.log(1, '(ScrollMagic.Scene) -> ERROR calling removeTween() due to missing Plugin \'animation.gsap\'. Please make sure to include plugins/animation.gsap.js');
        return this;
    }
    ScrollMagic.Scene.prototype.setVelocity = function () {
        ScrollMagic._util.log(1, '(ScrollMagic.Scene) -> ERROR calling setVelocity() due to missing Plugin \'animation.velocity\'. Please make sure to include plugins/animation.velocity.js');
        return this;
    }
    ScrollMagic.Scene.prototype.removeVelocity = function () {
        ScrollMagic._util.log(1, '(ScrollMagic.Scene) -> ERROR calling removeVelocity() due to missing Plugin \'animation.velocity\'. Please make sure to include plugins/animation.velocity.js');
        return this;
    }

    return ScrollMagic;
}));


/*!
 * VERSION: 1.9.2
 * DATE: 2019-02-07
 * UPDATES AND DOCS AT: http://greensock.com
 *
 * @license Copyright (c) 2008-2019, GreenSock. All rights reserved.
 * This work is subject to the terms at http://greensock.com/standard-license or for
 * Club GreenSock members, the software agreement that was issued with your membership.
 *
 * @author: Jack Doyle, jack@greensock.com
 **/
var _gsScope = "undefined" != typeof module && module.exports && "undefined" != typeof global ? global : this || window;
(_gsScope._gsQueue || (_gsScope._gsQueue = [])).push(function () {
    "use strict";
    var a = (_gsScope.document || {}).documentElement,
        b = _gsScope,
        c = function (c, d) {
            var e = "x" === d ? "Width" : "Height",
                f = "scroll" + e,
                g = "client" + e,
                h = document.body;
            return c === b || c === a || c === h ? Math.max(a[f], h[f]) - (b["inner" + e] || a[g] || h[g]) : c[f] - c["offset" + e]
        },
        d = function (a) {
            return "string" == typeof a && (a = TweenLite.selector(a)), a.length && a !== b && a[0] && a[0].style && !a.nodeType && (a = a[0]), a === b || a.nodeType && a.style ? a : null
        },
        e = function (c, d) {
            var e = "scroll" + ("x" === d ? "Left" : "Top");
            return c === b && (null != c.pageXOffset ? e = "page" + d.toUpperCase() + "Offset" : c = null != a[e] ? a : document.body),
                function () {
                    return c[e]
                }
        },
        f = function (c, f) {
            var g = d(c).getBoundingClientRect(),
                h = document.body,
                i = !f || f === b || f === h,
                j = i ? {
                    top: a.clientTop - (window.pageYOffset || a.scrollTop || h.scrollTop || 0),
                    left: a.clientLeft - (window.pageXOffset || a.scrollLeft || h.scrollLeft || 0)
                } : f.getBoundingClientRect(),
                k = {
                    x: g.left - j.left,
                    y: g.top - j.top
                };
            return !i && f && (k.x += e(f, "x")(), k.y += e(f, "y")()), k
        },
        g = function (a, b, d, e) {
            var g = typeof a;
            return isNaN(a) ? "string" === g && "=" === a.charAt(1) ? parseInt(a.charAt(0) + "1", 10) * parseFloat(a.substr(2)) + e : "max" === a ? c(b, d) : Math.min(c(b, d), f(a, b)[d]) : parseFloat(a)
        },
        h = _gsScope._gsDefine.plugin({
            propName: "scrollTo",
            API: 2,
            global: !0,
            version: "1.9.2",
            init: function (a, c, d) {
                return this._wdw = a === b, this._target = a, this._tween = d, "object" != typeof c ? (c = {
                    y: c
                }, "string" == typeof c.y && "max" !== c.y && "=" !== c.y.charAt(1) && (c.x = c.y)) : c.nodeType && (c = {
                    y: c,
                    x: c
                }), this.vars = c, this._autoKill = c.autoKill !== !1, this.getX = e(a, "x"), this.getY = e(a, "y"), this.x = this.xPrev = this.getX(), this.y = this.yPrev = this.getY(), null != c.x ? (this._addTween(this, "x", this.x, g(c.x, a, "x", this.x) - (c.offsetX || 0), "scrollTo_x", !0), this._overwriteProps.push("scrollTo_x")) : this.skipX = !0, null != c.y ? (this._addTween(this, "y", this.y, g(c.y, a, "y", this.y) - (c.offsetY || 0), "scrollTo_y", !0), this._overwriteProps.push("scrollTo_y")) : this.skipY = !0, !0
            },
            set: function (a) {
                this._super.setRatio.call(this, a);
                var d = this._wdw || !this.skipX ? this.getX() : this.xPrev,
                    e = this._wdw || !this.skipY ? this.getY() : this.yPrev,
                    f = e - this.yPrev,
                    g = d - this.xPrev,
                    i = h.autoKillThreshold;
                this.x < 0 && (this.x = 0), this.y < 0 && (this.y = 0), this._autoKill && (!this.skipX && (g > i || -i > g) && d < c(this._target, "x") && (this.skipX = !0), !this.skipY && (f > i || -i > f) && e < c(this._target, "y") && (this.skipY = !0), this.skipX && this.skipY && (this._tween.kill(), this.vars.onAutoKill && this.vars.onAutoKill.apply(this.vars.onAutoKillScope || this._tween, this.vars.onAutoKillParams || []))), this._wdw ? b.scrollTo(this.skipX ? d : this.x, this.skipY ? e : this.y) : (this.skipY || (this._target.scrollTop = this.y), this.skipX || (this._target.scrollLeft = this.x)), this.xPrev = this.x, this.yPrev = this.y
            }
        }),
        i = h.prototype;
    h.max = c, h.getOffset = f, h.buildGetter = e, h.autoKillThreshold = 7, i._kill = function (a) {
        return a.scrollTo_x && (this.skipX = !0), a.scrollTo_y && (this.skipY = !0), this._super._kill.call(this, a)
    }
}), _gsScope._gsDefine && _gsScope._gsQueue.pop()(),
    function (a) {
        "use strict";
        var b = function () {
            return (_gsScope.GreenSockGlobals || _gsScope)[a]
        };
        "undefined" != typeof module && module.exports ? (require("../TweenLite.min.js"), module.exports = b()) : "function" == typeof define && define.amd && define(["TweenLite"], b)
    }("ScrollToPlugin");

/*! ScrollMagic v2.0.7 | (c) 2019 Jan Paepke (@janpaepke) | license & info: http://scrollmagic.io */
! function (e, n) {
    "function" == typeof define && define.amd ? define(["ScrollMagic", "TweenMax", "TimelineMax"], n) : "object" == typeof exports ? (require("gsap"), n(require("scrollmagic"), TweenMax, TimelineMax)) : n(e.ScrollMagic || e.jQuery && e.jQuery.ScrollMagic, e.TweenMax || e.TweenLite, e.TimelineMax || e.TimelineLite)
}(this, function (e, s, u) {
    "use strict";
    e.Scene.addOption("tweenChanges", !1, function (e) {
        return !!e
    }), e.Scene.extend(function () {
        var i, o = this;
        o.on("progress.plugin_gsap", function () {
            a()
        }), o.on("destroy.plugin_gsap", function (e) {
            o.removeTween(e.reset)
        });
        var a = function () {
            if (i) {
                var e = o.progress(),
                    n = o.state();
                i.repeat && -1 === i.repeat() ? "DURING" === n && i.paused() ? i.play() : "DURING" === n || i.paused() || i.pause() : e != i.progress() && (0 === o.duration() ? 0 < e ? i.play() : i.reverse() : o.tweenChanges() && i.tweenTo ? i.tweenTo(e * i.duration()) : i.progress(e).pause())
            }
        };
        o.setTween = function (e, n, r) {
            var t;
            1 < arguments.length && (arguments.length < 3 && (r = n, n = 1), e = s.to(e, n, r));
            try {
                (t = u ? new u({
                    smoothChildTiming: !0
                }).add(e) : e).pause()
            } catch (e) {
                return o
            }
            return i && o.removeTween(), i = t, e.repeat && -1 === e.repeat() && (i.repeat(-1), i.yoyo(e.yoyo())), a(), o
        }, o.removeTween = function (e) {
            return i && (e && i.progress(0).pause(), i.kill(), i = void 0), o
        }
    })
});