<?php
/**
 * Security Handler Class
 *
 * @package PrecisionQA
 * @since 0.0.1
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Security handler for the plugin
 *
 * @since 0.0.1
 */
class PrecisionQA_Security {

	/**
	 * Plugin instance
	 *
	 * @var PrecisionQA
	 */
	private $plugin;

	/**
	 * Constructor
	 *
	 * @param PrecisionQA $plugin Plugin instance.
	 */
	public function __construct( $plugin ) {
		$this->plugin = $plugin;
	}

	/**
	 * Verify nonce for form submissions
	 *
	 * @param string $nonce_field Nonce field name.
	 * @param string $action Nonce action.
	 * @return bool
	 */
	public function verify_nonce( $nonce_field, $action ) {
		if ( ! isset( $_POST[ $nonce_field ] ) ) {
			return false;
		}

		return wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST[ $nonce_field ] ) ), $action );
	}

	/**
	 * Check if user has required capability
	 *
	 * @param string $capability Capability to check.
	 * @return bool
	 */
	public function user_can( $capability = 'manage_options' ) {
		return current_user_can( $capability );
	}

	/**
	 * Sanitize and validate form data
	 *
	 * @param array $data Form data to sanitize.
	 * @param array $fields Fields configuration.
	 * @return array
	 */
	public function sanitize_form_data( $data, $fields = array() ) {
		$sanitized = array();

		foreach ( $data as $key => $value ) {
			if ( isset( $fields[ $key ] ) ) {
				$field_type = $fields[ $key ];
				$sanitized[ $key ] = $this->sanitize_field( $value, $field_type );
			} else {
				// Default sanitization
				$sanitized[ $key ] = $this->sanitize_field( $value, 'text' );
			}
		}

		return $sanitized;
	}

	/**
	 * Sanitize individual field based on type
	 *
	 * @param mixed  $value Field value.
	 * @param string $type Field type.
	 * @return mixed
	 */
	private function sanitize_field( $value, $type ) {
		switch ( $type ) {
			case 'email':
				return sanitize_email( $value );
			case 'url':
				return esc_url_raw( $value );
			case 'int':
				return intval( $value );
			case 'float':
				return floatval( $value );
			case 'textarea':
				return sanitize_textarea_field( $value );
			case 'select':
			case 'radio':
				return sanitize_text_field( $value );
			case 'checkbox':
				return (bool) $value;
			case 'array':
				return is_array( $value ) ? array_map( 'sanitize_text_field', $value ) : array();
			default:
				return sanitize_text_field( $value );
		}
	}

	/**
	 * Escape output for display
	 *
	 * @param string $text Text to escape.
	 * @param string $context Context for escaping.
	 * @return string
	 */
	public function escape_output( $text, $context = 'html' ) {
		switch ( $context ) {
			case 'html':
				return esc_html( $text );
			case 'attr':
				return esc_attr( $text );
			case 'url':
				return esc_url( $text );
			case 'js':
				return esc_js( $text );
			default:
				return esc_html( $text );
		}
	}

	/**
	 * Validate and sanitize post ID
	 *
	 * @param mixed $post_id Post ID to validate.
	 * @return int|false
	 */
	public function validate_post_id( $post_id ) {
		$post_id = intval( $post_id );
		return get_post( $post_id ) ? $post_id : false;
	}

	/**
	 * Validate and sanitize form ID
	 *
	 * @param mixed $form_id Form ID to validate.
	 * @return int|false
	 */
	public function validate_form_id( $form_id ) {
		if ( ! $this->plugin->is_gravity_forms_active() ) {
			return false;
		}

		$form_id = intval( $form_id );
		$form = GFAPI::get_form( $form_id );
		return $form ? $form_id : false;
	}

	/**
	 * Log security events
	 *
	 * @param string $message Log message.
	 * @param string $level Log level.
	 */
	public function log_security_event( $message, $level = 'info' ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( sprintf( '[GF CPT Manager Security] %s: %s', strtoupper( $level ), $message ) );
		}
	}
}
