<?php
/**
 * Gravity Forms Handler Class
 *
 * @package PrecisionQA
 * @since 0.0.1
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Gravity Forms handler for the plugin
 *
 * @since 0.0.1
 */
class PrecisionQA_Gravity_Forms {

	/**
	 * Plugin instance
	 *
	 * @var PrecisionQA
	 */
	private $plugin;

	/**
	 * Constructor
	 *
	 * @param PrecisionQA $plugin Plugin instance.
	 */
	public function __construct( $plugin ) {
		$this->plugin = $plugin;
	}

	/**
	 * Generate form based on type
	 *
	 * @param string $form_type Type of form to generate.
	 * @return int|WP_Error Form ID on success, WP_Error on failure.
	 */
	public function generate_form( $form_type ) {
		if ( ! $this->plugin->is_gravity_forms_active() ) {
			return new WP_Error( 'gravity_forms_inactive', __( 'Gravity Forms is not installed or activated.', 'precisionqa' ) );
		}

		switch ( $form_type ) {
			case 'standard':
				return $this->generate_standard_form();
			case 'advanced':
				return $this->generate_advanced_form();
			case 'post_fields':
				return $this->generate_post_fields_form();
			case 'pricing':
				return $this->generate_pricing_form();
			case 'multipage':
				return $this->generate_multipage_form();
			case 'conditional':
				return $this->generate_conditional_form();
			case 'conditional_dropdown':
				return $this->generate_conditional_dropdown_form();
			case 'conditional_textfield':
				return $this->generate_conditional_text_field_form();
			case 'conditional_paragraph':
				return $this->generate_conditional_paragraph_form();
			case 'conditional_checkbox':
				return $this->generate_conditional_checkbox_form();
			case 'conditional_radio':
				return $this->generate_conditional_radio_form();
			case 'conditional_multiplechoice':
				return $this->generate_conditional_multiplechoice_form();
			default:
				return new WP_Error( 'invalid_form_type', __( 'Invalid form type specified.', 'precisionqa' ) );
		}
	}

	/**
	 * Generate a standard Gravity Form
	 *
	 * @return int|WP_Error Form ID on success, WP_Error on failure.
	 */
	private function generate_standard_form() {
		$rand = wp_rand( 1000, 9999 );
		$form = array(
			'title'       => sprintf( 
				/* translators: %d: random number for uniqueness. */
				__( 'Standard Fields Form %d', 'precisionqa' ), 
				$rand 
			),
			'description' => __( 'A sample form with standard fields.', 'precisionqa' ),
			'fields'      => array(
				array(
					'type'        => 'text',
					'label'       => __( 'Name', 'precisionqa' ),
					'id'          => 1,
					'isRequired'  => true,
					'adminLabel'  => 'name',
				),
				array(
					'type'        => 'email',
					'label'       => __( 'Email', 'precisionqa' ),
					'id'          => 2,
					'isRequired'  => true,
					'adminLabel'  => 'email',
				),
				array(
					'type'        => 'phone',
					'label'       => __( 'Phone', 'precisionqa' ),
					'id'          => 3,
					'adminLabel'  => 'phone',
				),
				array(
					'type'        => 'textarea',
					'label'       => __( 'Message', 'precisionqa' ),
					'id'          => 4,
					'adminLabel'  => 'message',
				),
			),
		);

		return $this->create_form( $form );
	}

	/**
	 * Generate conditional multiple choice form
	 *
	 * @return int|WP_Error Form ID on success, WP_Error on failure.
	 */
	private function generate_conditional_multiplechoice_form() {
		// Implementation similar to conditional_text_field_form but with multiple choice-specific logic
		return $this->generate_conditional_text_field_form();
	}

	/**
	 * Generate advanced form with comprehensive fields
	 *
	 * @return int|WP_Error Form ID on success, WP_Error on failure.
	 */
	private function generate_advanced_form() {
		$rand = wp_rand( 1000, 9999 );
		$form = array(
			'title'       => sprintf( 
				/* translators: %d: random number for uniqueness. */
				__( 'Advanced Fields Form %d', 'precisionqa' ), 
				$rand 
			),
			'description' => __( 'A comprehensive form with advanced fields for testing.', 'precisionqa' ),
			'fields'      => array(
				array(
					'type'        => 'text',
					'label'       => __( 'First Name', 'precisionqa' ),
					'id'          => 1,
					'isRequired'  => true,
					'adminLabel'  => 'first_name',
				),
				array(
					'type'        => 'text',
					'label'       => __( 'Last Name', 'precisionqa' ),
					'id'          => 2,
					'isRequired'  => true,
					'adminLabel'  => 'last_name',
				),
				array(
					'type'        => 'email',
					'label'       => __( 'Email Address', 'precisionqa' ),
					'id'          => 3,
					'isRequired'  => true,
					'adminLabel'  => 'email',
				),
				array(
					'type'        => 'phone',
					'label'       => __( 'Phone Number', 'precisionqa' ),
					'id'          => 4,
					'adminLabel'  => 'phone',
				),
				array(
					'type'        => 'address',
					'label'       => __( 'Address', 'precisionqa' ),
					'id'          => 5,
					'adminLabel'  => 'address',
				),
				array(
					'type'        => 'date',
					'label'       => __( 'Date of Birth', 'precisionqa' ),
					'id'          => 6,
					'adminLabel'  => 'dob',
				),
				array(
					'type'        => 'fileupload',
					'label'       => __( 'Profile Picture', 'precisionqa' ),
					'id'          => 7,
					'adminLabel'  => 'profile_picture',
				),
			),
		);

		return $this->create_form( $form );
	}

	/**
	 * Generate post fields form for content creation
	 *
	 * @return int|WP_Error Form ID on success, WP_Error on failure.
	 */
	private function generate_post_fields_form() {
		$rand = wp_rand( 1000, 9999 );
		$form = array(
			'title'       => sprintf( 
				/* translators: %d: random number for uniqueness. */
				__( 'Post Fields Form %d', 'precisionqa' ), 
				$rand 
			),
			'description' => __( 'A form designed for post creation with title, content, and category fields.', 'precisionqa' ),
			'fields'      => array(
				array(
					'type'        => 'text',
					'label'       => __( 'Post Title', 'precisionqa' ),
					'id'          => 1,
					'isRequired'  => true,
					'adminLabel'  => 'post_title',
				),
				array(
					'type'        => 'textarea',
					'label'       => __( 'Post Content', 'precisionqa' ),
					'id'          => 2,
					'isRequired'  => true,
					'adminLabel'  => 'post_content',
				),
				array(
					'type'        => 'text',
					'label'       => __( 'Post Excerpt', 'precisionqa' ),
					'id'          => 3,
					'adminLabel'  => 'post_excerpt',
				),
				array(
					'type'        => 'select',
					'label'       => __( 'Post Category', 'precisionqa' ),
					'id'          => 4,
					'choices'     => $this->get_category_choices(),
					'adminLabel'  => 'post_category',
				),
			),
		);

		return $this->create_form( $form );
	}

	/**
	 * Generate pricing form with calculations
	 *
	 * @return int|WP_Error Form ID on success, WP_Error on failure.
	 */
	private function generate_pricing_form() {
		$rand = wp_rand( 1000, 9999 );
		$form = array(
			'title'       => sprintf( 
				/* translators: %d: random number for uniqueness. */
				__( 'Pricing Fields Form %d', 'precisionqa' ), 
				$rand 
			),
			'description' => __( 'A form with pricing and product fields including calculations.', 'precisionqa' ),
			'fields'      => array(
				array(
					'type'        => 'text',
					'label'       => __( 'Product Name', 'precisionqa' ),
					'id'          => 1,
					'isRequired'  => true,
					'adminLabel'  => 'product_name',
				),
				array(
					'type'        => 'number',
					'label'       => __( 'Quantity', 'precisionqa' ),
					'id'          => 2,
					'isRequired'  => true,
					'min'         => 1,
					'adminLabel'  => 'quantity',
				),
				array(
					'type'        => 'number',
					'label'       => __( 'Unit Price', 'precisionqa' ),
					'id'          => 3,
					'isRequired'  => true,
					'min'         => 0,
					'step'        => 0.01,
					'adminLabel'  => 'unit_price',
				),
				array(
					'type'        => 'calculation',
					'label'       => __( 'Total Price', 'precisionqa' ),
					'id'          => 4,
					'formula'     => '{2} * {3}',
					'adminLabel'  => 'total_price',
				),
			),
		);

		return $this->create_form( $form );
	}

	/**
	 * Generate multipage form with pagination
	 *
	 * @return int|WP_Error Form ID on success, WP_Error on failure.
	 */
	private function generate_multipage_form() {
		$rand = wp_rand( 1000, 9999 );
		$form = array(
			'title'       => sprintf( 
				/* translators: %d: random number for uniqueness. */
				__( 'Multipage Form %d', 'precisionqa' ), 
				$rand 
			),
			'description' => __( 'A multipage form with pagination for testing multi-step workflows.', 'precisionqa' ),
			'fields'      => array(
				// Page 1
				array(
					'type'        => 'page',
					'label'       => __( 'Page 1', 'precisionqa' ),
					'id'          => 1,
				),
				array(
					'type'        => 'text',
					'label'       => __( 'First Name', 'precisionqa' ),
					'id'          => 2,
					'pageNumber'  => 1,
					'isRequired'  => true,
					'adminLabel'  => 'first_name',
				),
				array(
					'type'        => 'text',
					'label'       => __( 'Last Name', 'precisionqa' ),
					'id'          => 3,
					'pageNumber'  => 1,
					'isRequired'  => true,
					'adminLabel'  => 'last_name',
				),
				// Page 2
				array(
					'type'        => 'page',
					'label'       => __( 'Page 2', 'precisionqa' ),
					'id'          => 4,
				),
				array(
					'type'        => 'email',
					'label'       => __( 'Email Address', 'precisionqa' ),
					'id'          => 5,
					'pageNumber'  => 2,
					'isRequired'  => true,
					'adminLabel'  => 'email',
				),
				array(
					'type'        => 'textarea',
					'label'       => __( 'Comments', 'precisionqa' ),
					'id'          => 6,
					'pageNumber'  => 2,
					'adminLabel'  => 'comments',
				),
			),
		);

		return $this->create_form( $form );
	}

	/**
	 * Generate conditional form with basic logic
	 *
	 * @return int|WP_Error Form ID on success, WP_Error on failure.
	 */
	private function generate_conditional_form() {
		$rand = wp_rand( 1000, 9999 );
		$form = array(
			'title'       => sprintf( 
				/* translators: %d: random number for uniqueness. */
				__( 'Conditional Form %d', 'precisionqa' ), 
				$rand 
			),
			'description' => __( 'A form with conditional logic for testing dynamic field behavior.', 'precisionqa' ),
			'fields'      => array(
				array(
					'type'        => 'select',
					'label'       => __( 'Select Option', 'precisionqa' ),
					'id'          => 1,
					'choices'     => array(
						array( 'text' => __( 'Option A', 'precisionqa' ), 'value' => 'option_a' ),
						array( 'text' => __( 'Option B', 'precisionqa' ), 'value' => 'option_b' ),
					),
					'isRequired'  => true,
					'adminLabel'  => 'option_selector',
				),
				array(
					'type'             => 'text',
					'label'            => __( 'Conditional Field A', 'precisionqa' ),
					'id'               => 2,
					'conditionalLogic' => array(
						'actionType' => 'show',
						'logicType'  => 'all',
						'rules'      => array(
							array(
								'fieldId' => 1,
								'operator' => 'is',
								'value'    => 'option_a',
							),
						),
					),
					'adminLabel'       => 'conditional_field_a',
				),
				array(
					'type'             => 'text',
					'label'            => __( 'Conditional Field B', 'precisionqa' ),
					'id'               => 3,
					'conditionalLogic' => array(
						'actionType' => 'show',
						'logicType'  => 'all',
						'rules'      => array(
							array(
								'fieldId' => 1,
								'operator' => 'is',
								'value'    => 'option_b',
							),
						),
					),
					'adminLabel'       => 'conditional_field_b',
				),
			),
		);

		return $this->create_form( $form );
	}

	/**
	 * Generate conditional dropdown form with rtCamp specific logic
	 *
	 * @return int|WP_Error Form ID on success, WP_Error on failure.
	 */
	private function generate_conditional_dropdown_form() {
		$rand = wp_rand( 1000, 9999 );
		$form = array(
			'title'       => sprintf( 
				/* translators: %d: random number for uniqueness. */
				__( 'Conditional Dropdown Form %d', 'precisionqa' ), 
				$rand 
			),
			'description' => __( 'A form with standard fields and conditional logic based on dropdown value.', 'precisionqa' ),
			'fields'      => array(
				// Dropdown Field (Reference Field for Conditional Logic)
				array(
					'type'       => 'select',
					'label'      => __( 'Choose an Option', 'precisionqa' ),
					'id'         => 1,
					'choices'    => array(
						array( 'text' => 'rtCamp', 'value' => 'rtCamp' ),
						array( 'text' => 'Other', 'value' => 'Other' ),
					),
					'isRequired' => true,
					'adminLabel' => 'option_selector',
				),
				// Single Line Text
				array(
					'type'             => 'text',
					'label'            => __( 'Single Line Text', 'precisionqa' ),
					'id'               => 2,
					'conditionalLogic' => array(
						'actionType' => 'show',
						'logicType'  => 'all',
						'rules'      => array(
							array(
								'fieldId' => 1,
								'operator' => 'is',
								'value'    => 'rtCamp',
							),
						),
					),
					'adminLabel'       => 'single_line_text',
				),
				// Paragraph Text
				array(
					'type'             => 'textarea',
					'label'            => __( 'Paragraph Text', 'precisionqa' ),
					'id'               => 3,
					'conditionalLogic' => array(
						'actionType' => 'show',
						'logicType'  => 'all',
						'rules'      => array(
							array(
								'fieldId' => 1,
								'operator' => 'is',
								'value'    => 'rtCamp',
							),
						),
					),
					'adminLabel'       => 'paragraph_text',
				),
				// Number Field
				array(
					'type'             => 'number',
					'label'            => __( 'Number Field', 'precisionqa' ),
					'id'               => 4,
					'conditionalLogic' => array(
						'actionType' => 'show',
						'logicType'  => 'all',
						'rules'      => array(
							array(
								'fieldId' => 1,
								'operator' => 'is',
								'value'    => 'rtCamp',
							),
						),
					),
					'adminLabel'       => 'number_field',
				),
				// Checkbox Field
				array(
					'type'             => 'checkbox',
					'label'            => __( 'Checkbox Field', 'precisionqa' ),
					'id'               => 5,
					'choices'          => array( array( 'text' => __( 'Option 1', 'precisionqa' ), 'value' => 'Option 1' ) ),
					'conditionalLogic' => array(
						'actionType' => 'show',
						'logicType'  => 'all',
						'rules'      => array(
							array(
								'fieldId' => 1,
								'operator' => 'is',
								'value'    => 'rtCamp',
							),
						),
					),
					'adminLabel'       => 'checkbox_field',
				),
				// Radio Button Field
				array(
					'type'             => 'radio',
					'label'            => __( 'Radio Button Field', 'precisionqa' ),
					'id'               => 6,
					'choices'          => array( array( 'text' => __( 'Option 1', 'precisionqa' ), 'value' => 'Option 1' ) ),
					'conditionalLogic' => array(
						'actionType' => 'show',
						'logicType'  => 'all',
						'rules'      => array(
							array(
								'fieldId' => 1,
								'operator' => 'is',
								'value'    => 'rtCamp',
							),
						),
					),
					'adminLabel'       => 'radio_button_field',
				),
				// Multiple Choice Field
				array(
					'type'             => 'select',
					'label'            => __( 'Multiple Choice Field', 'precisionqa' ),
					'id'               => 7,
					'choices'          => array( array( 'text' => __( 'Choice 1', 'precisionqa' ), 'value' => 'Choice 1' ) ),
					'conditionalLogic' => array(
						'actionType' => 'show',
						'logicType'  => 'all',
						'rules'      => array(
							array(
								'fieldId' => 1,
								'operator' => 'is',
								'value'    => 'rtCamp',
							),
						),
					),
					'adminLabel'       => 'multiple_choice_field',
				),
			),
		);

		return $this->create_form( $form );
	}

	/**
	 * Generate conditional text field form with various operators
	 *
	 * @return int|WP_Error Form ID on success, WP_Error on failure.
	 */
	private function generate_conditional_text_field_form() {
		$rand = wp_rand( 1000, 9999 );
		$form = array(
			'title'       => sprintf( 
				/* translators: %d: random number for uniqueness. */
				__( 'Conditional Text Field Form %d', 'precisionqa' ), 
				$rand 
			),
			'description' => __( 'A form with standard fields and conditional logic based on a text field.', 'precisionqa' ),
			'fields'      => array(
				// Text field with condition logic
				array(
					'type'       => 'text',
					'label'      => __( 'Text Field (Enter value)', 'precisionqa' ),
					'id'         => 1,
					'isRequired' => true,
					'adminLabel' => 'text_field',
				),
				// Single Line Text with conditional logic
				array(
					'type'        => 'text',
					'label'       => __( 'Single Line Text', 'precisionqa' ),
					'id'          => 2,
					'isRequired'  => true,
					'conditionalLogic' => array(
						'actionType' => 'show',
						'rules'      => array(
							array(
								'fieldId'  => 1,
								'operator' => 'endsWith',
								'value'    => 'opensource',
							),
						),
					),
					'adminLabel'  => 'conditional_single_line',
				),
				// Paragraph Text with conditional logic
				array(
					'type'        => 'textarea',
					'label'       => __( 'Paragraph Text', 'precisionqa' ),
					'id'          => 3,
					'isRequired'  => true,
					'conditionalLogic' => array(
						'actionType' => 'show',
						'rules'      => array(
							array(
								'fieldId'  => 1,
								'operator' => 'is',
								'value'    => 'rtCamp',
							),
						),
					),
					'adminLabel'  => 'conditional_paragraph',
				),
				// Dropdown with conditional logic
				array(
					'type'        => 'select',
					'label'       => __( 'Dropdown', 'precisionqa' ),
					'id'          => 4,
					'choices'     => array(
						array( 'text' => 'rtCamp', 'value' => 'rtCamp' ),
						array( 'text' => 'WordCamp', 'value' => 'wordCamp' ),
						array( 'text' => 'Open Source', 'value' => 'openSource' ),
						array( 'text' => 'Obo', 'value' => 'obo' ),
					),
					'isRequired'  => true,
					'conditionalLogic' => array(
						'actionType' => 'show',
						'rules'      => array(
							array(
								'fieldId'  => 1,
								'operator' => 'is',
								'value'    => 'rtCamp',
							),
						),
					),
					'adminLabel'  => 'conditional_dropdown',
				),
				// Number field with conditional logic
				array(
					'type'        => 'number',
					'label'       => __( 'Number Field', 'precisionqa' ),
					'id'          => 5,
					'isRequired'  => true,
					'conditionalLogic' => array(
						'actionType' => 'show',
						'rules'      => array(
							array(
								'fieldId'  => 1,
								'operator' => 'startsWith',
								'value'    => 'wordcamp',
							),
						),
					),
					'adminLabel'  => 'conditional_number',
				),
				// Checkbox with conditional logic
				array(
					'type'        => 'checkbox',
					'label'       => __( 'Checkbox Field', 'precisionqa' ),
					'id'          => 6,
					'choices'     => array(
						array( 'text' => __( 'Option 1', 'precisionqa' ), 'value' => 'option1' ),
						array( 'text' => __( 'Option 2', 'precisionqa' ), 'value' => 'option2' ),
					),
					'isRequired'  => true,
					'conditionalLogic' => array(
						'actionType' => 'show',
						'rules'      => array(
							array(
								'fieldId'  => 1,
								'operator' => 'endsWith',
								'value'    => 'opensource',
							),
						),
					),
					'adminLabel'  => 'conditional_checkbox',
				),
				// Radio button with conditional logic
				array(
					'type'        => 'radio',
					'label'       => __( 'Radio Button Field', 'precisionqa' ),
					'id'          => 7,
					'choices'     => array(
						array( 'text' => __( 'Option 1', 'precisionqa' ), 'value' => 'option1' ),
						array( 'text' => __( 'Option 2', 'precisionqa' ), 'value' => 'option2' ),
					),
					'isRequired'  => true,
					'conditionalLogic' => array(
						'actionType' => 'show',
						'rules'      => array(
							array(
								'fieldId'  => 1,
								'operator' => 'contains',
								'value'    => 'obo',
							),
						),
					),
					'adminLabel'  => 'conditional_radio',
				),
				// Multiple choice with conditional logic
				array(
					'type'        => 'checkbox',
					'label'       => __( 'Multiple Choice', 'precisionqa' ),
					'id'          => 8,
					'choices'     => array(
						array( 'text' => __( 'Option 1', 'precisionqa' ), 'value' => 'option1' ),
						array( 'text' => __( 'Option 2', 'precisionqa' ), 'value' => 'option2' ),
					),
					'isRequired'  => true,
					'conditionalLogic' => array(
						'actionType' => 'show',
						'rules'      => array(
							array(
								'fieldId'  => 1,
								'operator' => 'is',
								'value'    => 'rtCamp',
							),
						),
					),
					'adminLabel'  => 'conditional_multiple_choice',
				),
			),
		);

		return $this->create_form( $form );
	}

	/**
	 * Generate conditional paragraph form
	 *
	 * @return int|WP_Error Form ID on success, WP_Error on failure.
	 */
	private function generate_conditional_paragraph_form() {
		// Implementation similar to conditional_text_field_form but with paragraph-specific logic
		return $this->generate_conditional_text_field_form();
	}

	/**
	 * Generate conditional checkbox form
	 *
	 * @return int|WP_Error Form ID on success, WP_Error on failure.
	 */
	private function generate_conditional_checkbox_form() {
		// Implementation similar to conditional_text_field_form but with checkbox-specific logic
		return $this->generate_conditional_text_field_form();
	}

	/**
	 * Generate conditional radio form
	 *
	 * @return int|WP_Error Form ID on success, WP_Error on failure.
	 */
	private function generate_conditional_radio_form() {
		// Implementation similar to conditional_text_field_form but with radio-specific logic
		return $this->generate_conditional_text_field_form();
	}

	/**
	 * Create the form using Gravity Forms API
	 *
	 * @param array $form_data Form data array.
	 * @return int|WP_Error Form ID on success, WP_Error on failure.
	 */
	private function create_form( $form_data ) {
		try {
			// Add default form structure if not provided
			if ( ! isset( $form_data['button'] ) ) {
				$form_data['button'] = array(
					'type' => 'text',
					'text' => __( 'Submit', 'precisionqa' ),
				);
			}

			// Add default confirmation if not provided
			if ( ! isset( $form_data['confirmations'] ) ) {
				$form_data['confirmations'] = array(
					'default_confirmation' => array(
						'id' => 'default_confirmation',
						'name' => __( 'Default Confirmation', 'precisionqa' ),
						'isDefault' => true,
						'type' => 'message',
						'message' => __( 'Thanks for contacting us!', 'precisionqa' ),
						'disableAutoformat' => false,
					),
				);
			}

			$form_id = GFAPI::add_form( $form_data );
			
			if ( is_wp_error( $form_id ) ) {
				$this->plugin->security->log_security_event( 'Form creation failed: ' . $form_id->get_error_message(), 'error' );
				return $form_id;
			}

			$this->plugin->security->log_security_event( 'Form created successfully with ID: ' . $form_id, 'info' );
			return $form_id;

		} catch ( Exception $e ) {
			$this->plugin->security->log_security_event( 'Form creation exception: ' . $e->getMessage(), 'error' );
			return new WP_Error( 'form_creation_failed', $e->getMessage() );
		}
	}

	/**
	 * Get category choices for post fields form
	 *
	 * @return array Category choices array.
	 */
	private function get_category_choices() {
		$categories = get_categories( array( 'hide_empty' => false ) );
		$choices    = array();

		foreach ( $categories as $category ) {
			$choices[] = array(
				'text'  => $category->name,
				'value' => $category->term_id,
			);
		}

		return $choices;
	}
}
